/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "convert.h"
#include "volume.h"
#include "logging.h"

/**
 *	convert_volume_button_activated - callback to convert a compatibility volume to an EVMS volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Convert button in the Convert compatibility
 *	volume dialog window is activated.
 */
int convert_volume_button_activated(struct menu_item *item)
{
	int rc;
	char *name;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	name = dialog->user_data;
	rc = evms_convert_to_evms_volume(get_selected_handle(selwin->clist), name);
	update_status(_("Convert"), rc);

	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	filter_convert_volume - keep compatibility volume we can convert to EVMS volume
 *	@handle: the volume handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given compatibility volume can be converted to an EVMS volume.
 */
int filter_convert_volume(object_handle_t handle, void *user_data)
{
	return evms_can_convert_to_evms_volume(handle);
}

/**
 *	convert_evms_volume_button_activated - callback to convert an EVMS volume to a compatibility volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Convert button in the Convert EVMS volume
 *	dialog window is activated.
 */
int convert_evms_volume_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;

	rc = evms_convert_to_compatibility_volume(get_selected_handle(selwin->clist));
	update_status(_("Convert"), rc);

	if (rc == 0) {
		((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	filter_convert_evms_volume - keep EVMS volumes that we can convert to compatibility volumes
 *	@handle: the volume handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can be use to create a native EVMS volume.
 */
int filter_convert_evms_volume(object_handle_t handle, void *user_data)
{
	return evms_can_convert_to_compatibility_volume(handle);
}

/**
 *	on_convert_volume_dialog_delete - callback invoked when dialog used to convert a volume is deleted
 *	@dialog: the dialog window
 *
 *	This routine handles cleanup for the dialog used to convert compat volumes.
 *	We free any string that may be associated with the dialog containing 
 *	a volume name.
 */
void on_convert_volume_dialog_delete(struct dialog_window *dialog)
{
	g_free(dialog->user_data);
}

/**
 *	create_convert_volume_dialog - create a dialog to handle volume conversions
 *	@handle: volume handle; non-zero when convert initiated from context popup menu
 *	@activate_cb: the function to invoke when the Convert button is activated
 *	@filter_func: the function to filter the volumes for conversion
 *	@select_cb: the function to invoke when a volume in the clist is selected
 *
 *	This routine creates a dialog window for use by the routines that allow the
 *	conversion of EVMS and compatibility type volumes.
 */
struct dialog_window *create_convert_volume_dialog(object_handle_t handle,
						menuitem_activate_cb activate_cb,
						clist_filter_func filter_func,
						clist_select_item_cb select_cb)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(select_cb ? _("Convert Compatibility Volume") : _("Convert EVMS Volume"),
					NULL, NULL,
					_("_Convert"),
					activate_cb,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_volumes, filter_func,
				format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);

	if (select_cb != NULL) {
		set_clist_select_item_cb(selwin->clist, select_cb);
		set_dialog_delete_cb(dialog, (dialog_delete_cb)on_convert_volume_dialog_delete);
	}

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	return dialog;
}

/**
 *	show_convert_evms_volume_dialog - display dialog to convert an EVMS volume
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to convert an EVMS
 *	volume to a compatibility volume.
 **/
int show_convert_evms_volume_dialog(object_handle_t handle)
{
	struct dialog_window *dialog;

	dialog = create_convert_volume_dialog(handle,
					(menuitem_activate_cb)convert_evms_volume_button_activated,
					filter_convert_evms_volume, NULL);
	process_modal_dialog(dialog);
	return 0;
}

/**
 *	volume_to_convert_selected - callback activated compatibility volume selected
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked when a volume in the clist is selected for conversion
 *	to an EVMS volume. We prompt for a volume name and store the name in the
 *	dialog->user_data replacing any previous name.
 */
int volume_to_convert_selected(struct clist *clist, struct clist_item *item)
{
	char *prev_name;
	struct dialog_window *dialog = clist->user_data;

	prev_name = dialog->user_data;
	dialog->user_data = get_volume_name(prev_name);
	if (dialog->user_data != NULL) {
		row_selected(clist, item);
		g_free(prev_name);
	}

	return (dialog->user_data != NULL ? 0 : ENODATA);
}

/**
 *	show_convert_volume_dialog - display dialog to convert a compatibility volume
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to convert a
 *	compatibility volume to a native EVMS volume.
 **/
int show_convert_volume_dialog(object_handle_t handle)
{
	struct dialog_window *dialog;

	dialog = create_convert_volume_dialog(handle,
					(menuitem_activate_cb)convert_volume_button_activated,
					filter_convert_volume,
					(clist_select_item_cb)volume_to_convert_selected);
	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_convert_vol_menuitem_activated - convert a given volume to an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to convert the selected compatibility volume to a
 *	native EVMS volume.
 */
int context_convert_vol_menuitem_activated(struct menu_item *item)
{
	return show_convert_volume_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_convert_vol_menuitem_activated - convert a compatibility volume to an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the "Convert->Compatibility Volume to EVMS Volume" Actions
 *	pulldown menu item.
 */
int actions_convert_vol_menuitem_activated(struct menu_item *item)
{
	return show_convert_volume_dialog(0);
}

/**
 *	context_convert_evms_vol_menuitem_activated - convert given volume to compatibility volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to convert the supplied EVMS volume to a compatibility
 *	volume.
 */
int context_convert_evms_vol_menuitem_activated(struct menu_item *item)
{
	return show_convert_evms_volume_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_convert_evms_vol_menuitem_activated - convert an EVMS volume to a compatibility volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the "Convert->EVMS Volume to Compatibility Volume" Actions
 *	pulldown menu item.
 */
int actions_convert_evms_vol_menuitem_activated(struct menu_item *item)
{
	return show_convert_evms_volume_dialog(0);
}
