/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: fs_jfs.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/wait.h>
#include "fsimjfs.h"

plugin_record_t * my_plugin_record = &jfs_plugin_record;
engine_functions_t *EngFncs = NULL;
char jfsutils_version[10];
u_int32_t jfsutils_support;

struct ExternalLogRecord {
	struct ExternalLogRecord * NextRecord;
	struct ExternalLogRecord * PrevRecord;
	logical_volume_t * fs_volume;
	logical_volume_t * log_volume;
};

typedef struct ExternalLogRecord ExtLogRec;
ExtLogRec * ExtLog_FirstRec;

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/


static int fs_setup( engine_functions_t *engine_function_table)
{
	int rc = 0;
	EngFncs = engine_function_table;

	LOG_ENTRY();

	jfsutils_support = 0;

	/*
	 * jfsutils must be at version 1.0.9 or greater to work properly with this FSIM.
	 * Also, initialize jfsutils_version and jfsutils_support.
	 */
	rc = fsim_test_version();

	if ( rc ) {
		LOG_WARNING( "jfsutils must be version 1.0.9 or later to function properly with this FSIM.\n" );
		LOG_WARNING( "jfsutils must also reside in the search path specified by PATH.\n" );
		LOG_WARNING( "Please get/install the current version of jfsutils from http://oss.software.ibm.com/jfs\n" );
		rc = ENOSYS;
	} else {
		ExtLog_FirstRec = (ExtLogRec *)NULL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Free all of the private data item we have left on volumes.
 */
static void fs_cleanup()
{
	int rc = 0;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;
	LOG_ENTRY();

	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			if (volume->private_data) {
				EngFncs->engine_free(volume->private_data);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}
	LOG_EXIT_VOID();
	return;
}


/*
 * Does this FSIM manage the file system on this volume?
 * Return 0 for "yes", else a reason code.
 */
static int fs_probe(logical_volume_t * volume)
{
	int  rc = 0;
	struct superblock *sb_ptr;

	LOG_ENTRY();

	volume->private_data = NULL;

	/* allocate space for copy of superblock in private data */
	sb_ptr = EngFncs->engine_alloc( SIZE_OF_SUPER );

	if ( !sb_ptr ) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* get and validate JFS superblock */
	rc = fsim_get_jfs_superblock( volume, sb_ptr );

	if ( !rc ) {
		/* store copy of valid JFS superblock in private data */
		volume->private_data = (void *)sb_ptr;
	} else if ( jfsutils_support & JFS_EXTERNAL_LOG ) {
		/* didn't find valid JFS superblock,  */
		/* look for JFS log superblock        */
		rc = fsim_get_log_superblock( volume, (logsuper_t *)sb_ptr );
		if ( !rc ) {
			/* store copy of valid JFS log superblock in private data */
			volume->private_data = (void *)sb_ptr;
			/* let others know that this volume cannot be mounted */
			volume->flags |= VOLFLAG_NOT_MOUNTABLE;
		}
	}

	if ( rc ) {
		EngFncs->engine_free( sb_ptr );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can mkfs this volume?
 */
static int fs_can_mkfs(logical_volume_t * volume)
{
	int  rc=0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't format. */
		rc = EBUSY;
	} else if ((volume->vol_size * PBSIZE) < MINJFS) {
		/* volume size must be >= MINJFS */
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can unmkfs this volume?
 */
static int fs_can_unmkfs(logical_volume_t * volume)
{
	int  rc=0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else if (Is_JFS_Log_Vol((logsuper_t *)volume->private_data)) {
		/* If JFS external log volume, can't unmkfs */
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Can fsck this volume?
 */
static int fs_can_fsck(logical_volume_t * volume)
{
	int  rc=0;

	LOG_ENTRY();

	/* can't fsck.jfs external log volumes owned by this plugin */
	if (Is_JFS_Log_Vol((logsuper_t *)volume->private_data)) {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the current size of this volume
 */
static int fs_get_fs_size( logical_volume_t * volume,
			   sector_count_t   * size    )
{
	int  rc = EINVAL;
	struct superblock *sb_ptr = (struct superblock *)volume->private_data;
	logsuper_t *log_sb_ptr = (logsuper_t *)volume->private_data;

	LOG_ENTRY();

	if (!sb_ptr) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (!Is_JFS_Log_Vol( log_sb_ptr )) {
		/* get and validate current JFS superblock */
		rc = fsim_get_jfs_superblock( volume, sb_ptr );

		if (!rc && sb_ptr) {
			/*
			* s_size == (number of physical blocks in fs) - (log space) - (fsck workspace)
			* s_logpxd.len  == number of file system blocks for log
			* s_fsckpxd.len == number of file system blocks for fsck workspace
			* So, convert log space and fsck space to physical blocks and add to s_size
			*/
			*size = sb_ptr->s_size + (((sb_ptr->s_logpxd.len + sb_ptr->s_fsckpxd.len)*PSIZE)/PBSIZE);
		}
	} else {
		/* JFS external log */
		*size = (log_sb_ptr->size*LOGPSIZE)/PBSIZE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Get the size limits for this volume.
 */
static int fs_get_fs_limits( logical_volume_t * volume,
			     sector_count_t   * min_size,
			     sector_count_t   * max_volume_size,
			     sector_count_t   * max_object_size)
{
	int rc = EINVAL;
	struct superblock *sb_ptr = (struct superblock *)volume->private_data;
	logsuper_t *log_sb_ptr = (logsuper_t *)volume->private_data;

	LOG_ENTRY();

	if (!sb_ptr) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (!Is_JFS_Log_Vol(log_sb_ptr)) {
		/* get and validate current JFS superblock */
		rc = fsim_get_jfs_superblock( volume, sb_ptr );

		if ( !rc ) {
			rc = fsim_get_volume_limits( sb_ptr, min_size, max_volume_size, max_object_size);
			LOG_EXTRA("volume:%s, min:%"PRIu64", max:%"PRIu64"\n",volume->dev_node, *min_size, *max_volume_size);
			LOG_EXTRA("fssize:%"PRIu64", vol_size:%"PRIu64"\n",volume->fs_size,volume->vol_size );

			if (*min_size > volume->vol_size) {
				LOG_ERROR("JFS FSIM returned min size > volume size, setting min size to volume size\n");
				*min_size = volume->vol_size;
			}
		}
	} else {
		/* JFS external log */
		*min_size = *max_volume_size = *max_object_size = (log_sb_ptr->size*LOGPSIZE)/PBSIZE;
		if (*min_size > volume->vol_size) {
			LOG_ERROR("JFS FSIM returned min size > volume size, setting min size to volume size\n");
			*min_size = volume->vol_size;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Expand the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int fs_expand( logical_volume_t * volume,
		      sector_count_t   * new_size )
{
	int     rc;
	char   *argv[5];
	pid_t   pidf;
	int     status, bytes_read;
	int     fds2[2];
	char    *buffer = NULL;
	char	mount_opts[100];
	u_int64_t new_size_in_blocks;
	struct superblock  *sb_ptr = (struct superblock *)volume->private_data;

	LOG_ENTRY();

	if (!(jfsutils_support & JFS_EXPAND)) {
		rc = ENOSYS;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* open pipe, alloc buffer for collecting fsck.jfs output */
	if ((rc = pipe(fds2))) {
		return rc;
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		return ENOMEM;
	}

	argv[0] = "mount";
		
	new_size_in_blocks = *new_size / (sb_ptr->s_bsize / EVMS_VSECTOR_SIZE);
	sprintf(mount_opts, "-oremount,resize=%"PRIu64, new_size_in_blocks);
	argv[1] = mount_opts;
	
	argv[2] = EngFncs->engine_strdup(volume->mount_point);
	argv[3] = NULL;

	fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);

	pidf = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidf != -1) {
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE(_("jfs expand output: \n%s"),buffer); // on prompt if there is one
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					MESSAGE(_("jfs expand output: \n%s"),buffer);
				}
			} while (bytes_read == MAX_USER_MESSAGE_LEN);
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("JFS expand completed with rc = %d \n",rc);
			} else {
				LOG_ERROR("JFS expand completed with rc = %d \n",rc);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = errno;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * unmkfs has been scheduled.  Do any setup work such as
 * releasing another volume that was used for an external log.
 */
static int fs_unmkfs_setup(logical_volume_t * volume)
{
	int rc = 0;
	struct superblock  *sb_ptr = (struct superblock *)volume->private_data;
	logsuper_t  *log_sb_ptr = NULL;
	list_anchor_t JFS_vols;
	list_element_t vol_list_iter;
	logical_volume_t * extlog_vol;
	ExtLogRec * ExtLog_ThisRec;

	LOG_ENTRY();

	/*
	 * Once an unmkfs is scheduled for a volume, that volume loses its rights to
	 * private data.  Thus, if a JFS volume has an external log, we have to
	 * remember it here so that the external log that belongs to that JFS volume
	 * can be unmkfs'ed appropriately when the JFS volume is unmkfs'ed.  We
	 * remember the JFS volume / JFS external log pair by building a doubly
	 * linked list of all JFS volumes and their corresponding external logs
	 */

	/* see if the volume to be unmkfs'ed has an external log */
	if ( (jfsutils_support & JFS_EXTERNAL_LOG) &&
	     (sb_ptr) && (sb_ptr->s_version == JFS_VERSION) ) {
		/* get list of JFS claimed volumes */
		rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &JFS_vols);

		if (rc) {
			LOG_EXIT_INT(rc);
			return rc;
		}

		/*
		 * Look through all of the volumes to ultimately find the one that is
		 * an external log for the JFS volume to be unmkfs'ed.  We do this by
		 * checking for a match of the actual uuid of the external log, and the
		 * uuid of the external log that is stored in the JFS volume's superblock.
		 */
		LIST_FOR_EACH(JFS_vols, vol_list_iter, extlog_vol) {
			/* look for the JFS claimed volumes that are JFS external logs */
			if ( Is_JFS_Log_Vol((logsuper_t *)extlog_vol->private_data) ) {
				log_sb_ptr = (logsuper_t *)extlog_vol->private_data;

				/* compare the log uuid stored in the JFS superblock with the       */
				/* actual uuid of the external log volume to see if we have a match */
				if (!uuid_compare(sb_ptr->s_loguuid, log_sb_ptr->uuid)) {
					/**************************************************************************
					 *  TODO: Error checking.  Traverse the list and make sure the volume in  *
					 *  question is not already in it.  NOTE: The EVMS GUI will prevent that  *
					 *  situation since the GUI will not allow the user to undo uncommitted   *
					 *  actions without exiting EVMS, which would eliminate the list.  This   *
					 *  is really only needed to protect those who use the file system        *
					 *  utilities from the command line while the EVMS GUI is running, even   *
					 *  though EVMS says you're on your own in that case.                     *
					 **************************************************************************/
					/* If we have a match, add the JFS vol/external log vol pair to the list */
					if ( (ExtLog_ThisRec = EngFncs->engine_alloc(sizeof(struct ExternalLogRecord))) ) {
						ExtLog_ThisRec->fs_volume = volume;
						ExtLog_ThisRec->log_volume = extlog_vol;
						if (ExtLog_FirstRec) {
							ExtLog_ThisRec->NextRecord = ExtLog_FirstRec->NextRecord;
							ExtLog_FirstRec->NextRecord = ExtLog_ThisRec;
							ExtLog_ThisRec->PrevRecord = ExtLog_FirstRec;
							ExtLog_ThisRec->NextRecord->PrevRecord = ExtLog_ThisRec;
						} else {
							ExtLog_FirstRec = ExtLog_ThisRec;
							ExtLog_FirstRec->NextRecord = ExtLog_ThisRec;
							ExtLog_FirstRec->PrevRecord = ExtLog_ThisRec;
						}
						/* unclaim and free external log volume from EVMS */
						extlog_vol->file_system_manager = NULL;
						EngFncs->engine_free(extlog_vol->private_data);
					} else {
						/* couldn't get memory to add JFS vol/external log vol to list */
						rc = ENOMEM;
					}
					/* found a match, don't need to search any more */
					break;
				}
			}
		}
		EngFncs->destroy_list(JFS_vols);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * "unmkfs" the volume
 *
 * I rewrote this routine 3 times and it's still ugly, but it works.
 * Feel free to have at it...
 */
static int fs_unmkfs_jfs(logical_volume_t * volume)
{
	int rc = EINVAL;
	struct superblock  *sb_ptr = NULL;
	ExtLogRec * ExtLog_ThisRec;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* If external log is supported, get and save a copy of the superblock for later... */
	if (jfsutils_support & JFS_EXTERNAL_LOG) {
		if ( (sb_ptr = EngFncs->engine_alloc( SIZE_OF_SUPER )) ) {
			if ( (rc = fsim_get_jfs_superblock( volume, sb_ptr )) ) {
				/* volume to be unmkfs'ed no longer contains a valid JFS file system */
				MESSAGE(_("Volume %s does not contain a valid JFS file system.\n"),
					volume->dev_node );
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
		} else {
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}

	/* unmkfs the volume */
	if ( !(rc = fsim_unmkfs_jfs(volume)) ) {
		if ( jfsutils_support & JFS_EXTERNAL_LOG ) {
			/*
			 * External log is supported, so check superblock to see if
			 * one is used.  If so, find its volume and unmkfs it as well.
			 */
			if (sb_ptr->s_version == JFS_VERSION) {
				/*
				 * Look through the list of JFS volume/JFS external log pairs
				 * to find the correlating external log for this volume.
				 */
				if (ExtLog_FirstRec) {
					ExtLog_ThisRec = ExtLog_FirstRec;
					do {
						if ( ExtLog_ThisRec->fs_volume == volume ) {
							/*************************************************************
							 *  TODO: Error Checking.  Now that we have found the dev    *
							 *  node of the external log, go get its superblock and      *
							 *  verify that it really is an external log.  NOTE: This    *
							 *  is only needed to protect those who use the file system  *
							 *  utilities from the command line while the EVMS GUI is    *
							 *  running, even though EVMS says you're on your own in     *
							 *  that case.                                               *
							 *************************************************************/
							if ( (rc = fsim_unmkfs_ext_log(ExtLog_ThisRec->log_volume)) != FSIM_SUCCESS ) {
								LOG_WARNING("Could not unmkfs JFS external log on volume %s.\n",
									    ExtLog_ThisRec->log_volume->dev_node);
							}
							/* remove ThisRec from the linked list */
							if (ExtLog_ThisRec->NextRecord == ExtLog_ThisRec) {
								/* only one element in the linked list */
								ExtLog_FirstRec = (ExtLogRec *)NULL;
							} else {
								/* adjust Next and Prev pointers accordingly */
								ExtLog_ThisRec->NextRecord->PrevRecord = ExtLog_ThisRec->PrevRecord;
								ExtLog_ThisRec->PrevRecord->NextRecord = ExtLog_ThisRec->NextRecord;
							}
							/* free ThisRec */
							EngFncs->engine_free(ExtLog_ThisRec);
							/* we found the record, stop searching */
							break;
						} else {
							ExtLog_ThisRec = ExtLog_ThisRec->NextRecord;
						}
					} while ( ExtLog_ThisRec != ExtLog_FirstRec  );
				} else {
					/* JFS thinks it has an external log, but we have no record of one */
					/* Log a message to that effect and unmkfs the volume.             */
					LOG_WARNING("The JFS file system on volume %s thought it had an external log, but one was not found.\n",
						    volume->dev_node);
					LOG_WARNING("Volume %s was unmkfs'ed anyway.\n",
						    volume->dev_node);
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Shrink the volume to new_size.  If the volume is not expanded exactly to
 * new_size, set new_size to the new_size of the volume.
 */
static int fs_shrink( logical_volume_t * volume,
		      sector_count_t     requested_size,
		      sector_count_t   * new_size )
{
	/* unsupported at this time */
	int  rc = ENOSYS;

	LOG_ENTRY();
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * mkfs has been scheduled.  Do any setup work such as claiming another
 * volume for an external log.
 */
static int fs_mkfs_setup(logical_volume_t * volume,
			 option_array_t   * options)
{
	int  i, rc = 0;
	char  * ext_log_vol = NULL;
	list_anchor_t  avail_ext_logs;
	list_element_t logs_list_iter;
	logical_volume_t * vol;

	LOG_ENTRY();

	/* if there is no external log, nothing to do */
	if (!(jfsutils_support & JFS_EXTERNAL_LOG)) {
		LOG_EXIT_INT(0);
		return 0;
	}

	for ( i=0; i<options->count; i++ ) {
		if ( options->option[i].is_number_based ) {
			if (options->option[i].number == MKFS_JOURNAL_VOL_INDEX) {
				/* 'set journal volume' option */
				if ( options->option[i].value.s && strcmp(options->option[i].value.s, NO_SELECTION) ) {
					ext_log_vol = options->option[i].value.s;
				}
			}
		} else {
			if ( !strcmp(options->option[i].name, "journalvol") ) {
				/* 'set journal volume' option */
				if ( options->option[i].value.s && strcmp(options->option[i].value.s, NO_SELECTION) ) {
					ext_log_vol = options->option[i].value.s;
				}
			}
		}
	}

	if (ext_log_vol) {
		/*
		 * Search through the volumes to find and validate the one
		 * selected for an external log, and if valid, claim it.
		 */
		rc = EngFncs->get_volume_list(NULL, NULL, 0, &avail_ext_logs);
		if (!rc) {
			LIST_FOR_EACH(avail_ext_logs, logs_list_iter, vol) {
				/*************************************************************
				 *  TODO: When a min log volume size is put into mkfs.jfs,   *
				 *  add that requirement here for validation of a volume to  *
				 *  be considered for a JFS external log.                    *
				 *************************************************************/
				if ( (vol->file_system_manager == NULL ) &&
				     (!EngFncs->is_mounted(vol->dev_node, NULL)) &&
				     (!strcmp(vol->dev_node, ext_log_vol))  ) {
					vol->file_system_manager = &jfs_plugin_record;
					/* tell EVMS engine to re-probe this volume */
					vol->flags |= VOLFLAG_PROBE_FS;
				}
			}
			EngFncs->destroy_list(avail_ext_logs);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 * Format the volume.
 */
static int fs_mkfs_jfs(logical_volume_t * volume, option_array_t * options )
{
	int rc = 0;

	LOG_ENTRY();

	if (!jfsutils_support) {
		int answer = 0;
		char * choice_text[2] = {_("Ok"), NULL};
		QUESTION(&answer, choice_text,
			 _("The jfsutils must be version 1.0.9 or later to function properly with this FSIM.  "
			   "The jfsutils must also reside in the search path specified by PATH.  "
			   "Please get/install the current version of jfsutils from http://oss.software.ibm.com/jfs\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}
	/* don't format if mounted */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	rc = fsim_mkfs(volume, options);

	/* probe to set up private data */
	if ( !rc ) {
		rc = fs_probe(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Forget about this volume.  Don't remove the file system.  Just clean
 * up any data structures you may have associated with it.
 */
static int fs_discard(logical_volume_t * volume)
{
	LOG_ENTRY();

	if (volume->private_data) {
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Run fsck on the volume.
 */
static int fs_fsck(logical_volume_t * volume, option_array_t * options )
{
	int rc = EINVAL;

	LOG_ENTRY();

	if (!jfsutils_support) {
		int answer = 0;
		char * choice_text[2] = {_("Ok"), NULL};
		QUESTION(&answer, choice_text,
                         _("The jfsutils must be version 1.0.9 or later to function properly with this FSIM.  "
			   "The jfsutils must also reside in the search path specified by PATH.  "
			   "Please get/install the current version of jfsutils from http://oss.software.ibm.com/jfs\n"));
		LOG_EXIT_INT(E_CANCELED);
		return E_CANCELED;
	}

	rc = fsim_fsck( volume, options );

	/*
	 * If fsck.jfs returns FSCK_CORRECTED, the
	 * file system is clean, so return FSCK_OK.
	 */
	if ( rc == FSCK_CORRECTED ) {
		rc = FSCK_OK;
		/*
		 * The value of FSCK_CORRECTED is the same as
		 * EPERM, so fsim_fsck will return -1 for EPERM.
		 */
	} else if (rc == -1) {
		rc = EPERM;
	}

	/*
	 * If the volume is mounted, fsck.jfs checked JFS READ ONLY
	 * regardless of options specified.  If the check was READ ONLY
	 * and errors were found, let the user know how to fix them.
	 */
	if (EngFncs->is_mounted(volume->dev_node, NULL) && rc) {
		MESSAGE(_("%s is mounted.  "
			  "fsck.jfs checked the volume READ ONLY and found, but did not fix, errors.  "
			  "Unmount %s and run fsck.jfs again to repair the file system.\n"),
			volume->dev_node, volume->dev_node );
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return the total number of supported options for the specified task.
 */
static int fs_get_option_count(task_context_t * context)
{
	int count = 0;

	LOG_ENTRY();

	switch (context->action) {
		case EVMS_Task_mkfs:
			count = MKFS_JFS_OPTIONS_COUNT;
			break;
		case EVMS_Task_fsck:
			count = FSCK_JFS_OPTIONS_COUNT;
			break;
		case EVMS_Task_Expand:
			count = 0;
			break;
		default:
			count = -1;
			break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Get the list of volumes on the system that we can use. */
static int get_volume_list( value_list_t ** value_list,
			    list_anchor_t   selected_volumes )
{
	int rc = 0;
	int count, i;
	logical_volume_t * vol;
	list_element_t tmp_list_iter;
	list_anchor_t tmp_list, selected_tmp_list;

	LOG_ENTRY();

	if (!EngFncs->list_empty(selected_volumes)) {
		vol = EngFncs->first_thing(selected_volumes, NULL);
		if (vol->disk_group != NULL) {
			EngFncs->get_volume_list(NULL, vol->disk_group, 0, &tmp_list);
		} else {
			EngFncs->get_volume_list(NULL, NULL, VOL_NO_DISK_GROUP, &tmp_list);
		}
	} else {
		EngFncs->get_volume_list(NULL, NULL, 0, &tmp_list);
	}

	/* populate temp list with actual 'selected' volumes */
	selected_tmp_list = EngFncs->copy_list(selected_volumes);
	if (!selected_tmp_list) {
		LOG_ERROR("Error copying list\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/*
	 * Loop through 'selected' volumes, remove them from temp list.
	 */
	LIST_FOR_EACH(selected_tmp_list, tmp_list_iter, vol) {
		LOG_DETAILS("Volume %s selected, removing from 'available' list\n",vol->name);
		EngFncs->remove_thing(tmp_list, vol);
	}
	EngFncs->destroy_list(selected_tmp_list);

	/* if 'available volumes' list already exists, free it and start over */
	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}
	*value_list = NULL;

	count = EngFncs->list_count(tmp_list);
	/* Increment count to hold the 'None' selection. */
	count++;
	/* allocate and populate 'available volumes' list */
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));
	if (!rc) {
		list_element_t vol_list_iter;

		i = 0;
		(*value_list)->value[i].s = EngFncs->engine_strdup(NO_SELECTION);
		i++;
		LIST_FOR_EACH(tmp_list, vol_list_iter, vol) {
			/* add proper volumes to 'available volumes' list */
			/*************************************************************
			 *  TODO: When a min log volume size is put into mkfs.jfs,   *
			 *  add that requirement here for validation of a volume to  *
			 *  be considered for a JFS external log.                    *
			 *************************************************************/
			if ( (vol->file_system_manager == NULL ) &&
			     (!EngFncs->is_mounted(vol->dev_node, NULL)) ) {
				(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(vol->name) + 1);
				strcpy((*value_list)->value[i].s, vol->name);
				i++;
			}
		}
		(*value_list)->count = i;
	}
	EngFncs->destroy_list(tmp_list);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int fs_init_mkfs_acceptable_objects(task_context_t * context)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	if (context->volume != NULL) {
		if (context->volume->disk_group != NULL) {
			rc = EngFncs->get_volume_list(NULL, context->volume->disk_group, 0, &global_volumes);
		} else {
			rc = EngFncs->get_volume_list(NULL, NULL, VOL_NO_DISK_GROUP, &global_volumes);
		}
	} else {
		rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	}

	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* only mkfs unformatted, unmounted, 'large enough' volumes */
			if ( (volume->file_system_manager == NULL) &&
			     ((volume->vol_size * PBSIZE) >= MINJFS) &&
			     (!EngFncs->is_mounted(volume->dev_node, NULL))  ) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum number of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 */
static int fs_init_task( task_context_t * context )
{
	int  rc = 0;

	LOG_ENTRY();

	context->min_selected_objects = 0;
	context->max_selected_objects = 0;
	context->option_descriptors->count = 0;

	/* Parameter check */
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
		
		case EVMS_Task_mkfs:

			rc = fs_init_mkfs_acceptable_objects(context);
			if (rc) {
				break;
			}

			if (!jfsutils_support) {
				int answer = 0;
				char * choice_text[2] = {_("Ok"), NULL};
				QUESTION(&answer, choice_text,
					 _("The jfsutils must be version 1.0.9 or later to function properly with this FSIM.  "
					   "The jfsutils must also reside in the search path specified by PATH.  "
					   "Please get/install the current version of jfsutils from http://oss.software.ibm.com/jfs\n"));
				rc = E_CANCELED;
				break;
			}
			context->option_descriptors->count = MKFS_JFS_OPTIONS_COUNT;

			/* check for bad blocks option */
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].name = EngFncs->engine_strdup("badblocks" );
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].title = EngFncs->engine_strdup(_("Check For Bad Blocks"));
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].tip = EngFncs->engine_strdup(_("Check the volume for bad blocks before building the file system."));
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].help = NULL;
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[MKFS_CHECKBB_INDEX].value.b = FALSE;

			/* case insensitive option */
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].name = EngFncs->engine_strdup("caseinsensitive" );
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].title = EngFncs->engine_strdup(_("Case Insensitive"));
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].tip = EngFncs->engine_strdup(_("Mark the file system as case insensitive (for OS/2 compatibility)."));
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].help = NULL;
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[MKFS_CASEINSENS_INDEX].value.b = FALSE;

			/* Set Volume Label option */
			context->option_descriptors->option[MKFS_SETVOL_INDEX].name = EngFncs->engine_strdup("vollabel" );
			context->option_descriptors->option[MKFS_SETVOL_INDEX].title = EngFncs->engine_strdup(_("Volume Label"));
			context->option_descriptors->option[MKFS_SETVOL_INDEX].tip = EngFncs->engine_strdup(_("Set the volume label for the file system."));
			context->option_descriptors->option[MKFS_SETVOL_INDEX].help = NULL;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].type = EVMS_Type_String;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].min_len = 1;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].max_len = LV_NAME_SIZE;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[MKFS_SETVOL_INDEX].value.s = EngFncs->engine_alloc(LV_NAME_SIZE + 1);
			if (context->option_descriptors->option[MKFS_SETVOL_INDEX].value.s == NULL) {
				LOG_EXIT_INT(ENOMEM);
				return ENOMEM;
			}

			/* Set Journal Volume option */
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].name = EngFncs->engine_strdup("journalvol" );
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].title = EngFncs->engine_strdup(_("External Journal Volume"));
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].tip = EngFncs->engine_strdup(_("Set the volume for the journal if other than the JFS file system volume."));
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].help = NULL;
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].type = EVMS_Type_String;
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].min_len = 1;
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].max_len = EVMS_VOLUME_NAME_SIZE + 1;
			if ( jfsutils_support & JFS_EXTERNAL_LOG ) {
				context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			} else {
				context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			}
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].constraint_type = EVMS_Collection_List;
			get_volume_list((value_list_t **)&context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].constraint.list,
					context->selected_objects);
			context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE + 1);
			if (context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].value.s == NULL) {
				LOG_EXIT_INT(ENOMEM);
				return ENOMEM;
			}

			/* Log size option */
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].name = EngFncs->engine_strdup("logsize" );
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].title = EngFncs->engine_strdup(_("Inline Log Size"));
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].tip = EngFncs->engine_strdup(_("Set inline log size (in megabytes).  Default log size is 0.4% of volume size.  Used only if no external journal volume is specified."));
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].help = NULL;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].type = EVMS_Type_Int32;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].unit = EVMS_Unit_Megabytes;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint_type = EVMS_Collection_Range;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
			if (context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range == NULL) {
				LOG_EXIT_INT(ENOMEM);
				return ENOMEM;
			}
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->min.ui = 1;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->max.ui = 32;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->increment.ui = 1;
			context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].value.ui = 1;

			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
			break;

		case EVMS_Task_fsck:

			if (!jfsutils_support) {
				int answer = 0;
				char * choice_text[2] = {_("Ok"), NULL};
				QUESTION(&answer, choice_text,
					 _("The jfsutils must be version 1.0.9 or later to function properly with this FSIM.  "
					   "The jfsutils must also reside in the search path specified by PATH.  "
					   "Please get/install the current version of jfsutils from http://oss.software.ibm.com/jfs\n"));
				rc = E_CANCELED;
				break;
			}

			context->option_descriptors->count = FSCK_JFS_OPTIONS_COUNT;

			/* force check option */
			context->option_descriptors->option[FSCK_FORCE_INDEX].name = EngFncs->engine_strdup("force" );
			context->option_descriptors->option[FSCK_FORCE_INDEX].title = EngFncs->engine_strdup(_("Force Check, Repair All Problems"));
			context->option_descriptors->option[FSCK_FORCE_INDEX].tip = EngFncs->engine_strdup(_("Force complete file system check and automatically repair all problems."));
			context->option_descriptors->option[FSCK_FORCE_INDEX].help = NULL;
			context->option_descriptors->option[FSCK_FORCE_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[FSCK_FORCE_INDEX].unit = EVMS_Unit_None;
			/* if volume is mounted, only possible fsck.jfs options is READONLY */
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				context->option_descriptors->option[FSCK_FORCE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			} else {
				context->option_descriptors->option[FSCK_FORCE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			}
			context->option_descriptors->option[FSCK_FORCE_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[FSCK_FORCE_INDEX].value.b = FALSE;

			/* read-only check option */
			context->option_descriptors->option[FSCK_READONLY_INDEX].name = EngFncs->engine_strdup("readonly" );
			context->option_descriptors->option[FSCK_READONLY_INDEX].title = EngFncs->engine_strdup(_("Check Read-Only"));
			context->option_descriptors->option[FSCK_READONLY_INDEX].tip = EngFncs->engine_strdup(_("Check the file system READ ONLY.  Report but do not correct errors."));
			context->option_descriptors->option[FSCK_READONLY_INDEX].help = NULL;
			context->option_descriptors->option[FSCK_READONLY_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[FSCK_READONLY_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[FSCK_READONLY_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[FSCK_READONLY_INDEX].constraint_type = EVMS_Collection_None;
			/* if volume is mounted, only possible fsck.jfs options is READONLY */
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				context->option_descriptors->option[FSCK_READONLY_INDEX].value.b = TRUE;
			} else {
				context->option_descriptors->option[FSCK_READONLY_INDEX].value.b = FALSE;
			}

			/* omit log replay option */
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].name = EngFncs->engine_strdup("omitlog" );
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].title = EngFncs->engine_strdup(_("Omit Log Replay"));
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].tip = EngFncs->engine_strdup(_("Omit transaction log replay.  Use only if log is corrupt."));
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].help = NULL;
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].unit = EVMS_Unit_None;
			/* if volume is mounted, only possible fsck.jfs options is READONLY */
			if (EngFncs->is_mounted(context->volume->dev_node, NULL)) {
				context->option_descriptors->option[FSCK_OMITLOG_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			} else {
				context->option_descriptors->option[FSCK_OMITLOG_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			}
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[FSCK_OMITLOG_INDEX].value.b = FALSE;

			/* verbose output option */
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].name = EngFncs->engine_strdup("verbose" );
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].title = EngFncs->engine_strdup(_("Verbose Output"));
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].tip = EngFncs->engine_strdup(_("Log fsck.jfs details and debug information."));
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].help = NULL;
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[FSCK_VERBOSE_INDEX].value.b = FALSE;

			/* display version option */
			context->option_descriptors->option[FSCK_VERSION_INDEX].name = EngFncs->engine_strdup("version" );
			context->option_descriptors->option[FSCK_VERSION_INDEX].title = EngFncs->engine_strdup(_("Display fsck.jfs Version Only (No Check)"));
			context->option_descriptors->option[FSCK_VERSION_INDEX].tip = EngFncs->engine_strdup(_("Display fsck.jfs version and exit without checking the file system, regardless of other chosen options."));
			context->option_descriptors->option[FSCK_VERSION_INDEX].help = NULL;
			context->option_descriptors->option[FSCK_VERSION_INDEX].type = EVMS_Type_Boolean;
			context->option_descriptors->option[FSCK_VERSION_INDEX].unit = EVMS_Unit_None;
			context->option_descriptors->option[FSCK_VERSION_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
			context->option_descriptors->option[FSCK_VERSION_INDEX].constraint_type = EVMS_Collection_None;
			context->option_descriptors->option[FSCK_VERSION_INDEX].value.b = FALSE;

			context->min_selected_objects = 0;
			context->max_selected_objects = 0;

                        break;

		case EVMS_Task_Expand:
			context->min_selected_objects = 0;
			context->max_selected_objects = 0;
			break;

		default:
			rc = EINVAL;
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted
 * if necessary/allowed. If so, set the effect return value accordingly.
 */
static int fs_set_option( task_context_t * context,
			  u_int32_t        index,
			  value_t        * value,
			  task_effect_t  * effect )
{
	int  rc= 0;

	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		return EFAULT;
	}

	switch (context->action) {
		
		case EVMS_Task_mkfs:
			switch (index) {
				
				case MKFS_CHECKBB_INDEX:
					/* 'check for bad blocks' option set? */
					context->option_descriptors->option[index].value.b = value->b;
					break;

				case MKFS_CASEINSENS_INDEX:
					/* 'case insensitive' option set? */
					context->option_descriptors->option[index].value.b = value->b;
					break;

				case MKFS_SETVOL_INDEX:
					/* 'set volume label' option set? */
					strncpy(context->option_descriptors->option[index].value.s, value->s, LV_NAME_SIZE);
					break;

				case MKFS_JOURNAL_VOL_INDEX:
					/* 'set journal volume' option set? */
					strncpy(context->option_descriptors->option[index].value.s, value->s, EVMS_VOLUME_NAME_SIZE);
					if ( strcmp(value->s, NO_SELECTION) ) {
						/* If JFS external log volume specified, 'log size' is ignored */
						context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
						*effect |= EVMS_Effect_Reload_Options;
					} else if (context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].flags == EVMS_OPTION_FLAGS_INACTIVE) {
						/* JFS external log deselected, so re-activate 'log size' option */
						context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
						*effect |= EVMS_Effect_Reload_Options;
					}
					break;

				case MKFS_SETLOGSIZE_INDEX:
					/* 'set log size' option set? */
					context->option_descriptors->option[index].value.i32 = value->i32;
					break;

				default:
					break;
			}
			break;

		case EVMS_Task_fsck:
			switch (index) {
				
				case FSCK_FORCE_INDEX:
					/* 'force check' option set? */

					/*
					 * When using the EVMS CLI, if the volume is mounted
					 * and the user sets force=TRUE, the resultant EVMS
					 * message is "One of the options specified was
					 * incorrect."  Provide a little more useful info.
					 */
					if ( EngFncs->is_mounted(context->volume->dev_node, NULL) ) {
						MESSAGE(_("The 'force' option is not allowed when the volume is mounted.\n"));
						break;
					}

					context->option_descriptors->option[index].value.b = value->b;

					/* if 'force check' is selected... */
					if (value->b) {
						/* ... and 'check read only' is set... */
						if (context->option_descriptors->option[FSCK_READONLY_INDEX].value.b) {
							/* ... turn 'check read only' off */
							context->option_descriptors->option[FSCK_READONLY_INDEX].value.b = 0;
							*effect |= EVMS_Effect_Reload_Options;
						}
						/* ... and 'display version only' is set... */
						if (context->option_descriptors->option[FSCK_VERSION_INDEX].value.b) {
							/* ... turn 'display version only' off */
							context->option_descriptors->option[FSCK_VERSION_INDEX].value.b = 0;
							*effect |= EVMS_Effect_Reload_Options;
						}
					}
					break;

				case FSCK_READONLY_INDEX:
					/* 'check read only' option set? */
					context->option_descriptors->option[index].value.b = value->b;

					/* If mounted, only allow 'yes' for check read only */
					if (EngFncs->is_mounted(context->volume->dev_node, NULL) && !value->b) {
						context->option_descriptors->option[FSCK_READONLY_INDEX].value.b = TRUE;
						*effect |= EVMS_Effect_Reload_Options;
					}

					/* if 'check read only' is selected... */
					if (value->b) {
						/* ... and 'force check' is set... */
						if (context->option_descriptors->option[FSCK_FORCE_INDEX].value.b) {
							/* ... turn 'force check' off */
							context->option_descriptors->option[FSCK_FORCE_INDEX].value.b = 0;
							*effect |= EVMS_Effect_Reload_Options;
						}
						/* ... and 'display version only' is set... */
						if (context->option_descriptors->option[FSCK_VERSION_INDEX].value.b) {
							/* ... turn 'display version only' off */
							context->option_descriptors->option[FSCK_VERSION_INDEX].value.b = 0;
							*effect |= EVMS_Effect_Reload_Options;
						}
					}
					break;

				case FSCK_OMITLOG_INDEX:
					/* 'omit log replay' option set? */
					context->option_descriptors->option[index].value.b = value->b;
					break;

				case FSCK_VERBOSE_INDEX:
					/* 'verbose' option set? */
					context->option_descriptors->option[index].value.b = value->b;
					break;

				case FSCK_VERSION_INDEX:
					/* 'display version only' option set? */
					context->option_descriptors->option[index].value.b = value->b;

					/* if 'display version only' is selected, turn of all other options */
					if (value->b) {
						context->option_descriptors->option[FSCK_FORCE_INDEX].value.b = 0;
						context->option_descriptors->option[FSCK_READONLY_INDEX].value.b = 0;
						context->option_descriptors->option[FSCK_OMITLOG_INDEX].value.b = 0;
						context->option_descriptors->option[FSCK_VERBOSE_INDEX].value.b = 0;
						*effect |= EVMS_Effect_Reload_Options;
					}
					break;

				default:
					break;
			}
			break;

		default:
			break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Validate the volumes in the selected_objects list in the task context.
 * Remove from the selected objects lists any volumes which are not
 * acceptable.  For unacceptable volumes, create a declined_handle_t
 * structure with the reason why it is not acceptable, and add it to the
 * declined_volumes list.  Modify the acceptable_objects list in the task
 * context as necessary based on the selected objects and the current
 * settings of the options.  Modify any option settings as necessary based
 * on the selected objects.  Return the appropriate task_effect_t settings
 * if the object list(s), minimum or maximum objects selected, or option
 * settings have changed.
 */
static int fs_set_volumes( task_context_t * context,
			   list_anchor_t    declined_volumes,
			   task_effect_t  * effect )
{
	int  rc = 0;
	int64_t  log_size;
	unsigned int  max_log_size;
	logical_volume_t * vol;

	LOG_ENTRY();

	if (context->action == EVMS_Task_mkfs) {

		/* get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);

		if (vol) {
			if (EngFncs->is_mounted(vol->dev_node, NULL)) {
				/* If mounted, can't mkfs.jfs. */
				rc = EBUSY;
			} else {
				if ( (vol->vol_size * PBSIZE) < MINJFS) {

					/***************************************************
					 *  TODO - move this volume to unacceptable list   *
					 ***************************************************/
					char number_buffer[64];

					sprintf(number_buffer, "%"PRIu64, vol->vol_size * PBSIZE);
					MESSAGE(_("The size of volume %s is %s bytes.  "
						  "mkfs.jfs requires a minimum of %u bytes to build the JFS file system.\n"),
						vol->dev_node, number_buffer, MINJFS);
					rc = EINVAL;
				} else {
					/*
					 * Default log size is .4% of volume size, which
					 * for simplicity is 4/1024 == 2**2/2**10 == 1/2**8
					 *
					 * vol->vol_size is in sectors, so in bytes, it is:
					 * (vol->vol_size)(512 bytes/sector) == (vol->vol_size)(2**9)
					 *
					 * Combining, we have (vol->vol_size)(2**9)/(2**8), or
					 * (vol->vol_size)(2)
					 *
					 * Then round logsize up to a megabyte boundary
					 */
					log_size = ( (vol->vol_size * 2) + MEGABYTE - 1) & ~(MEGABYTE - 1);  /* round up to meg */
					log_size = log_size / MEGABYTE;

					/* calculate and set upper log size bound */
					/* -1 on volsize is to force rounding down due to bug in mkfs.jfs for == 10% */
					max_log_size = (unsigned int)((((vol->vol_size - 1) * PBSIZE) * (MAX_LOG_PERCENTAGE/100.0))/MEGABYTE);
					max_log_size = (max_log_size < MEGABYTE32/MEGABYTE) ? max_log_size : MEGABYTE32/MEGABYTE;
					context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->max.ui = max_log_size;

					/* make sure log size is not greater than max log size */
					if ( log_size > max_log_size ) {
						log_size = max_log_size;
					}
					/* initialize log size option */
					context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].value.ui = (unsigned int)log_size;
					*effect |= EVMS_Effect_Reload_Options;

					/* Remove selected volume from external log selection list. */
					if ( jfsutils_support & JFS_EXTERNAL_LOG ) {
						get_volume_list((value_list_t **)&context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].constraint.list,
								context->selected_objects);
					}
				}
			}
		} else {
			rc = ENODATA;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return any additional information that you wish to provide about the
 * volume.  The Engine provides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of extended_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the storage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * responsibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 */
static int fs_get_volume_info( logical_volume_t        * volume,
			       char                    * info_name,
			       extended_info_array_t * * info )
{
	int rc = EINVAL;
	extended_info_array_t  *Info;
	struct superblock      *sb_ptr = (struct superblock *)volume->private_data;
	logsuper_t *log_sb_ptr = (logsuper_t *)volume->private_data;
	char magic[12];


	LOG_ENTRY();

	if (!sb_ptr) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (!Is_JFS_Log_Vol(log_sb_ptr)) {
		/* get and validate current JFS superblock */
		rc = fsim_get_jfs_superblock( volume, sb_ptr );

		if ( (info_name == NULL) && !rc ) {
			fs_get_fs_limits( volume,	       // reset limits.
					  &volume->min_fs_size,
					  &volume->max_vol_size,
					  &volume->max_fs_size);

			Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 7 * sizeof(extended_info_t) ) );
			if (Info) {

				Info->count = 7;

				Info->info[0].name = EngFncs->engine_strdup( "MagicNumber" );
				Info->info[0].title = EngFncs->engine_strdup( _("JFS Magic Number") );
				Info->info[0].desc = EngFncs->engine_strdup( _("JFS Magic Number.") );
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				Info->info[0].value.s = EngFncs->engine_strdup(sb_ptr->s_magic);
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset( &Info->info[0].group, 0, sizeof(group_info_t));

				Info->info[1].name = EngFncs->engine_strdup( "Version" );
				Info->info[1].title = EngFncs->engine_strdup( _("JFS Version Number") );
				Info->info[1].desc = EngFncs->engine_strdup( _("JFS Version Number.") );
				Info->info[1].type               = EVMS_Type_Unsigned_Int32;
				Info->info[1].unit               = EVMS_Unit_None;
				Info->info[1].value.ui32         = sb_ptr->s_version;
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset( &Info->info[1].group, 0, sizeof(group_info_t));

				Info->info[2].name = EngFncs->engine_strdup( "State" );
				Info->info[2].title = EngFncs->engine_strdup( _("JFS State") );
				Info->info[2].desc = EngFncs->engine_strdup( _("The state of JFS.") );
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				switch (sb_ptr->s_state) {
					case FM_CLEAN:
					case FM_MOUNT:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Clean"));
						break;
					case FM_DIRTY:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Dirty, run fsck.jfs"));
						break;
					case FM_LOGREDO:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Logredo failed, run fsck.jfs"));
						break;
					case FM_EXTENDFS:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Extend FS in progress"));
						break;
					default:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Unknown"));
						break;
				}
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset( &Info->info[2].group, 0, sizeof(group_info_t));

				Info->info[3].name = EngFncs->engine_strdup( "VolLabel" );
				Info->info[3].title = EngFncs->engine_strdup( _("Volume Label") );
				Info->info[3].desc = EngFncs->engine_strdup( _("File system volume label chosen via mkfs.jfs.") );
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				Info->info[3].value.s = EngFncs->engine_strdup(sb_ptr->s_fpack);
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset( &Info->info[3].group, 0, sizeof(group_info_t));

				/********************************************************************
				 *  TODO: If this JFS volume has an external log volume, display    *
				 *  information about that volume here instead of Inline Log Size.  *
				 ********************************************************************/
				Info->info[4].name = EngFncs->engine_strdup( "LogSize" );
				Info->info[4].title = EngFncs->engine_strdup( _("Inline Log Size") );
				Info->info[4].desc = EngFncs->engine_strdup( _("Size of the inline file system log (included in the file system size).") );
				Info->info[4].type               = EVMS_Type_Unsigned_Int64;
				Info->info[4].unit               = EVMS_Unit_Sectors;
				Info->info[4].value.ui64         = ((sb_ptr->s_logpxd.len)*PSIZE)/PBSIZE;
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset( &Info->info[4].group, 0, sizeof(group_info_t));

				Info->info[5].name = EngFncs->engine_strdup( "Workspace" );
				Info->info[5].title = EngFncs->engine_strdup( _("fsck.jfs Workspace Size") );
				Info->info[5].desc = EngFncs->engine_strdup( _("Size of fsck.jfs workspace (included in the file system size.") );
				Info->info[5].type               = EVMS_Type_Unsigned_Int64;
				Info->info[5].unit               = EVMS_Unit_Sectors;
				Info->info[5].value.ui64         = ((sb_ptr->s_fsckpxd.len)*PSIZE)/PBSIZE;
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset( &Info->info[5].group, 0, sizeof(group_info_t));

				Info->info[6].name = EngFncs->engine_strdup( "jfsutils" );
				Info->info[6].title = EngFncs->engine_strdup( _("JFS Utilities Version") );
				Info->info[6].desc = EngFncs->engine_strdup( _("JFS utilities version number.") );
				Info->info[6].type               = EVMS_Type_String;
				Info->info[6].unit               = EVMS_Unit_None;
				Info->info[6].value.s = EngFncs->engine_strdup(jfsutils_version);
				Info->info[6].collection_type    = EVMS_Collection_None;
				memset( &Info->info[6].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information about any of the descriptors. */
			rc = EINVAL;
		}
	} else {
		/* get and validate current JFS external log superblock */
		rc = fsim_get_log_superblock( volume, log_sb_ptr );

		if ( (info_name == NULL) && !rc ) {
			Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 6 * sizeof(extended_info_t) ) );
			if (Info) {

				Info->count = 6;

				Info->info[0].name = EngFncs->engine_strdup( "LogMagicNumber" );
				Info->info[0].title = EngFncs->engine_strdup( _("JFS External Log Magic Number") );
				Info->info[0].desc = EngFncs->engine_strdup( _("JFS External Log Magic Number.") );
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				sprintf(magic, "0x%x", log_sb_ptr->magic);
				Info->info[0].value.s = EngFncs->engine_strdup(magic);
				Info->info[0].value.s = EngFncs->engine_strdup(magic);
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset( &Info->info[0].group, 0, sizeof(group_info_t));

				Info->info[1].name = EngFncs->engine_strdup( "LogVersion" );
				Info->info[1].title = EngFncs->engine_strdup( _("JFS External Log Version Number") );
				Info->info[1].desc = EngFncs->engine_strdup( _("JFS External Log Version Number.") );
				Info->info[1].type               = EVMS_Type_Unsigned_Int32;
				Info->info[1].unit               = EVMS_Unit_None;
				Info->info[1].value.i32          = log_sb_ptr->version;
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset( &Info->info[1].group, 0, sizeof(group_info_t));

				Info->info[2].name = EngFncs->engine_strdup( "LogState" );
				Info->info[2].title = EngFncs->engine_strdup( _("JFS Log State") );
				Info->info[2].desc = EngFncs->engine_strdup( _("The state of JFS log.") );
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				switch (log_sb_ptr->state) {
					case LOGMOUNT:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Log Mounted"));
						break;
					case LOGREDONE:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Log Redo Completed"));
						break;
					case LOGWRAP:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Log Wrapped"));
						break;
					case LOGREADERR:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Log Read Error Detected"));
						break;
					default:
						Info->info[2].value.s = EngFncs->engine_strdup(_("Unknown"));
						break;
				}
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset( &Info->info[2].group, 0, sizeof(group_info_t));

				Info->info[3].name = EngFncs->engine_strdup( "JournalLabel" );
				Info->info[3].title = EngFncs->engine_strdup( _("Journal Label") );
				Info->info[3].desc = EngFncs->engine_strdup( _("External log volume label") );
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				Info->info[3].value.s = EngFncs->engine_strdup(log_sb_ptr->label);
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset( &Info->info[3].group, 0, sizeof(group_info_t));

				Info->info[4].name = EngFncs->engine_strdup( "LogSize" );
				Info->info[4].title = EngFncs->engine_strdup( _("Log Size") );
				Info->info[4].desc = EngFncs->engine_strdup( _("Size of the file system log (included in the file system size)") );
				Info->info[4].type               = EVMS_Type_Unsigned_Int64;
				Info->info[4].unit               = EVMS_Unit_Sectors;
				Info->info[4].value.ui64         = (log_sb_ptr->size*LOGPSIZE)/PBSIZE;
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset( &Info->info[4].group, 0, sizeof(group_info_t));

				Info->info[5].name = EngFncs->engine_strdup( "jfsutils" );
				Info->info[5].title = EngFncs->engine_strdup( _("JFS Utilities Version") );
				Info->info[5].desc = EngFncs->engine_strdup( _("JFS utilities version number") );
				Info->info[5].type               = EVMS_Type_String;
				Info->info[5].unit               = EVMS_Unit_None;
				Info->info[5].value.s = EngFncs->engine_strdup(jfsutils_version);
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset( &Info->info[5].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information about any of the descriptors. */
			rc = EINVAL;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Returns Plugin specific information ...
 */
static int fs_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
	int                      rc = EINVAL;
	extended_info_array_t   *Info;
	char                     version_string[64];
	char                     required_engine_api_version_string[64];
	char                     required_fsim_api_version_string[64];

	LOG_ENTRY();

	if (info) {

		if (descriptor_name == NULL) {
			*info = NULL;	  // init to no info returned

			Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
			if (Info) {

				Info->count = 6;

				sprintf(version_string, "%d.%d.%d",
					MAJOR_VERSION,
					MINOR_VERSION,
					PATCH_LEVEL );

				sprintf(required_engine_api_version_string, "%d.%d.%d",
					my_plugin_record->required_engine_api_version.major,
					my_plugin_record->required_engine_api_version.minor,
					my_plugin_record->required_engine_api_version.patchlevel );

				sprintf(required_fsim_api_version_string, "%d.%d.%d",
					my_plugin_record->required_plugin_api_version.fsim.major,
					my_plugin_record->required_plugin_api_version.fsim.minor,
					my_plugin_record->required_plugin_api_version.fsim.patchlevel );

				Info->info[0].name = EngFncs->engine_strdup( "Short Name" );
				Info->info[0].title = EngFncs->engine_strdup( _("Short Name") );
				Info->info[0].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
				Info->info[0].type               = EVMS_Type_String;
				Info->info[0].unit               = EVMS_Unit_None;
				Info->info[0].value.s = EngFncs->engine_strdup( my_plugin_record->short_name );
				Info->info[0].collection_type    = EVMS_Collection_None;
				memset( &Info->info[0].group, 0, sizeof(group_info_t));

				Info->info[1].name = EngFncs->engine_strdup( "Long Name" );
				Info->info[1].title = EngFncs->engine_strdup( _("Long Name") );
				Info->info[1].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
				Info->info[1].type               = EVMS_Type_String;
				Info->info[1].unit               = EVMS_Unit_None;
				Info->info[1].value.s = EngFncs->engine_strdup( my_plugin_record->long_name );
				Info->info[1].collection_type    = EVMS_Collection_None;
				memset( &Info->info[1].group, 0, sizeof(group_info_t));

				Info->info[2].name = EngFncs->engine_strdup( "Type" );
				Info->info[2].title = EngFncs->engine_strdup( _("Plug-in Type") );
				Info->info[2].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
				Info->info[2].type               = EVMS_Type_String;
				Info->info[2].unit               = EVMS_Unit_None;
				Info->info[2].value.s = EngFncs->engine_strdup( _("File System Interface Module") );
				Info->info[2].collection_type    = EVMS_Collection_None;
				memset( &Info->info[2].group, 0, sizeof(group_info_t));

				Info->info[3].name = EngFncs->engine_strdup( "Version" );
				Info->info[3].title = EngFncs->engine_strdup( _("Plug-in Version") );
				Info->info[3].desc = EngFncs->engine_strdup( _("This is the version number of the plug-in.") );
				Info->info[3].type               = EVMS_Type_String;
				Info->info[3].unit               = EVMS_Unit_None;
				Info->info[3].value.s = EngFncs->engine_strdup( version_string );
				Info->info[3].collection_type    = EVMS_Collection_None;
				memset( &Info->info[3].group, 0, sizeof(group_info_t));

				Info->info[4].name = EngFncs->engine_strdup( "Required Engine Services API Version" );
				Info->info[4].title = EngFncs->engine_strdup( _("Required Engine Services API Version") );
				Info->info[4].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that this plug-in requires. "
									       " It will not run on older versions of the Engine services.") );
				Info->info[4].type               = EVMS_Type_String;
				Info->info[4].unit               = EVMS_Unit_None;
				Info->info[4].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
				Info->info[4].collection_type    = EVMS_Collection_None;
				memset( &Info->info[4].group, 0, sizeof(group_info_t));

				Info->info[5].name = EngFncs->engine_strdup( "Required Engine FSIM API Version" );
				Info->info[5].title = EngFncs->engine_strdup( _("Required Engine FSIM API Version") );
				Info->info[5].desc = EngFncs->engine_strdup( _("This is the version of the Engine FSIM API that this plug-in requires.  "
									       "It will not run on older versions of the Engine FSIM API.") );
				Info->info[5].type               = EVMS_Type_String;
				Info->info[5].unit               = EVMS_Unit_None;
				Info->info[5].value.s = EngFncs->engine_strdup( required_fsim_api_version_string );
				Info->info[5].collection_type    = EVMS_Collection_None;
				memset( &Info->info[5].group, 0, sizeof(group_info_t));

				*info = Info;

				rc = 0;
			} else {
				rc = ENOMEM;
			}

		} else {
			/* There is no more information on any of the descriptors. */
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * How much can file system expand?
 */
static int fs_can_expand_by(logical_volume_t * volume,
			    sector_count_t   * delta)
{
	int  rc = 0;

	LOG_ENTRY();
	if (jfsutils_support & JFS_EXPAND) {
		if (!EngFncs->is_mounted(volume->dev_node, NULL)) {
			/* If not mounted, can't expand. online only. */
			rc = EBUSY;
		} else if ( *delta > volume->max_fs_size - volume->fs_size) {
			*delta = volume->max_fs_size - volume->fs_size;
		}
	} else {
		rc = ENOSYS;
	}
	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * How much can file system shrink?
 */
static int fs_can_shrink_by(logical_volume_t * volume,
			    sector_count_t * delta)
{
	/* unsupported at this time */
	int  rc = ENOSYS;

	LOG_ENTRY();
	LOG_EXIT_INT(rc);
	return rc;
}

static fsim_functions_t fsim_ops = {
	.setup_evms_plugin	= fs_setup,
	.cleanup_evms_plugin	= fs_cleanup,
	.probe			= fs_probe,
	.can_mkfs		= fs_can_mkfs,
	.can_unmkfs		= fs_can_unmkfs,
	.can_fsck		= fs_can_fsck,
	.get_fs_size		= fs_get_fs_size,
	.get_fs_limits		= fs_get_fs_limits,
	.can_expand_by		= fs_can_expand_by,
	.can_shrink_by		= fs_can_shrink_by,
	.expand			= fs_expand,
	.shrink			= fs_shrink,
	.mkfs			= fs_mkfs_jfs,
	.mkfs_setup		= fs_mkfs_setup,
	.discard		= fs_discard,
	.fsck			= fs_fsck,
	.unmkfs			= fs_unmkfs_jfs,
	.unmkfs_setup		= fs_unmkfs_setup,
	.get_option_count	= fs_get_option_count,
	.init_task		= fs_init_task,
	.set_option		= fs_set_option,
	.set_volumes		= fs_set_volumes,
	.get_volume_info	= fs_get_volume_info,
	.get_plugin_info	= fs_get_plugin_info
};

plugin_record_t jfs_plugin_record = {
	.id = EVMS_JFS_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 11,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = EVMS_JFS_PLUGIN_SHORT_NAME,
	.long_name = EVMS_JFS_PLUGIN_LONG_NAME,
	.oem_name = EVMS_IBM_OEM_NAME,
	.functions = {
		.fsim = &fsim_ops
	},
	.container_functions = NULL
};

plugin_record_t *evms_plugin_records[] = {
	&jfs_plugin_record,
	NULL
};

