/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: defsegmgr.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "segs.h"
#include "checks.h"
#include "discovery.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "segoptions.h"
#include "dm.h"
#include "move.h"

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t segmgr_plugin_record;

plugin_record_t                *Seg_My_PluginRecord_Ptr=&segmgr_plugin_record;

struct engine_functions_s      *EngFncs=NULL;

/*
 * Called to revert back to original drive geometry that was
 * reported by the kernel. This is done when we have been
 * using an alternate geometry and have deleted ALL data segs
 * off the disk and can now revert to the proper drive geometry.
 */
static int revert_drive_geometry( LOGICALDISK *ld )
{
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	int                rc=EINVAL;
	DISKSEG           *mbr;
	DISKSEG           *freespace;
	DISKSEG           *tseg;
	int                metadata_count=0;
	int                freespace_count=0;
	int                data_count=0;
	list_element_t     iter;


	LOG_ENTRY();

	if (disk_pdata) {

		if ( ( disk_pdata->geometry.cylinders != ld->geometry.cylinders) ||
		     ( disk_pdata->geometry.heads     != ld->geometry.heads) ||
		     ( disk_pdata->geometry.sectors_per_track != ld->geometry.sectors_per_track) ) {

			LIST_FOR_EACH( ld->parent_objects, iter, tseg ) {

				if ( tseg->data_type == FREE_SPACE_TYPE ) {
					++freespace_count;
				} else if (tseg->data_type == META_DATA_TYPE ) {
					++metadata_count;
				} else if (tseg->data_type == DATA_TYPE) {
					++data_count;
				}

			}

			// test if we can revert the geometry
			if ( data_count > 0 ) {
				rc = EPERM;    // operation not permitted
				LOG_EXIT_INT(rc);
				return rc;
			}

			// ask the user if it is Ok
			{
				char * choices[] = {_("Yes"), _("No"), NULL};
				int answer = 0;	    // Initialize to Yes ... revert to kernel reported geometry

				QUESTION( &answer, choices,
					  "\nQuestion: The only remaining segment on drive %s is being deleted. Since we used an alternate "
					  "geometry for this drive, in order to get through segment discovery, you now have an opportunity "
					  "to revert back to using the drive geometry that was reported by the kernel.  The recommendation is "
					  "that you reply YES and revert back to the kernel reported geometry.\n\n"
					  "Do you want to revert back to using the kernel reported geometry?\n",
					  ld->name);

				if (answer == 1) { // if user replied NO
					rc = EPERM;    // then return ... operation not permitted
					LOG_EXIT_INT(rc);
					return rc;
				}

			}

			// revert old drive geometry
			memcpy(&disk_pdata->geometry, &ld->geometry, sizeof(geometry_t));

			// turn off LBA addressing flags
			disk_pdata->flags &= ~DISK_USES_LBA_ADDRESSING;
			disk_pdata->flags &= ~DISK_HAS_FORCED_LBA_ADDRESSING;

			// resize the mbr and freespace segments if needed
			mbr = get_mbr_from_seglist( ld->parent_objects );
			if ( ( mbr ) &&
			     ( mbr->size != disk_pdata->geometry.sectors_per_track ) &&
			     ( metadata_count == 1) &&
			     ( freespace_count == 1)) {

				freespace        = get_freespace_following_seg( mbr );

				mbr->size        = ld->geometry.sectors_per_track;

				freespace->start = mbr->size;
				freespace->size  = ld->size - mbr->size;

			}

			rc = 0;
		} else {
			rc = -1;  // non destructive error
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
static int SEG_SetupEVMSPlugin( engine_functions_t * engine_functions)
{
	int rc = EINVAL;
	EngFncs  = engine_functions;

	LOG_ENTRY();

	Disk_PrivateData_List = EngFncs->allocate_list();

	if ( Disk_PrivateData_List == NULL ) {
		rc = ENOMEM;
	} else {
		EngFncs->register_name( "/dev/evms/os2" ); // prevent anyone from building objects
		EngFncs->register_name( "/dev/evms/OS2" ); // with a name of OS2
		rc = 0;
	}

	LOG_EXIT_INT(rc);

	return rc;
}


static void SEG_Cleanup(void)
{
	list_anchor_t seglist=EngFncs->allocate_list();
	int rc;
	DISKSEG *seg;
	list_element_t iter;
	DISK_PRIVATE_DATA * disk_pdata;

	LOG_ENTRY();

	if (seglist != NULL) {

		// cleanup all seg private data areas
		rc = EngFncs->get_object_list( SEGMENT,
					       0,
					       Seg_My_PluginRecord_Ptr,
					       NULL,
					       0,
					       &seglist );

		LIST_FOR_EACH( seglist, iter, seg ) {

			if ( seg->data_type == META_DATA_TYPE ) {

				if ( ((SEG_PRIVATE_DATA *)seg->private_data)->dlat != NULL ) {
					free ( ((SEG_PRIVATE_DATA *)seg->private_data)->dlat );
				}

			}

			if (seg->private_data) free(seg->private_data);
		}

		EngFncs->destroy_list(seglist);
	}


	// cleanup all disk private data areas
	if ( Disk_PrivateData_List != NULL ) {
		LIST_FOR_EACH( Disk_PrivateData_List, iter, disk_pdata ) {
			free(disk_pdata);
		}
		EngFncs->destroy_list( Disk_PrivateData_List );
	}

	LOG_EXIT_VOID();
}




/*
 *  I will allow the object to be made into a volume (or reverted) if ...
 *
 *  (1) I actually own the object
 *  (2) Which means it is a segment and has necessarey ctl blocks
 *
 */
static int SEG_can_set_volume(storage_object_t * seg, boolean flag )
{
	int rc = EINVAL;
	LOGICALDISK *ld;
	DISK_PRIVATE_DATA *disk_pdata;

	LOG_ENTRY();

	if ( ( seg ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) ) {

		ld = get_logical_disk(seg);

		if (ld) {

			if (flag == TRUE) {  // CREATE VOLUME

				disk_pdata = get_disk_private_data(ld);

				if (disk_pdata) {

					rc = 0;

				}

			} else {      // REVERT VOLUME
				rc = 0;
			}
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  I can delete this segment if ...
 *
 *  (1) I own the segment
 *  (2) It is either a data segment or else an mbr segment and
 *      there are no data segments.
 *  (3) It isnt the first logical drive in an ebr chain
 */
static int SEG_CanDestroy( DISKSEG * seg )
{
	int rc = EINVAL;

	LOG_ENTRY();

	if ( ( seg ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) &&
	     ( disk_move_pending( seg ) == FALSE ) &&
	     ( i_can_modify_seg( seg ) == TRUE )) {

		rc = 0;

	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  I can expand the object if ...
 *
 *  (1) it is a data segment
 *  (2) I own the segment
 *  (3) the logical disk and segment info is Ok
 *  (4) a freespace segment immediately follows it
 *
 */
static int SEG_CanExpand( DISKSEG         *seg,		      // data segment
			  sector_count_t  expand_limit,      // ?
			  list_anchor_t          expansion_points ) // list to place expand object on
{
	DISKSEG              *freespace;
	LOGICALDISK          *ld;
	int                   rc = EINVAL;
	expand_object_info_t *expand_object;
	SEG_PRIVATE_DATA     *pdata;

	LOG_ENTRY();

	if ( ( expansion_points ) &&
	     ( seg ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) &&
	     ( disk_move_pending( seg ) == FALSE ) &&
	     ( i_can_modify_seg( seg ) == TRUE ) &&
	     ( seg_is_volitile(seg)==TRUE) ) {

		freespace = get_freespace_following_seg( seg );
		ld        = get_logical_disk(seg);
		pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

		if ( freespace && ld ) {

			// we can only expand in cylinder size amounts
			if ( freespace->size >= get_cylinder_size(ld) &&
			     expand_limit >= get_cylinder_size(ld)) {

				expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
				if (expand_object) {

					expand_object->object          = seg;
					expand_object->max_expand_size = min(freespace->size, expand_limit);

					if (EngFncs->insert_thing(expansion_points,expand_object,INSERT_BEFORE,NULL)) {
						rc = 0;
					} else {
						rc = EPERM;
						EngFncs->engine_free( expand_object );
					}

				} else {
					LOG_ERROR("\nerror, alloc of expand object failed\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;

}


/*
 * I can expand the object by size sectors if:
 *
 *  (1) i own the object
 *  (2) the segment can be expanded
 *  (3) the freespace has at least a cylinder of space
 *
 * If I cannot expand because the freespace is too small
 * then I'll reduce the expand sector count to my maximum.
 *
 */
static int SEG_CanExpandBy(storage_object_t * seg, sector_count_t *size)
{
	int                        rc = EINVAL;
	LOGICALDISK               *ld;
	DISKSEG                   *freespace;
	sector_count_t             SectorsPerCylinder;
	sector_count_t             max_expand_sectors=0;
	lba_t                      freespace_end_lba=0;
	SEG_PRIVATE_DATA          *pdata;


	LOG_ENTRY();

	if ( ( i_can_modify_seg(seg)==TRUE ) &&
	     ( disk_move_pending( seg ) == FALSE ) &&
	     ( seg_is_volitile(seg)==TRUE  )) {

		freespace = get_freespace_following_seg( seg );
		ld        = get_logical_disk(seg);
		pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

		if ( freespace && ld ) {

			SectorsPerCylinder = get_cylinder_size(ld);

			// partitions end on a cylinder boundary. if freespace doesnt end on
			// a cylinder bdy then round it down to a cyl bdy before testing if
			// freespace can handle the ExpandBy.
			if (ends_on_cylinder_boundary(ld, freespace->start+freespace->size-1) == TRUE) {
				freespace_end_lba = freespace->start + freespace->size - 1;
			} else {
				freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
			}

			// calculate the max useable size of the freespace area, i.e. the
			// max area that ends on a cylinder boundary.
			if (freespace_end_lba > freespace->start) {
				max_expand_sectors = freespace_end_lba - freespace->start + 1;
			} else {
				max_expand_sectors = 0;
			}

			// we expand in cylinder size chunks ... if max useable freespace
			// size is less than 1 cylinder then we cant do any expand at all.
			if (max_expand_sectors >= SectorsPerCylinder) {

				if ( max_expand_sectors >= *size ) {

					if ( max_expand_sectors == *size) {
						rc = 0;
					} else {
						freespace_end_lba = roundup_to_cylinder_boundary(ld, freespace->start + *size - 1 );
						*size      = freespace_end_lba - freespace->start + 1;
					}

				} else {
					*size = max_expand_sectors;
					rc = EINVAL;
				}

			}

		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}




/*
 * I can shrink a seg if ...
 *
 *  (1) i own the object
 *  (2) it is a data segment
 *  (3) if I chop off a cylinder, the seg will still have
 *      a minimum of 1 cylinder of space
 *
 *  If not exact set new_size to closest higher value possible.
 */
static int SEG_CanShrink( storage_object_t * seg,		// object to shrink
			  sector_count_t     shrink_limit,	// a delta size
			  list_anchor_t            shrink_points )    // of type shrink_object_info_t,
{
	int                   rc = EINVAL;
	sector_count_t        SectorsPerCylinder;
	LOGICALDISK          *ld;
	shrink_object_info_t *shrink_object=NULL;
	SEG_PRIVATE_DATA     *pdata;


	LOG_ENTRY();

	if ( ( seg ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type == DATA_TYPE ) &&
	     ( disk_move_pending( seg ) == FALSE ) &&
	     ( i_can_modify_seg( seg ) == TRUE ) &&
	     ( seg_is_volitile(seg)==TRUE )) {


		pdata = (SEG_PRIVATE_DATA *)seg->private_data;
		ld    = get_logical_disk(seg);

		if ( ld ) {

			SectorsPerCylinder = get_cylinder_size(ld);

			// see if caller is trying to shrink smaller than min partition size
			if ( seg->size > SectorsPerCylinder &&
			     shrink_limit >= SectorsPerCylinder) {

				if (shrink_points) {

					shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
					if (shrink_object) {

						// we are suppose to return the max amount we can shrink. since the
						// minimum partition size is 1 cylinder ... then anything more than
						// a cylinder is the max we can shrink.
						shrink_object->object          = seg;
						shrink_object->max_shrink_size = min(seg->size - SectorsPerCylinder, shrink_limit);

						if (EngFncs->insert_thing(shrink_points,shrink_object,INSERT_BEFORE,NULL)) {
							rc = 0;
						} else {
							LOG_ERROR("\nerror, inserting object into shrink points failed\n");
							rc = EPERM;
						}

					} else {
						LOG_ERROR("\nerror, failed to alloc shrink_object\n");
						rc = ENOMEM;
					}
				}
			}

		}
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  I can allow the storage object to shrink by the specified amount if ...
 *
 *  (1) the shrink point is a segment and I own it
 *  (2) the segment is large enough to allow a shrink
 *  (3) i can shrink it and end the segment on a cylinder boundary
 *
 *  If there are any problems then I'll return an error and
 *  report the amount that we could shrink by.
 *
 */
static int SEG_CanShrinkBy( storage_object_t * seg,
			    sector_count_t   * size )
{
	int               rc = EINVAL;
	sector_count_t    SectorsPerCylinder=0;
	LOGICALDISK      *ld=NULL;
	sector_count_t    delta=0;
	SEG_PRIVATE_DATA *pdata;

	LOG_ENTRY();

	if ( ( seg ) &&
	     ( size ) &&
	     (*size > 0 ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type == DATA_TYPE ) &&
	     ( disk_move_pending( seg ) == FALSE ) &&
	     ( i_can_modify_seg( seg ) == TRUE ) &&
	     ( seg_is_volitile(seg)==TRUE )) {


		pdata = (SEG_PRIVATE_DATA *)seg->private_data;
		ld    = get_logical_disk(seg);

		if ( ld ) {

			SectorsPerCylinder = get_cylinder_size(ld);

			// seg cant shrink if it is a cylinder or less in size
			if (seg->size > SectorsPerCylinder) {

				if ( *size < seg->size ) {

					if (*size < SectorsPerCylinder) {
						delta = SectorsPerCylinder;
					} else {
						delta = (*size / SectorsPerCylinder)*SectorsPerCylinder;
					}

				} else {
					delta = seg->size - SectorsPerCylinder;
				}

				if ( delta == *size  ) {
					rc = 0;
				} else {
					*size = delta;
					rc = EINVAL;
				}

			} else {
				*size =0;
				rc = EINVAL;
			}

		} else {
			*size = 0;
			rc = EINVAL;
		}

	} else {
		if (size != NULL) *size = 0;
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static boolean isa_discardable_ld( LOGICALDISK *ld )
{
	boolean result;

	LOG_ENTRY();

	result = EngFncs->list_empty( ld->parent_objects );
	
	LOG_EXIT_BOOL(result);
	return result;
}

/*
 * Forget about these objects.  Don't delete them.  Just clean up any
 * data structures you may have associated with them.  The Engine will
 * call to deactivate the objects durring commit.
*/
static int SEG_Discard(list_anchor_t objects)
{
	int rc=0;
	LOGICALDISK *ld;
	DISKSEG *seg;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH( objects, iter, seg ) {
		if (seg->plugin == Seg_My_PluginRecord_Ptr ) {

			ld = get_logical_disk(seg);
			if (ld) {
				remove_diskseg_from_list( ld->parent_objects, seg );
				EngFncs->remove_thing( seg->child_objects, ld );
				if (isa_discardable_ld(ld)==TRUE) {
					delete_disk_private_data( ld );
				}
			}
			if (seg->private_data) {
				free(seg->private_data);
				seg->private_data = NULL;
			}
			EngFncs->free_segment(seg);

		}
		else {
			rc = EINVAL;  // but continue !
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to run discovery code on list of evms objects that
 *  the engine has found so far.  We essentially need to walk the
 *  list of objects, looking for Logical Disks, and see if we
 *  recognize partitioning schemes.  If so, consume the logical
 *  disk by removing it from the list, and place all new segment
 *  objects on the output_object list. Any object we dont like in
 *  the input_object list must be copied to the output_object list.
 *
 */
static int SEG_Discover( list_anchor_t input_objects, list_anchor_t output_objects, boolean final_call)
{
	int  rc = 0;
	uint  count = 0;
	storage_object_t *object;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH( input_objects, iter, object) {
		rc = dos_segment_discovery( object, output_objects, &count );
	}

	// if we got a least a single object created ... return the count ...
	// else return the RC from the discovery code.
	if (count > 0) {
		rc = count;
	}

	LOG_EXIT_INT(rc);
	return rc;
}




/*
 *  Called with CREATE api for a DATA type object.  This means we assign
 *  ourselves to the disk by laying down our partition scheme, creating
 *  an MBR metadata segment and a FREESPACE segment.
 */
static int Assign_SegmentManager_ToDisk( LOGICALDISK *ld,
					 boolean      isa_os2_disk,
					 char        *disk_name )
{
	int                rc;
	DISKSEG           *freespace;
	DISK_PRIVATE_DATA *disk_pdata=NULL;

	LOG_ENTRY();

	// we should not have a private data area for this disk yet because we were just
	// told to install on it.  we must have been passed a logical disk we are already
	// using and this is very wrong.
	if ( get_disk_private_data(ld) ) {
		LOG_ERROR("attempting to reinstall on disk (%s)\n", ld->name);
		rc = EINVAL;
	} else {

		// create disk private data area
		rc = create_disk_private_data( ld );
		if (!rc) {
			disk_pdata = get_disk_private_data(ld);
			if (disk_pdata==NULL) {
				rc=ENOMEM;
			}
		}

		if (rc == 0) {

			// toss any items that may have been left in the parent list
			EngFncs->delete_all_elements(ld->parent_objects);

			// create freespace on the disk
			rc = find_freespace_on_disk( ld );
			if ( rc == 0) {

				// now get the freespace segment we just created
				freespace = get_first_freespace_seg_in_list( ld->parent_objects );
				if ( freespace ) {

					// create an MBR storage object for the new disk
					// it will be marked dirty for later commit.
					rc = create_mbr_For_Disk( ld, disk_name, isa_os2_disk );
					if (!rc) {

						// clear 1st track of any possible dlat or disk labels
						KILL_SECTORS( ld, 0, ld->geometry.sectors_per_track);

						disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
					} else {
						free_disk_segment(freespace);
						delete_disk_private_data( ld );
					}

				} else {
					rc = ENODEV;
					delete_disk_private_data( ld );
					LOG_ERROR("failed to create any freespace storage objects on disk %s\n", ld->name);
				}

			} else {
				delete_disk_private_data( ld );
				LOG_ERROR("unable to establish free space on disk %s\n", ld->name);
			}
		} else {
			LOG_ERROR("unable to malloc disk (%s) private data area\n", ld->name);
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}




/*  Called to create the specified segment.
 *
 *  - allocate memory for segment,
 *  - add segment to seg list of logical disk.
 *  - fill in seg info
 */
static int CreateDiskSegment( DISKSEG               * free_space,	    /* a free space disk segment      */
			      DISKSEG             * * new_segment,	    /* resulting data segment         */
			      sector_count_t          size,		    /* size in sectors                */
			      sector_count_t          sector_offset,	    /* sector offset in freespace seg */
			      u_int32_t               partition_type,	    /* type, e.g. 0x83=linux ext2     */
			      boolean                 Bootable,		    /* partition record active flag   */
			      boolean                 primary_partition,    /* TRUE if it must be primary     */
			      char                  * partition_name,	    /* for OS2 partitions             */
			      char                  * volume_name,	    /* for OS2 partitions             */
			      char                  * drive_letter )	    /* for OS2 partitions             */
{
	int                 rc=0;
	DISKSEG            *seg=NULL;
	SEG_PRIVATE_DATA   *seg_pdata;
	SEG_PRIVATE_DATA   *free_pdata;
	LOGICALDISK        *ld;
	DISK_PRIVATE_DATA  *disk_pdata;
	DISKSEG            *mbr;
	SEG_PRIVATE_DATA   *mbr_pdata=NULL;
	sector_count_t      seg_size;
	lba_t               seg_start_lba;
	lba_t               seg_end_lba;
	DLA_Entry           dla;

	sector_count_t      sizeof_freespace = free_space->size;
	lba_t               startof_freespace = free_space->start;

	boolean             create_seg_on_cylinder_boundary = FALSE;
	sector_count_t      cylsize;
	sector_count_t      trksize;


	LOG_ENTRY();


	/*
	 *  Get private data areas
	 */
	free_pdata = (SEG_PRIVATE_DATA *) free_space->private_data;
	ld         = get_logical_disk( free_space );
	disk_pdata = get_disk_private_data(ld);
	cylsize    = disk_pdata->geometry.heads * disk_pdata->geometry.sectors_per_track;
	trksize    = disk_pdata->geometry.sectors_per_track;

	/*
	 *  Check for an MBR metadata segment
	 *
	 *  The disk must have an MBR on it else we dont have any
	 *  partition table to start with.
	 *
	 */
	mbr = get_mbr_from_seglist( ld->parent_objects );
	if (mbr==NULL) {
		LOG_ERROR("disk has no MBR \n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	} else {
		mbr_pdata = (SEG_PRIVATE_DATA *)mbr->private_data;
		if (mbr_pdata==NULL) {
			LOG_ERROR("mbr segment is missing private data area\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
	}

	/*
	 *  Dump debug info to engine log
	 */
	LOG_DEBUG("New Seg Parms ...\n");
	LOG_DEBUG("         Size: %"PRIu64"\n", size);
	LOG_DEBUG("       Offset: %"PRIu64"\n", sector_offset );
	LOG_DEBUG("FreeSpace ...\n");
	LOG_DEBUG("    Start LBA: %"PRIu64"\n", free_space->start);
	LOG_DEBUG("         Size: %"PRIu64"\n", free_space->size);
	LOG_DEBUG("     Cyl Size: %"PRIu64"\n", get_cylinder_size(ld) );
	if (starts_on_cylinder_boundary( ld, free_space->start )==TRUE) {
		LOG_DEBUG("  Cyl Bdy: Yes ... starts on cylinder boundary\n" );
	} else {
		LOG_DEBUG("  Cyl Bdy: No ... does not start on cylinder boundary\n");
	}

	/*
	 * Determine the kind of allignment we should impose on the starting
	 * LBA of this partition. All partitions will end on a cylinder
	 * boundary.
	 *
	 * Use track allignment for starting LBA if ...
	 *
	 * (1) we are creating the 1st primary partition following the mbr
	 *     and it will start within the 1st cylinder on the drive.
	 *
	 * (2) we are creating a logical drive that will start at the very
	 *     beginning of an established extd partition
	 *
	 * Everything else will start on cylinder allignment.
	 *
	 */
	if ( ( primary_partition == TRUE ) &&
	     ( (free_space->start+sector_offset) < cylsize )) {

		create_seg_on_cylinder_boundary = FALSE;

	} else if ( ( primary_partition == FALSE ) &&
		    ( seg_is_within_or_adjacant_to_extended_partition( ld, free_space ) == TRUE ) ) {

		if ( (free_space->start+sector_offset >= disk_pdata->extd_partition_lba) &&
		     (free_space->start+sector_offset <  disk_pdata->extd_partition_lba+trksize)) {
			create_seg_on_cylinder_boundary = FALSE;
		} else {
			create_seg_on_cylinder_boundary = TRUE;
		}

	} else {
		create_seg_on_cylinder_boundary = TRUE;
	}

	/*
	 *  Start segment on a cylinder boundary if required
	 */
	if ( ( create_seg_on_cylinder_boundary == TRUE ) &&
	     ( starts_on_cylinder_boundary(ld, free_space->start+sector_offset) == FALSE)) {

		if ( sector_offset < cylsize ) {
			seg_start_lba   = roundup_to_cylinder_boundary(ld, free_space->start + sector_offset)+1;
		} else {
			seg_start_lba   = rounddown_to_cylinder_boundary(ld, free_space->start + sector_offset);
		}

	} else {
		seg_start_lba = free_space->start + sector_offset;
	}


	/*
	 *  End segment on a cylinder boundary
	 */
	seg_end_lba = seg_start_lba + size - 1;

	if ( ends_on_cylinder_boundary(ld, seg_end_lba) == FALSE ) {

		// if less than a cylinder ... round up to next cyl boundary
		if ( ((seg_end_lba - seg_start_lba + 1) / cylsize) == 0) {
			seg_end_lba = roundup_to_cylinder_boundary(ld, seg_end_lba);
		} else {
			seg_end_lba = rounddown_to_cylinder_boundary(ld, seg_end_lba) - 1;
		}

		// if we goofed somehow and rounded down to start of seg then fix it!
		if ( seg_start_lba >= seg_end_lba ) {
			seg_end_lba     = roundup_to_cylinder_boundary(ld, seg_start_lba+1);
		}

	}

	/*
	 *  Now get its actual size
	 */
	seg_size = seg_end_lba - seg_start_lba + 1;


	/*
	 * Test starting LBA
	 */
	if (  seg_start_lba < free_space->start ) {

		LOG_ERROR("error, with cylinder allignment, the new seg would start before the free_space segment\n");
		*new_segment = NULL;
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;

	}

	/*
	 *  Test ending LBA
	 */
	if (  seg_end_lba > (free_space->start + free_space->size -1) ) {

		LOG_ERROR("error, with cylinder allignment the new segment would run past end of free_space segment\n");
		*new_segment = NULL;
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;

	}


	/*
	 *  Dump debug info to engine log
	 */
	LOG_DEBUG("Create Parms:\n");
	LOG_DEBUG("FreeSpace: start_lba= %"PRIu64"  sector_offset= %"PRIu64"  size= %"PRIu64"  name= %s\n", free_space->start, sector_offset, free_space->size,free_space->name );
	LOG_DEBUG("   NewSeg: start_lba= %"PRIu64"  end_lba= %"PRIu64"  size= %"PRIu64"\n", seg_start_lba, seg_end_lba, seg_size);


	/*
	 *  Allocate memory for the new disk data segment (DISKSEG)
	 */
	seg = allocate_disk_segment(ld);
	if (seg==NULL) {
		*new_segment = NULL;
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	} else {
		seg_pdata = (SEG_PRIVATE_DATA *) seg->private_data;
	}

	/*
	 *  Now were ready to fill out the new DISKSEG
	 */
	seg->size                 = seg_size;
	seg->start                = seg_start_lba;

	if (Bootable) {
		seg->flags           |= SOFLAG_BIOS_READABLE  ;
		seg_pdata->boot_ind   = ACTIVE_PARTITION;
	}

	seg_pdata->sys_id         = partition_type;


	/*
	 *  Figure out if it will be added as a primary partition or
	 *  as a logical drive.
	 */
	if ( ( primary_partition == TRUE ) || ( Bootable == TRUE) ) {

		if ( seg_is_within_the_extended_partition( ld, seg ) == FALSE ) {

			seg_pdata->flags |= SEG_IS_PRIMARY_PARTITION;
			rc = 0;
		} else {
			rc = EINVAL;
			free_disk_segment(seg);
			*new_segment = NULL;
			LOG_ERROR("primary partition specified but it falls within an extended partition on the drive\n");
			LOG_EXIT_INT(rc);
			return rc;
		}

	} else {

		if ( ( disk_has_extended_partition( ld ) == FALSE ) ||
		     ( seg_is_within_or_adjacant_to_extended_partition( ld, seg ) == TRUE )) {

			seg_pdata->flags |= SEG_IS_LOGICAL_PARTITION;
			rc = 0;

		} else {
			rc = EINVAL;
			free_disk_segment(seg);
			*new_segment = NULL;
			LOG_ERROR("primary partition not specified but cannot add segment as a logical partition on the drive\n");
			LOG_EXIT_INT(rc);
			return rc;
		}

	}


	/*
	 *  If OS2 disk ... Build a DLAT entry for the partition
	 */
	if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

		memset(&dla, 0, sizeof(DLA_Entry));

		dla.Partition_Size           = seg->size;
		dla.Partition_Start          = seg->start;
		dla.Drive_Letter             = *drive_letter;
		dla.Partition_Serial_Number  = seg_gen_serial_number((u_int32_t)&dla.Partition_Serial_Number);
		if ( strlen(partition_name) > 0) {
			strncpy(&dla.Partition_Name[0], partition_name, PARTITION_NAME_SIZE);
		}
		if ( strlen(volume_name) > 0) {
			strncpy(&dla.Volume_Name[0], volume_name, VOLUME_NAME_SIZE);
		}

	}


	/*
	 *  Calculate new dimensions for the existing freespace segment
	 */
	if ( seg->start == free_space->start ) {
		free_space->size  -= seg->size;
		free_space->start += seg->size;
	} else {
		free_space->size   = seg->start - free_space->start;
	}


	LOG_DEBUG("New FreeSpace: start_lba= %"PRIu64" size= %"PRIu64"\n", free_space->start, free_space->size);

	/*
	 *  If the new segment is a logical partition then we need to create a
	 *  Logical Drive equivalent in our LIST, i.e. an EBR track, with
	 *  an EBR table and dlat, and a logical partition.
	 */
	if ( seg_pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

		rc = create_logical_partition( ld, seg, &dla, free_space, sector_offset );

	} else {

		rc = create_primary_partition( ld, seg, &dla );
	}


	if (rc) {
		free_space->size  = sizeof_freespace;
		free_space->start = startof_freespace;

		free_disk_segment(seg);
		*new_segment = NULL;
		LOG_EXIT_INT(rc);
		return rc;
	}


	/*
	 *  Remove the free_space seg from the list, if we used up the entire
	 *  segment for the new data segment.
	 */
	if ( free_space->size == 0 ) {
		LOG_DEBUG("removing freespace segment from disk list because we used it all up\n");
		remove_diskseg_from_list(ld->parent_objects, free_space );
		free_disk_segment(free_space);
	}

	find_freespace_on_disk( ld );  // re-examine the disk for any free space and expose it by
				       // hanging free space segments.

	mbr->flags |= SOFLAG_DIRTY;    // mark mbr dirty so we get called for a commit

	*new_segment = seg;	       // return the new data segment to the caller

	rc = 0;

	LOG_EXIT_INT(rc);
	return rc;
}



static int GetCreateOptions(  DISKSEG         * seg,
			      LOGICALDISK     * ld,
			      option_array_t  * options,
			      sector_count_t  * size,
			      lsn_t           * sector_offset,
			      u_int32_t       * type,
			      boolean         * bootable,
			      boolean         * primary_partition,
			      char            * partition_name,
			      char            * volume_name,
			      char            * drive_letter )
{
	int i;
	int rc = EINVAL;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	LOG_ENTRY();

	// check parms
	if ((ld==NULL)||(seg==NULL)||(options==NULL)||(disk_pdata==NULL)) {
		LOG_DEBUG("error, NULL ptr in parms\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// init the options before beginning
	*(drive_letter)=0x00;
	*(volume_name) =0x00;
	*(partition_name)=0x00;
	*primary_partition = FALSE;
	*bootable = FALSE;
	*type = 0;
	*sector_offset = 0;
	*size = 0;

	// pickup option values
	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case SEG_CREATE_OPTION_SIZE_INDEX:
				*size = options->option[i].value.ui64;
				break;

			case SEG_CREATE_OPTION_OFFSET_INDEX:
				*sector_offset = options->option[i].value.ui64;
				break;

			case SEG_CREATE_OPTION_TYPE_INDEX:
				*type = (u_int32_t) options->option[i].value.uc;
				break;

			case SEG_CREATE_OPTION_BOOTABLE_INDEX:
				*bootable = options->option[i].value.uc;
				break;

			case SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX:
				*primary_partition = options->option[i].value.b;
				break;

			case SEG_CREATE_OPTION_PARTITIONNAME_INDEX:
				strcpy( partition_name, options->option[i].value.s );
				break;

			case SEG_CREATE_OPTION_VOLUMENAME_INDEX:
				strcpy( volume_name, options->option[i].value.s );
				break;

			case SEG_CREATE_OPTION_DRIVELETTER_INDEX:
				strncpy(drive_letter,options->option[i].value.s,1);
				break;

			default:
				break;

			}

		} else {

			if (strcmp(options->option[i].name, SEG_CREATE_OPTION_SIZE_NAME) == 0) {
				*size = options->option[i].value.ui64;
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_OFFSET_NAME) == 0) {
				*sector_offset = options->option[i].value.ui64;
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_TYPE_NAME) == 0) {
				*type = options->option[i].value.ui32;
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_BOOTABLE_NAME) == 0) {
				*bootable = options->option[i].value.uc;
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_PRIMARY_PARTITION_NAME ) == 0) {
				*primary_partition = options->option[i].value.uc;
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_PARTITIONNAME_NAME ) == 0) {
				strcpy( partition_name, options->option[i].value.s );
			} else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_VOLUMENAME_NAME ) == 0) {
				strcpy( volume_name, options->option[i].value.s );
			} else if (strcmp(options->option[i].name,SEG_CREATE_OPTION_DRIVELETTER_NAME) == 0) {
				strncpy(drive_letter,options->option[i].value.s,1);
			}

		}
	}


	LOG_DEBUG("Create Options ...\n");
	LOG_DEBUG("          seg name: %s\n", partition_name);
	LOG_DEBUG("              size: %"PRIu64"\n", *size);
	LOG_DEBUG("     sector offset: %"PRIu64"\n", *sector_offset);
	LOG_DEBUG("              type: %d\n", *type);
	if (*bootable==TRUE)
		LOG_DEBUG("     make bootable: YES\n");
	else
		LOG_DEBUG("     make bootable: NO\n");
	if (*primary_partition==TRUE)
		LOG_DEBUG("      make primary: YES\n");
	else
		LOG_DEBUG("      make primary: NO\n");
	LOG_DEBUG("      drive letter: %s\n", drive_letter );
	LOG_DEBUG("       volume name: %s\n", volume_name );


	// test results
	if ( (*size > 0) &&
	     (*sector_offset >= 0) &&
	     ( (*size + *sector_offset) <= seg->size ) &&
	     (*type != 0 ) &&
	     ((*bootable == TRUE)||(*bootable==FALSE)) &&
	     ((*primary_partition==TRUE)||(*primary_partition==FALSE))) {

		// OS2 partitions must have a partition name
		if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

			if ( strlen(partition_name) > 0 ) {
				rc = 0;
			}
		} else {
			rc = 0;
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}

static int GetAssignOptions(  option_array_t * options,  char  * disk_name , boolean *isa_os2_disk )
{
	int i;
	int rc = EINVAL;


	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case SEG_ASSIGN_OPTION_TYPENAME_INDEX:
				if ( strlen(options->option[i].value.s) ) {

					if ( strncmp(options->option[i].value.s, "OS/2",4) == 0) {
						*isa_os2_disk = TRUE;
					} else {
						*isa_os2_disk = FALSE;
					}
					rc =0;
				}
				break;


			case SEG_ASSIGN_OPTION_DISKNAME_INDEX:
				if ( ( strlen(options->option[i].value.s) > 0 ) &&
				     ( strlen(options->option[i].value.s) <= DISK_NAME_SIZE )) {

					strcpy( disk_name, options->option[i].value.s );
					rc = 0;
				} else {
					rc = EINVAL;
				}
				break;

			default:
				break;

			}

		} else {

			if (strcmp(options->option[i].name,SEG_ASSIGN_OPTION_TYPENAME_NAME) == 0) {

				if ( strlen(options->option[i].value.s) ) {

					if ( strncmp(options->option[i].name, "OS/2",4) == 0) {
						*isa_os2_disk = TRUE;
					} else {
						*isa_os2_disk = FALSE;
					}
					rc =0;
				}

			}

			if (strcmp(options->option[i].name,SEG_ASSIGN_OPTION_DISKNAME_NAME) == 0) {

				if ( ( strlen(options->option[i].value.s) > 0 ) &&
				     ( strlen(options->option[i].value.s) <= DISK_NAME_SIZE )) {
					strncpy( disk_name, options->option[i].value.s, DISK_NAME_SIZE );
					rc = 0;
				} else {
					rc = EINVAL;
				}

				break;
			}

		}

	}

/*
    LOG_DEBUG("Assign Options ... rc= %d\n", rc);
    LOG_DEBUG("          disk type: %d\n", *isa_os2_disk);
    LOG_DEBUG("          disk name: %s\n", disk_name);
*/

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Create storage_object_t(s) from the list of objects using the given
 * options.  Return the newly allocated storage_object_t(s) in new_objects
 * list.
 */
static int SEG_CreateSegment( list_anchor_t          input_objects,
			      option_array_t * options,
			      list_anchor_t          new_objects)
{
	int                 rc=EINVAL;
	uint                object_count=0;

	DISKSEG            *free_space_seg;
	DISKSEG            *newseg;

	//  Options ...
	sector_count_t      size;
	sector_count_t      sector_offset;
	u_int32_t           type;
	boolean             bootable;
	boolean             primary_partition;
	boolean             created_disk = FALSE;
	char                partition_name[EVMS_NAME_SIZE+1];
	char                volume_name[EVMS_VOLUME_NAME_SIZE+1];
	char                drive_letter[16];
	LOGICALDISK        *ld=NULL;
	DISK_PRIVATE_DATA  *disk_pdata=NULL;


	LOG_ENTRY();

	// we should only be called with a single input object
	object_count = EngFncs->list_count( input_objects );

	if (object_count==1) {

		free_space_seg = EngFncs->first_thing( input_objects,NULL );

		if (free_space_seg) {

			if ( ( i_can_modify_seg(free_space_seg) == TRUE ) &&
			     ( free_space_seg->data_type == FREE_SPACE_TYPE )) {

				ld         = get_logical_disk( free_space_seg);
				disk_pdata = get_disk_private_data(ld);

				if (ld && disk_pdata) {
					rc = GetCreateOptions( free_space_seg,
							       ld,
							       options,
							       &size,
							       &sector_offset,
							       &type,
							       &bootable,
							       &primary_partition,
							       partition_name,
							       volume_name,
							       drive_letter );
				} else {
					rc = EINVAL;
				}

				if (rc) {
					LOG_ERROR("error, invalid create options\n");
					LOG_EXIT_INT(rc);
					return EINVAL;
				}

				rc = CreateDiskSegment( free_space_seg,
							&newseg,
							size,
							sector_offset,
							type,
							bootable,
							primary_partition,
							partition_name,
							volume_name,
							drive_letter );

				if (rc ==0) {
					if (EngFncs->insert_thing(new_objects,newseg,INSERT_BEFORE,NULL)) {
						rc = 0;
					} else {
						rc = EPERM;
					}
				}

			} else {
				LOG_ERROR("object, to be consumed by create, has the wrong data_type\n");
				rc = EINVAL;
			}

		}
	} else {
		LOG_ERROR("expected 1 object in the input list but found %d\n", object_count);
		rc = EINVAL;
	}


	// free disk private data if we just created the disk but failed with the segment create
	if ( (rc) && (created_disk==TRUE) ) {
		delete_disk_private_data( ld );
	}

	// if successful mark changes pending on the disk
	if (!rc) {
		disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int SEG_Assign( storage_object_t * input_object,
		       option_array_t * options)
{

	int                 rc;
	char                disk_name[EVMS_NAME_SIZE+1];
	boolean             isa_os2_disk=FALSE;
	LOGICALDISK        *ld = (LOGICALDISK *) input_object;

	LOG_ENTRY();

	// see if we are installing on a DISK and therefore are assigning the segment
	// manager to the DISK ...
	// added checks for DATA SEGS for s390 work. we consume data segs on s390 so as
	// to provide an msdos partitioning capability ... yuck.
	if ( ( ld->object_type == DISK )||
	     ( ld->object_type == SEGMENT && ld->data_type == DATA_TYPE )) {

		// get disk assign options
		rc = GetAssignOptions( options, disk_name, &isa_os2_disk );
		if (rc) {
			LOG_ERROR("invalid options\n");
			rc = EINVAL;
		} else {
			rc = Assign_SegmentManager_ToDisk( ld,
							   isa_os2_disk,
							   disk_name );
		}

	} else {
		LOG_ERROR("object, to be consumed by assign, has the wrong data_type\n");
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to test if we can unassign from the specified
 *  object.
 */
static int SEG_CanUnassign( storage_object_t *object )
{
	int rc=EINVAL;

	LOG_ENTRY();

	if ( i_can_modify_disk( object ) == TRUE ) {

		rc = 0;

	}

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Called to remove the msdos partitioning scheme from the
 *  specified object.
 */
static int SEG_UnAssign( LOGICALDISK *ld )
{
	int rc=EINVAL;
	DISKSEG *seg;
	list_element_t iter,iter2;
	SEG_PRIVATE_DATA *pdata;


	LOG_ENTRY();

	// commit changes if we own the segments on the disk
	if ( i_can_modify_disk( ld ) == TRUE ) {

		LIST_FOR_EACH_SAFE( ld->parent_objects, iter, iter2, seg ) {
			pdata = seg->private_data;

			if (pdata->flags & SEG_IS_EBR) {

				// blast EBR sector
				KILL_SECTORS( ld, seg->start, 1);

				// blast possible DLAT sector
				KILL_SECTORS( ld, roundup_to_track_boundary(ld, seg->start), 1);

			}

			free_disk_segment(seg);

			EngFncs->delete_element(iter);
		}

		// blast MBR sector
		KILL_SECTORS( ld, 0, 1);

		// blast DLAT sector
		KILL_SECTORS( ld,ld->geometry.sectors_per_track-1,1);

		// toss our disk private data area
		delete_disk_private_data( ld );

		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to free an embedded data segment.
 */
static int SEG_DestroyEmbeddedSegment( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata, DISKSEG * seg )
{
	int rc=EINVAL;
	DISKSEG *container_seg=NULL;
	DISKSEG *freespace=NULL;
	uint     segcount;


	LOG_ENTRY();

	// get container seg that was consumed by the embedded
	// partition segments.
	container_seg = only_child(seg);

	if (container_seg) {
		LOG_DEBUG("container seg is %s\n", container_seg->name );
	} else {
		LOG_DEBUG("container seg not found\n");
	}

	if ( container_seg ) {

		// ask the user if it is Ok
		{
			char * choices[] = {_("Yes"), _("No"), NULL};
			int answer = 0;	    // Initialize to Yes

			QUESTION( &answer, choices,
				  "\n\nYou are about to destroy a segment that was discovered in an embedded partition.  "
				  "This segment can be destroyed and removed from the embedded partition table, however, new \n"
				  "segments cannot be created within the embedded partition table.\n\n"
				  "The reason you may wish to destroy embedded segments is so that you can recover the primary \n"
				  "partition within which the embedded segments are found.  This is accomplished by destroying all of \n"
				  "the embedded segments that consumed the primary partition.  Once they are destroyed, the primary \n"
				  "partition will become a top most object and available for use by EVMS.\n\n"
				  "Do you want to continue and destroy the embedded segment?\n");

			if (answer == 1) {
				rc = EINVAL;
				LOG_EXIT_INT(rc);
				return rc;
			}

		}

		// remove embedded partition segment from disk seg list and unregister its name
		rc = remove_diskseg_from_list( ld->parent_objects, seg );

		if (rc == 0) {

			// remove embedded partition from the container segment
			EngFncs->remove_thing( container_seg->parent_objects, seg );

			// reduce count of embedded partitions on this drive.
			--disk_pdata->embedded_partition_count;

			// if all embedded partitions are deleted from the
			// container seg then we can revert the container seg
			segcount = EngFncs->list_count( container_seg->parent_objects );

			if (segcount==0) {

				// we are about to insert the primary segment back into
				// the disk seglist. however, a freespace sement may be
				// occupying the area in which we need to insert the
				// data segment that represents the primary partition.
				// the solution is to remove freespace segments from
				// the disk prior to doing the insert.
				while ( (freespace=get_first_freespace_seg_in_list(ld->parent_objects))!=NULL) {

					remove_diskseg_from_list( ld->parent_objects, freespace );

					free_disk_segment(freespace);

				}

				// remove container seg from the container seglist for the disk
				revert_container_segment( container_seg );

				// and blast the front of the partition to remove the disk label
				KILL_SECTORS( ld, container_seg->start, 2);

			}

			// actually free embedded partition storage object
			free_disk_segment(seg);

			// rename the logical partitions
			fixup_logical_partition_names(ld);

			// reclaim any freespace on the disk
			find_freespace_on_disk( ld );

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called to free a data segment.  If the corresponding disk partition
 *  is a logical drive then we also need to free the EBR segment
 *  as well.
 */
static int SEG_DestroySegment( DISKSEG * seg, list_anchor_t child_objects )
{
	int                        rc = EINVAL;
	SEG_PRIVATE_DATA          *pdata;
	LOGICALDISK               *ld=NULL;
	DISK_PRIVATE_DATA         *disk_pdata=NULL;
	storage_object_t          *mbr = NULL;


	LOG_ENTRY();

	if ( ( seg ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) &&
	     ( i_can_modify_seg( seg ) == TRUE)) {

		ld         = get_logical_disk(seg);
		disk_pdata = get_disk_private_data(ld);
		pdata      = (SEG_PRIVATE_DATA *) seg->private_data;
		mbr        = get_mbr_from_seglist(ld->parent_objects);

		LOG_DEBUG("seg: %s\n", seg->name );

		// test private data ptrs
		if ( pdata==NULL || disk_pdata==NULL ) {
			LOG_EXIT_INT(rc);
			return rc;
		}

		// handle special case of embedded segment
		if ( pdata->flags & SEG_IS_EMBEDDED ) {

			rc = SEG_DestroyEmbeddedSegment( ld, disk_pdata, seg );

			if (!rc) {
				disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
				mbr->flags |= SOFLAG_DIRTY;
			}

			LOG_EXIT_INT(rc);
			return rc;
		}

		// remove segment from list and unregister its name
		rc = remove_diskseg_from_list( ld->parent_objects, seg );
		if (rc == 0) {

			if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

				// remove EBR segment from list and unregister its name
				rc = remove_diskseg_from_list( ld->parent_objects, pdata->ebr );

				if (rc == 0) {

					// make sure to fixup the ebr chain
					fixup_EBR_Chain( ld );

					// zap the ebr sector
					KILL_SECTORS( ld, pdata->ebr->start, 1);

					// zap dlat sector
					KILL_SECTORS( ld, roundup_to_track_boundary(ld,pdata->ebr->start), 1);

					// free memory
					free_disk_segment(pdata->ebr);

					// rename the logical partitions
					fixup_logical_partition_names(ld);
				} else {

					// error removing ebr ... put seg back on list
					insert_diskseg_into_list( ld->parent_objects, seg );
					fixup_EBR_Chain(ld);
				}

			}

			if (rc == 0) {

				// release the dlat entry if this is an OS2 disk
				if ( pdata->dla_entry != NULL) memset( pdata->dla_entry, 0, sizeof(DLA_Entry));

				// free actual storage object memory
				free_disk_segment(seg);

				// merge free space on disk
				find_freespace_on_disk( ld );

			}

		}

	}

	// if we have removed all the data segments from this disk
	// then we'll revert back to the original drive geometry in
	// case we were using an alternate geometry due to bogus
	// partitioning by some tool.
	if (rc == 0) {

		rc = revert_drive_geometry(ld);
		if (rc == 0) {
			char number_buffer[64];

			// tell the user
			sprintf(number_buffer, "%"PRIu64, disk_pdata->geometry.cylinders);
			MESSAGE( _("Information: reverting to kernel reported geometry for drive %s.\n"
				   "The new geometry is C= %s H= %d S= %d\n"),
				 ld->name,
				 number_buffer,
				 disk_pdata->geometry.heads,
				 disk_pdata->geometry.sectors_per_track );
		}

		rc = 0;
	}

	// mark mbr dirty
	if (!rc) {
		disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
		mbr->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return  rc;
}




static void GetExpandOptions( option_array_t * options,  sector_count_t  * size)
{
	int i;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			if (options->option[i].number == SEG_EXPAND_OPTION_SIZE_INDEX) {
				*size = options->option[i].value.ui64;
			}

		} else {

			if (strcmp(options->option[i].name, SEG_EXPAND_OPTION_SIZE_NAME) == 0) {
				*size = options->option[i].value.ui64;
			}

		}
	}


	LOG_EXIT_VOID();
}


/*
 *  Called to expand a data segment.  The segment will be expanded
 *  into the freespace segment that follows the data segment.
 */
static int SEG_Expand( DISKSEG *seg, DISKSEG *expand_object, list_anchor_t  objects, option_array_t *options )
{
	int               rc = EINVAL;
	sector_count_t    expand_sectors=0;
	DISKSEG          *freespace;
	lba_t             end_lba;
	sector_count_t    old_seg_size;
	LOGICALDISK      *ld=NULL;
	SEG_PRIVATE_DATA *pdata=NULL;
	sector_count_t    SectorsPerCylinder=0;
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	LOG_ENTRY();


	// initial checks to see if we can do the expand
	if ( ( seg ) &&
	     ( seg == expand_object ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) &&
	     ( i_can_modify_seg( seg ) == TRUE ) &&
	     ( seg_is_volitile(seg)==TRUE ) ) {

		pdata      = (SEG_PRIVATE_DATA *)seg->private_data;
		ld         = get_logical_disk(seg);
		disk_pdata = get_disk_private_data(ld);
		freespace  = get_freespace_following_seg( seg );

		GetExpandOptions( options,  &expand_sectors);

		if ( freespace && ld && disk_pdata && expand_sectors  ) {

			LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
			LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
			LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
			LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
			LOG_DEBUG("              Start: %"PRIu64"\n", freespace->start);
			LOG_DEBUG("               Size: %"PRIu64"\n", freespace->size);

			SectorsPerCylinder = get_cylinder_size(ld);

			// just in case ...
			if ( freespace->size < SectorsPerCylinder ) {
				LOG_ERROR("error, trying to expand into free space that is less than 1 cylinder\n");
				LOG_EXIT_INT(rc);
				return rc;
			}

			// you just never know ...
			if ( expand_sectors > freespace->size ) {
				expand_sectors = freespace->size;
			}

			// we expand in cylinder size chunks
			if (expand_sectors < SectorsPerCylinder) {
				expand_sectors = SectorsPerCylinder;
			} else {
				expand_sectors = (expand_sectors/SectorsPerCylinder)*SectorsPerCylinder;
			}

			// do cylinder alignment
			end_lba = seg->start + seg->size + expand_sectors - 1;

			if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
				end_lba = roundup_to_cylinder_boundary( ld, end_lba );
			}

			// now adjust downwards if too big for freespace ...
			if ( end_lba > (freespace->start + freespace->size - 1) ) {
				end_lba = rounddown_to_cylinder_boundary(ld, end_lba-1 ) - 1;
			}


			// test if we can expand the data seg into the freespace area
			if ( ( end_lba > freespace->start ) &&
			     ( end_lba <= freespace->start+freespace->size - 1) ) {

				// calc actual expand sector count
				expand_sectors = end_lba - freespace->start + 1;

				// ask the engine if this expand amount is ok.
				rc = EngFncs->can_expand_by(seg, &expand_sectors);
				if (rc) {
					LOG_ERROR("Shrink of segment %s rejected by "
						  "the engine.\n", seg->name);
					LOG_EXIT_INT(rc);
					return rc;
				}

				// expand the data segment
				old_seg_size = seg->size;
				seg->size   += expand_sectors;

				// shrink the freespace segment
				freespace->start  += expand_sectors;
				freespace->size   -= expand_sectors;

				// success.
				rc = 0;

				// if we used up all the free space then discard the freespace seg
				if ( freespace->size == 0 ) {

					rc = remove_diskseg_from_list( ld->parent_objects, freespace );
					if (rc==0) {
						free_disk_segment(freespace);
					} else {   // error ... backout the expand
						LOG_ERROR("error, unable to remove the freespace segment from the disk list\n");
						seg->size = old_seg_size;
						freespace->start  -= expand_sectors;
						freespace->size   += expand_sectors;
					}

				}
			}
		}
	}


	if (!rc) {
		disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
		if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) fixup_disk_extd_partition_dimensions( ld );
		seg->flags |= SOFLAG_DIRTY;
		if (seg->flags & SOFLAG_ACTIVE) {
			seg->flags |= SOFLAG_NEEDS_ACTIVATE;
		}
		
	}


	LOG_EXIT_INT(rc);
	return rc;
}


static void GetShrinkOptions( option_array_t * options, sector_count_t * size)
{
	int i;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			if (options->option[i].number == SEG_SHRINK_OPTION_SIZE_INDEX) {
				*size = options->option[i].value.ui64;
			}

		} else {

			if (strcmp(options->option[i].name, SEG_SHRINK_OPTION_SIZE_NAME) == 0) {
				*size = options->option[i].value.ui64;
			}
		}

	}

	LOG_EXIT_VOID();
}


/*
 *  Called to shrink a data segment to new_size or next smaller increment.
 *  Then, update appropriate fields in the segment and make
 *  changes to freespace segments as needed.
 *
 */
static int SEG_Shrink( storage_object_t * seg,
		       storage_object_t * shrink_object,
		       list_anchor_t            objects,
		       option_array_t   * options )
{
	int               rc = EINVAL;
	sector_count_t    shrink_sector_count=0;
	u_int64_t         end_lba;
	LOGICALDISK      *ld=NULL;
	sector_count_t    new_seg_size=0;
	SEG_PRIVATE_DATA *pdata=NULL;
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	sector_count_t    SectorsPerCylinder=0;

	LOG_ENTRY();

	// initial checks to see if we can do the shrink
	if ( ( seg ) &&
	     ( seg == shrink_object ) &&
	     ( seg->object_type == SEGMENT ) &&
	     ( seg->data_type   == DATA_TYPE ) &&
	     ( i_can_modify_seg( seg ) == TRUE ) &&
	     ( seg_is_volitile(seg)==TRUE ) ) {

		pdata      = (SEG_PRIVATE_DATA *)seg->private_data;
		ld         = get_logical_disk(seg);
		disk_pdata = get_disk_private_data(ld);

		GetShrinkOptions( options,  &shrink_sector_count);


		if ( (ld != NULL) &&
		     (disk_pdata != NULL) &&
		     (shrink_sector_count > 0) &&
		     (shrink_sector_count < seg->size ) ) {

			LOG_DEBUG("     Data Seg  Name: %s\n",   seg->name);
			LOG_DEBUG("              Start: %"PRIu64"\n", seg->start);
			LOG_DEBUG("               Size: %"PRIu64"\n", seg->size);
			LOG_DEBUG("Shrink Sector Count: %"PRIu64"\n", shrink_sector_count );

			SectorsPerCylinder = get_cylinder_size(ld);

			// we shrink in cylinder size chunks
			if (shrink_sector_count < SectorsPerCylinder) {
				shrink_sector_count = SectorsPerCylinder;
			} else {
				shrink_sector_count = (shrink_sector_count/SectorsPerCylinder)*SectorsPerCylinder;
			}

			// ask the engine if it's ok to shrink by this amount.
			rc = EngFncs->can_shrink_by(seg, &shrink_sector_count);
			if (rc) {
				LOG_ERROR("Shrink of segment %s rejected by "
					  "the engine.\n", seg->name);
				LOG_EXIT_INT(rc);
				return rc;
			}

			// resulting seg size
			new_seg_size = seg->size - shrink_sector_count;

			// make sure it ends on cylinder boundary
			end_lba = seg->start + new_seg_size - 1;
			if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
				end_lba = rounddown_to_cylinder_boundary( ld, end_lba ) - 1;
			}

			if ( end_lba >= (seg->start + seg->size - 1) ) {
				end_lba = rounddown_to_cylinder_boundary(ld, end_lba ) - 1;
			}

			// final test if we can shrink the segment
			if (  ( end_lba > seg->start ) &&
			      ( end_lba < (seg->start+seg->size-1)) ) {

				// actual new seg size
				new_seg_size = end_lba - seg->start + 1;

				// shrink the data segment
				seg->size = new_seg_size;

				// success.
				rc = 0;

			}

			if (!rc) {

				// mark disk has changes for commit code
				disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;

				// fixup logical drive info
				if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) fixup_disk_extd_partition_dimensions( ld );

				// collect any freespace we released
				find_freespace_on_disk( ld );

				// mark seg dirty for commit
				seg->flags |= SOFLAG_DIRTY;
				if (seg->flags & SOFLAG_ACTIVE) {
					seg->flags |= SOFLAG_NEEDS_ACTIVATE;
				}

			}

		} else {
			LOG_ERROR("error, something wrong with shrink sector count, cant shrink segment\n");
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("error, something wrong with parms\n");
		rc = EINVAL;
	}


	LOG_EXIT_INT(rc);

	return rc;
}



static int SEG_AddSectorsToKillList( DISKSEG *seg, lsn_t lsn, sector_count_t count)
{
	int                         rc = EINVAL;
	LOGICALDISK                *ld;
	struct plugin_functions_s  *Fncs;

	LOG_ENTRY();

	if ( lsn + count > seg->size ) {
		rc = EINVAL;
	} else {

		ld = get_logical_disk(seg);

		if (ld) {

			Fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

			rc   = Fncs->add_sectors_to_kill_list( ld, seg->start+lsn, count);
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int SEG_CommitChanges( storage_object_t *obj, uint phase )
{
	int          rc = 0;
	LOGICALDISK *ld=NULL;
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	SEG_PRIVATE_DATA *pdata=(SEG_PRIVATE_DATA *)obj->private_data;
	DISKSEG     *clean_seg;
	DISKSEG     *kseg;
	list_element_t iter, iter2;


	LOG_ENTRY();
	LOG_DEBUG("object= %s  commit phase= %d\n", obj->name, phase );

	// get logical disk
	if (obj->object_type == DISK) {
		ld = obj;
	} else if ( obj->object_type == SEGMENT ) {
		ld = get_logical_disk(obj);
	}

	// commit changes if we own the segments on the disk
	if ( i_can_modify_disk( ld ) == TRUE ) {

		disk_pdata = get_disk_private_data(ld);

		// are there any deactivate objects on this disk?
		if ( disk_pdata->flags & DISK_HAS_DEACTIVATE_OBJECTS ) {

			LOG_DEBUG("walking deactivate object list\n");

			LIST_FOR_EACH_SAFE( disk_pdata->deactivate_object_list, iter, iter2, kseg ) {
				EngFncs->dm_deactivate(kseg);
				if (kseg->private_data)	free(kseg->private_data);
				free(kseg);
				EngFncs->delete_element( iter );
			}

			disk_pdata->flags &= ~DISK_HAS_DEACTIVATE_OBJECTS;
			rc = 0;
		}

		if (phase == MOVE) {
			if ( disk_pdata->flags & DISK_HAS_MOVE_PENDING ) {
				LOG_DEBUG("committing move on the disk\n");
				rc = dos_move_segment_commit( obj, pdata->move_target, disk_pdata->copy_job );
				if (disk_pdata->copy_job) free(disk_pdata->copy_job );
				disk_pdata->copy_job = NULL;
				disk_pdata->flags   &= ~DISK_HAS_MOVE_PENDING;
			} else {
				rc = 0;
			}

		} else if ( phase==FIRST_METADATA_WRITE || phase==SECOND_METADATA_WRITE ) {

			if ( obj->flags & SOFLAG_DIRTY  &&
			     disk_pdata->flags & DISK_HAS_CHANGES_PENDING &&
			     !(disk_pdata->flags & DISK_HAS_MOVE_PENDING) ) {

				rc = Commit_Disk_Partition_Tables(ld, obj, FALSE);

				// if successful ... mark all segments on the disk clean
				if (rc == 0) {

					LIST_FOR_EACH( ld->parent_objects, iter, clean_seg ) {
						pdata = (SEG_PRIVATE_DATA *)clean_seg->private_data;
						if (pdata) {
							if (!(pdata->flags & SEG_IS_MOVE_TARGET)) {
								clean_seg->flags &= ~SOFLAG_DIRTY;
							}
						}
					}

					disk_pdata->flags &= ~DISK_HAS_CHANGES_PENDING;
				}

			} else {
				rc = 0;
			}

		} else {
			rc = 0;
		}

	} else {
		rc = EINVAL;
	}


	LOG_EXIT_INT(rc);
	return rc;
}


static int SEG_BackupMetadata(DISKSEG *seg)
{
	LOGICALDISK *ld = get_logical_disk(seg);
	int rc = 0;

	LOG_ENTRY();

	if (i_can_modify_disk(ld) == TRUE &&
	    seg->data_type != FREE_SPACE_TYPE) {
		rc = Commit_Disk_Partition_Tables(ld, seg, TRUE);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int SEG_Read( DISKSEG        *seg,
		     lsn_t           offset,
		     sector_count_t  count,
		     void           *buffer )
{
	int                         rc = ENODEV;
	LOGICALDISK                *ld;
	struct plugin_functions_s  *fncs;

	LOG_ENTRY();

	if ( offset + count > seg->size ) {
		rc = EINVAL;
	} else {

		ld = get_logical_disk(seg);

		if (ld) {

			fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

			rc   = fncs->read( ld, seg->start+offset, count, buffer);
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int SEG_Write( DISKSEG        *seg,
		      lsn_t           offset,
		      sector_count_t  count,
		      void           *buffer )
{
	int                         rc = ENODEV;
	LOGICALDISK                *ld;
	struct plugin_functions_s  *fncs;


	LOG_ENTRY();

	if ( offset + count > seg->size ) {
		rc = EINVAL;
	} else {

		ld = get_logical_disk(seg);

		if (ld) {

			fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

			rc   = fncs->write( ld, seg->start+offset, count, buffer);
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * This call notifies you that your object is being made into (or part of)
 * a volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or
 * removed (FALSE).
 */
static void SEG_set_volume(storage_object_t * object, boolean flag)
{
	return;
}


/*
 * Execute the private action on the object.
 */
static  int SEG_plugin_function( storage_object_t * object,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options)
{
	int rc=EINVAL;
	DISKSEG *freespace=NULL;


	LOG_ENTRY();

	switch (action) {
	
	case EVMS_Task_Dos_Move_Segment:

		if ( EngFncs->list_count( objects ) == 1 ) {

			freespace = EngFncs->first_thing(objects,NULL);

			if (freespace) {
				rc = dos_move_segment(object,freespace);
			}

		}
		break;


	default:
		rc = ENOSYS;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Return an array of private actions that you support for this object.
 */
static int SEG_get_plugin_functions( storage_object_t        * object,
				     function_info_array_t * * actions)
{
	int                    rc = EINVAL;
	function_info_array_t *func_info=NULL;

	LOG_ENTRY();


	func_info = EngFncs->engine_alloc( sizeof(function_info_array_t) + sizeof(function_info_t) );
	if (func_info) {

		func_info->count = 0;

		rc = dos_can_move_segment( object );
		if (!rc) {

			func_info->count = 1;

			func_info->info[SEG_MOVE_OPTION_INDEX].function = EVMS_Task_Dos_Move_Segment;

			func_info->info[SEG_MOVE_OPTION_INDEX].title = EngFncs->engine_strdup( "Move" );
			func_info->info[SEG_MOVE_OPTION_INDEX].verb = EngFncs->engine_strdup( _("Move") );
			func_info->info[SEG_MOVE_OPTION_INDEX].name = EngFncs->engine_strdup( _("Move") );
			func_info->info[SEG_MOVE_OPTION_INDEX].help = EngFncs->engine_strdup( _("Use this function to move a data segment.") );

		}

		rc = 0;
	} else {
		rc = ENOMEM;
	}

	*actions = func_info;

	LOG_EXIT_INT(rc);
	return rc;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s sft={

	// the following routines are found above
	setup_evms_plugin:                   SEG_SetupEVMSPlugin,
	cleanup_evms_plugin:                 SEG_Cleanup,
	can_set_volume:                      SEG_can_set_volume,
	can_delete:                          SEG_CanDestroy,
	can_expand:                          SEG_CanExpand,
	can_expand_by:                       SEG_CanExpandBy,
	can_shrink:                          SEG_CanShrink,
	can_shrink_by:                       SEG_CanShrinkBy,
	discard:                             SEG_Discard,
	discover:                            SEG_Discover,
	create:                              SEG_CreateSegment,
	assign:                              SEG_Assign,
	can_unassign:                        SEG_CanUnassign,
	unassign:                            SEG_UnAssign,
	delete:                              SEG_DestroySegment,
	expand:                              SEG_Expand,
	shrink:                              SEG_Shrink,
	add_sectors_to_kill_list:            SEG_AddSectorsToKillList,
	commit_changes:                      SEG_CommitChanges,
	read:                                SEG_Read,
	write:                               SEG_Write,
	set_volume:                          SEG_set_volume,
	can_activate:                        SEG_can_activate,
	activate:                            SEG_activate,
	deactivate:                          SEG_deactivate,
	can_deactivate:                      SEG_can_deactivate,
	get_plugin_functions:                SEG_get_plugin_functions,
	plugin_function:                     SEG_plugin_function,
	backup_metadata:                     SEG_BackupMetadata,

	// the following routines found in segoptions.c
	get_option_count:                    SEG_GetOptionCount,
	init_task:                           SEG_InitTask,
	set_option:                          SEG_SetOption,
	set_objects:                         SEG_SetObjects,
	get_info:                            SEG_GetInfo,
	get_plugin_info:                     SEG_GetPluginInfo
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t segmgr_plugin_record = {

	id:                                 EVMS_DOS_PLUGIN_ID,

	version:                            {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

	required_engine_api_version:        {15,0,0},
	required_plugin_api_version:        {plugin: {13,1,0}},

	short_name:                         EVMS_DOS_PLUGIN_SHORT_NAME,
	long_name:                          EVMS_DOS_PLUGIN_LONG_NAME,
	oem_name:                           EVMS_IBM_OEM_NAME,

	functions:                          {plugin: &sft},

	container_functions:                NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
	&segmgr_plugin_record,
	NULL
};
