/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: evmsd_worker.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <pthread.h>
#include <libgen.h>
#include <string.h>
#include <semaphore.h>

#include <frontend.h>

#include "../dmonmsg.h"
#include "../flatten.h"

#define LOG_CRITICAL(msg, args...)              evms_write_log_entry(CRITICAL, prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_SERIOUS(msg, args...)               evms_write_log_entry(SERIOUS,  prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_ERROR(msg, args...)                 evms_write_log_entry(ERROR,    prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_WARNING(msg, args...)               evms_write_log_entry(WARNING,  prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEFAULT(msg, args...)               evms_write_log_entry(DEFAULT,  prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DETAILS(msg, args...)               evms_write_log_entry(DETAILS,  prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEBUG(msg, args...)                 evms_write_log_entry(DEBUG,    prog_name, "%s: " msg, __FUNCTION__ , ## args)
#define LOG(level, msg, args...)                evms_write_log_entry(level,    prog_name, "%s: " msg, __FUNCTION__ , ## args)

#define LOG_PROC_ENTRY()                        evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Enter.\n", __FUNCTION__);

#define LOG_PROC_EXIT_INT(rc)                   evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.  Return value is %d.\n", __FUNCTION__, rc);
#define LOG_PROC_EXIT_UINT(rc)                  evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.  Return value is %u (0x%08x).\n", __FUNCTION__, rc, rc);
#define LOG_PROC_EXIT_PTR(ptr)                  evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.  Returned pointer is %p.\n", __FUNCTION__, ptr);
#define LOG_PROC_EXIT_BOOLEAN(result)           evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.  Result is %s.\n", __FUNCTION__, result ? "TRUE" : "FALSE");
#define LOG_PROC_EXIT_BOOLEAN_INT(result, rc)   evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.  Result is %s.  Return value is %d.\n", __FUNCTION__, result ? "TRUE" : "FALSE",rc);
#define LOG_PROC_EXIT_VOID()                    evms_write_log_entry(ENTRY_EXIT, prog_name, "%s: Exit.\n", __FUNCTION__);

#define PRINTF(msg, args...)						\
	sprintf(msg_buf, "worker: %s: " msg, __FUNCTION__ , ## args);	\
	write(2, msg_buf, strlen(msg_buf));

static char * prog_name;

static evms_version_t prog_version = {major:      MAJOR_VERSION,
				      minor:      MINOR_VERSION,
				      patchlevel: PATCH_LEVEL};

static char msg_buf[2048];

static inline void log_buffer(void * in_buff, uint len) {

	int i;
	int byte_count = 0;

	unsigned char * byte = (unsigned char *) in_buff;
	char * p = msg_buf;

	for (i = 0; i < len; i++) {
		sprintf(p, "%02x", *byte);
		p += strlen(p);
		byte++;
		byte_count++;
		if (byte_count % 16 == 0) {
			LOG_DEBUG("%s\n", msg_buf);
			p = msg_buf;
		} else {
			if (byte_count % 4 == 0) {
				strcat(p, " ");
				p++;
			}
		}
	}

	if (p != msg_buf) {
		LOG_DEBUG("%s\n", msg_buf);
	}
}

static void show_help(void) {
	printf("%s version %d.%d.%d\n", prog_name, prog_version.major, prog_version.minor, prog_version.patchlevel);
	printf("This program should only be lauched by evmsd.\n");
}


static void * get_msg(uint * cmd, size_t * size, void * buffer) {

	msg_hdr_t msg_hdr;
	size_t bytes_read;

	LOG_PROC_ENTRY();

	do {
		LOG_DEBUG("Request to read %d bytes from stdin.\n", sizeof(msg_hdr));
		bytes_read = read(0, &msg_hdr, sizeof(msg_hdr));

		if (bytes_read == -1) {
			LOG_SERIOUS("Read of message failed with errno %d: %s.\n", errno, strerror(errno));
		} else {
			LOG_DEBUG("%d of %d bytes read.\n", bytes_read, sizeof(msg_hdr));
		}

		if (bytes_read != sizeof(msg_hdr)) {
			LOG_WARNING("Insufficient number of bytes read. I'm dropping the message.\n");
		}

		if (msg_hdr.signature != MSG_HDR_SIG) {
			LOG_WARNING("Message header does not have the correct signature.\n");
		}

	} while (msg_hdr.signature != MSG_HDR_SIG);

	LOG_DEBUG("Command: %#x (%s %s)  Size: %u\n",
		  msg_hdr.cmd, msg_cmd_name(msg_hdr.cmd), (msg_hdr.cmd & COMMAND_RESPONSE) ? "response" : "request", msg_hdr.size);

	*cmd  = (uint)   msg_hdr.cmd;
	*size = (size_t) msg_hdr.size;

	if (msg_hdr.size > 0) {

		if (buffer == NULL) {
			buffer = calloc(1, msg_hdr.size);
		}

		if (buffer != NULL) {
			LOG_DEBUG("Request to read %d bytes from stdin.\n", msg_hdr.size);
			bytes_read = read(0, buffer, msg_hdr.size);

			if (bytes_read == -1) {
				LOG_SERIOUS("Read of message failed with errno %d: %s.\n", errno, strerror(errno));
			} else {
				LOG_DEBUG("%d of %d bytes read.\n", bytes_read, msg_hdr.size);
			}

			if (bytes_read < msg_hdr.size) {
				LOG_WARNING("Insufficient number of bytes read. I'm dropping the message.\n");
			}

			if (msg_hdr.signature != MSG_HDR_SIG) {
				LOG_WARNING("Message header does not have the correct signature.\n");
			}
		}
	}

	LOG_PROC_EXIT_PTR(buffer);
	return buffer;
}


static void put_msg(uint cmd, size_t size, void * message) {

	msg_hdr_t msg_hdr;
	size_t bytes_written;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Command: %#x (%s %s)  Size: %u  Message: %p\n",
		  cmd, msg_cmd_name(cmd), (cmd & COMMAND_RESPONSE) ? "response" : "request", size, message);

	msg_hdr.signature = MSG_HDR_SIG;
	msg_hdr.cmd       = (u_int32_t) cmd;
	msg_hdr.size      = (u_int32_t) size;

	LOG_DEBUG("Request to write %u bytes to stdout.\n", sizeof(msg_hdr));
	bytes_written = write(1, &msg_hdr, sizeof(msg_hdr));

	if (bytes_written == -1) {
		LOG_SERIOUS("Write of message header failed with errno %d: %s.\n", errno, strerror(errno));
	} else {
		LOG_DEBUG("%d of %d bytes written.\n", bytes_written, sizeof(msg_hdr));
	}

	if (size != 0) {
		LOG_DEBUG("Request to write %u bytes to stdout.\n", size);
		bytes_written = write(1, message, size);

		if (bytes_written == -1) {
			LOG_SERIOUS("Write of message failed with errno %d: %s.\n", errno, strerror(errno));
		} else {
			LOG_DEBUG("%d of %d bytes written.\n", bytes_written, size);
		}
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_engine_api_version(void) {

	size_t size;
	evms_version_t engine_version = {0};
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_sizeof_host_to_net(&size, evms_version_f, engine_version.major, engine_version.minor, engine_version.patchlevel);

	return_buf = calloc(1, size);

	if (return_buf != NULL) {
		evms_get_api_version(&engine_version);

		evms_host_to_net(return_buf, evms_version_f, engine_version.major, engine_version.minor, engine_version.patchlevel);

		put_msg(EVMS_GET_API_VERSION | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);
	}

	LOG_PROC_EXIT_VOID();
}


static int message_callback(char   * message_text,
			    int    * answer,
			    char * * choices) {

	int rc = 0;
	size_t size;
	void * send_buf = NULL;
	uint cmd;
	void * reply;
	int dummy;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&size, user_message_args_f, message_text, answer, choices);
	LOG_DEBUG("Flat user message is of size %u.\n", size);
	send_buf = calloc(1, size);

	if (send_buf != NULL) {
		evms_host_to_net(send_buf, user_message_args_f,
				 message_text,
				 answer,
				 choices);

		put_msg(USER_MESSAGE, size, send_buf);

		do {
			reply = get_msg(&cmd, &size, NULL);

			if (cmd != (USER_MESSAGE | COMMAND_RESPONSE)) {
				LOG_SERIOUS("Message is unexpected.  I'm dropping the message.\n");
				if (reply != NULL) {
					free(reply);
				}
			}

		} while (cmd != (USER_MESSAGE | COMMAND_RESPONSE));

		if (reply != NULL) {
			if (answer != NULL) {
				evms_net_to_host(reply, user_message_rets_f,
						 &rc,
						 answer);
			} else {
				evms_net_to_host(reply, user_message_rets_f,
						 &rc,
						 &dummy);
			}

			free(reply);
		}

		free(send_buf);

	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int progress_callback(progress_t * progress) {

	int rc = 0;
	size_t size;
	void * send_buf;
	uint cmd;
	void * reply;

	LOG_PROC_ENTRY();

	evms_sizeof_host_to_net(&size, user_progress_args_f, progress);

	send_buf = calloc(1, size);

	if (send_buf != NULL) {
		evms_host_to_net(send_buf, user_progress_args_f, progress);

		put_msg(PROGRESS, size, send_buf);

		do {
			reply = get_msg(&cmd, &size, NULL);

			if (cmd != (PROGRESS | COMMAND_RESPONSE)) {
				LOG_SERIOUS("Message is unexpected.  I'm dropping the message.\n");
				if (reply != NULL) {
					free(reply);
				}
			}

		} while (cmd != (PROGRESS | COMMAND_RESPONSE));

		if (reply != NULL) {
			evms_net_to_host(reply, user_progress_rets_f,
					 &rc,
					 &progress->id,
					 &progress->ui_private_data);
			free(reply);
		}

		free(send_buf);

	} else {
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void status_callback(char * message) {

	size_t size;
	uint cmd;
	void * reply;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Status is: %s\n", message);

	/*
	 * Status is a character string.  It goes out on the wire without any
	 * conversion to network byte order.
	 */
	put_msg(STATUS, strlen(message) + 1, message);

	do {
		reply = get_msg(&cmd, &size, NULL);

		if (cmd != (STATUS | COMMAND_RESPONSE)) {
			LOG_SERIOUS("Message is unexpected.  I'm dropping the message.\n");
			if (reply != NULL) {
				free(reply);
			}
		}

	} while (cmd != (STATUS | COMMAND_RESPONSE));

	if (reply != NULL) {
		free(reply);
	}

	LOG_PROC_EXIT_VOID();
}


static ui_callbacks_t callbacks = {
	user_message:       message_callback,
	user_communication: NULL,
	progress:           progress_callback,
	status:             status_callback};


static void worker_open_engine(void * msg) {

	char * node_name;
	engine_mode_t mode;
	void * caller_callbacks;
	debug_level_t debug_level;
	char * log_file;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

     	evms_net_to_host(msg, evms_open_engine_args_f,
			 &node_name,
			 &mode,
			 &caller_callbacks,
			 &debug_level,
			 &log_file);

	/*
	 * Don't send back status.  Mutiple nodes may be opening the Engine
	 * at the same time.  Mixed status messages from all the nodes would
	 * be confusing.
	 */
	callbacks.status = NULL;

	LOG_DEBUG("mode: %#x  debug_level: %#x  log_file: %s\n", mode, debug_level, log_file);
	rc = evms_open_engine(NULL,
			      mode | ENGINE_WORKER,
			      &callbacks,
			      debug_level,
			      log_file);

	LOG_DEBUG("Return code from evms_open_engine() is %d: %s\n", rc, evms_strerror(rc));

	/* It's OK to send status now. */
	callbacks.status = status_callback;

	evms_host_to_net(&net_rc, evms_open_engine_rets_f, rc);

	put_msg(EVMS_OPEN_ENGINE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_plugin_list(void * msg) {

	evms_plugin_code_t    type;
	plugin_search_flags_t flags;
	handle_array_t      * ha = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_plugin_list_args_f,
			 &type,
			 &flags);

	rc = evms_get_plugin_list(type, flags, &ha);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_plugin_list_rets_f,
					     rc,
					     ha);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}

	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_plugin_list_rets_f,
				 rc,
				 ha);

		put_msg(EVMS_GET_PLUGIN_LIST | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_PLUGIN_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(ha);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_plugin_by_ID(void * msg) {

	plugin_id_t plugin_ID;
	plugin_handle_t plugin_handle;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_plugin_by_ID_args_f,
			 &plugin_ID);

	rc = evms_get_plugin_by_ID(plugin_ID, &plugin_handle);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_plugin_by_ID_rets_f,
					     rc,
					     plugin_handle);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_plugin_by_ID_rets_f,
				 rc,
				 plugin_handle);

		put_msg(EVMS_GET_PLUGIN_BY_ID | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_PLUGIN_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_plugin_by_name(void * msg) {

	plugin_handle_t plugin_handle;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	/*
	 * Strings get passed as-is on the wire.  No need to extract
	 * the plugin_name from the msg.  the msg is the name.
	 */
	rc = evms_get_plugin_by_name((char *) msg, &plugin_handle);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_plugin_by_ID_rets_f,
					     rc,
					     plugin_handle);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_plugin_by_ID_rets_f,
				 rc,
				 plugin_handle);

		put_msg(EVMS_GET_PLUGIN_BY_NAME | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_PLUGIN_BY_NAME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_changes_pending() {

	boolean result;
	change_record_array_t * changes = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

        rc = evms_changes_pending(&result, &changes);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_changes_pending_rets_f,
					     rc,
					     result,
					     changes);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_changes_pending_rets_f,
				 rc,
				 result,
				 changes);

		put_msg(EVMS_CHANGES_PENDING | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_CHANGES_PENDING | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(changes);

	LOG_PROC_EXIT_VOID();
}


static void worker_set_debug_level(void * msg) {

	debug_level_t level;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_set_debug_level_args_f,
			 &level);

	rc = evms_set_debug_level(level);

	evms_host_to_net(&net_rc, evms_set_debug_level_rets_f,
			 rc);

	put_msg(EVMS_SET_DEBUG_LEVEL | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_commit_changes() {

	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

        rc = evms_commit_changes();

	evms_host_to_net(&net_rc, evms_commit_changes_rets_f,
			 rc);

	put_msg(EVMS_COMMIT_CHANGES | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
}


static void worker_close_engine(void) {

	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	rc = evms_close_engine();

	evms_host_to_net(&net_rc, evms_close_engine_rets_f, rc);

	put_msg(EVMS_CLOSE_ENGINE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_expand_points(void * msg) {

	int rc;
	object_handle_t thing;
	expand_handle_array_t * expand_points = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_expand_points_args_f,
			 &thing);

	rc = evms_get_expand_points(thing, &expand_points);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_expand_points_rets_f,
					     rc,
					     expand_points);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_expand_points_rets_f,
				 rc,
				 expand_points);

		put_msg(EVMS_GET_EXPAND_POINTS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_EXPAND_POINTS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(expand_points);

	LOG_PROC_EXIT_VOID();
}


static void worker_expand(void * msg) {

	object_handle_t thing;
	handle_array_t * objects = NULL;
	option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_expand_args_f,
			 &thing,
			 &objects,
			 &options);

	rc = evms_expand(thing, objects, options);

	evms_host_to_net(&net_rc, evms_expand_rets_f, rc);

	put_msg(EVMS_EXPAND | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(objects);
	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_shrink_points(void * msg) {

	int rc;
	object_handle_t thing;
	shrink_handle_array_t * shrink_points = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_shrink_points_args_f,
			 &thing);

	rc = evms_get_shrink_points(thing, &shrink_points);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_shrink_points_rets_f,
					     rc,
					     shrink_points);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_shrink_points_rets_f,
				 rc,
				 shrink_points);

		put_msg(EVMS_GET_SHRINK_POINTS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_SHRINK_POINTS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(shrink_points);

	LOG_PROC_EXIT_VOID();
}


static void worker_shrink(void * msg) {

	object_handle_t thing;
	handle_array_t * objects = NULL;
	option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_shrink_args_f,
			 &thing,
			 &objects,
			 &options);

	rc = evms_shrink(thing, objects, options);

	evms_host_to_net(&net_rc, evms_shrink_rets_f, rc);

	put_msg(EVMS_SHRINK | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(objects);
	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_handle_object_type(void * msg) {

	object_handle_t thing;
	int rc;
	object_type_t type;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_handle_object_type_args_f,
			 &thing);

        rc = evms_get_handle_object_type(thing, &type);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_handle_object_type_rets_f,
					     rc,
					     type);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_handle_object_type_rets_f,
				 rc,
				 type);

		put_msg(EVMS_GET_HANDLE_OBJECT_TYPE | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_HANDLE_OBJECT_TYPE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_info(void * msg) {

	object_handle_t thing;
	handle_object_info_t * user_info = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_info_args_f,
			 &thing);

	rc = evms_get_info(thing, &user_info);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_info_rets_f,
					     rc,
					     user_info);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_info_rets_f,
				 rc,
				 user_info);

		put_msg(EVMS_GET_INFO | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_INFO | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(user_info);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_extended_info(void * msg) {

	object_handle_t thing;
	char * name;
	extended_info_array_t * eia = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_extended_info_args_f,
			 &thing,
			 &name);

	rc = evms_get_extended_info(thing, name, &eia);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_extended_info_rets_f,
					     rc,
					     eia);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_extended_info_rets_f,
				 rc,
				 eia);

		put_msg(EVMS_GET_EXTENDED_INFO | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_EXTENDED_INFO | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(eia);

	LOG_PROC_EXIT_VOID();
}


static void worker_set_info(void * msg) {

	object_handle_t thing;
	option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_set_info_args_f,
			 &thing,
			 &options);

	rc = evms_set_info(thing, options);

	evms_host_to_net(&net_rc, evms_set_info_rets_f, rc);

	put_msg(EVMS_SET_INFO | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_object_handle_for_name(void * msg) {

	object_type_t   type;
	char          * name;
	object_handle_t object_handle;
	int rc;
	u_int32_t return_buf[2];

	evms_net_to_host(msg, evms_get_object_handle_for_name_args_f,
			 &type,
			 &name);

	rc = evms_get_object_handle_for_name(type, name, &object_handle);

	evms_host_to_net(return_buf, evms_get_object_handle_for_name_rets_f,
			 rc,
			 object_handle);

	put_msg(EVMS_GET_OBJECT_HANDLE_FOR_NAME | COMMAND_RESPONSE, sizeof(u_int32_t) * 2, return_buf);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_feature_list(void * msg) {

	object_handle_t thing;
	handle_array_t * ha = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_feature_list_args_f,
			 &thing);

	rc = evms_get_feature_list(thing, &ha);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_feature_list_rets_f,
					     rc,
					     ha);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_feature_list_rets_f,
				 rc,
				 ha);

		put_msg(EVMS_GET_FEATURE_LIST | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_FEATURE_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(ha);

	LOG_PROC_EXIT_VOID();
}


static void worker_create(void * msg) {

	plugin_handle_t  plugin;
	handle_array_t * input_objects = NULL;
	option_array_t * options = NULL;
	handle_array_t * output_objects = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_create_args_f,
			 &plugin,
			 &input_objects,
			 &options);

	rc = evms_create(plugin, input_objects, options, &output_objects);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_create_rets_f,
					     rc,
					     output_objects);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_create_rets_f,
				 rc,
				 output_objects);

		put_msg(EVMS_CREATE | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_CREATE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(input_objects);
	evms_free(options);
	evms_free(output_objects);

	LOG_PROC_EXIT_VOID();
}


static void worker_assign(void * msg) {

	object_handle_t  object;
	plugin_handle_t  plugin;
	option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_assign_args_f,
			 &object,
			 &plugin,
			 &options);

	rc = evms_assign(object, plugin, options);

	evms_host_to_net(&net_rc, evms_assign_rets_f,
			 rc);

	put_msg(EVMS_ASSIGN | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_object_list(void * msg) {

	object_type_t         object_type;
	data_type_t           data_type;
	plugin_handle_t       plugin;
	object_handle_t       disk_group;
	object_search_flags_t flags;
	handle_array_t      * ha = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_object_list_args_f,
			 &object_type,
			 &data_type,
			 &plugin,
			 &disk_group,
			 &flags);

	rc = evms_get_object_list(object_type, data_type, plugin, disk_group, flags, &ha);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_object_list_rets_f,
					     rc,
					     ha);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_object_list_rets_f,
				 rc,
				 ha);

		put_msg(EVMS_GET_OBJECT_LIST | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_OBJECT_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(ha);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_plugin_functions(void * msg) {

	engine_handle_t         thing;
	function_info_array_t * actions = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_plugin_functions_args_f,
			 &thing);

	rc = evms_get_plugin_functions(thing, &actions);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_plugin_functions_rets_f,
					     rc,
					     actions);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_plugin_functions_rets_f,
				 rc,
				 actions);

		put_msg(EVMS_GET_PLUGIN_FUNCTIONS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_PLUGIN_FUNCTIONS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(actions);

	LOG_PROC_EXIT_VOID();
}


static void worker_do_plugin_function(void * msg) {

	engine_handle_t  thing;
	task_action_t    action;
	handle_array_t * objects = NULL;
	option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_do_plugin_function_args_f,
			 &thing,
			 &action,
			 &objects,
			 &options);

	rc = evms_do_plugin_function(thing, action, objects, options);

	evms_host_to_net(&net_rc, evms_do_plugin_function_rets_f,
			 rc);

	put_msg(EVMS_DO_PLUGIN_FUNCTION | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(objects);
	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_create_volume(void * msg) {

	object_handle_t object;
	char          * name;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_create_volume_args_f,
			 &object,
			 &name);

	rc = evms_create_volume(object, name);

	evms_host_to_net(&net_rc, evms_create_volume_rets_f,
			 rc);

	put_msg(EVMS_CREATE_VOLUME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_set_volume_name(void * msg) {

	object_handle_t object;
	char          * name;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_set_volume_name_args_f,
			 &object,
			 &name);

	rc = evms_set_volume_name(object, name);

	evms_host_to_net(&net_rc, evms_set_volume_name_rets_f,
			 rc);

	put_msg(EVMS_SET_VOLUME_NAME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_convert_to_evms_volume(void * msg) {

	object_handle_t object;
	char          * name;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_convert_to_evms_volume_args_f,
			 &object,
			 &name);

	rc = evms_convert_to_evms_volume(object, name);

	evms_host_to_net(&net_rc, evms_convert_to_evms_volume_rets_f,
			 rc);

	put_msg(EVMS_CONVERT_TO_EVMS_VOLUME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_add_feature_to_volume(void * msg) {

	object_handle_t  volume;
	plugin_handle_t  feature;
        option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_add_feature_to_volume_args_f,
			 &volume,
			 &feature,
			 &options);

	rc = evms_add_feature_to_volume(volume, feature, options);

	evms_host_to_net(&net_rc, evms_add_feature_to_volume_rets_f,
			 rc);

	put_msg(EVMS_ADD_FEATURE_TO_VOLUME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_mkfs(void * msg) {

	object_handle_t  volume;
	plugin_handle_t  feature;
        option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_mkfs_args_f,
			 &volume,
			 &feature,
			 &options);

	rc = evms_mkfs(volume, feature, options);

	evms_host_to_net(&net_rc, evms_mkfs_rets_f,
			 rc);

	put_msg(EVMS_MKFS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_fsck(void * msg) {

	object_handle_t  volume;
        option_array_t * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_fsck_args_f,
			 &volume,
			 &options);

	rc = evms_fsck(volume, options);

	evms_host_to_net(&net_rc, evms_fsck_rets_f,
			 rc);

	put_msg(EVMS_FSCK | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_volume_list(void * msg) {

	object_handle_t       fsim_handle;
	object_handle_t       disk_group;
	volume_search_flags_t flags;
	handle_array_t      * ha = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_volume_list_args_f,
			 &fsim_handle,
			 &disk_group,
			 &flags);

	rc = evms_get_volume_list(fsim_handle, disk_group, flags, &ha);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_volume_list_rets_f,
					     rc,
					     ha);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_volume_list_rets_f,
				 rc,
				 ha);

		put_msg(EVMS_GET_VOLUME_LIST | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_VOLUME_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(ha);

	LOG_PROC_EXIT_VOID();
}


static void worker_mount(void * msg) {

	object_handle_t volume;
	char          * mount_point;
        char          * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_mount_args_f,
			 &volume,
			 &mount_point,
			 &options);

	rc = evms_mount(volume, mount_point, options);

	evms_host_to_net(&net_rc, evms_mount_rets_f,
			 rc);

	put_msg(EVMS_MOUNT | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_remount(void * msg) {

	object_handle_t volume;
        char          * options = NULL;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_remount_args_f,
			 &volume,
			 &options);

	rc = evms_remount(volume, options);

	evms_host_to_net(&net_rc, evms_remount_rets_f,
			 rc);

	put_msg(EVMS_REMOUNT | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_create_container(void * msg) {

	plugin_handle_t   plugin;
	handle_array_t  * input_objects = NULL;
	option_array_t  * options = NULL;
	object_handle_t   output_container;
	int rc;
	u_int32_t response_buf[2];

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_create_container_args_f,
			 &plugin,
			 &input_objects,
			 &options);

	rc = evms_create_container(plugin, input_objects, options, &output_container);

	evms_host_to_net(response_buf, evms_create_container_rets_f,
			 rc,
			 output_container);

	put_msg(EVMS_CREATE_CONTAINER | COMMAND_RESPONSE, sizeof(u_int32_t) * 2, response_buf);

	evms_free(input_objects);
	evms_free(options);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_container_list(void * msg) {

	object_handle_t          plugin_handle;
	object_handle_t          disk_group;
	container_search_flags_t flags;
	handle_array_t         * ha = NULL;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_container_list_args_f,
			 &plugin_handle,
			 &disk_group,
			 &flags);

	rc = evms_get_container_list(plugin_handle, disk_group, flags, &ha);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_container_list_rets_f,
					     rc,
					     ha);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_container_list_rets_f,
				 rc,
				 ha);

		put_msg(EVMS_GET_CONTAINER_LIST | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_CONTAINER_LIST | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(ha);

	LOG_PROC_EXIT_VOID();
}


static void worker_create_task(void * msg) {

	engine_handle_t thing;
	task_action_t   action;
	task_handle_t   new_task_context;
	int rc;
	u_int32_t response_buf[2];

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_create_task_args_f,
			 &thing,
			 &action);

	rc = evms_create_task(thing, action, &new_task_context);

	evms_host_to_net(response_buf, evms_create_task_rets_f,
			 rc,
			 new_task_context);

	put_msg(EVMS_CREATE_TASK | COMMAND_RESPONSE, sizeof(u_int32_t) * 2, response_buf);

	LOG_PROC_EXIT_VOID();
}


static void worker_invoke_task(void * msg) {

	int rc;
	task_handle_t    task;
	handle_array_t * resulting_objects = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_invoke_task_args_f,
			 &task);

	rc = evms_invoke_task(task, &resulting_objects);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_invoke_task_rets_f,
					     rc,
					     resulting_objects);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_invoke_task_rets_f,
				 rc,
				 resulting_objects);

		put_msg(EVMS_INVOKE_TASK | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_INVOKE_TASK | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(resulting_objects);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_task_action(void * msg) {

	task_handle_t handle;
	task_action_t action;

	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_task_action_args_f,
			 &handle);

        rc = evms_get_task_action(handle, &action);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_task_action_rets_f,
					     rc,
					     action);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_task_action_rets_f,
				 rc,
				 action);

		put_msg(EVMS_GET_TASK_ACTION | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_TASK_ACTION | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_acceptable_objects(void * msg) {

	int rc;
	task_handle_t    task;
	handle_array_t * acceptable_object_list = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_acceptable_objects_args_f,
			 &task);

	rc = evms_get_acceptable_objects(task, &acceptable_object_list);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_acceptable_objects_rets_f,
					     rc,
					     acceptable_object_list);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_acceptable_objects_rets_f,
				 rc,
				 acceptable_object_list);

		put_msg(EVMS_GET_ACCEPTABLE_OBJECTS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_ACCEPTABLE_OBJECTS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(acceptable_object_list);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_selected_objects(void * msg) {

	int rc;
	task_handle_t    task;
	handle_array_t * selected_object_list = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_selected_objects_args_f,
			 &task);

	rc = evms_get_selected_objects(task, &selected_object_list);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_selected_objects_rets_f,
					     rc,
					     selected_object_list);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_selected_objects_rets_f,
				 rc,
				 selected_object_list);

		put_msg(EVMS_GET_SELECTED_OBJECTS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_SELECTED_OBJECTS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(selected_object_list);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_selected_object_limits(void * msg) {

	int rc;
	task_handle_t task;
	u_int32_t     minimum;
	u_int32_t     maximum;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_selected_object_limits_args_f,
			 &task);

	rc = evms_get_selected_object_limits(task, &minimum, &maximum);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_selected_object_limits_rets_f,
					     rc,
					     minimum,
					     maximum);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_selected_object_limits_rets_f,
				 rc,
				 minimum,
				 maximum);

		put_msg(EVMS_GET_SELECTED_OBJECT_LIMITS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_SELECTED_OBJECT_LIMITS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_set_selected_objects(void * msg) {

	int rc;
	task_handle_t             task;
	handle_array_t          * selected_object_list = NULL;
	declined_handle_array_t * declined_list = NULL;
	task_effect_t             effect;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_set_selected_objects_args_f,
			 &task,
			 &selected_object_list);

	rc = evms_set_selected_objects(task, selected_object_list, &declined_list, &effect);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_set_selected_objects_rets_f,
					     rc,
					     declined_list,
					     effect);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_set_selected_objects_rets_f,
				 rc,
				 declined_list,
				 effect);

		put_msg(EVMS_SET_SELECTED_OBJECTS | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_SET_SELECTED_OBJECTS | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(selected_object_list);
	evms_free(declined_list);

	LOG_PROC_EXIT_VOID();
}


static void worker_get_option_count(void * msg) {

	task_handle_t task;
	int           count;
	int rc;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_option_count_args_f,
			 &task);

        rc = evms_get_option_count(task, &count);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_option_count_rets_f,
					     rc,
					     count);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_option_count_rets_f,
				 rc,
				 count);

		put_msg(EVMS_GET_OPTION_COUNT | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

                evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_OPTION_COUNT | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_option_descriptor(void * msg) {

	int rc;
	task_handle_t         task;
	u_int32_t             option;
	option_descriptor_t * descriptor = NULL;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_option_descriptor_args_f,
			 &task,
			 &option);

	rc = evms_get_option_descriptor(task, option, &descriptor);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_option_descriptor_rets_f,
					     rc,
					     descriptor);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_option_descriptor_rets_f,
				 rc,
				 descriptor);

		put_msg(EVMS_GET_OPTION_DESCRIPTOR | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_OPTION_DESCRIPTOR | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(descriptor);

	LOG_PROC_EXIT_VOID();
}


static void worker_set_option_value(void * msg) {

	int rc;
	task_handle_t task;
	u_int32_t     option;
	value_t       value;
	task_effect_t effect;
	option_descriptor_t * od;
	value_type_t value_type = 0;
	boolean value_is_list = FALSE;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	memset(&value, 0, sizeof(value_t));

	evms_net_to_host(msg, evms_set_option_value_args_f,
			 &task,
			 &option,
			 &value);

	rc = evms_set_option_value(task, option, &value, &effect);

	if (rc == 0) {
		/*
		 * We need to know the value type to know how to send the value over
		 * the wire.  Get the option descriptor which has the type.
		 */
		rc = evms_get_option_descriptor(task, option, &od);
		if (rc != 0) {
			LOG_ERROR("Error getting option descriptor index %d.  Return code is %d: %s\n", option, rc, evms_strerror(rc));
		}

		if (rc == 0) {
			value_type = od->type;
			value_is_list = od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST;
			evms_free(od);

			rc = evms_sizeof_host_to_net(&size, evms_set_option_value_rets_f,
						     rc,
						     value_type,
						     value_is_list,
						     value,
						     effect);

			if (rc == 0) {
				return_buf = calloc(1, size);

				if (return_buf == NULL) {
					LOG_CRITICAL("Error getting memory for a return buffer.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_set_option_value_rets_f,
				 rc,
				 value_type,
				 value_is_list,
				 value,
				 effect);

		put_msg(EVMS_SET_OPTION_VALUE | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_SET_OPTION_VALUE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_get_option_descriptor_by_name(void * msg) {

	int rc;
	task_handle_t         task;
	char                * option;
	option_descriptor_t * descriptor;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_get_option_descriptor_by_name_args_f,
			 &task,
			 &option);

	rc = evms_get_option_descriptor_by_name(task, option, &descriptor);

	if (rc == 0) {
		rc = evms_sizeof_host_to_net(&size, evms_get_option_descriptor_by_name_rets_f,
					     rc,
					     descriptor);

		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_get_option_descriptor_by_name_rets_f,
				 rc,
				 descriptor);

		put_msg(EVMS_GET_OPTION_DESCRIPTOR_BY_NAME | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_GET_OPTION_DESCRIPTOR_BY_NAME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	evms_free(descriptor);

	LOG_PROC_EXIT_VOID();
}


static void worker_set_option_value_by_name(void * msg) {

	int rc;
	task_handle_t task;
	char        * option;
	value_t       value;
	task_effect_t effect;
	option_descriptor_t * od;
	value_type_t value_type = 0;
	boolean value_is_list = FALSE;
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	memset(&value, 0, sizeof(value_t));

	evms_net_to_host(msg, evms_set_option_value_by_name_args_f,
			 &task,
			 &option,
			 &value);

	rc = evms_set_option_value_by_name(task, option, &value, &effect);

	if (rc == 0) {
		/*
		 * We need to know the value type to know how to send the value over
		 * the wire.  Get the option descriptor which has the type.
		 */
		rc = evms_get_option_descriptor_by_name(task, option, &od);
		if (rc != 0) {
			LOG_ERROR("Error getting option descriptor named \"%s\".  Return code is %d: %s\n", option, rc, evms_strerror(rc));
		}

		if (rc == 0) {
			value_type = od->type;
			value_is_list = od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST;
			evms_free(od);

			rc = evms_sizeof_host_to_net(&size, evms_set_option_value_by_name_rets_f,
						     rc,
						     value_type,
						     value_is_list,
						     value,
						     effect);

			if (rc == 0) {
				return_buf = calloc(1, size);

				if (return_buf == NULL) {
					LOG_CRITICAL("Error getting memory for a return buffer.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, evms_set_option_value_by_name_rets_f,
				 rc,
				 value_type,
				 value_is_list,
				 value,
				 effect);

		put_msg(EVMS_SET_OPTION_VALUE_BY_NAME | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, int_f, rc);

		put_msg(EVMS_SET_OPTION_VALUE_BY_NAME | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	LOG_PROC_EXIT_VOID();
}


static void worker_mark_for_rediscover(void * msg) {

	int rc;
	char * name;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_mark_for_rediscover_args_f,
			 &name);

	rc = evms_mark_for_rediscover(name);

	evms_host_to_net(&net_rc, evms_mark_for_rediscover_rets_f,
			 rc);

	put_msg(EVMS_MARK_FOR_REDISCOVER | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_rediscover(void * msg) {

	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	/*
	 * Don't send back status.  Mutiple nodes may be rediscovering at the
	 * same time.  Mixed status messages from all the nodes would be
	 * confusing.
	 */
	callbacks.status = NULL;

	rc = evms_rediscover();

	/* It's OK to send status now. */
	callbacks.status = status_callback;

	evms_host_to_net(&net_rc, evms_rediscover_rets_f,
			 rc);

	put_msg(EVMS_REDISCOVER | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_one_handle_arg(uint cmd, void * msg) {

	object_handle_t thing;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, object_handle_f,
			 &thing);

	switch (cmd) {
		case EVMS_CAN_DELETE:
			rc = evms_can_delete(thing);
			break;

		case EVMS_CAN_DESTROY:
			rc = evms_can_destroy(thing);
			break;

		case EVMS_CAN_EXPAND:
			rc = evms_can_expand(thing);
			break;

		case EVMS_CAN_SHRINK:
			rc = evms_can_shrink(thing);
			break;

		case EVMS_CAN_SET_INFO:
			rc = evms_can_set_info(thing);
			break;

		case EVMS_DELETE:
			rc = evms_delete(thing);
			break;

		case EVMS_DESTROY:
			rc = evms_destroy(thing);
			break;

		case EVMS_CAN_UNASSIGN:
			rc = evms_can_unassign(thing);
			break;

		case EVMS_UNASSIGN:
			rc = evms_unassign(thing);
			break;

		case EVMS_CAN_CREATE_VOLUME:
			rc = evms_can_create_volume(thing);
			break;

		case EVMS_CAN_CREATE_COMPATIBILITY_VOLUME:
			rc = evms_can_create_compatibility_volume(thing);
			break;

		case EVMS_CAN_SET_VOLUME_NAME:
			rc = evms_can_set_volume_name(thing);
			break;

		case EVMS_CREATE_COMPATIBILITY_VOLUME:
			rc = evms_create_compatibility_volume(thing);
			break;

		case EVMS_CONVERT_TO_COMPATIBILITY_VOLUME:
			rc = evms_convert_to_compatibility_volume(thing);
			break;

		case EVMS_CAN_CONVERT_TO_EVMS_VOLUME:
			rc = evms_can_convert_to_evms_volume(thing);
			break;

		case EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME:
			rc = evms_can_convert_to_compatibility_volume(thing);
			break;

		case EVMS_CAN_UNMKFS:
			rc = evms_can_unmkfs(thing);
			break;

		case EVMS_CAN_FSCK:
			rc = evms_can_fsck(thing);
			break;

		case EVMS_CAN_MOUNT:
			rc = evms_can_mount(thing);
			break;

		case EVMS_CAN_UNMOUNT:
			rc = evms_can_unmount(thing);
			break;

		case EVMS_UNMOUNT:
			rc = evms_unmount(thing);
			break;

		case EVMS_CAN_REMOUNT:
			rc = evms_can_remount(thing);
			break;

		case EVMS_DESTROY_TASK:
			rc = evms_destroy_task(thing);
			break;

		case EVMS_CAN_ACTIVATE:
			rc = evms_can_activate(thing);
			break;

		case EVMS_ACTIVATE:
			rc = evms_activate(thing);
			break;

		case EVMS_CAN_DEACTIVATE:
			rc = evms_can_deactivate(thing);
			break;

		case EVMS_DEACTIVATE:
			rc = evms_deactivate(thing);
			break;

		default:
			LOG_SERIOUS("Got unexpected command %#x (%s).\n",
				    cmd, msg_cmd_name(cmd));
			rc = EINVAL;
	}

	evms_host_to_net(&net_rc, int_f,
			 rc);

	put_msg(cmd | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_two_handle_arg(uint cmd, void * msg) {

	object_handle_t thing1;
	object_handle_t thing2;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, object_handle_f object_handle_f,
			 &thing1,
			 &thing2);

	switch(cmd) {
		case EVMS_CAN_REPLACE:
			rc = evms_can_replace(thing1, thing2);
			break;

		case EVMS_REPLACE:
			rc = evms_replace(thing1, thing2);
			break;

		case EVMS_CAN_ADD_FEATURE_TO_VOLUME:
			rc = evms_can_add_feature_to_volume(thing1, thing2);
			break;

		case EVMS_CAN_MKFS:
			rc = evms_can_mkfs(thing1, thing2);
			break;

		default:
			LOG_SERIOUS("Got unexpected command %#x (%s).\n",
				    cmd, msg_cmd_name(cmd));
			rc = EINVAL;
	}

	evms_host_to_net(&net_rc, int_f,
			 rc);

	put_msg(cmd | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static void worker_metadata_backup(void * msg) {

	char * dir;
	int rc;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_metadata_backup_args_f,
			 &dir);

	rc = evms_metadata_backup(dir);

	evms_host_to_net(&net_rc, evms_metadata_backup_rets_f,
			 rc);

	put_msg(EVMS_METADATA_BACKUP | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	LOG_PROC_EXIT_VOID();
}


static int worker_engine_read(void * msg) {

	int rc = 0;
	object_handle_t handle;
	lsn_t           lsn;
	sector_count_t  length;
	u_int64_t       buff_size;
	void          * buffer;
	char rets_format[32];
	size_t size;
	void * return_buf = NULL;

	LOG_PROC_ENTRY();

	evms_net_to_host(msg, evms_engine_read_args_f,
			 &handle,
			 &lsn,
			 &length);

	buff_size = length << EVMS_VSECTOR_SIZE_SHIFT;
	buffer = calloc(1, buff_size);
	if (buffer == NULL) {
		u_int32_t net_rc;
		evms_host_to_net(&net_rc, int_f,
				 ENOMEM);

		put_msg(EVMS_ENGINE_READ | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	rc = evms_engine_read(handle, lsn, length, buffer);

	if (rc == 0) {
		sprintf(rets_format, "%sp{x%"PRIu64"}", evms_engine_read_rets_f, buff_size);

		rc = evms_sizeof_host_to_net(&size, rets_format,
					     rc,
					     buffer);
		if (rc == 0) {
			return_buf = calloc(1, size);

			if (return_buf == NULL) {
				LOG_CRITICAL("Error getting memory for a return buffer.\n");
				rc = ENOMEM;
			}
		}
	}

	if (rc == 0) {
		evms_host_to_net(return_buf, rets_format,
				 rc,
				 buffer);

		put_msg(EVMS_ENGINE_READ | COMMAND_RESPONSE, size, return_buf);

		free(return_buf);

	} else {
		u_int32_t net_rc;

		evms_host_to_net(&net_rc, evms_engine_read_rets_f,
				 rc);

		put_msg(EVMS_ENGINE_READ | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);
	}

	free(buffer);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int worker_engine_write(void * msg) {

	int rc = 0;
	char args_format[32];
	object_handle_t handle;
	lsn_t           lsn;
	sector_count_t  length;
	void          * buffer = NULL;
	u_int32_t net_rc;

	LOG_PROC_ENTRY();

	/*
	 * evms_engine_write_args_f does not include the "p{x<n>}" for the
	 * fourth parameter which is the buffer of data to be written.
	 * We must first get the length to know how big of a buffer to get.
	 */
	evms_net_to_host(msg, evms_engine_write_args_f,
			 &handle,
			 &lsn,
			 &length);

	/*
	 * Now append the "p{x<n>}" to evms_engine_write_args_f and get all of the
	 * arguments.
	 */
	sprintf(args_format, "%sp{x%"PRIu64"}",
		evms_engine_write_args_f, length << EVMS_VSECTOR_SIZE_SHIFT);

	evms_net_to_host(msg, args_format,
			 &handle,
			 &lsn,
			 &length,
			 &buffer);

	rc = evms_engine_write(handle, lsn, length, buffer);

	evms_host_to_net(&net_rc, int_f,
			 rc);

	put_msg(EVMS_ENGINE_WRITE | COMMAND_RESPONSE, sizeof(net_rc), &net_rc);

	evms_free(buffer);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void dispatch_commands(void) {

	uint cmd;
	size_t size;
	void * msg;

	LOG_PROC_ENTRY();

	while (TRUE) {

		msg = get_msg(&cmd, &size, NULL);
		LOG_DEBUG("Command %#x (%s %s)  size: %u\n",
			  cmd, msg_cmd_name(cmd), (cmd & COMMAND_RESPONSE) ? "response" : "request", size);

		switch (cmd) {
			case EVMS_GET_API_VERSION:
				worker_get_engine_api_version();
				break;

			case EVMS_OPEN_ENGINE:
				worker_open_engine(msg);
				break;

			case EVMS_GET_PLUGIN_LIST:
				worker_get_plugin_list(msg);
				break;

			case EVMS_GET_PLUGIN_BY_ID:
				worker_get_plugin_by_ID(msg);
				break;

			case EVMS_GET_PLUGIN_BY_NAME:
				worker_get_plugin_by_name(msg);
				break;

			case EVMS_CHANGES_PENDING:
				worker_changes_pending();
				break;

			case EVMS_WRITE_LOG_ENTRY:
				break;

			case EVMS_SET_DEBUG_LEVEL:
				worker_set_debug_level(msg);
				break;

			case EVMS_COMMIT_CHANGES:
				worker_commit_changes();
				break;

			case EVMS_CLOSE_ENGINE:
				worker_close_engine();
				break;

			case EVMS_CAN_DELETE:
			case EVMS_CAN_DESTROY:
			case EVMS_CAN_EXPAND:
			case EVMS_CAN_SHRINK:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_CAN_REPLACE:
				worker_two_handle_arg(cmd, msg);
				break;

			case EVMS_CAN_SET_INFO:
			case EVMS_DELETE:
			case EVMS_DESTROY:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_GET_EXPAND_POINTS:
				worker_get_expand_points(msg);
				break;

			case EVMS_EXPAND:
				worker_expand(msg);
				break;

			case EVMS_GET_SHRINK_POINTS:
				worker_get_shrink_points(msg);
				break;

			case EVMS_SHRINK:
				worker_shrink(msg);
				break;

			case EVMS_REPLACE:
				worker_two_handle_arg(cmd, msg);
				break;

			case EVMS_GET_HANDLE_OBJECT_TYPE:
				worker_get_handle_object_type(msg);
				break;

			case EVMS_GET_INFO:
				worker_get_info(msg);
				break;

			case EVMS_GET_EXTENDED_INFO:
				worker_get_extended_info(msg);
				break;

			case EVMS_SET_INFO:
				worker_set_info(msg);
				break;

			case EVMS_GET_OBJECT_HANDLE_FOR_NAME:
				worker_get_object_handle_for_name(msg);
				break;

			case EVMS_GET_FEATURE_LIST:
				worker_get_feature_list(msg);
				break;

			case EVMS_CREATE:
				worker_create(msg);
				break;

			case EVMS_ASSIGN:
				worker_assign(msg);
				break;

			case EVMS_CAN_UNASSIGN:
			case EVMS_UNASSIGN:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_GET_OBJECT_LIST:
				worker_get_object_list(msg);
				break;

			case EVMS_GET_PLUGIN_FUNCTIONS:
				worker_get_plugin_functions(msg);
				break;

			case EVMS_DO_PLUGIN_FUNCTION:
				worker_do_plugin_function(msg);
				break;

			case EVMS_CAN_MOUNT:
			case EVMS_CAN_UNMOUNT:
			case EVMS_CAN_REMOUNT:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_CAN_ADD_FEATURE_TO_VOLUME:
				worker_two_handle_arg(cmd, msg);
				break;

			case EVMS_CAN_CREATE_VOLUME:
			case EVMS_CAN_CREATE_COMPATIBILITY_VOLUME:
			case EVMS_CAN_SET_VOLUME_NAME:
			case EVMS_CAN_CONVERT_TO_EVMS_VOLUME:
			case EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_CREATE_VOLUME:
				worker_create_volume(msg);
				break;

			case EVMS_SET_VOLUME_NAME:
				worker_set_volume_name(msg);
				break;

			case EVMS_CREATE_COMPATIBILITY_VOLUME:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_CONVERT_TO_EVMS_VOLUME:
				worker_convert_to_evms_volume(msg);
				break;

			case EVMS_CONVERT_TO_COMPATIBILITY_VOLUME:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_ADD_FEATURE_TO_VOLUME:
				worker_add_feature_to_volume(msg);
				break;

			case EVMS_CAN_MKFS:
				worker_two_handle_arg(cmd, msg);
				break;

			case EVMS_CAN_UNMKFS:
			case EVMS_CAN_FSCK:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_MKFS:
				worker_mkfs(msg);
				break;

			case EVMS_UNMKFS:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_FSCK:
				worker_fsck(msg);
				break;

			case EVMS_GET_VOLUME_LIST:
				worker_get_volume_list(msg);
				break;

			case EVMS_MOUNT:
				worker_mount(msg);
				break;

			case EVMS_UNMOUNT:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_REMOUNT:
				worker_remount(msg);
				break;

			case EVMS_CREATE_CONTAINER:
				worker_create_container(msg);
				break;

			case EVMS_GET_CONTAINER_LIST:
				worker_get_container_list(msg);
				break;

			case EVMS_CREATE_TASK:
				worker_create_task(msg);
				break;

			case EVMS_INVOKE_TASK:
				worker_invoke_task(msg);
				break;

			case EVMS_DESTROY_TASK:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_GET_TASK_ACTION:
				worker_get_task_action(msg);
				break;

			case EVMS_GET_ACCEPTABLE_OBJECTS:
				worker_get_acceptable_objects(msg);
				break;

			case EVMS_GET_SELECTED_OBJECTS:
				worker_get_selected_objects(msg);
				break;

			case EVMS_GET_SELECTED_OBJECT_LIMITS:
				worker_get_selected_object_limits(msg);
				break;

			case EVMS_SET_SELECTED_OBJECTS:
				worker_set_selected_objects(msg);
				break;

			case EVMS_GET_OPTION_COUNT:
				worker_get_option_count(msg);
				break;

			case EVMS_GET_OPTION_DESCRIPTOR:
				worker_get_option_descriptor(msg);
				break;

			case EVMS_SET_OPTION_VALUE:
				worker_set_option_value(msg);
				break;

			case EVMS_GET_OPTION_DESCRIPTOR_BY_NAME:
				worker_get_option_descriptor_by_name(msg);
				break;

			case EVMS_SET_OPTION_VALUE_BY_NAME:
				worker_set_option_value_by_name(msg);
				break;

			case EVMS_MARK_FOR_REDISCOVER:
				worker_mark_for_rediscover(msg);
				break;

			case EVMS_REDISCOVER:
				worker_rediscover(msg);
				break;

			case EVMS_CAN_ACTIVATE:
			case EVMS_ACTIVATE:
			case EVMS_CAN_DEACTIVATE:
			case EVMS_DEACTIVATE:
				worker_one_handle_arg(cmd, msg);
				break;

			case EVMS_METADATA_BACKUP:
				worker_metadata_backup(msg);
				break;

			case EVMS_ENGINE_READ:
				worker_engine_read(msg);
				break;

			case EVMS_ENGINE_WRITE:
				worker_engine_write(msg);
				break;

			default:
				if (cmd >= INVALID_COMMAND) {
					LOG_WARNING("%d is not a valid command.\n", cmd);

				} else {
					LOG_WARNING("The worker does not know how to handle command %d.\n", cmd);
				}

				put_msg(MSG_INVALID_CMD | COMMAND_RESPONSE, 0, NULL);
				break;
		}

		free(msg);
	}

	LOG_PROC_EXIT_VOID();
}


int main(int argc, char * * argv, char * * env){

	prog_name = basename(argv[0]);

	if (argc > 1) {
		show_help();
		return EINVAL;
	}

	dispatch_commands();

	sprintf(msg_buf, "%s: I'm out of here.\n", prog_name);
	write(2, msg_buf, strlen(msg_buf));
	return 0;
}

