/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: commit.c
 */ 

#include <unistd.h>
#include <sys/types.h>
#include <pthread.h> 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "commit.h"
#include "logging.h"
#include "pixmap.h"
#include "views.h"

/*
 * File scope globals
 */
static gboolean commit_in_progress = FALSE;

/*
 * Structure containing information needed by commit thread.
 */
typedef struct evmsgui_commit_info_s {
    gint                  rc;
    gboolean              exit_after_commit;
    GtkProgressBar       *progress_bar;
} evmsgui_commit_info_t;

gboolean is_commit_in_progress (void)
{    
    return commit_in_progress;
}

/*
 *
 *   void commit_cleanup (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine handles cleanup after a commit
 *      changes has completed.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Clears progress bar and re-enables main window 
 *      menu bar sensitivity.
 *
 */
void commit_cleanup (evmsgui_commit_info_t *commit_info)
{
    GtkWidget *main_menubar;

    /*
     * Clear the progress bar with the following code.
     */

    gtk_progress_set_activity_mode (GTK_PROGRESS (commit_info->progress_bar), FALSE);
    gtk_progress_bar_update (commit_info->progress_bar, 0.0);

    /*
     * Re-enable the main window menu bar
     */

    main_menubar = gtk_object_get_data (GTK_OBJECT (get_main_window_id ()), "main_menubar");
    gtk_widget_set_sensitive (main_menubar, TRUE);

    g_free (commit_info);
}

/*
 *
 *   void process_post_commit (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine handles post commit processing.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Displays results screen if necessary and calls
 *      cleanup routine. If exit_after_commit flag is
 *      set, it emits the signal that destroys the 
 *      main window.
 *
 */
void process_post_commit (evmsgui_commit_info_t *commit_info)
{
    if (commit_info->rc != SUCCESS)
    {
        display_results_window (commit_info->rc, _("Commit Changes"),
                                _("An error was encountered committing changes."),
                                NULL, TRUE, get_main_window_id ());
    }
    else
    {
        set_status_bar_message (_("All changes were committed successfully."));

        /*
         * If this is a commit and exit, present a popup to show commit
         * succeeded. Connect the gtk_widget_destroy of the main window
         * with the results window's close button to close the app when
         * clicked.
         */

        if (commit_info->exit_after_commit)
        {
            GtkWidget *window;
            GtkWidget *close_button;

            window = display_results_window (commit_info->rc, _("Commit Changes"),
                                             _("All changes were committed successfully."),
                                             NULL, FALSE, get_main_window_id ());

            close_button = gtk_object_get_data (GTK_OBJECT (window), "results_window_button");

            gtk_signal_disconnect_by_func (GTK_OBJECT (close_button),
                                           GTK_SIGNAL_FUNC (on_button_clicked_destroy_window),
                                           NULL);

            gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked", gtk_widget_destroy, 
                                       GTK_OBJECT (get_main_window_id ()));
        }
        else
        {
            refresh_main_window_views ();
        }
    }

    commit_cleanup (commit_info);
}

/*
 *
 *   static gboolean monitor_commit_progress (evmsgui_commit_info_t *)
 *   
 *   Description:
 *      This routine is called periodically to update the
 *      main progress bar mainly for lengthy commit operations.
 *      If commit is no longer in progress it calls the
 *      function that processes the commit results and
 *      cleanup.
 * 
 *   Entry:
 *      commit_info - contains information about commit
 *
 *   Exit:
 *      Progress bar is updated by 5% if commit in progress
 *      or call post commit function if commit ended.
 *
 */
static gboolean monitor_commit_progress (evmsgui_commit_info_t *commit_info)
{
    gboolean keep_timeout=TRUE;
    static float pos = 0;
    static int orientation = 0;

    gdk_threads_enter ();
    
    if (commit_in_progress)
    {
        /*
         * Make updates in increments of 5%. Once we get to 100%,
         * reverse direction and start progress at 5%.
         */

    	pos += 0.05;

    	if (pos >= 0.99)
    	{
            if (orientation == 0)
                gtk_progress_bar_set_orientation (commit_info->progress_bar, GTK_PROGRESS_RIGHT_TO_LEFT);
            else
                gtk_progress_bar_set_orientation (commit_info->progress_bar, GTK_PROGRESS_LEFT_TO_RIGHT);
    		        
            orientation = !orientation;
            pos = 0.05;
        }

        gtk_progress_bar_update (commit_info->progress_bar, pos);
    }
    else
    {
        /*
         * Commit is done. Check its return code and
         * do any cleanup work then return FALSE to
         * have this timeout callback deregistered.
         */

        process_post_commit (commit_info);
        keep_timeout = FALSE;
    }

    gdk_threads_leave ();
    
    return keep_timeout;
}

/*
 *
 *   gboolean update_status_bar_text_from_main_thread (gchar *)
 *
 *   Description:
 *      This routine is a one-time execution idle function that 
 *      updates the status bar while running under the control of
 *      the main event loop. 
 *
 *   Entry:
 *      text - text to place in the status bar and then free
 *
 *   Exit:
 *      The current message in the status bar is removed and the
 *      given text is added.
 *
 */
gboolean update_status_bar_text_from_main_thread (gchar *text)
{
    gdk_threads_enter ();
    
    remove_status_bar_message ();
    set_status_bar_message (g_strstrip (text));
    
    gdk_threads_leave ();
    
    g_free (text);
    
    return FALSE;
}

/*
 *
 *   void commit_status_callback (gchar *)
 *   
 *   Description:
 *      This routine is called by the engine during commit
 *      to provide status of the commit phases.
 * 
 *   Entry:
 *      status - a pointer to a string containing the commit status
 *
 *   Exit:
 *      
 *      or call post commit function if commit ended.
 *
 */
void commit_status_callback (gchar *status)
{
    if (status)
        gtk_idle_add ((GtkFunction) update_status_bar_text_from_main_thread,
                      g_strdup (status));
}

/*
 *
 *   void *commit_thread (void *)
 *   
 *   Description:
 *      This routine invokes the commit API. Once it
 *      the API call is complete it updates the return
 *      code in the commit info structure it was given
 *      and the thread terminates.
 * 
 *   Entry:
 *      args - address of argument given at thread create
 *
 *   Exit:
 *      Commit API is invoked, return code is saved and
 *      commit_in_progress flag is reset to FALSE.
 *
 */
void *commit_thread(void *arg)
{
    evmsgui_commit_info_t *commit_info;
        
    commit_info = (evmsgui_commit_info_t *)arg;

    commit_info->rc = evms_commit_changes (&commit_status_callback);

    commit_in_progress = FALSE;

    return NULL;
}

/*
 *
 *   void initiate_commit_changes (gboolean)
 *   
 *   Description:
 *      This routine starts a thread that invokes the commit API.
 *      It also registers a timeout handler that increments the
 *      main window progress indicator and monitors for commit
 *      completion.
 * 
 *   Entry:
 *      exit_after_commit - flag that indicates whether after a
 *                          successful commit we exit the app
 *
 *   Exit:
 *      Thread that invokes commit is started and progress bar
 *      timeout handler is added.
 *
 */
void initiate_commit_changes (gboolean exit_after_commit)
{
    gint                   rc;
    guint                  handler_id;
    pthread_t              tid;
    evmsgui_commit_info_t *info;
    GtkWidget             *main_menubar;
    
    commit_in_progress = TRUE;

    info = g_malloc (sizeof(evmsgui_commit_info_t));

    /*
     * Lookup the main_progress_bar and set activity mode to allow
     * a steady, rhythmic progress. Add the monitor_commit_progress
     * callback to a main event loop timeout scheduler queue.  
     * Set the main window menubar to be insensitive while commit
     * is in progress. Finally, start the commit thread.
     */

    set_status_bar_message (_("Committing changes..."));

    main_menubar = gtk_object_get_data (GTK_OBJECT (get_main_window_id ()), "main_menubar");
    gtk_widget_set_sensitive (main_menubar, FALSE);

    info->exit_after_commit = exit_after_commit;
    info->progress_bar = gtk_object_get_data (GTK_OBJECT (get_main_window_id ()), "main_progress_bar");

    gtk_progress_set_activity_mode (GTK_PROGRESS (info->progress_bar), TRUE);
    handler_id = gtk_timeout_add (50, (GtkFunction) monitor_commit_progress, info);

    rc = pthread_create (&tid, NULL, commit_thread, info);

    if (rc != SUCCESS)
    {
        /*
         * This stinks. I can't run commit on a separate thread.
         * The best I can do is issue it here and pray it doesn't
         * freeze the app for very long. BUGBUG: There should be
         * a better way or should I just report the error and 
         * not commit?
         */

        log_error ("%s: pthread_create() failed! Return code is %d.\n", __FUNCTION__, rc);

        commit_thread (info);
        gtk_timeout_remove (handler_id);
        process_post_commit (info);
    }
}

/*
 *
 *   void display_nothing_to_commit_popup (void)
 *   
 *   Description:
 *      This routine display a small popup that indicates
 *      that there are no changes available to be committed.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Popup is displayed awaiting dismissal.
 *
 */               
void display_nothing_to_commit_popup (void)
{
    display_popup_window (_("Commit Changes"), _("There are no changes available to be committed."));
}

/*
 *
 *   void on_commit_button_clicked (GtkButton *, gpointer *)
 *   
 *   Description:
 *      This routine calls the function that initiates the
 *      commit changes. It also dismisses the dialog 
 *      containing the button that caused this callback 
 *      to get invoked.
 * 
 *   Entry:
 *      button    - the commit button clicked
 *      user_data - contains either a gboolean which
 *                  determines whether an exit should
 *                  occur after a successful commit                  
 *
 *   Exit:
 *      Commit changes is invoked and the button's parent dialog
 *      is destroyed.
 *
 */
void on_commit_button_clicked (GtkButton *button, gpointer *user_data)
{
    gboolean exit_after_commit = GPOINTER_TO_INT (user_data);

    initiate_commit_changes (exit_after_commit);
    gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/*
 *
 *   void display_commit_changes_popup (void)
 *   
 *   Description:
 *      This routine creates and displays a popup to 
 *      notify the user that changes are pending.
 *      The popup then waits for a response from the
 *      user on whether to commit changes or exit 
 *      and lose those changes.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a popup that asks the user whether they
 *      would like to commit changes, exiting without saving,
 *      or cancel exit and return to main window.
 *
 */
void display_commit_changes_popup (void)
{
    GtkWidget   *commit_changes_window;
    GtkWidget   *vbox;
    GtkWidget   *label;
    GtkWidget   *question_label;
    GtkWidget   *hbuttonbox;
    GtkWidget   *commit_button;
    GtkWidget   *exit_button;
    GtkWidget   *cancel_button;
    GtkWidget   *pixmap;
    GdkBitmap   *mask;
    GdkPixmap   *gdk_pixmap;    
    GtkTooltips *tooltips;
    
    tooltips      = gtk_tooltips_new ();
    
    get_dialog_pixmap (WARNING_PIXMAP, &gdk_pixmap, &mask);
    
    commit_changes_window = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_window_set_title (GTK_WINDOW (commit_changes_window), _("Unsaved Changes Detected"));
    gtk_window_set_position (GTK_WINDOW (commit_changes_window), GTK_WIN_POS_CENTER);
    gtk_window_set_modal (GTK_WINDOW (commit_changes_window), TRUE);
    
    vbox           = gtk_vbox_new (FALSE, 0);
    label          = gtk_label_new (_("Unsaved changes have been detected."));
    question_label = gtk_label_new (_("Would you like to save these changes and quit, quit without saving changes, or cancel quitting?"));
    hbuttonbox     = gtk_hbutton_box_new ();
    commit_button  = gtk_button_new_with_label (_("Save and Quit"));
    exit_button    = gtk_button_new_with_label (_("Don't Save"));
    cancel_button  = gtk_button_new_with_label (_("Don't Quit"));
    pixmap         = gtk_pixmap_new (gdk_pixmap, mask);
    
    gtk_container_add (GTK_CONTAINER (commit_changes_window), vbox);
    
    gtk_box_pack_start (GTK_BOX (vbox), pixmap, FALSE, FALSE, 0);
    gtk_misc_set_alignment (GTK_MISC (pixmap), 0.10, 1);
    gtk_misc_set_padding (GTK_MISC (pixmap), 0, 5);
    
    gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
    gtk_misc_set_padding (GTK_MISC (label), 63, 10);
    
    gtk_box_pack_start (GTK_BOX (vbox), question_label, FALSE, FALSE, 0);
    gtk_label_set_line_wrap (GTK_LABEL (question_label), TRUE);
    gtk_misc_set_padding (GTK_MISC (question_label), 63, 20);
    
    gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox), 10);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 14);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox), -1, 50);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), cancel_button);
    GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, cancel_button, _("Don't quit the application."), NULL);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), exit_button);
    GTK_WIDGET_SET_FLAGS (exit_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, exit_button, _("Quit the application without saving changes"), NULL);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), commit_button);
    GTK_WIDGET_SET_FLAGS (commit_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, commit_button, _("Save changes then quit the application"), NULL);
        
    gtk_signal_connect (GTK_OBJECT (commit_button), "clicked", on_commit_button_clicked, GINT_TO_POINTER (TRUE));
    gtk_signal_connect_object (GTK_OBJECT (exit_button), "clicked", gtk_widget_destroy, GTK_OBJECT (get_main_window_id ()));
    gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked", gtk_widget_destroy, GTK_OBJECT (commit_changes_window));
    
    gtk_widget_grab_default (commit_button);
    
    gtk_widget_show_all (commit_changes_window);
    gdk_beep ();
}

/*
 *
 *   void display_commit_changes_confirmation_dialog (void)
 *   
 *   Description:
 *      This routine creates and displays a dialog to 
 *      allow the user to confirm they want to commit
 *      changes or cancel and return to the main window.
 * 
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a dialog that asks the user whether they
 *      would like to commit changes or cancel and return
 *      to the main window.
 *
 */
void display_commit_changes_confirmation_dialog (void)
{
    GtkWidget   *commit_confirmation_window;
    GtkWidget   *vbox;
    GtkWidget   *question_label;
    GtkWidget   *hbuttonbox;
    GtkWidget   *commit_button;
    GtkWidget   *cancel_button;
    GtkWidget   *pixmap;
    GdkBitmap   *mask;
    GdkPixmap   *gdk_pixmap;        
    GtkTooltips *tooltips;
    
    tooltips      = gtk_tooltips_new ();
    
    get_dialog_pixmap (QUESTION_PIXMAP, &gdk_pixmap, &mask);
    
    commit_confirmation_window = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_window_set_title (GTK_WINDOW (commit_confirmation_window), _("Commit Changes Confirmation"));
    gtk_window_set_position (GTK_WINDOW (commit_confirmation_window), GTK_WIN_POS_CENTER);
    gtk_window_set_modal (GTK_WINDOW (commit_confirmation_window), TRUE);
    
    vbox           = gtk_vbox_new (FALSE, 0);
    question_label = gtk_label_new (_("Do you wish to commit changes now or cancel?"));
    hbuttonbox     = gtk_hbutton_box_new ();
    commit_button  = gtk_button_new_with_label (_("Commit"));
    cancel_button  = gtk_button_new_with_label (_("Cancel"));
    pixmap         = gtk_pixmap_new (gdk_pixmap, mask);
    
    gtk_container_add (GTK_CONTAINER (commit_confirmation_window), vbox);
    
    gtk_box_pack_start (GTK_BOX (vbox), pixmap, FALSE, FALSE, 0);
    gtk_misc_set_alignment (GTK_MISC (pixmap), 0.10, 1);
    gtk_misc_set_padding (GTK_MISC (pixmap), 0, 5);
        
    gtk_box_pack_start (GTK_BOX (vbox), question_label, FALSE, FALSE, 0);
    gtk_label_set_line_wrap (GTK_LABEL (question_label), TRUE);
    gtk_misc_set_padding (GTK_MISC (question_label), 63, 20);
    
    gtk_box_pack_start (GTK_BOX (vbox), hbuttonbox, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox), 10);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 14);
    gtk_button_box_set_child_size (GTK_BUTTON_BOX (hbuttonbox), -1, 50);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), cancel_button);
    GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, cancel_button, _("Cancel commit and dismiss this dialog"), NULL);
    
    gtk_container_add (GTK_CONTAINER (hbuttonbox), commit_button);
    GTK_WIDGET_SET_FLAGS (commit_button, GTK_CAN_DEFAULT);
    gtk_tooltips_set_tip (tooltips, commit_button, _("Commit pending changes"), NULL);
    
    gtk_signal_connect (GTK_OBJECT (commit_button), "clicked", on_commit_button_clicked, GINT_TO_POINTER (FALSE));
    gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked", gtk_widget_destroy, GTK_OBJECT (commit_confirmation_window));
    
    gtk_widget_grab_default (commit_button);
    
    gtk_widget_show_all (commit_confirmation_window);
}
