/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_pvdisplay.c
 *
 *	Emulates LVM's 'pvdisplay' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_segment_handle.c"
#include "helpers/is_lvm_segment.c"
#include "helpers/open_engine.c"
#include "helpers/print_ext_info_item.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s - Physical Volume Display\n\n", cmd);
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-c/--colon]\n");
	printf("\t[-d/--debug]\n");
	printf("\t[-h/-?/--help]\n");
	printf("\t[-s/--short]\n");
	printf("\t[-v[v]/--verbose [--verbose]]\n");
	printf("\t[-V/--version]\n");
	printf("\tPhysicalVolumePath [PhysicalVolumePath...]\n\n");
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "cdh?svV";
	struct option	long_opts[] = {
				{ "colon",	no_argument, NULL, 'c'},
				{ "debug",	no_argument, NULL, 'd'},
				{ "help",	no_argument, NULL, 'h'},
				{ "short",	no_argument, NULL, 's'},
				{ "verbose",	no_argument, NULL, 'v'},
				{ "version",	no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'c':
			// -c is ignored by EVMS
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 's':
			// -s is ignored by EVMS
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t			segment_handle;
	object_handle_t			container_handle;
	handle_object_info_t		* segment_info = NULL;
	extended_info_array_t		* container_ext_info = NULL;
	extended_info_array_t		* pv_name_info = NULL;
	extended_info_array_t		* pv_ext_info = NULL;
	extended_info_array_t		* pv_pe_info = NULL;
	char				* pv_names[256] = {NULL};
	char				* buf = NULL;
	int				number_of_pvs;
	int				log_level = DEFAULT;
	int				i, j, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for PV names
	if ( optind == argc ) {
		printf("%s -- please enter a physical volume path\n\n", cmd);
		return EINVAL;
	}
	number_of_pvs = argc - optind;

	// Copy the VG names from the command line, detecting any duplicates.
	remove_duplicates(&argv[optind], pv_names, number_of_pvs);

	// Open the engine.
	rc = open_engine(ENGINE_READONLY, log_level);
	if (rc) {
		return rc;
	}

	// Examine each physical volume
	for ( i = 0; i < number_of_pvs; i++ ) {

		// Get the handle for this PV.
		rc = get_segment_handle(pv_names[i], &segment_handle);
		if (rc) {
			continue;
		}

		// Does this region belong to LVM?
		if ( ! is_lvm_segment(segment_handle) ) {
			printf("%s -- Object %s does not belong to LVM.\n", cmd, pv_names[i]);
			continue;
		}

		// Get the basic info for the PV.
		rc = evms_get_info(segment_handle, &segment_info);
		if (rc) {
			printf("%s -- Error getting info for object %s (%d)\n", cmd, pv_names[i], rc);
			continue;
		}

		// Get the extended info for the PV's VG.
		container_handle = segment_info->info.segment.consuming_container;
		rc = evms_get_extended_info(container_handle, NULL, &container_ext_info);
		if (rc) {
			printf("%s -- Error getting extended info for the container for object %s (%d)\n", cmd, pv_names[i], rc);
			evms_free(segment_info);
			continue;
		}

		evms_free(segment_info);

		// Get the list of PV names from this VG.
		rc = evms_get_extended_info(container_handle, "Current_PVs", &pv_name_info);
		if (rc) {
			printf("%s -- Error getting list of PV names from VG %s (%d)\n", cmd, container_ext_info->info[0].value.s, rc);
			evms_free(container_ext_info);
			continue;
		}

		// Search the list of PVs to find the one we're looking for
		buf = NULL;
		for ( j = 0; j < pv_name_info->count; j++ ) {
			if ( ! strcmp(translate_dev_name(pv_names[i]), pv_name_info->info[j].value.s)) {
				buf = pv_name_info->info[j].name;
				break;
			}
		}

		if ( ! buf ) {
			printf("%s -- Error finding object %s in list of PVs in VG %s\n", cmd, pv_names[i], container_ext_info->info[0].value.s);
			evms_free(container_ext_info);
			continue;
		}
		evms_free(container_ext_info);

		// Get the extended info for this PV
		rc = evms_get_extended_info(container_handle, buf, &pv_ext_info);
		if (rc) {
			printf("%s -- Error getting extended info for PV %s (%d)\n", cmd, pv_names[i], rc);
			continue;
		}
		evms_free(pv_name_info);

		// Display all available information
		printf("--- Physical Volume ---\n");
		for ( j = 0; j < pv_ext_info->count; j++ ) {
			print_ext_info_item(&(pv_ext_info->info[j]));
		}
		printf("\n");

		if ( opts.verbose ) {
			// Get the list of extents
			rc = evms_get_extended_info(container_handle, pv_ext_info->info[7].name, &pv_pe_info);
			if (rc) {
				printf("%s -- Error getting PE mapping information for PV %s (%d)\n", cmd, pv_names[i], rc);
				continue;
			}

			for ( j = 0; j < pv_pe_info->count; j++ ) {
				print_ext_info_item(&(pv_pe_info->info[j]));
			}
			printf("\n");
			evms_free(pv_pe_info);
		}
		evms_free(pv_ext_info);
	}

	evms_close_engine();
	return 0;
}

