/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_lvremove.c
 *
 *	Emulates LVM's 'lvremove' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvremove command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_region_handle.c"
#include "helpers/is_lvm_region.c"
#include "helpers/open_engine.c"
#include "helpers/remove_duplicates.c"
#include "helpers/revert_volume.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	printf("%s - Logical Volume Remove\n\n", cmd);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-A/--autobackup y/n]\n");
	printf("\t[-d/--debug]\n");
	printf("\t[-f/--force]\n");
	printf("\t[-h/-?/--help]\n");
	printf("\t[-v/--verbose]\n");
	printf("\t[-V/--version]\n");
	printf("\tLogicalVolume[Path] [LogicalVolume[Path]...]\n\n");
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "Adfh?vV";
	struct option	long_opts[] = {
				{ "autobackup",	required_argument, NULL, 'A'},
				{ "debug",	no_argument,       NULL, 'd'},
				{ "force",	no_argument,       NULL, 'f'},
				{ "help",	no_argument,       NULL, 'h'},
				{ "verbose",	no_argument,       NULL, 'v'},
				{ "version",	no_argument,       NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'A':
			// -A is ignored by EVMS
			if ( optarg[0] != 'n' && optarg[0] != 'y' ) {
				printf( "%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'f':
			// -f is ignored by EVMS
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t		lv_handle;
	char			* lv_names[256]	= {0};
	int			number_of_lvs;
	int			log_level = DEFAULT;
	int			i, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Check for LV names
	if ( optind == argc) {
		printf("%s -- please enter a logical volume path\n\n", cmd);
		return EINVAL;
	}
	number_of_lvs = argc - optind;

	// Copy the LV names from the command line, detecting any duplicates.
	rc = remove_duplicates(&argv[optind], lv_names, number_of_lvs);
	if (rc) {
		printf("%s -- Duplicate LVs specified. Please only specify each LV once\n", cmd);
		return rc;
	}

	// Open the EVMS Engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		return rc;
	}

	// For each LV specified on the command line, find the Engine handle
	// and make sure it belongs to LVM. If the LV is a compatibility
	// volume, call evms_destroy. If it has no volume, call evms_remove.
	for ( i = 0; i < number_of_lvs; i++ ) {
		rc = get_region_handle(lv_names[i], &lv_handle);
		if (rc) {
			evms_close_engine();
			return rc;
		}

		// Does this region belong to LVM?
		if ( ! is_lvm_region(lv_handle) ) {
			printf("%s -- %s is not an LVM LV...skipping\n", cmd, lv_names[i]);
			continue;
		}

		// If the region has a compatibility volume, the volume must be
		// destroyed. If there is no volume, just remove the region. If
		// there is an EVMS volume, don't delete.
		rc = revert_volume_from_region(lv_handle);
		if (rc) {
			printf("%s -- Error removing LV %s (%d)\n", cmd, lv_names[i], rc);
			evms_close_engine();
			return rc;
		}

		printf("%s -- logical volume %s successfully removed\n", cmd, lv_names[i]);
	}

	// Write everything to disk.
	rc = evms_commit_changes(NULL);
	if (rc) {
		printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		evms_close_engine();
		return rc;
	}

	if ( opts.verbose ) {
		printf("%s -- All changes committed to disk\n", cmd);
	}

	evms_close_engine();
	return 0;
}

