/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: evms.c
 */

/*
 * Change History:
 *
 * 7/2001  B. Rafanello  Initial version.
 *
 */

/*
 *
 */

/* Identify this file. */
#define EVMS_C

/*--------------------------------------------------
 * Necessary include files
 --------------------------------------------------*/
#include <sys/types.h>
#include <string.h>        /* strcat, strcpy */
#include <ctype.h>         /* toupper */
#include <stdio.h>         /* printf */
#include <stdlib.h>        /* strtol */

#include "dlist.h"          /* dlist_t */
#include "error.h"         /* ReportError */
#include "token.h"
#include "scanner.h"       /* SetInput */
#include "parser.h"        /* Parse */
#include "interpreter.h"
#include "frontend.h"

#ifdef DEBUG_PARSER

#include "screener.h"

#endif


/*--------------------------------------------------
 * Global Variables
 --------------------------------------------------*/
BOOLEAN                     Interactive = TRUE;      /* Controls whether or not EVMS will prompt the user for commands.
                                                        If it is FALSE, then input is expected from a file.             */
engine_message_functions_t  Messaging_Functions;     /* Used to pass call back functions to the EVMS Engine.  These 
                                                        functions allow the EVMS Engine to provide status and error messages
                                                        for display to the user during complicated operations.               */

/*--------------------------------------------------
 * Private Constants
 --------------------------------------------------*/
#define VERSION_NUMBER    "1.0.0"
#define REQUIRED_ENGINE_VERSION_MAJOR   4
#define REQUIRED_ENGINE_VERSION_MINOR   0
#define REQUIRED_ENGINE_VERSION_PATCH   1



/*--------------------------------------------------
 * There are No Private Type Definitions
 --------------------------------------------------*/




/*--------------------------------------------------
  Private Global Variables.
--------------------------------------------------*/
static char                        CommandLine[4096];   /* Buffer to store the reconstructed command line in. */
#ifdef DEBUG_PARSER

char *   IntegerDataStr = "INTEGER DATA";
char *   RealNumberDataStr = "REAL NUMBER DATA";
char *   StringDataStr = "STRING DATA";
char *   FilterStr = "FILTER";
char *   FeatureNameStr = "FEATURE NAME";
char *   PluginNameStr = "PLUG-IN NAME";
#endif




/*--------------------------------------------------
 Local Function Prototypes
--------------------------------------------------*/
static void Display_Command_Line_Help( void );
#ifndef DEBUG_PARSER
static int Display_Engine_Message(char   * message_text,
                                  int    * answer,
                                  char * * choices);
#endif
#ifdef DEBUG_PARSER
static int Map_Command_List_Entries (ADDRESS   Object,
                                     TAG       ObjectTag,
                                     uint      ObjectSize,
                                     ADDRESS   ObjectHandle,
                                     ADDRESS   Parameters);
#endif


/*--------------------------------------------------
 * Public Functions Available
 --------------------------------------------------*/


/*********************************************************************/
/*                                                                   */
/*   Function Name: main                                             */
/*                                                                   */
/*   Descriptive Name:  The entry point for the EVMS.EXE program.    */
/*                                                                   */
/*   Input: int argc - The number of parameters on the command line. */
/*          char * argv - An array of pointers to the text of each   */
/*                        parameter found on the command line.       */
/*                                                                   */
/*   Output: If Success : The program's exit code will be 0.         */
/*                                                                   */
/*           If Failure : The program's exit code will be 1.         */
/*                                                                   */
/*   Error Handling: If an error occurs, all memory allocated by this*/
/*                   program is freed and an error message is output */
/*                   to the user.                                    */
/*                                                                   */
/*   Side Effects: The state of the disks, partitions, and/or volumes*/
/*                 in the system may be altered.  Nodes may be       */
/*                 created or removed from the /dev tree.            */
/*                                                                   */
/*   Notes:                                                          */
/*                                                                   */
/*********************************************************************/
int main ( int argc, char * argv[])
{
   evms_version_t              Engine_Version;          /* Used to check the version of the EVMS Engine to ensure we have a version we can work with. */
   uint                        ParameterIndex;          /* Used to access the item in argv being processed. */
   dlist_t                     Command_List;            /* Used to hold the list of command nodes returned by the
                                                           parser.  This list will be given to the Stack Machine,
                                                           which will then act as an interpreter for the command
                                                           nodes.                                                 */
   int                         ReturnValue = 0;         /* The value to return.  A value is returned in case this
                                                           program was called from a batch file.  A return value
                                                           of 0 indicates successful program completion while
                                                           any non-zero value is a program defined error code.
                                                           We will assume success and initialize ReturnValue to 0. */
   int                         Index = 0;               /* Used when converting strings to uppercase. */
   int                         Length = 0;              /* Used when converting strings to uppercase. */
   char *                      Token;                   /* Used for parameter processing. */
   BOOLEAN                     One_Commit = FALSE;      /* Flag to control how often commits are performed.  If TRUE,
                                                           then only one commit operation will be performed, just
                                                           before the program exits.  If FALSE, then a commit is
                                                           performed after each command passed into the stack machine. */
   BOOLEAN                     Continuous = TRUE;       /* Flag to control how many commands will be prompted for.
                                                           If the flag is FALSE, then we will only prompt once for a
                                                           command.  If TRUE, then we will continuously prompt for a
                                                           command until the command line returned is NULL.            */
   BOOLEAN                     ParseOnly = FALSE;       /* Controls whether or not the output of the parser is set to the
                                                           interpreter for execution.                                      */
   BOOLEAN                     Verbose = FALSE;         /* Has verbose mode been specified? */
   debug_level_t               Debug_Level = DEFAULT;   /* Controls the level of debugging information output to
                                                           the user.                                              */

#ifdef DEBUG_PARSER
   uint     Indentation_Level = 0; /* Used when displaying command nodes produced by the parser. */

#endif

   printf("\n\nEVMS Command Line Interpreter Version %s\n\n",VERSION_NUMBER);

  if ( argc > 1 )
  {

    for ( ParameterIndex = 1; ParameterIndex < argc; ParameterIndex++ )
    {

      Token = argv[ParameterIndex];

      if ( ( strcmp("-c", Token) == 0) ||
           ( strcmp("-C", Token) == 0) ||
           ( strcmp("--c", Token) == 0) ||
           ( strcmp("--C", Token) == 0)
         )
      {
        if ( One_Commit )
          printf("\n\nThe -c command line option has been specified more than once!\n\n");

        One_Commit = TRUE;

      }

      if ( ( strcmp("-v", Token) == 0) ||
           ( strcmp("-V", Token) == 0) ||
           ( strcmp("--V", Token) == 0) ||
           ( strcmp("--v", Token) == 0)
         )
      {
        if ( Verbose )
          printf("\n\nMore than one of the verbose options (-v, -v0, -v1,-v2) have been specified!\n\n");

        Verbose = TRUE;
#ifndef DEBUG_PARSER
        Verbose_Mode = 2;
#endif
      }
      
      if ( ( strcmp("-v0", Token) == 0) ||
           ( strcmp("-V0", Token) == 0) ||
           ( strcmp("--V0", Token) == 0) ||
           ( strcmp("--v0", Token) == 0)
         )
      {
        if ( Verbose )
          printf("\n\nMore than one of the verbose options (-v, -v0, -v1,-v2) have been specified!\n\n");

        Verbose = TRUE;
#ifndef DEBUG_PARSER
        Verbose_Mode = 0;
#endif
      }
      
      if ( ( strcmp("-v1", Token) == 0) ||
           ( strcmp("-V1", Token) == 0) ||
           ( strcmp("--V1", Token) == 0) ||
           ( strcmp("--v1", Token) == 0)
         )
      {
        if ( Verbose )
          printf("\n\nMore than one of the verbose options (-v, -v0, -v1,-v2) have been specified!\n\n");

        Verbose = TRUE;
#ifndef DEBUG_PARSER
        Verbose_Mode = 1;
#endif
      }
      
      if ( ( strcmp("-v2", Token) == 0) ||
           ( strcmp("-V2", Token) == 0) ||
           ( strcmp("--V2", Token) == 0) ||
           ( strcmp("--v2", Token) == 0)
         )
      {
        if ( Verbose )
          printf("\n\nMore than one of the verbose options (-v, -v0, -v1,-v2) have been specified!\n\n");

        Verbose = TRUE;
#ifndef DEBUG_PARSER
        Verbose_Mode = 2;
#endif
      }
      
      if ( ( strcmp("-p", Token) == 0) ||
           ( strcmp("-P", Token) == 0) ||
           ( strcmp("--p", Token) == 0) ||
           ( strcmp("--P", Token) == 0)
         )
      {
        if ( ParseOnly )
          printf("\n\nThe -p command line option has been specified more than once!\n\n");

        ParseOnly = TRUE;

      }

      if ( ( strcmp("-S", Token) == 0) ||
           ( strcmp("-s", Token) == 0) ||
           ( strcmp("--S", Token) == 0) ||
           ( strcmp("--s", Token) == 0)
         )
      {

        /* Have we seen this command line option already? */
        if ( ! Continuous )
          printf("\n\nThe -s command line option has been specified more than once!\n\n");

        Continuous = FALSE;

      }

      if ( ( strcmp("-H", Token) == 0) ||
           ( strcmp("-h", Token) == 0) ||
           ( strcmp("--H", Token) == 0) ||
           ( strcmp("--h", Token) == 0) ||
           ( strcasecmp("-HELP",Token) == 0) ||
           ( strcasecmp("--HELP",Token) == 0)
         )
      {
        
        /* We have a help request.  This takes precedence over all other command line input.
           Output a help message and abort.                                                   */
        Display_Command_Line_Help();

        exit(0);

      }

      if ( ( strcmp("-F", Token) == 0) ||
           ( strcmp("-f", Token) == 0) ||
           ( strcmp("--f", Token) == 0) ||
           ( strcmp("--F", Token) == 0)
         )
      {
        /* We should have a filename as the next argv. */

        /* Use the scanner's SetInput function to set up the file as the source of input for parsing. */
        if ( ( ( ParameterIndex + 1 ) >= argc ) ||
             ( ! SetInput(TRUE, argv[ParameterIndex + 1]) )
           )
        {

          printf("\n\nBad filename specified.\n\n");

          exit(ENOENT);

        }

        Interactive = FALSE;
        ParameterIndex++;

      }


      if ( ( strcmp("-D", Token) == 0) ||
           ( strcmp("-d", Token) == 0) ||
           ( strcmp("--d", Token) == 0) ||
           ( strcmp("--D", Token) == 0)
         )
      {
        /* We should have a debug level as the next argv. */
        
        if ( ( ParameterIndex + 1 ) >= argc ) 
        {

          printf("\n\nNo debug level specified.\n\n");

          exit(EINVAL);

        }

        ParameterIndex++;

        /* Get the debug level to use. */
        Token = argv[ParameterIndex];
        Length = strlen(Token);

        /* Convert to upper case. */
        for ( Index = 0; Index < Length; Index++)
           Token[Index] = toupper(Token[Index]);

        /* Look for a valid debug level. */
        if ( strcmp("DEFAULT", Token) == 0 )
          Debug_Level = DEFAULT;
        else if ( strcmp("CRITICAL", Token) == 0 )
          Debug_Level = CRITICAL;
        else if ( strcmp("SERIOUS", Token) == 0 )
          Debug_Level = SERIOUS;
        else if ( strcmp("ERROR", Token) == 0 )
          Debug_Level = ERROR;
        else if ( strcmp("WARNING", Token) == 0 )
          Debug_Level = WARNING;
        else if ( strcmp("DETAILS", Token) == 0 )
          Debug_Level = DETAILS;              
        else if ( strcmp("EXTRA", Token) == 0 )
          Debug_Level = EXTRA;
        else if ( strcmp("ENTRY_EXIT", Token) == 0 )
          Debug_Level = ENTRY_EXIT;
        else if ( strcmp("EVERYTHING", Token) == 0 )
          Debug_Level = EVERYTHING;
        else if ( strcmp("DEBUG", Token) == 0 )
          Debug_Level = DEBUG;
        else
        {

          printf("\n\nInvalid debug level specified.\n\n");

          exit(EINVAL);

        }

      }

    }

  }

  /* Do we have conflicting command line options specified? */
  if ( ( ! Interactive ) && ( ! Continuous ) )
    printf("\n\nThe -f and -s options can not be specified together!  Ignoring the -s option!\n\n");

#ifndef DEBUG_PARSER

   if ( !ParseOnly )
   {

     /* Get the version of the EVMS Engine. */
     evms_get_api_version(&Engine_Version);

     /* Is this a version of the EVMS Engine we can work with? */
     if ( ( Engine_Version.major != REQUIRED_ENGINE_VERSION_MAJOR ) ||
          ( Engine_Version.minor < REQUIRED_ENGINE_VERSION_MINOR ) ||
          ( ( Engine_Version.minor == REQUIRED_ENGINE_VERSION_MINOR ) &&
            ( Engine_Version.patchlevel < REQUIRED_ENGINE_VERSION_PATCH )
          )
        ) {
       printf("\n\nIncorrect version of the EVMS Engine.  This version of the EVMS Command Line\n");
       printf("Interpreter requires a version of the EVMS Engine which supports the EVMS\n");
       printf("Application Programming Interface version %d, minor level %d (or greater).\n",REQUIRED_ENGINE_VERSION_MAJOR,REQUIRED_ENGINE_VERSION_MINOR);
       printf("If using version %d, minor %d, then the patch level must be at least %d.\n",REQUIRED_ENGINE_VERSION_MAJOR,REQUIRED_ENGINE_VERSION_MINOR,REQUIRED_ENGINE_VERSION_PATCH);

       printf("\nThe version of the EVMS Engine currently loaded supports the EVMS Application\n");
       printf("Programming Interface version %d, minor level %d, patch level %d.\n\n",Engine_Version.major,Engine_Version.minor,Engine_Version.patchlevel);

       exit(EPERM);
     }

     /* Set up the messaging functions for the EVMS Engine. */
     Messaging_Functions.user_message = Display_Engine_Message;
     Messaging_Functions.user_communication = NULL;

     /* Open the EVMS Engine. */
     ReturnValue = evms_open_engine(ENGINE_READWRITE,&Messaging_Functions,Debug_Level,DEFAULT_LOG_FILE);

     if ( ReturnValue != 0 )      /* What should the return value be!!!!  Get the engine guys to clearly document the return codes! */
     {
       printf("\n\nUnable to open the EVMS Engine.  Engine returns code %i.\n\n",ReturnValue);
       exit(ReturnValue);

     }

   }
#endif

  /* Are we interactive or batch? */
  if ( Interactive )
  {

    /* Get command lines from user. */
    do
    {

      /* Initialize ReturnValue in case we leave the loop early due to the user not providing a command to interpret. */
      ReturnValue = 0;

      /* Set the input source. */
      if ( ! SetInput(FALSE, CommandLine) )
      {

        printf("\n\nInternal Error!!!!\n\n");

        abort();

      }

      /* Null out the buffer so we have a clean buffer to work with. */
      for ( ParameterIndex = 0; ParameterIndex < 4096; ParameterIndex++ )
        CommandLine[ParameterIndex] = 0;

      /* Put out our command prompt. */
      printf("EVMS: ");

      /* Get a command line from the user. */
      fgets(CommandLine, sizeof(CommandLine), stdin);

      if ( ( strlen(CommandLine) <= 1 ) &&
           ( ( CommandLine[0] == 0 ) ||
             ( CommandLine[0] == '\n' )
           )
         )
        break;

      /* Now lets parse the command line. */
      Command_List = Parse_Command_Line();

      /* Was there a parsing error? */
      if ( Command_List != NULL )
      {

#ifdef DEBUG_PARSER
        ReturnValue = ForEachItem(Command_List, Map_Command_List_Entries, &Indentation_Level, TRUE);
#else
        if ( !ParseOnly )
        {
          /* Now lets execute the command line.  Remember to tell the interpreter whether
             or not to commit changes to disk after each command.                          */
          ReturnValue = Interpret_Commands(Command_List, One_Commit, Interactive, Debug_Level);
        }

#endif

      }
      else
        if ( ! Continuous )
          ReturnValue = -1;

    } while ( Continuous );

  }
  else
  {
    /* Now lets parse the command line. */
      Command_List = Parse_Command_Line();

    /* Did we succeed? */
    if ( Command_List != NULL )
    {

#ifdef DEBUG_PARSER
      ReturnValue = ForEachItem(Command_List, Map_Command_List_Entries, &Indentation_Level, TRUE);
#else
      if ( !ParseOnly )
      {
        /* Now lets execute the command line.  Remember to tell the interpreter whether
           or not to commit changes to disk after each command.                          */
        ReturnValue = Interpret_Commands(Command_List, One_Commit, Interactive, Debug_Level);

      }

#endif

    }
    else
      ReturnValue = -1;

  }

#ifndef DEBUG_PARSER
  /* If we have not already committed the changes to disk, do so now. */
  if ( (ReturnValue == 0) && One_Commit )
  {

    /* Commit the changes made to disk. */
    ReturnValue = Commit_Changes();
  }


  if (!ParseOnly)
    evms_close_engine();
#endif

  return ReturnValue;

}



/*--------------------------------------------------
 * Private Functions Available
 --------------------------------------------------*/

void Display_Command_Line_Help( void )
{

  printf("\n\nThe EVMS Command Line Interpreter is invoked using the following syntax:\n");
  printf("\n");
  printf("evms [ -c ] [ -d <Debug Level> ] [ -f <Filename> ] [ -s ] [ -p ] \n     [ -v | -v0 | -v1 | -v2 ] [ -h ] \n");
  printf("\n");
  printf("where:\n");
  printf("\n");
  printf("     -c indicates that changes are not to be committed to disk until the\n");
  printf("        user exits the EVMS Command Line Interpreter.  Normally changes\n");
  printf("        are committed to disk after each individual command is completed,\n");
  printf("        even when there are multiple commands specified on the same command\n");
  printf("        line.\n");
  printf("     -d <Debug Level> sets the debug level, which controls how much information\n");
  printf("        is logged in the evmsEngine.log file located in /var/log directory.\n");
  printf("        <Debug Level> is either Default, Critical, Serious, Error, Warning, \n");
  printf("        Details, Extra, Entry_Exit, Everything, or Debug.\n");
  printf("     -f <Filename> tells the EVMS Command Line Interpreter to accept input \n");
  printf("        from the file specified by <Filename>.  When all of the commands in\n");
  printf("        <Filename> have been processed, the EVMS Command Line Interpreter\n");
  printf("        will exit.\n");
  printf("     -s tells the EVMS Command Line Interpreter to prompt for a single command\n");
  printf("        line, execute the command line, and then exit.  The EVMS Command Line\n");
  printf("        Interpreter normally prompts for command lines until it finds an\n");
  printf("        empty command line, at which point it exits.\n");
  printf("     -p tells the EVMS Command Line Interpeter to parse commands only.\n");
  printf("        Errors found during parsing will be reported.  Commands will NOT\n");
  printf("        be executed.\n");
  printf("     -v Enable verbose mode.  Currently, this causes EVMS Engine status\n");
  printf("        messages to be displayed whenever changes are committed to disk.\n");
  printf("        The messages are displayed using a single line on the screen - i.e.\n");
  printf("        the current message is erased and the next message is displayed in\n");
  printf("        in its place.\n");
  printf("    -v0 Disable verbose mode.  Currently, this causes EVMS Engine status\n");
  printf("        messages to be discarded whenever changes are committed to disk.\n");
  printf("    -v1 Enable verbose mode 1.  Currently, this causes EVMS Engine status\n");
  printf("        messages to be displayed whenever changes are committed to disk.\n");
  printf("        The messages are displayed one per line on the screen.\n");
  printf("    -v2 Enable verbose mode 2.  Currently, this causes EVMS Engine status\n");
  printf("        messages to be displayed whenever changes are committed to disk.\n");
  printf("        The messages are displayed using a single line on the screen - i.e.\n");
  printf("        the current message is erased and the next message is displayed in\n");
  printf("        in its place.\n");
  printf("     -h invokes this help screen.  To get help on commands accepted by the\n");
  printf("        EVMS Command Line Interpreter, invoke the interpeter, enter the\n");
  printf("        word 'help' at the prompt, and press enter.\n");
  printf("\n\n");

  return;
}

#ifndef DEBUG_PARSER

static int Display_Engine_Message(char   * message_text,
                                  int    * answer,
                                  char * * choices)
{
  int        Index;
  int        I;
  long int   UserChoice;
  char       Buffer[10];
  char *     Bad_Characters;

  if ( message_text != NULL )
    printf("\n\n%s\n\n",message_text);

  if ( ( answer != NULL ) && ( choices != NULL ) )
  {
    /* A choice needs to be made.  The default should be set in answer already.
       If we are in batch mode (i.e. processing a command file) then we will
       automatically accept the default and just print a message that the 
       default has been accepted and what the default is.  If we are not in
       batch mode, then we will display a list of the choices and let the
       user choose one.                                                           */
    if (  ! Interactive )
    {

      printf("The EVMS Command Line Interpreter is processing a command file - therefore user\n");
      printf("interaction is not allowed.  Thus, the EVMS Command Line Interpreter has\n");
      printf("accepted the default choice.  The default choice is:\n\n%s\n\n",choices[*answer]);

    }
    else
    {
      /* Initialize our loop control variables. */
      Bad_Characters = NULL;
      UserChoice = 1;
      Index = UserChoice;

      /* Remain in the loop until the user makes a valid selection. */
      do
      {

        if ( (Bad_Characters != NULL) || (UserChoice > Index) || (UserChoice == 0) )
          printf("\n\n%s does not correspond to any of the available responses.\nPlease try again.\n\n", Buffer);

        printf("The following responses are available:\n\n");

        for (Index = 0; choices[Index] != NULL; Index++)
        {
          if (Index != *answer )
            printf(" %d = %s\n",Index + 1, choices[Index]);
          else
            printf("*%d = %s\n",Index + 1, choices[Index]);
        }

        printf("\nThe default choice is marked with an *.\n");
        printf("Please enter the number corresponding to your choice: ");

        /* Get the user's choice. */
        Bad_Characters = NULL;
        UserChoice = 0;
        for (I = 0; I < sizeof(Buffer); I++)
          Buffer[I] = 0;
        fgets(Buffer, sizeof(Buffer), stdin);

        /* Did the user make a choice? */
        if ( ( strlen(Buffer) == 0 ) || ( strcmp(Buffer,"\n") == 0 ) )
        {
          /* No choice made -- use the default. */
          UserChoice = *answer + 1;
          break;
        }

        /* The user made a choice.  Is it valid? */
        UserChoice = strtol(Buffer, &Bad_Characters, 10);
        if ( strcmp(Bad_Characters,"\n") == 0 )
          Bad_Characters = NULL;

      } while ( (Bad_Characters != NULL) || (UserChoice > Index) || (UserChoice == 0) );

      /* Save the user's choice. */
      *answer = UserChoice - 1;

    }

  }

  return 0;
}

#endif

#ifdef DEBUG_PARSER

static int Map_Command_List_Entries (ADDRESS   Object,
                                         TAG       ObjectTag,
                                         uint      ObjectSize,
                                         ADDRESS   ObjectHandle,
                                         ADDRESS   Parameters)
{

  /* Establish easy access to the current node. */
  Executable_Node * Node_To_Map = (Executable_Node *) Object;
  uint *            Indentation_Level = (uint *) Parameters;
  uint              count;
  int               Error;
  char *            Node_Name = NULL;

  /* Sanity check!  Is the node of the expected type? */
  if ( ObjectTag != STACK_NODE )
  {

    /* We have an illegal node in the list!  Abort! */
    return DLIST_ITEM_TAG_WRONG;

  }

  /* Map the current node. */

  if ( *Indentation_Level == 0 )
    printf("\n\n");

  /* Do the indentation spaces first */
  if ( *Indentation_Level > 5 )
    for (count = 0; count < *Indentation_Level; count++)
      printf(" ");

  /* Now do the vertical bar. */
  if ( *Indentation_Level > 0)
  {
    printf("|");

    /* Now do the dashes. */
    for ( count = 0; count < 4; count++)
      printf("-");

  }

  /* Now output the information for the node. */
  switch ( Node_To_Map->NodeType )
  {
    case StringData:
      Node_Name = StringDataStr;
      break;
    case IntegerData:
      Node_Name = IntegerDataStr;
      break;
    case RealNumberData:
      Node_Name = RealNumberDataStr;
      break;
    case Assign_Node:
      Node_Name = AssignStr;
      break;
    case Remove_Node:
      Node_Name = RemoveStr;
      break;
    case Allocate_Node:
      Node_Name = AllocateStr;
      break;
    case Probe_Node:
      Node_Name = ProbeStr;
      break;
    case Create_Node:
      Node_Name = CreateStr;
      break;
    case Set_Node:
      Node_Name = SetStr;
      break;
    case Delete_Node:
      Node_Name = DeleteStr;
      break;
    case Expand_Node:
      Node_Name = ExpandStr;
      break;
    case Shrink_Node:
      Node_Name = ShrinkStr;
      break;
    case Rename_Node:
      Node_Name = RenameStr;
      break;
    case Revert_Node:
      Node_Name = RevertStr;
      break;
    case Query_Node:
      Node_Name = QueryStr;
      break;
    case Help_Node:
      Node_Name = HelpStr;
      break;
    case Filter:
      Node_Name = FilterStr;
      break;
    case Feature_ID:
      Node_Name = FeatureStr;
      break;
    case Feature_Name:
      Node_Name = FeatureNameStr;
      break;
    case Plugin_ID:
      Node_Name = PluginStr;
      break;
    case Plugin_Name:
      Node_Name = PluginNameStr;
      break;
    case Format_Node:
      Node_Name = FormatStr;
      break;
    case Unformat_Node:
      Node_Name = UnformatStr;
      break;
    case Check_Node:
      Node_Name = CheckStr;
      break;
    case Defragment_Node:
      Node_Name = DefragmentStr;
      break;
    case Name_Value_Pair:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_Value_Pair.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_Value_Pair.  Node Value is NULL!\n");
      break;
    case ObjectList:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: ObjectList.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: ObjectList.  Node Value is NULL!\n");
      break;
    case Translate_Name_To_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Translate_Name_To_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Translate_Name_To_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Volume_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Volume_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Volume_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Object_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Object_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Object_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Region_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Region_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Region_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Container_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Container_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Container_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Segment_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Segment_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Segment_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Disk_Handle:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Name_To_Disk_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Disk_Handle.  Node Value is NULL!\n");
      break;
    case Name_To_Plugin_Handle:
      if ( Node_To_Map->NodeValue != NULL )
        printf("Node: Name_To_Plugin_Handle.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Name_To_Plugin_Handle.  Node Value is NULL!\n");
      break;
    case Units_Node:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Units_Node.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Units_Node.  Node Value is NULL!\n");
      break;
    case Task_Adjust:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Task_Adjust.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Task_Adjust.  Node Value is NULL!\n");
      break;
    case Topmost_Objects:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Topmost_Objects.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Topmost_Objects.  Node Value is NULL!\n");
      break;
    case Precision_Node:
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Precision_Node.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Precision_Node.  Node Value is NULL!\n");
      break;
    case Query_Extended_Info :
      if (Node_To_Map->NodeValue != NULL)
        printf("Node: Query_Extended_Info.  Node Value: %s\n",(char *) Node_To_Map->NodeValue);
      else
        printf("Node: Query_Extended_Info.  Node Value is NULL!\n");
      break;
    default:
      printf("Invalid node type!\n");
    break;
  }

  if ( Node_Name != NULL )
  {

    if (Node_To_Map->NodeValue != NULL)
      printf("Node: %s.  Node Value: %s\n", Node_Name,(char *) Node_To_Map->NodeValue);
    else
      printf("Node: %s.  Node Value is NULL!\n", Node_Name);
    
  }

  /* If the node has children, we must map them also. */
  if ( Node_To_Map->Children != NULL )
  {

    /* Map the children. */
    *Indentation_Level = *Indentation_Level + 5;
    Error = ForEachItem(Node_To_Map->Children, Map_Command_List_Entries, Indentation_Level, TRUE);
    *Indentation_Level = *Indentation_Level - 5;

    if ( Error != DLIST_SUCCESS )
      return Error;

  }

  return DLIST_SUCCESS;

}

#endif
