/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid0_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid0 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#define MY_PLUGIN raid0_plugin
#include "md.h"
#include "raid0_mgr.h"
#include "raid0_discover.h"



// Global variables
static engine_mode_t            open_mode;	// Still don't know what this is for. :)



/* Function: raid0_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid0_setup_evms_plugin(engine_mode_t        mode,
				   engine_functions_t * functions) {
	int rc = 0;

	/* Parameter check */
	if (!functions) {
		return EINVAL;
	}

	open_mode = mode;
	EngFncs = functions;

	LOG_ENTRY;
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	RETURN(rc);
}


/****** Region Checking Functions ******/


/* All of the following md_can_* functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid0_can_delete
 *
 * Can we remove the specified MD logical volume?  Yes.
 */
static int raid0_can_delete( storage_object_t * region ) {

	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid0_can_expand
 *
 * No expand, just return 0 to stop recursion.
 */
static int raid0_can_expand(storage_object_t    * region,
			    u_int64_t               * expand_limit,
			    dlist_t                 expansion_points ) {
	
	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid0_can_expand_by
 *
 * RAID0 does not allow its children to expand.
 */

static int raid0_can_expand_by( storage_object_t * child_object,
				u_int64_t        * size ) {

	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_can_shrink
 *
 * Just like can_expand, but in the opposite direction.
 */
static int raid0_can_shrink(storage_object_t * region,
			    u_int64_t        * shrink_limit,
			    dlist_t            shrink_points ) {
	
	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid0_can_shrink_by
 *
 * RAID0 does not allow its children to shrink.
 */
static int raid0_can_shrink_by( storage_object_t * region,
				u_int64_t        * size ) {
	
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_can_move
 *
 */
static int raid0_can_move( storage_object_t * region ) {
	
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_can_set_volume
 *
 */
static int raid0_can_set_volume( storage_object_t * region,
				 boolean            flag ) {
	
	LOG_ENTRY;
	RETURN(0);
}


/* Function: raid0_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid0_discover( dlist_t input_list,
			   dlist_t output_list,
			   boolean final_call ) {
	int count = 0;

	LOG_ENTRY;

	// Parameter check
	if (!input_list || !output_list) {
		RETURN(EFAULT);
	}

	md_discover_volumes(input_list, output_list);
	LOG_DETAILS("PV discovery complete.\n");

	// LV discovery and exporting
	raid0_discover_regions(output_list, &count, final_call);
	LOG_DETAILS("Volume (region) discovery complete.\n");

	RETURN(count);
}



/****** Region Functions ******/


static int get_create_options( option_array_t * options,
			       u_int32_t      * chunk_size) {
	int i;
	int rc = 0;

	LOG_ENTRY;

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case MD_OPTION_CHUNK_SIZE_INDEX:
				// Option is in 1K byte blocks
				*chunk_size = options->option[i].value.ui32;
				break;

			default:
				break;
			}

		} else {

			if (strcmp(options->option[i].name, MD_OPTION_CHUNK_SIZE_NAME) == 0) {
				// Option is in 1K byte blocks
				*chunk_size = options->option[i].value.ui32;
			}
		}
	}

	RETURN(rc);
}


/* Function: raid0_create
 *
 *  Create a new MD RAID0 volume using the specified objects and options.
 */
static int raid0_create( dlist_t          objects,
			 option_array_t * options,
			 dlist_t          new_region_list ) {

	md_volume_t      * volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int tag, waste;
	int i, spare_disks = 0, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	int chunk_size_blocks = MD_DEFAULT_CHUNK_SIZE;

	LOG_ENTRY;

	// Parameter check
	if (!objects || !options || !new_region_list) {
		RETURN(EFAULT);
	}

	rc = GetListSize(objects, &nr_disks);
	if (rc) {
		LOG_CRITICAL("DLIST error getting number of objects. rc = %d.\n",rc);
		RETURN(EINVAL);
	}

	if (nr_disks > MAX_MD_DEVICES) {
		LOG_ERROR("Too many objects (%d) given. Maximum is %d.\n", nr_disks, MAX_MD_DEVICES);
		RETURN(EINVAL);
	}

	if (md_allocate_memory((void**)&volume, sizeof(md_volume_t) )) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		RETURN(ENOMEM);
	}

	while (!(rc = BlindExtractObject(objects, &waste, (TAG *)&tag, NULL, (void *)&object))) {
		size = min(size, object->size);	 // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
	}

	get_create_options(options, &chunk_size_blocks);

	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size_blocks, ((size << EVMS_VSECTOR_SIZE_SHIFT) / BLOCK_SIZE));

	md_create_first_superblock(volume, disk, pers_to_level(RAID0), chunk_size_blocks, size, nr_disks, spare_disks, (1 << MD_SB_CLEAN));

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
	}

	volume->personality = RAID0;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;

	rc = raid0_create_region(volume, new_region_list, TRUE);
	RETURN(rc);
}


/* Function: raid0_delete
 *
 *  Remove the specified region.
 */
static int raid0_delete( storage_object_t * region,
			 dlist_t            children ) {

	md_volume_t * volume;
	int           rc;

	LOG_ENTRY;

	// Check that this region can be removed.
	if ((rc = raid0_can_delete(region))) {
		RETURN(rc);
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume);
	EngFncs->free_region(region);

	RETURN(rc);
}


/* Function: raid0_expand
 */

static int raid0_expand( storage_object_t * region,
			 storage_object_t * expand_object,
			 dlist_t            input_objects,
			 option_array_t   * options ) {
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_shrink
 */
static int raid0_shrink( storage_object_t * region,
			 storage_object_t * shrink_object,
			 dlist_t            input_objects,
			 option_array_t   * options ) {
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_move
 */
static int raid0_move( storage_object_t * source,
		       storage_object_t * target,
		       option_array_t   * options ) {
	LOG_ENTRY;
	RETURN(ENOSYS);
}


/* Function: raid0_set_volume
 *
 *  MD doesn't really care if its regions are made into volumes.
 */
static void raid0_set_volume( storage_object_t * region,
			      boolean            flag ) {
	LOG_ENTRY;
	LOG_EXIT(0);
}


static int get_child_run( md_volume_t       * volume,
			  lsn_t               lsn,
			  sector_count_t      count,
			  storage_object_t ** child_object,
			  lsn_t             * child_lsn,
			  sector_count_t    * child_count) {

	unsigned int sect_in_chunk;
	unsigned int chunk_size_in_sectors;
	unsigned int chunksize_bits;
	raid0_conf_t *conf = mdvol_to_conf(volume);
	struct raid0_hash * hash;
	struct strip_zone * zone;
	unsigned long chunk;

	LOG_ENTRY;

	chunk_size_in_sectors = volume->super_array[0]->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
	chunksize_bits = calc_log2(chunk_size_in_sectors);
	hash = conf->hash_table + (lsn / conf->smallest_zone->size);

	/* Sanity checks */
	if (!hash) {
		LOG_WARNING("hash == NULL for lsn %ld\n", lsn);
		RETURN(EIO);
	}

	if (!hash->zone0) {
		LOG_WARNING("hash->zone0 == NULL for lsn %ld\n", lsn);
		RETURN(EIO);
	}

	if (lsn >= (hash->zone0->size + hash->zone0->zone_offset)) {
		if (!hash->zone1) {
			LOG_WARNING("hash->zone1 == NULL for lsn %ld\n", lsn);
			RETURN(EIO);
		}
		zone = hash->zone1;
	} else {
		zone = hash->zone0;
	}

	sect_in_chunk = lsn & (chunk_size_in_sectors - 1);
	chunk = (lsn - zone->zone_offset) / (zone->nb_dev << chunksize_bits);
	*child_object = zone->dev[(lsn >> chunksize_bits) % zone->nb_dev];
	*child_lsn = ((chunk << chunksize_bits) + zone->dev_offset) + sect_in_chunk;
	*child_count = min(count, chunk_size_in_sectors - sect_in_chunk);

	RETURN(0);
}


/* Function: raid0_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid0_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count ) {

	md_volume_t     *volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	storage_object_t * child_object;
	lsn_t           child_lsn;
	sector_count_t  child_count;

	LOG_ENTRY;

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect.\n ",volume->name);
		RETURN(EIO);
	}

	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume,lsn,count,&child_object,&child_lsn,&child_count);
		if (rc == 0) {
			rc = KILL_SECTORS(child_object, child_lsn, child_count);
			count -= child_count;
		}
	}
	RETURN(rc);
}


/* Function: raid0_commit_changes
 *
 *  Newly created regions may have the ZERO flag set. In this case, clear
 *  the first 1k of the LV.
 *
 *  All other commit operations are done in commit_container_changes.
 */
static int raid0_commit_changes( storage_object_t * region,
				 uint               phase ) {

	md_volume_t * volume = (md_volume_t *)region->private_data;
	int         rc = 0;

	LOG_ENTRY;

	// Make sure this region belongs to MD, and is dirty
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		RETURN(EINVAL);
	}
	if (!(region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		RETURN(0);
	}

	switch (phase) {
	case FIRST_METADATA_WRITE:
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;	     // mark clean
		break;
	default :
		break;
	}

	RETURN(rc);
}


/* Function: raid0_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid0_get_option_count( task_context_t * task ) {
	int count = 0;

	LOG_ENTRY;

	switch (task->action) {
	case EVMS_Task_Create:
		count = 1;
		break;

	default:
		count = 0;
		break;
	}

	RETURN(count);
}


/* Function: raid0_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid0_init_task( task_context_t * context ) {

	int rc = 0;
	dlist_t tmp_list;

	LOG_ENTRY;

	// Parameter check.
	if (!context) {
		RETURN(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = 1;

		// option 0 is chunk size
		context->option_descriptors->option[0].flags = 0;
		SET_POWER2_LIST(context->option_descriptors->option[0].constraint.list, MD_MIN_CHUNK_SIZE, MD_MAX_CHUNK_SIZE);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		SET_STRING(context->option_descriptors->option[0].name,MD_OPTION_CHUNK_SIZE_NAME);
		context->option_descriptors->option[0].size = sizeof(u_int32_t);
		SET_STRING(context->option_descriptors->option[0].tip, "Size of IO to each member of the array (also refered to as stripe size)." );
		SET_STRING(context->option_descriptors->option[0].title, "Chunk Size:" );
		context->option_descriptors->option[0].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[0].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[0].value.ui32 = MD_DEFAULT_CHUNK_SIZE;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		DestroyList(&tmp_list, FALSE);

		context->min_selected_objects = 2;
		context->max_selected_objects = MAX_MD_DEVICES;
		break;

	default:
		break;
	}

	RETURN(rc);
}


/* Function: raid0_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid0_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect ) {
	int rc = 0;

	LOG_ENTRY;

	// Parameter check.
	if (!context || !value || !effect) {
		RETURN(EFAULT);
	}

	*effect = 0;

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case MD_OPTION_CHUNK_SIZE_INDEX:
			// Option is in kilobytes
			context->option_descriptors->option[index].value.ui32 = value->ui32;
			break;

		default:
			break;
                }

	default:
		break;
	}
	RETURN(rc);
}


/* Function: raid0_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid0_set_objects( task_context_t * context,
			      dlist_t          declined_objects,
			      task_effect_t  * effect ) {
	int rc = 0;

	LOG_ENTRY;

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		RETURN(EFAULT);
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		// Assume front end has selected valid objects from the
		// acceptable_objects list.  Could add safety checking.
		break;

	default:
		break;
	}

	RETURN(rc);
}


/* Function: raid0_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid0_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array ) {

	md_volume_t * volume = NULL;
	int           rc= 0;

	LOG_ENTRY;

	// Parameter check
	if (!info_array) {
		RETURN(EFAULT);
	}

	// Make sure this is an MD region
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		RETURN(EINVAL);
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	RETURN(rc);
}


/* Function: raid0_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid0_get_plugin_info( char                   * name,
				  extended_info_array_t ** info_array ) {

	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;
	int rc = 0;

	LOG_ENTRY;

	// Parameter check
	if (!info_array) {
		RETURN(EFAULT);
	}

	if (!name) {
		// Get memory for the info array
		if (!(info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*4))) {
			LOG_ERROR("Error allocating memory for info array\n");
			RETURN(ENOMEM);
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid0_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid0_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required API Version
		SET_STRING(info->info[i].name, "Required_Version");
		SET_STRING(info->info[i].title, "Required Plugin API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine that the plugin requires. It will not run on older versions of the Engine.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid0_plugin->required_api_version.major, raid0_plugin->required_api_version.minor, raid0_plugin->required_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		RETURN(EINVAL);
	}

	info->count = i;
	*info_array = info;
	RETURN(0);
}


/* Function: raid0_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid0_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer ) {

	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	LOG_ENTRY;

	// Parameter check.
	if (!buffer) {
		RETURN(EFAULT);
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n ",volume->name);
		RETURN(0);
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%d\n ",volume->name,lsn+count);
		RETURN(EFAULT);
	}

	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume, lsn, count, &child_object, &child_lsn, &child_count);
		if (rc == 0) {
			rc = READ(child_object, child_lsn, child_count, buffer);
			lsn += child_count;
			count -= child_count;
			buffer += child_count << EVMS_VSECTOR_SIZE_SHIFT;
		}
	}

	RETURN(rc);
}


/* Function: raid0_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid0_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer ) {

	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	LOG_ENTRY;

	// Parameter check.
	if (!buffer) {
		RETURN(EFAULT);
	}

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD Object %s is corrupt, writing data is not allowed\n ",volume->name);
		RETURN(EIO);
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%d\n ",volume->name,lsn+count);
		RETURN(EFAULT);
	}

	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume, lsn, count, &child_object, &child_lsn, &child_count);
		if (rc == 0) {
			rc = WRITE(child_object, child_lsn, child_count, buffer);
			lsn += child_count;
			count -= child_count;
			buffer += child_count << EVMS_VSECTOR_SIZE_SHIFT;
		}
	}

	RETURN(rc);
}


/* Function: raid0_direct_plugin_communication
 */
static int raid0_direct_plugin_communication( void    * thing,
					      boolean   target_kernel_plugin,
					      void    * arg ) {

	LOG_ENTRY;
	LOG("Not yet implemented\n");
	RETURN(ENOSYS);
}


static int free_region (ADDRESS object,
			TAG     object_tag,
			uint    object_size,
			ADDRESS object_handle,
			ADDRESS parameters) {

	int i;
	int nr_disks = 0;
	storage_object_t * region = (storage_object_t *) object;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	raid0_conf_t * conf = mdvol_to_conf(volume);

	if (conf != NULL) {
		md_deallocate_memory(conf->strip_zone);
		md_deallocate_memory(conf->hash_table);
	}

	md_deallocate_memory(conf);

	for (i = 0; (i < MAX_MD_DEVICES) && (nr_disks < volume->nr_disks); i++) {
		if (volume->super_array[i]) {
			nr_disks++;
			md_deallocate_memory(volume->super_array[i]);
		}
	}
        md_deallocate_memory(volume->super_block);
	if (volume->added_disks != NULL) {
		DestroyList(&volume->added_disks, TRUE);
	}
	if (volume->removed_disks != NULL) {
		DestroyList(&volume->removed_disks, TRUE);
	}
	if (volume->activated_disks != NULL) {
		DestroyList(&volume->activated_disks, TRUE);
	}
	if (volume->deactivated_disks != NULL) {
		DestroyList(&volume->deactivated_disks, TRUE);
	}
	md_remove_volume_from_list(volume);
	md_deallocate_memory(volume);

	RETURN(DLIST_SUCCESS);
}


void raid0_plugin_cleanup() {

	int rc;
	dlist_t raid0_regions_list;

	LOG_ENTRY;

	rc = EngFncs->get_object_list(REGION, DATA_TYPE, MY_PLUGIN, 0, &raid0_regions_list);

	if (rc == 0) {
		ForEachItem(raid0_regions_list, free_region, NULL, TRUE);

		DestroyList(&raid0_regions_list, FALSE);
	}

        LOG_EXIT(0);
	return;
}


/* Function table for the MD RAID0 Region Manager */
static plugin_functions_t raid0_functions = {
	cleanup_evms_plugin             : raid0_plugin_cleanup,
	setup_evms_plugin               : raid0_setup_evms_plugin,
	can_delete                      : raid0_can_delete,
	can_expand                      : raid0_can_expand,
	can_expand_by                   : raid0_can_expand_by,
	can_shrink                      : raid0_can_shrink,
	can_shrink_by                   : raid0_can_shrink_by,
	can_move                        : raid0_can_move,
	can_set_volume                  : raid0_can_set_volume,
	discover                        : raid0_discover,
	create                          : raid0_create,
	delete                          : raid0_delete,
	expand                          : raid0_expand,
	shrink                          : raid0_shrink,
	move                            : raid0_move,
	set_volume                      : raid0_set_volume,
	add_sectors_to_kill_list        : raid0_add_sectors_to_kill_list,
	commit_changes                  : raid0_commit_changes,
	get_option_count                : raid0_get_option_count,
	init_task                       : raid0_init_task,
	set_option                      : raid0_set_option,
	set_objects                     : raid0_set_objects,
	get_info                        : raid0_get_info,
	get_plugin_info                 : raid0_get_plugin_info,
	read                            : raid0_read,
	write                           : raid0_write,
	direct_plugin_communication     : raid0_direct_plugin_communication
};



/*
 *  Initialize the local plugin record
 */

plugin_record_t raid0_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 6),

	version:                {major:         MAJOR_VERSION,
				 minor:         MINOR_VERSION,
				 patchlevel:    PATCH_LEVEL},

	required_api_version:   {major:         3,
				 minor:         0,
				 patchlevel:    0},

	short_name:             "MDRaid0RegMgr",
	long_name:              "MD Raid 0 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin:        &raid0_functions},

	container_functions:    NULL
};

