/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_dlist.c
 *
 * Description: This file contains all functions related to manipulating the
 *              dlists that are used by the MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#define MY_PLUGIN my_plugin
#include <sys/ioctl.h>
#include "md.h"



/* Function: md_append_region_to_object
 *
 *	Associate the specified region and object as parent/child. Add the
 *	region to the object's "parent" dlist, and add the object to the
 *	region's "child" dlist. Also need to check for duplicates, so the
 *	same region/object don't get associated more than once.
 */
int md_append_region_to_object(	storage_object_t	* region,
				storage_object_t	* object )
{
	int	rc;
	void	* handle;

	// No LOG_ENTRY or RETURN calls. Writes too many messages to the log.

	rc = ExclusiveInsertObject(object->parent_objects,
				sizeof(storage_object_t),
				region,
				REGION_TAG,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding region %s as a parent to object %s\n",
			region->name, object->name);
		return rc;
	}

	rc = ExclusiveInsertObject(region->child_objects,
				sizeof(storage_object_t),
				object,
				object->object_type,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding object %s as a child to region %s\n",
			object->name, region->name);
		DeleteObject(object->parent_objects, region);
	}

	return rc;
}


/* Function: md_remove_region_from_object
 *
 *	Remove the parent/child association between the specified region
 *	and object. Remove the region from the object's "parent" list,
 *	and remove the object from the region's "child" list.
 */
int md_remove_region_from_object(	storage_object_t	* region,
					storage_object_t	* object )
{
	int	rc;
	void	* handle;

	LOG_ENTRY;

	rc = DeleteObject(object->parent_objects, region);
	if (rc) {
		LOG_SERIOUS("Error removing region %s from object %s\n",
			region->name, object->name);
		RETURN(rc);
	}

	rc = DeleteObject(region->child_objects, object);
	if (rc) {
		LOG_SERIOUS("Error removing object %s from region %s\n",
			object->name, region->name);
		// Should we put the region back on the object list???
		ExclusiveInsertObject(object->parent_objects,
					sizeof(storage_object_t),
					region,
					REGION_TAG,
					NULL,
					AppendToList,
					TRUE,
					&handle);
	} else{
		object->volume = NULL; // null out the volume pointer.
	}


	RETURN(rc);
}


/* Function: md_clear_child_list
 *
 *	Remove all objects from this region's child list.
 */
int md_clear_child_list( storage_object_t * region , dlist_t return_list)
{
	storage_object_t	* object = NULL;
	TAG			tag;
	int			size;
	int			rc;

	LOG_ENTRY;

	rc = GoToStartOfList(region->child_objects);
	if (rc) {
		LOG_SERIOUS("Problem with child list in region %s\n", region->name);
		RETURN(rc);
	}

	while ( ! BlindGetObject(region->child_objects, &size, &tag, NULL, FALSE, (void**)&object) ) {
		if ( object ) {
			if ( md_remove_region_from_object(region, object) ) {
				LOG_SERIOUS("Could not clear all objects from child list in region %s\n", region->name);
			}
			if (md_add_object_to_list(object, return_list)) {
				LOG_SERIOUS("Could not add all objects to the return list for%s\n", region->name);
			}
		}
	}

	RETURN(0);
}



/* Function: md_add_object_to_list
 *
 *	Add the specified object to the specified list.
 */
int md_add_object_to_list(	storage_object_t	* object,
				dlist_t			objects )
{
	int	rc;
	void	* handle;

	LOG_ENTRY;

	rc = InsertObject(objects,
			sizeof(storage_object_t),
			object,
			object->object_type,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding object %s to output list\n", object->name);
	}

	RETURN(rc);
}


/* Function: md_transfer_lists
 *
 *	This function simply removes each item from the input list and adds it
 *	to the output list.
 */
int md_transfer_list(	dlist_t	input,
			dlist_t	output )
{
	int rc;

	LOG_ENTRY;

	rc = AppendList(output, input);

	if ((rc == DLIST_EMPTY) || (rc == DLIST_END_OF_LIST)) {
		rc = DLIST_SUCCESS;
	}

	RETURN(rc);
}


void md_add_volume_to_list(md_volume_t * volume)
{
	LOG_ENTRY;
	volume->next = volume_list_head;
	volume_list_head = volume;
	LOG_EXIT(0);
        return;
}

void md_remove_volume_from_list(md_volume_t * volume)
{
	md_volume_t * tmp;
	LOG_ENTRY;
	if (volume == volume_list_head) {
		volume_list_head = volume_list_head->next;
	}else{
		for (tmp = volume_list_head; tmp != NULL; tmp = tmp->next) {
			if (volume == tmp->next) {
				tmp->next = tmp->next->next;
			}
		}
	}
	LOG_EXIT(0);
        return;
}


/*
 * Find all the Linux volumes that are using this MD volume and call the Engine
 * to rediscover them.
 * The parameters are structured so that this function can call itself
 * recursively using the dlist ForEachItem list processor.
 */
static int md_rediscover_region_volumes(ADDRESS object,
                                        TAG     object_tag,
                                        uint    object_size,
                                        ADDRESS object_handle,
                                        ADDRESS parameters)
{
	storage_object_t * region = (storage_object_t *) object;

	LOG_ENTRY;

	/*
	 * If the region's volume pointer is not NULL, then this region is fully
	 * contained in that volume, so all that needs to be done is issue a
	 * rediscover on that volume.
	 */
	if (region->volume != NULL) {
                EngFncs->rediscover_volume(region->volume, FALSE);

	} else {
	       	/*
		 * Either this region is not part of a volume at all (e.g., it's
		 * part of a storage_object that has not been made into a
		 * volume) or it is, or is part of, an object that is consumed
		 * by a container.  If this object is part of a container, then
		 * go to the container and call this function recursively on all
		 * the regions that are produced by that container.
		 */
		if (region->consuming_container != NULL) {
			ForEachItem(region->consuming_container->objects_produced,
				    md_rediscover_region_volumes,
				    NULL,
				    TRUE);

		} else {
			/*
			 * The region itself is not consumed by a container,
			 * but it may be part of a parent object that is
			 * consumed by a container.  Call this function
			 * recursively on this region's parent list to see
			 * if there is a container somewhere up the stack.
			 */
			ForEachItem(region->parent_objects,
				    md_rediscover_region_volumes,
				    NULL,
				    TRUE);
		}
	}

	RETURN(DLIST_SUCCESS);
}

/* Front end function for the worker function above */
void md_rediscover_volumes_for_region(storage_object_t * region) {

	LOG_ENTRY;

	md_rediscover_region_volumes(region,
				     REGION_TAG,
				     sizeof(storage_object_t),
				     NULL,
				     NULL);

	LOG_EXIT(0);
}


static BOOLEAN remove_md_kdev(ADDRESS   object,
                              TAG       object_tag,
                              uint      object_size,
                              ADDRESS   object_handle,
                              ADDRESS   parameters,
                              BOOLEAN * free_memory,
                              uint    * error) {

	evms_md_kdev_t * md_kdev = (evms_md_kdev_t *) object;
	evms_md_kdev_t * search_md_kdev = (evms_md_kdev_t *) parameters;

	LOG_ENTRY;

	if ((md_kdev->major == search_md_kdev->major) &&
	    (md_kdev->minor == search_md_kdev->minor)) {
		/*
		 * Free the md_kdev_t structure when removing it
		 * from the list.
		 */
		*free_memory = TRUE;

		/* Flag that the object was found in the list. */
		*error = DLIST_OBJECT_ALREADY_IN_LIST;

		/* Remove the item from the list. */
		RETURN(TRUE);

	} else {
		*free_memory = TRUE;	/* just to be safe */

		/* Continue searching */
		*error = DLIST_SUCCESS;

		/* Keep the item in the list. */
		RETURN(FALSE);

	}
}

// Adds an entry into the specified list for an IOCTL action to be done later.
int md_add_modify_object_to_list(md_volume_t * volume, int cmd, int major, int minor) {

	int rc = 0;
	dlist_t list = NULL;
	evms_md_kdev_t * parms;
	void	* handle;

	LOG_ENTRY;

	if (md_allocate_memory((void **)&parms,sizeof(evms_md_kdev_t))) {
		RETURN(ENOMEM);
	}
	parms->major = major;
	parms->minor = minor;

	switch (cmd) {
	case EVMS_MD_ADD:
		// If the object is on the removed_disks list, remove it.
		rc = PruneList(volume->removed_disks, remove_md_kdev, parms);
		if (rc == DLIST_SUCCESS) {
			// The object was not on the removed_disks list.
			list = volume->added_disks;
		}
		break;
	case EVMS_MD_REMOVE:
		// If the object is on the added_disks list, remove it.
		rc = PruneList(volume->added_disks, remove_md_kdev, parms);
		if (rc == DLIST_SUCCESS) {
			// The object was not on the added_disks list.
			list = volume->removed_disks;
		}
		break;
	case EVMS_MD_ACTIVATE:
		// If the object is on the deactivated_disks list, remove it.
		rc = PruneList(volume->deactivated_disks, remove_md_kdev, parms);
		if (rc == DLIST_SUCCESS) {
			// The object was not on the deactivated_disks list.
			list = volume->activated_disks;
		}
		break;
	case EVMS_MD_DEACTIVATE:
		// If the object is on the activated_disks list, remove it.
		rc = PruneList(volume->activated_disks, remove_md_kdev, parms);
		if (rc == DLIST_SUCCESS) {
			// The object was not on the activated_disks list.
			list = volume->deactivated_disks;
		}
		break;
	default:
		LOG_SERIOUS("Error, invalid command for process list %d\n",cmd);
		RETURN(EINVAL);
	}

	if (rc == DLIST_OBJECT_ALREADY_IN_LIST) {
		// The md_kdev was found in another list and deleted from that list.
		// Don't put it in a new list.  Free the memory and return 0.
		md_deallocate_memory(parms);
		RETURN(0);
	}

	/* Make sure the md_kdev is not already on the list. */
	PruneList(list, remove_md_kdev, parms);

	rc = InsertObject(list,
			sizeof(evms_md_kdev_t),
			parms,
			MD_MODIFY_TAG,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding object to ioctl list\n");
	}

	RETURN(rc);
}

// removes all entries from the specified list and IOCTls to the kernel for each.
int md_process_modify_list(md_volume_t * volume, int cmd) {

	int rc = 0;
	evms_md_kdev_t * parms;
	int size, tag;
	evms_md_ioctl_t         md_ioctl = {0};
	evms_plugin_ioctl_t     plugin_ioctl = {0};
	dlist_t list;
	LOG_ENTRY;

	switch (cmd) {
	case EVMS_MD_ADD:
		list = volume->added_disks;
		break;
	case EVMS_MD_REMOVE:
		list = volume->removed_disks;
		break;
	case EVMS_MD_ACTIVATE:
		list = volume->activated_disks;
		break;
	case EVMS_MD_DEACTIVATE:
		list = volume->deactivated_disks;
	default:
		LOG_SERIOUS("Error, invalid command for process list %d\n",cmd);
		RETURN(EINVAL);

	}

	if (list) {
		while (BlindExtractObject(list, &size, (TAG *) &tag, NULL, (void**)&parms)==0) {
			/* Use the MD init io ioctl to write the data.  If the ioctl fails, do it manually. */
			plugin_ioctl.feature_id = SetPluginID(IBM_OEM_ID, EVMS_REGION_MANAGER, 0x4),
						  plugin_ioctl.feature_command = cmd,
			plugin_ioctl.feature_ioctl_data = &md_ioctl;

			md_ioctl.mddev_idx = volume->super_block->md_minor;
			md_ioctl.cmd = 0; //not used here
			md_ioctl.arg = parms;

			LOG_EXTRA("Calling kernel MD ioctl %d for major %d minor %d.\n",cmd, parms->major,parms->minor);
			rc = EngFncs->ioctl_evms_kernel(EVMS_PLUGIN_IOCTL, &plugin_ioctl);

			/* If the ioctl failed we can try do to the write by hand. */
			if (rc != 0) {
				/*
				 * Get a real error code other than the -1 which ioctl()
				 * returns.
				 */
				if (plugin_ioctl.status != 0) {
					rc = plugin_ioctl.status;
				} else {
					rc = errno;
				}

				/* Check for acceptable error codes. */
				if ((abs(rc) == ENOPKG) ||	/* MD kernel module is not loaded */
				    (abs(rc) == ENODEV)) {	/* MD region does not exist */
					rc = 0;
				} else {
					LOG_ERROR("Calling kernel MD ioctl %d for major %d minor %d, rc=%d\n",cmd, parms->major,parms->minor,rc);
				}
			}

		}
	}
	RETURN(rc);
}


