/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: mdregmgr.h
 */

#ifndef _MD_REGION_MANAGER_H_
#define _MD_REGION_MANAGER_H_ 1

#include <linux/evms/evms_md_p.h>

// Datatypes and Defs used in mdregmgr.c
#define SECTOR_SIZE_BITS            9
#define PAGE_SIZE_SECTORS           16      // for 8k pages
#define PE_SIZE_ALIGNMENT_SECTORS   128     // for 64k aligned PEs
#define MD_NAME_SPACE               EVMS_DEV_NODE_PATH "md"
#define MD_DEV_DIRECTORY            "md/"
#define MAX_MD_DEVICES              27
#define MAX_MD_MINORS               255
#define MD_MIN_CHUNK_SIZE           4       // in kilobytes
#define MD_MAX_CHUNK_SIZE           4096    // in kilobytes = 4MB
#define MD_DEFAULT_CHUNK_SIZE       32      // in kilobytes
#define BLOCK_SIZE                  1024    // needed by md size macros

typedef u_int8_t boolean;

#define MD_RESERVED       0UL
#define LINEAR            1UL
#define RAID0             2UL
#define RAID1             3UL
#define RAID5             4UL
#define TRANSLUCENT       5UL
#define HSM               6UL
#define MULTIPATH         7UL
#define MAX_PERSONALITY   8UL

#define INVALID_LEVEL    99UL

static inline int pers_to_level (int pers) {
	switch (pers) {
	case MULTIPATH:     return -4;
	case HSM:           return -3;
	case TRANSLUCENT:   return -2;
	case LINEAR:        return -1;
	case RAID0:         return 0;
	case RAID1:         return 1;
	case RAID5:         return 5;
	}
	return INVALID_LEVEL;
}

static inline int level_to_pers (int level) {
	switch (level) {
	case -3: return HSM;
	case -2: return TRANSLUCENT;
	case -1: return LINEAR;
	case 0:  return RAID0;
	case 1:  return RAID1;
	case 4:
	case 5:  return RAID5;
	}
	return MD_RESERVED;
}
// md_volume_t flags
#define MD_DISCOVERED	(1 << 0)
#define MD_DEGRADED	(1 << 1)
#define MD_CORRUPT	(1 << 2)
#define MD_DIRTY	(1 << 3)
#define MD_ACTIVE	(1 << 4)  // This md volume is active in the kernel

// commit_flags
#define MD_COMMIT_USE_DISK (1 << 0)  	// Set this flag in modifies if you want commit
					// to preserve the existing on disk SB.state

typedef struct md_volume_s {
	storage_object_t * region;
	storage_object_t * child_object[MAX_MD_DEVICES];
	mdp_super_t      * super_array[MAX_MD_DEVICES];
	char               name[128];
	u_int32_t          commit_flag;	 //if set, use SB state from disk, else use from Master SB
	u_int32_t          uuid;
	u_int32_t          nr_disks;
	u_int32_t          flags;
	u_int32_t          personality;	 // uses the positive defines, not level
	mdp_super_t *      super_block;	 // master superblock
	struct md_volume_s * next;
	dlist_t            removed_disks;
	dlist_t            added_disks;
	dlist_t            activated_disks;
	dlist_t            deactivated_disks;
	void             * private_data;
} md_volume_t;


#define MD_MODIFY_TAG 99  // tag for evms_md_kdev_t in DLIST

// Global data used by mdregmgr
extern engine_functions_t   * EngFncs;
extern plugin_record_t      * linear_plugin;
extern plugin_record_t        linear_plugin_record;
extern plugin_record_t      * raid1_plugin;
extern plugin_record_t        raid1_plugin_record;
extern plugin_record_t      * raid0_plugin;
extern plugin_record_t        raid0_plugin_record;
extern plugin_record_t      * raid5_plugin;
extern plugin_record_t        raid5_plugin_record;
extern plugin_record_t      * my_plugin;
extern md_volume_t          * volume_list_head;	// List of real volume groups, indexed by vg_number.
extern char                   message_buffer[1024];


// Log message macros
#define LOG_CRITICAL(msg, args...)  EngFncs->write_log_entry(CRITICAL,  my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_SERIOUS(msg, args...)   EngFncs->write_log_entry(SERIOUS,   my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_ERROR(msg, args...)     EngFncs->write_log_entry(ERROR,     my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_WARNING(msg, args...)   EngFncs->write_log_entry(WARNING,   my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG(msg, args...)           EngFncs->write_log_entry(DEFAULT,   my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DETAILS(msg, args...)   EngFncs->write_log_entry(DETAILS,   my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEBUG(msg, args...)     EngFncs->write_log_entry(DEBUG,     my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EXTRA(msg, args...)     EngFncs->write_log_entry(EXTRA,     my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_PROC(msg, args...)      EngFncs->write_log_entry(ENTRY_EXIT,    my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EVERYTHING(msg, args...)    EngFncs->write_log_entry(EVERYTHING,    my_plugin, "%s: " msg, __FUNCTION__ , ## args)
#define MESSAGE(msg, args...)       EngFncs->user_message(my_plugin, NULL, NULL, msg, ##args)

#define SET_STRING(a,b) a = EngFncs->engine_alloc( strlen(b)+1 );if (a ) { strcpy(a, b); } else { rc = ENOMEM; LOG_EXIT(rc);}

#define LOG_ENTRY       my_plugin = MY_PLUGIN; LOG_PROC("Entering\n")
#define LOG_EXIT(x)     LOG_PROC("Exiting: rc = %d\n", x)
#define RETURN(x)       {LOG_EXIT(x); return(x);}

#ifndef min
	#define min(a,b) (((a)<(b))?(a):(b))
#endif
#ifndef max
	#define max(a,b) (((a)>(b))?(a):(b))
#endif

extern long calc_log2( long arg );
/* Macro: SET_POWER2_LIST
 */
// Set a contraint list that increments each item by a power of 2.
// b is the starting value, c is ending value (both must be power of 2).
#define SET_POWER2_LIST(a, b, c) \
{ \
    if ( ((b) & ((b)-1)) || ((c) & ((c)-1)) ) { \
        RETURN(EINVAL); \
    } \
    (a) = EngFncs->engine_alloc(calc_log2(c)*sizeof(value_t)+1); \
    if (a) { \
        long i = b; \
        (a)->count = 0; \
        while ( i <= c ) { \
            (a)->value[(a)->count].ui32 = i; \
            (a)->count++; \
            i <<= 1; \
        } \
    } else { \
        RETURN(ENOMEM); \
    } \
}


#define MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(chunk_size_in_sectors, size_in_sectors) (MD_NEW_SIZE_SECTORS(size_in_sectors) & ~(chunk_size_in_sectors - 1))
#define MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size_in_blocks, size_in_blocks) (MD_NEW_SIZE_BLOCKS(size_in_blocks) & ~(chunk_size_in_blocks - 1))


// These need to follow all of the structure definitions.
#include <linux/evms/evms_md_u.h>
#include "md_allocation.h"
#include "md_discover.h"
#include "md_dlist.h"
#include "md_io.h"
#include "md_create.h"
#include "md_info.h"

#endif

