/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_snapshots.c
 *
 * Description: This file contains all functions related to managing
 *		LVM snapshot volumes.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <plugin.h>
#include "lvmregmgr.h"


/* Function: lvm_rediscover_snapshot_chain
 *
 *	Add all volumes on the snapshot chain for the original to the EVMS
 *	rediscover list. If a snapshot is being created or deleted, make sure
 *	it is NOT on the chain before calling this function.
 */
int lvm_rediscover_snapshot_chain( lvm_logical_volume_t * org_volume )
{
	lvm_logical_volume_t * next_snap;

	LOG_ENTRY;

	// First add all snapshots.
	for ( next_snap = org_volume->snapshot_next; next_snap; next_snap = next_snap->snapshot_next ) {
		if ( ! next_snap->region->volume ) {
			LOG_ERROR("Region %s is not a Compatibility Volume.\n", next_snap->region->name);
			LOG_ERROR("Cannot force it to be rediscovered!\n");
			RETURN(EINVAL);
		}
		lvm_engine->rediscover_volume(next_snap->region->volume, FALSE);
	}

	// Add the original last.
	if ( ! org_volume->region->volume ) {
		LOG_ERROR("Region %s is not a Compatibility Volume.\n", org_volume->region->name);
		LOG_ERROR("Cannot force it to be rediscovered!\n");
		RETURN(EINVAL);
	}
	lvm_engine->rediscover_volume(org_volume->region->volume, TRUE);

	RETURN(0);
}


/* Function: lvm_create_snapshot_volume
 *
 *	Perform all necessary steps for preparing a volume to be snapshotted.
 *	This will involve modifying the original's metadata as appropriate,
 *	adding the snapshot to the chain, and forcing a rediscover on the
 *	existing snapshot chain.
 */
int lvm_create_snapshot_volume( lvm_logical_volume_t	* snap_volume,
				lvm_logical_volume_t	* org_volume )
{
	int rc = 0;

	LOG_ENTRY;

	// Sanity check
	if ( ! (snap_volume->lv->lv_access & LV_SNAPSHOT) ) {
		LOG_ERROR("Region %s is not a snapshot.\n", snap_volume->region->name);
		RETURN(EINVAL);
	}

	// Force kernel-rediscovers on all existing volumes in the chain. This
	// is still necessary for creates of new snapshots.
	rc = lvm_rediscover_snapshot_chain(org_volume);
	if (rc) {
		LOG_ERROR("Error forcing rediscover of snapshot chain for original %s\n", org_volume->region->name);
		LOG_ERROR("Cannot create snapshot region %s\n", snap_volume->region->name);
		RETURN(rc);
	}
	
	// Set the lv_access flag for the original
	org_volume->lv->lv_access |= LV_SNAPSHOT_ORG;
	org_volume->region->flags |= SOFLAG_MUST_BE_VOLUME | SOFLAG_READ_ONLY;

	// Add the new snapshot to the original's chain.
	snap_volume->snapshot_next = org_volume->snapshot_next;
	snap_volume->snapshot_org = org_volume;
	org_volume->snapshot_next = snap_volume;
	snap_volume->region->associated_object = org_volume->region;

	RETURN(0);
}


/* Function: lvm_delete_snapshot_volume
 *
 *	Perform all necessary steps for preparing a snapshot volume to be
 *	deleted. This will involve adding the entire snapshot chain to the
 *	list of volumes to quiesce, delete, and rediscover. It also involves
 *	removing this volume from the snapshot chain, and updating the
 *	metadata for the original if this is the only snapshot.
 */
int lvm_delete_snapshot_volume( lvm_logical_volume_t * snap_volume )
{
	lvm_logical_volume_t	* org_volume;
	lvm_logical_volume_t	* next_snap;

	LOG_ENTRY;

	// Sanity check
	if ( ! (snap_volume->lv->lv_access & LV_SNAPSHOT) ) {
		LOG_ERROR("Region %s is not a snapshot.\n", snap_volume->region->name);
		RETURN(EINVAL);
	}

	org_volume = snap_volume->snapshot_org;

	// Take the snapshot out of the chain.
	for ( next_snap = org_volume; next_snap; next_snap = next_snap->snapshot_next ) {
		if ( next_snap->snapshot_next == snap_volume ) {
			next_snap->snapshot_next = snap_volume->snapshot_next;
			break;
		}
	}
	snap_volume->snapshot_next = NULL;
	snap_volume->snapshot_org = NULL;
	snap_volume->region->associated_object = NULL;

	// If this was the only snapshot, update the original's metadata.
	if ( ! org_volume->snapshot_next ) {
		org_volume->lv->lv_access &= ~LV_SNAPSHOT_ORG;
		org_volume->region->flags &= ~SOFLAG_MUST_BE_VOLUME;
		if ( org_volume->lv->lv_access & LV_WRITE ) {
			org_volume->region->flags &= ~SOFLAG_READ_ONLY;
		}
	}

	// No longer need to do a rediscover of this snapshot chain. The engine
	// will now take care of quiescing and unquiescing the original before/
	// after the hard-delete of the snapshot.

	RETURN(0);
}


/* Function: lvm_count_snapshot_volumes
 *
 *	Return the count of snapshot volumes in this group
 */
int lvm_count_snapshot_volumes( lvm_volume_group_t * group )
{
	int i, count = 0;

	LOG_ENTRY;

	for ( i = 1; i <= MAX_LV; i++ ) {
		if ( group->volume_list[i] &&
		     group->volume_list[i]->lv->lv_access & LV_SNAPSHOT ) {
			count++;
		}
	}

	RETURN(count);
}


/* Function: lvm_update_snapshot_stats
 *
 *	In order to determine the status of a snapshot volume, we need to
 *	know if the kernel thinks there is any space left on it. Call a
 *	direct ioctl to the LVM kernel plugin, and get the starting LBA of
 *	the next available snapshot chunk. If this LBA is >= the size of
 *	the volume, then the snapshot is full, and should be deleted.
 */
int lvm_update_snapshot_stats( lvm_logical_volume_t * volume )
{
	evms_plugin_ioctl_t		arg;
	lvm_snapshot_stat_ioctl_t	snap_stats;
	int				rc = 0;

	LOG_ENTRY;

	if ( ! (volume->lv->lv_access & LV_SNAPSHOT) ) {
		LOG_ERROR("Region %s is not a snapshot. Cannot get stats\n", volume->region->name);
		RETURN(EINVAL);
	}

	memcpy(snap_stats.vg_uuid, volume->group->vg->vg_uuid, UUID_LEN);
	snap_stats.lv_number = volume->number;
	snap_stats.next_free_chunk = 0;
	snap_stats.lv_status = 0;

	arg.feature_id = lvm_plugin->id;
	arg.feature_command = EVMS_LVM_SNAPSHOT_STAT_IOCTL;
	arg.status = 0;
	arg.feature_ioctl_data = &snap_stats;

	// Call the direct ioctl.
	if ( (rc = lvm_engine->ioctl_evms_kernel(EVMS_PLUGIN_IOCTL, &arg)) ||
	     (rc = arg.status) < 0 ) {
		LOG_ERROR("Ioctl error (%d).\n", rc);
		LOG_ERROR("Kernel could not get snapshot stats for region %s\n", volume->region->name);
	}
	else if ( rc > 0 ) {
		LOG_ERROR("Snapshot region %s not found in kernel\n", volume->region->name);
	}
	else {
		// Update the volume info
		volume->next_free_chunk = snap_stats.next_free_chunk;
		volume->lv->lv_status = snap_stats.lv_status;
		if ( volume->next_free_chunk >= volume->lv->lv_size ) {
			LOG_ERROR("Snapshot region %s is full. Deactivating\n", volume->region->name);
			volume->lv->lv_status &= ~LV_ACTIVE;
		}
		else if ( ! (volume->lv->lv_status & LV_ACTIVE) ) {
			LOG_ERROR("Snapshot region %s has been deactivated in the kernel.\n", volume->region->name);
		}
	}

	RETURN(rc);
}


/* Function: lvm_link_snapshot_volumes
 *
 *	Examine the list of logical volumes in this group and set up the
 *	necessary pointers to link snapshots and their originals. A singly-
 *	linked list is created starting with the original volume. All
 *	snapshots also point directly back to their original.
 */
int lvm_link_snapshot_volumes( lvm_volume_group_t * group )
{
	lvm_logical_volume_t	* org_volume;
	lvm_logical_volume_t	* snap_volume;
	unsigned long		org_minor;
	int			i,j;

	LOG_ENTRY;

	for ( i = 1; i <= MAX_LV; i++ ) {

		// Only process snapshot-originals
		if ( group->volume_list[i] && 
		     group->volume_list[i]->lv->lv_access & LV_SNAPSHOT_ORG ) {

			// For each original, look for all other volumes that
			// claim to be snapshotting it. For each one that is
			// found, insert it at the start of the original's list
			// of snapshots.
			org_volume			= group->volume_list[i];
			org_minor			= org_volume->minor;
			org_volume->snapshot_next	= NULL;
			for ( j = 1; j <= MAX_LV; j++ ) {
				if ( group->volume_list[j] &&
				     group->volume_list[j]->lv->lv_access & LV_SNAPSHOT &&
				     group->volume_list[j]->lv->lv_snapshot_minor == org_minor ) {
					snap_volume				= group->volume_list[j];
					snap_volume->snapshot_org		= org_volume;
					snap_volume->snapshot_next		= org_volume->snapshot_next;
					snap_volume->region->associated_object 	= org_volume->region;
					org_volume->snapshot_next		= snap_volume;
					LOG_DETAILS("Linking snapshot %s to original %s\n",
						snap_volume->region->name, org_volume->region->name);
				}
			}

			// If no snapshots were found for a volume that claims to be
			// under snapshot, we need to inform the user. Just print an
			// error message for now.
			// Do we want to reset lv_access here?
			if ( ! org_volume->snapshot_next ) {
				LOG_ERROR("No snapshots found for original %s\n",org_volume->region->name);
			}
		}
	}
	RETURN(0);
}


