/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_names.c
 *
 * Description: This file contains all functions related to translating names
 *		between the LVM style and the EVMS style. This applies to
 *		VGs/containers and LVs/regions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

#define DEV_DIRECTORY		DEV_PATH "/"
#define EVMS_DEV_DIRECTORY	EVMS_DIR_NAME "/"

/* Function: lvm_translate_vg_name_to_container_name
 *
 *	In LVM, volume groups have simple names, like "SystemGroup". In EVMS,
 *	the container representing that VG should appear as "lvm/SystemGroup".
 *	Thus, the name from the VG needs to be translated before copying to the
 *	associated container. The container_name parameter should point to a
 *	string buffer with sufficient space for copying the name.
 */
int lvm_translate_vg_name_to_container_name(	char	* vg_name,
						char	* container_name )
{
	LOG_ENTRY;

	strncpy(container_name, LVM_DEV_DIRECTORY, EVMS_NAME_SIZE);
	strncat(container_name, vg_name, EVMS_NAME_SIZE-strlen(container_name));

	RETURN(0);
}


/* Function: lvm_translate_container_name_to_vg_name
 *
 *	In LVM, volume groups have simple names, like "SystemGroup". In EVMS,
 *	the container representing that VG will have the name "lvm/SystemGroup".
 *	The container name needs to be translated to the VG name before the
 *	name can be copied to the group's metadata. The vg_name parameter
 *	should point to a string buffer with sufficient space for copying the
 *	name.
 */
int lvm_translate_container_name_to_vg_name(	char	* container_name,
						char	* vg_name )
{
	char * ptr;

	LOG_ENTRY;

	// Make sure the string starts with "lvm/" and skip over it.
	ptr = strstr(container_name, LVM_DEV_DIRECTORY);
	if ( ptr != container_name ) {
		LOG_ERROR("Invalid container name: %s\n", container_name);
		RETURN(EINVAL);
	}
	ptr = &(ptr[strlen(LVM_DEV_DIRECTORY)]);

	// ptr should now be pointing at "group_name".
	// Use this to create the name for the LV.	
	strncpy(vg_name, ptr, EVMS_NAME_SIZE);

	RETURN(0);
}


/* Function: lvm_translate_lv_name_to_region_name
 *
 *	In LVM, volumes have names based on their dev-node, which follow the
 *	pattern /dev/group_name/volume_name. In EVMS, the region representing
 *	that volume should appear as lvm/group_name/volume_name. Thus, the name
 *	from the lv_disk_t needs to be translated before copying to the
 *	associated region. The region_name parameter should point to a string
 *	buffer with sufficient space for copying the name.
 */
int lvm_translate_lv_name_to_region_name(char	* lv_name,
					char	* region_name )
{
	char * ptr;

	LOG_ENTRY;

	// Make sure the string starts with /dev/, and skip over it.
	ptr = strstr(lv_name, DEV_DIRECTORY);
	if ( ptr != lv_name ) {
		LOG_ERROR("Invalid LV name: %s\n", lv_name);
		RETURN(EINVAL);
	}
	ptr = &(ptr[strlen(DEV_DIRECTORY)]);

	// The ptr should now be point to "group_name/volume_name".
	// Use this to create the name for the region.
	strncpy(region_name, LVM_DEV_DIRECTORY, EVMS_NAME_SIZE);
	strncat(region_name, ptr, EVMS_NAME_SIZE-strlen(region_name));

	RETURN(0);
}


/* Function: lvm_translate_region_name_to_lv_name
 *
 *	In LVM, volumes have names based on their dev-node, which follow the
 *	pattern /dev/group_name/volume_name. In EVMS, the region representing
 *	that volume should appear as lvm/group_name/volume_name. Thus, the name
 *	from the region needs to be translated before copying to the associated
 *	lv_disk_t. The lv_name parameter should point to a string buffer with
 *	sufficient space for copying the name.
 */
int lvm_translate_region_name_to_lv_name(char	* region_name,
					char	* lv_name )
{
	char * ptr;

	LOG_ENTRY;

	// Make sure the string starts with "lvm/" and skip over it.
	ptr = strstr(region_name, LVM_DEV_DIRECTORY);
	if ( ptr != region_name ) {
		LOG_ERROR("Invalid region name: %s\n", region_name);
		RETURN(EINVAL);
	}
	ptr = &(ptr[strlen(LVM_DEV_DIRECTORY)]);

	// ptr should now be pointing at "group_name/volume_name".
	// Use this to create the name for the LV.	
	strcpy(lv_name, DEV_DIRECTORY);
	strncat(lv_name, ptr, EVMS_NAME_SIZE-strlen(lv_name));

	RETURN(0);
}


/* Function: lvm_make_lv_name
 *
 *	This function will be used during creation of a new volume. The user
 *	will specify a simple name for the volume, and we need to generate
 *	an LV name, using the specified group.
 */
int lvm_make_lv_name(	char			* simple_name,
			lvm_volume_group_t	* group,
			char			* lv_name )
{
	char vg_name[EVMS_NAME_SIZE+1] = {0};

	LOG_ENTRY;

	lvm_translate_container_name_to_vg_name(group->container->name, vg_name);

	strcpy(lv_name, DEV_DIRECTORY);
	strncat(lv_name, vg_name, EVMS_NAME_SIZE-strlen(lv_name));
	strncat(lv_name, "/", EVMS_NAME_SIZE-strlen(lv_name));
	strncat(lv_name, simple_name, EVMS_NAME_SIZE-strlen(lv_name));

	RETURN(0);
}


/* Function: lvm_translate_dev_name
 *
 *	Copy input string to output string, removing any leading "/dev",
 *	"/dev/evms/", or "evms/". Make sure output points to a buffer
 *	of suitable size.
 */
int lvm_translate_dev_name(	char	* input,
				char	* output )
{
	char * ptr = input;

	LOG_ENTRY;

	// Does ptr start with "/dev/"?
	if ( strstr(ptr, DEV_DIRECTORY) == ptr ) {
		ptr = &(ptr[strlen(DEV_DIRECTORY)]);
	}

	// Does ptr start with "evms/"?
	if ( strstr(ptr, EVMS_DEV_DIRECTORY) == ptr ) {
		ptr = &(ptr[strlen(EVMS_DEV_DIRECTORY)]);
	}

	output[0] = 0;

	// Does ptr start with "lvm/"?
	if ( strstr(ptr, LVM_DEV_DIRECTORY) != ptr ) {
		strncat(output, LVM_DEV_DIRECTORY, EVMS_NAME_SIZE);
	}
	strncat(output, ptr, EVMS_NAME_SIZE-strlen(output));

	RETURN(0);
}

