/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aix_dlist.c
 *
 * Description: Managing the various dlists that are used by the AIX region
 *		manager.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"


/* aix_get_list_item
 *
 *	Get the current item from the specified list.
 */
void * aix_get_list_item( dlist_t list )
{
	void	* object;
	TAG	tag;
	int	size;

	LOG_ENTRY;
	BlindGetObject(list, &size, &tag, NULL, TRUE, &object);
	RETURN(object);
}


/* aix_add_object_to_list
 *
 *	Add the specified object to the end of the specified D-list.
 */
int aix_add_object_to_list(	storage_object_t	* object,
				dlist_t			list )
{
	void	* handle;
	int	rc;

	LOG_ENTRY;

	rc = InsertObject(list,
			sizeof(storage_object_t),
			object,
			object->object_type,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding %s to list\n", object->name);
	}

	RETURN(rc);
}


/* aix_remove_object_from_list
 *
 *	Remove the specified object from the specified D-list.
 */
int aix_remove_object_from_list(storage_object_t	* object,
				dlist_t			list )
{
	int	rc;

	LOG_ENTRY;

	rc = DeleteObject(list, object);

	if (rc) {
		LOG_SERIOUS("Error removing %s from list\n", object->name);
	}

	RETURN(rc);
}


/* aix_add_container_to_list
 *
 *	Add the specified AIX container to the global list.
 */
int aix_add_container_to_list( storage_container_t * container )
{
	void	* handle;
	int	rc;

	LOG_ENTRY;

	rc = InsertObject(aix_container_list,
			sizeof(storage_container_t),
			container,
			CONTAINER_TAG,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding %s to list\n", container->name);
	}

	RETURN(rc);
}


/* aix_remove_container_from_list
 *
 *	Remove the specified AIX container from the global list.
 */
int aix_remove_container_from_list( storage_container_t * container )
{
	int rc;

	LOG_ENTRY;

	rc = DeleteObject(aix_container_list, container);

	if (rc) {
		LOG_SERIOUS("Error removing %s from list\n", container->name);
	}

	RETURN(rc);
}


/* aix_append_region_to_object
 *
 *	Associate the specified region and object as parent and child. Add the
 *	region to the object's "parent" list, and add the object to the region's
 *	"child" list.
 */
int aix_append_region_to_object(storage_object_t	* region,
				storage_object_t	* object )
{
	int	rc;
	void	* handle;

	rc = ExclusiveInsertObject(object->parent_objects,
				sizeof(storage_object_t),
				region,
				region->object_type,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding region %s as a parent to object %s\n",
			region->name, object->name);
		return rc;
	}

	rc = ExclusiveInsertObject(region->child_objects,
				sizeof(storage_object_t),
				object,
				object->object_type,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding object %s as a child to region %s\n",
			object->name, region->name);
		DeleteObject(object->parent_objects, region);
	}

	return rc;
}

