/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: task.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>
#include "task.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "option.h"
#include "memman.h"


/*
 *
 *   static inline void FreeTaskMemory(task_handle_t, task_context_t *)
 *
 *   Description:
 *      This routine frees all memory dynamically allocated by the task
 *      API referenced in the task context structure. This specifically
 *      includes the parameters, option descriptors and array, and the
 *      option values and array.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      All task context memory is freed
 *
 */
static inline void FreeTaskMemory (task_context_t *task) {

    LOG_PROC_ENTRY();

    if (task != NULL) {

        int i;
        /*
         * Avoiding memory leaks is of great importance here.
         */
        if (task->selected_objects != NULL) {
            /*
             * Destroy the parameter list. Don't free the
             * objects in the list as they are on someone
             * else's list.
             */
            DestroyList(&(task->selected_objects), FALSE);
        }
        if (task->option_descriptors != NULL) {
            /*
             * All strings in the descriptor structure should be dynamically
             * allocated.  This includes, constraint lists containing strings
             * and range structures for numeric ranges.  We free all of these
             * before freeing the option descriptor array.
             */
            for (i=0; i < task->option_descriptors->count; i++) {
                free_option_descriptor_contents(&(task->option_descriptors->option[i]));
            }
            free(task->option_descriptors);
        }
        free(task);
    }

    LOG_PROC_EXIT_VOID();
    return;
}


/*
 *
 *   static inline int GetTaskOptionsCount(task_context_t *)
 *
 *   Description:
 *      This routine calls the GetOptionsCount plug-in function to
 *      retrieve the maximum backen options available for a
 *      specific operation/action.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      Returns count of maximum backend options for a specific task
 *
 */
static inline int GetTaskOptionsCount (task_context_t *task) {
    int count;

    LOG_PROC_ENTRY();

    switch (GetPluginType(task->plugin->id)) {
        case EVMS_DEVICE_MANAGER:
        case EVMS_SEGMENT_MANAGER:
        case EVMS_REGION_MANAGER:
        case EVMS_FEATURE:
        case EVMS_ASSOCIATIVE_FEATURE:
            count = task->plugin->functions.plugin->get_option_count(task);
            break;

        case EVMS_FILESYSTEM_INTERFACE_MODULE:
            count = task->plugin->functions.fsim->get_option_count(task);
            break;

        case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
        case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
        default:
            count = 0;
            break;
    }

    LOG_PROC_EXIT_INT(count);
    return count;
}


/*
 *
 *   static inline int InitTaskOptions(task_context_t *)
 *
 *   Description:
 *      This routine calls the InitOptions plug-in function to allow
 *      the backend to initialize the option descriptor and option
 *      values arrays.
 *
 *   Entry:
 *      task - address of task context structure created by evms_create_task()
 *
 *   Exit:
 *      On success, returns a return code of 0 and option values and
 *      option descriptor arrays are initialized.
 *      On error, returns an error code < 0
 *
 */
static inline int InitTaskOptions (task_context_t *task) {
    int rc = 0;

    LOG_PROC_ENTRY();

    switch (GetPluginType(task->plugin->id)) {
        case EVMS_DEVICE_MANAGER:
        case EVMS_SEGMENT_MANAGER:
        case EVMS_REGION_MANAGER:
        case EVMS_FEATURE:
        case EVMS_ASSOCIATIVE_FEATURE:
            rc = task->plugin->functions.plugin->init_task(task);
            break;

        case EVMS_FILESYSTEM_INTERFACE_MODULE:
            rc = task->plugin->functions.fsim->init_task(task);
            break;

        case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
        case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
        default:
            rc = 0;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_create_task(plugin_handle_t, object_handle_t, task_action_t *, task_handle_t *)
 *
 *   Description:
 *      This routine creates the task context structure used to bind parameters
 *      and option data for a specific operation, e.g. CreateObject().
 *
 *   Entry:
 *      plugin             - handle of plug-in to communicate task to
 *      referential_object - handle of object used in decision of acceptable
 *                           parameters
 *      action             - operation/low-level API to eventually get invoked
 *      new_task_context   - address for task context handle to be returned
 *
 *   Exit:
 *      On success, returns a return code of 0 and *new_task_context is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_create_task (engine_handle_t thing,
                      task_action_t   action,
                      task_handle_t * new_task_context) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            plugin_record_t * plugin = NULL;
            storage_object_t * obj = NULL;
            storage_container_t * con = NULL;
            logical_volume_t * vol = NULL;

            switch (type) {
                case PLUGIN:
                    if ((action == EVMS_Task_Create) ||
                        (action == EVMS_Task_Create_Container) ||
                        (action == EVMS_Task_Assign_Plugin) ||
                        (action == EVMS_Task_mkfs)) {

                        plugin = (plugin_record_t *) object;

                    } else {
                        LOG_ERROR("Command %d cannot be targeted at a plug-in.\n", action);
                        rc = EINVAL;
                    }
                    break;

                case EVMS_OBJECT:
                case REGION:
                case SEGMENT:
                case DISK:
                    if ((action == EVMS_Task_Set_Info) ||
                        (action == EVMS_Task_Expand) ||
                        (action == EVMS_Task_Shrink)) {

                        obj = (storage_object_t *) object;
                        plugin = obj->plugin;

                    } else {
                        LOG_ERROR("Command %d cannot be targeted at an object.\n", action);
                        rc = EINVAL;
                    }
                    break;

                case CONTAINER:
                    if ((action == EVMS_Task_Expand_Container) ||
                        (action == EVMS_Task_Set_Info)) {

                        con = (storage_container_t *) object;
                        plugin = con->plugin;

                    } else {
                        LOG_ERROR("Command %d cannot be targeted at a container.\n", action);
                        rc = EINVAL;
                    }
                    break;

                case VOLUME:
                    if ((action == EVMS_Task_fsck) ||
                        (action == EVMS_Task_defrag) ||
                        (action == EVMS_Task_Set_Info)) {

                        vol = (logical_volume_t *) object;

                        if (vol->file_system_manager != NULL) {
                            plugin = vol->file_system_manager;

                        } else {
                            LOG_ERROR("Command %d cannot be executed on volume %s because the volume has no File System Interface Module assigned to it.\n", action, vol->name);
                            rc = EINVAL;
                        }

                    } else {
                        LOG_ERROR("Command %d cannot be targeted at a volume.\n", action);
                        rc = EINVAL;
                    }
                    break;

                default:
                    LOG_ERROR("%d is not a valid handle for creating a task.\n", thing);
                    break;
            }

            if (rc == 0) {
                if (new_task_context != NULL) {
                    task_context_t * task = calloc(1, sizeof(task_context_t));

                    if (task != NULL) {
                        int size;
                        int count;

                        /*
                         * Initialize static members of the task context
                         * structure
                         */

                        task->plugin = plugin;
                        task->object = obj;
                        task->container = con;
                        task->volume = vol;
                        task->action = action;

                        /*
                         * Determine option count and allocate option
                         * descriptor and option value arrays.
                         */

                        count = GetTaskOptionsCount(task);

                        if (count > 1) {
                            size = sizeof(option_desc_array_t) +
                                   (sizeof(option_descriptor_t) * (count - 1));
                        } else {
                            size = sizeof(option_desc_array_t);
                        }

                        task->option_descriptors = (option_desc_array_t *) calloc(1, size);

                        if (task->option_descriptors != NULL) {

                            /* Initialize task lists. */
                            task->acceptable_objects = CreateList();
                            task->selected_objects = CreateList();
                            rc = InitTaskOptions(task);

                            if (rc == 0) {
                                rc = create_handle(task,
                                                   TASK_TAG,
                                                   new_task_context);

                                if (rc < 0) {
                                    LOG_WARNING("create_handle() error!\n");
                                    FreeTaskMemory(task);
                                }

                            } else {
                                LOG_WARNING("Error initializing options.\n");
                                FreeTaskMemory(task);
                            }

                        } else {
                            LOG_WARNING("Allocation of option descriptor array failed.\n");
                            FreeTaskMemory(task);
                            rc = ENOMEM;
                        }

                    } else {
                        LOG_WARNING("Memory allocation of task_context_t failed.\n");
                        rc = ENOMEM;
                    }

                } else {
                    LOG_ERROR("Address for context handle can not be NULL.\n");
                    rc = EINVAL;
                }
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * This function does the work for the EVMS_Task_Expand_Container task.
 * The code is broken out into a separate function to keep the switch
 * statement in evms_ivoke_task() tidy.
 */
static int do_expand_container_task(handle_array_t   * selected_ha,
                                    task_context_t   * task,
                                    option_array_t   * option_array) {
    int rc = 0;
    int i;
    int final_rc = 0;

    LOG_PROC_ENTRY();


    /*
     * evms_transfer() can only handle transferring one object at a time.  If
     * multiple objects are selected, issue an emvs_transfer() for each object.
     * If the evms_transfer() succeeds, put the transferred object on the
     * resulting_objects list so that the caller can know which transfers
     * succeeded in case of errors.
     */

    for (i = 0; i <= selected_ha->count; i++) {
        rc = evms_transfer(selected_ha->handle[i],
                           task->plugin->app_handle,
                           task->container->app_handle,
                           option_array);

        if (rc != 0) {

            /* Save the first non-zero error code. */
            if (final_rc != 0) {
                final_rc = rc;
            }
        }
    }

    rc = final_rc;

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_invoke_task(task_handle_t handle, handle_array_t * * resulting_objects)
 *
 *   Description:
 *      This routine uses the current option values and parameters in
 *      the task context to make the appropriate call to the low-level
 *      API for the task defined action/operation.
 *
 *   Entry:
 *      handle        - handle to task context created by evms_create_task()
 *      returned_data - pointer to any data returned as a result of
 *                      low-level API call
 *
 *   Exit:
 *      On success, returns a return code of 0 and possibly *returned_data
 *      is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_invoke_task(task_handle_t handle, handle_array_t * * resulting_objects) {
    int rc, tmp_rc, size, i;
    void * object;
    object_type_t type;
    option_array_t * option_array;

    LOG_PROC_ENTRY();

    *resulting_objects = NULL;

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if (type == TASK_TAG) {
                int j;
                handle_array_t * selected_ha;
                task_context_t * task = (task_context_t *) object;
                if (task->option_descriptors->count > 1) {
                    size = sizeof(option_desc_array_t) +
                           (sizeof(option_descriptor_t) * (task->option_descriptors->count - 1));
                } else {
                    size = sizeof(option_desc_array_t);
                }
                option_array = (option_array_t *) calloc(1, size);

                if (option_array != NULL) {
                    for (i=0,j=0; i< task->option_descriptors->count; i++) {
                        /* Only copy active values */
                        if (EVMS_OPTION_IS_ACTIVE(task->option_descriptors->option[i].flags)) {
                            option_array->option[j].type = task->option_descriptors->option[i].type;
                            option_array->option[j].value = task->option_descriptors->option[i].value;
                            option_array->option[j].is_number_based = TRUE;
                            option_array->option[j].number = i;
                            option_array->count++;
                            j++;
                        }
                    }

                    rc = make_handle_array(task->selected_objects, &selected_ha);

                    if (rc == 0) {
                        switch (task->action) {
                            case EVMS_Task_Create:

                                rc = evms_create(task->plugin->app_handle,
                                                 selected_ha,
                                                 option_array,
                                                 resulting_objects);
                                break;

                            case EVMS_Task_Create_Container:
                                {
                                    object_handle_t object_handle = 0;

                                    rc = evms_create_container(task->plugin->app_handle,
                                                               selected_ha,
                                                               option_array,
                                                               &object_handle);

                                    if (rc == 0) {
                                        if (resulting_objects != NULL) {
                                            *resulting_objects = malloc(sizeof(handle_array_t));

                                            if (*resulting_objects != NULL) {
                                                (*resulting_objects)->count = 1;
                                                (*resulting_objects)->handle[0] = object_handle;

                                            } else {
                                                LOG_CRITICAL("Error getting memory for a return handle array.\n");
                                                rc = ENOMEM;
                                            }
                                        }
                                    }
                                }
                                break;

                            case EVMS_Task_Assign_Plugin:
                                /*
                                 * Assignment of a plug-in to manage an object
                                 * is handled through the evms_create() API.
                                 * The plug-in in effect creates its meta data,
                                 * free space, etc., out of the object(s) given.
                                 */
                                rc = evms_create(task->plugin->app_handle,
                                                 selected_ha,
                                                 option_array,
                                                 resulting_objects);
                                break;

                            case EVMS_Task_Expand_Container:

                                rc = do_expand_container_task(selected_ha,
                                                              task,
                                                              option_array);
                                break;

                            case EVMS_Task_Set_Info:

                                if (task->object != NULL) {
                                    rc = evms_set_info(task->object->app_handle,
                                                       option_array);
                                } else if (task->container != NULL) {
                                    rc = evms_set_info(task->container->app_handle,
                                                       option_array);
                                } else if (task->volume != NULL) {
                                    rc = evms_set_info(task->volume->app_handle,
                                                       option_array);
                                } else {
                                    LOG_ERROR("Task has no target object, container, nor volume.\n");
                                    rc = EINVAL;
                                }
                                break;

                            case EVMS_Task_Expand:

                                rc = evms_expand(task->object->app_handle,
                                                 selected_ha,
                                                 option_array);
                                break;

                            case EVMS_Task_Shrink:

                                rc = evms_shrink(task->object->app_handle,
                                                 selected_ha,
                                                 option_array);
                                break;

                            case EVMS_Task_mkfs:

                                /* The FSIM may allow multiple selections of the
                                 * volumes to be mkfs'd, but the emms_mkfs() API
                                 * only takes a single volume.  Run through the
                                 * list of selected volumes and issue an
                                 * evms_mkfs() for it.
                                 */
                                for (i = 0; i < selected_ha->count; i++) {
                                    tmp_rc = evms_mkfs(selected_ha->handle[i],
                                                  task->plugin->app_handle,
                                                  option_array);

                                    if ((tmp_rc != 0) && (rc == 0)) {
                                        rc = tmp_rc;
                                    }
                                }
                                break;

                            case EVMS_Task_fsck:

                                rc = evms_fsck(task->volume->app_handle,
                                               option_array);
                                break;

                            case EVMS_Task_defrag:

                                rc = evms_defrag(task->volume->app_handle,
                                                 option_array);
                                break;

                            case EVMS_Task_Slide:
                            case EVMS_Task_Move:
                                LOG_WARNING("Task action %d is not supported yet.\n", task->action);

                            default:
                                LOG_ERROR("Unknown task action %d.\n", task->action);
                                rc = EINVAL;
                                break;
                        }

                        free (selected_ha);
                    }

                    free (option_array);

                } else {
                    LOG_CRITICAL("Error allocating memory for building the option array.\n");
                    rc = ENOMEM;
                }

            } else {
                LOG_ERROR("%d is not a task context handle.\n", handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_destroy_task(task_handle_t)
 *
 *   Description:
 *      This routine frees up all resources/memory associated with the
 *      task handle and invalidates the task handle upon successful
 *      completion.
 *
 *   Entry:
 *      handle - handle to task context created by evms_create_task()
 *
 *   Exit:
 *      On success, returns a return code of 0 and task context is no more
 *      On error, returns an error code < 0
 *
 */
int evms_destroy_task (task_handle_t handle) {
    int rc;
    void * object;
    object_type_t type;

    /*
     * Question: What mechanism do we have to avoid
     * race conditions between two or more threads
     * calling us with the same handle?
     */

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                /*
                 * Empty out the parameter list and deallocate all
                 * memory, including the task_context_t and finally
                 * destroy the task context handle.
                 */

                FreeTaskMemory(task);

                rc = destroy_handle(handle);

            } else {
                LOG_ERROR("%d is not a task context handle.\n", handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_get_task_action (task_handle_t, task_action_t *)
 *
 *   Description:
 *      This routine
 *
 *   Entry:
 *      handle - handle to task context created by evms_create_task()
 *      action -
 *
 *   Exit:
 *      On success, returns a return code of 0 and action is returned
 *      On error, returns an error code != 0
 *
 */
int evms_get_task_action (task_handle_t handle, task_action_t *action) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (action != NULL)
                    *action = task->action;
                else
                    rc = EINVAL;

            } else {
                LOG_ERROR("%d is not a task context handle.\n", handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Add the declined_handle for a given object to a declined_handle_array_t.
 * The parameters of this function are structured so that it can be called by
 * ForEachItem().
 */
int make_declined_handle_entry(ADDRESS object,
                               TAG     object_tag,
                               uint    object_size,
                               ADDRESS object_handle,
                               ADDRESS parameters) {
    int rc = 0;

    declined_object_t * pDecObj = (declined_object_t *) object;
    declined_handle_array_t * dha = (declined_handle_array_t *) parameters;

    LOG_PROC_ENTRY();

    if (object_tag == DECLINED_OBJECT_TAG) {
        storage_object_t * pObj = pDecObj->object;

        engine_write_log_entry(DEBUG, "Number of entries in declined handle array:  %d.\n", dha->count);

        rc = ensure_app_handle(pObj,
                               pObj->object_type,
                               &pObj->app_handle);
        if (rc == 0) {
            dha->declined[dha->count].handle = pObj->app_handle;
            dha->declined[dha->count].reason = pDecObj->reason;
            dha->count++;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make a user (it has a Engine memory block header) array of handles
 * (handle_array_t) for the objects in a dlist_t.
 */
int make_user_declined_handle_array(dlist_t declined_objects_list, declined_handle_array_t * * dha) {
    int rc = 0;
    uint count;
    uint size;

    LOG_PROC_ENTRY();

    rc = GetListSize(declined_objects_list, &count);

    if (rc == 0) {
        engine_write_log_entry(DEBUG, "Number of objects in the list:  %d\n", count);
        if (count > 1) {
            size = sizeof(declined_handle_array_t) + ((count -1) * sizeof(declined_handle_t));
        } else {
            size = sizeof(declined_handle_array_t);
        }

        *dha = alloc_app_struct(size, NULL);
        if (*dha != NULL) {
            rc = ForEachItem(declined_objects_list,
                             make_declined_handle_entry,
                             *dha,
                             TRUE);
        } else {
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_get_acceptable_objects(task_handle_t, handle_array_t **)
 *
 *   Description:
 *      This routine retrieves handles to objects, e.g. regions,
 *      segments, containers, etc., deemed "acceptable" to the
 *      backend for the specified task context.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      acceptable_list - address of pointer to contain acceptable parameter
 *                        handles
 *
 *   Exit:
 *      On success, returns a return code of 0 and **acceptable_list is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_acceptable_objects(task_handle_t      handle,
                                handle_array_t * * acceptable_object_list) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {

        /* The caller must provide an acceptable_object_list. */
        if (acceptable_object_list != NULL) {

            *acceptable_object_list = NULL;

            rc = translate_handle(handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {

                if (type == TASK_TAG) {
                    task_context_t * task = (task_context_t *) object;

                    rc = make_user_handle_array(task->acceptable_objects, acceptable_object_list);

                } else {
                    LOG_ERROR("The handle given is not for a task context.\n");
                    rc = EINVAL;
                }
            } else {
                LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("The pointer to the acceptable objects list cannot be NULL.\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_get_selected_objects(task_handle_t, handle_array_t **)
 *
 *   Description:
 *      This routine retrieves handles to objects, e.g. regions,
 *      segments, containers, etc., that are currently selected by the user.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      acceptable_list - address of pointer to contain selected object
 *                        handles
 *
 *   Exit:
 *      On success, returns a return code of 0 and **acceptable_list is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_selected_objects(task_handle_t      handle,
                              handle_array_t * * selected_object_list) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {

        /* The caller must provide an selected_object_list. */
        if (selected_object_list != NULL) {

            *selected_object_list = NULL;

            rc = translate_handle(handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {

                if (type == TASK_TAG) {
                    task_context_t * task = (task_context_t *) object;

                    rc = make_user_handle_array(task->selected_objects, selected_object_list);

                } else {
                    LOG_ERROR("The handle given is not for a task context.\n");
                    rc = EINVAL;
                }
            } else {
                LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("The pointer to the selected objects list cannot be NULL.\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_get_selected_object_limits(task_handle_t, u_int32_t *, u_int32_t *)
 *
 *   Description:
 *      This routine retrieves limits, minimum and maximum, for the number of
 *      objects that the user can select.
 *
 *   Entry:
 *      handle          - handle to task context created by evms_create_task()
 *      pMininum        - address of where to put the minimum
 *      pMaximum        - address of where to put the maximum
 *
 *   Exit:
 *      On success, returns a return code of 0 and *minimum and *maximum are
 *      filled in.
 *      On error, returns an error code < 0
 *
 */
int evms_get_selected_object_limits(task_handle_t handle,
                                    u_int32_t   * pMinimum,
                                    u_int32_t   * pMaximum) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {

        /* The caller must provide valid pointers for the limits */
        if ((pMinimum != NULL) && (pMaximum != NULL)) {

            *pMinimum = 0;
            *pMaximum = 0;

            rc = translate_handle(handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {

                if (type == TASK_TAG) {
                    task_context_t * task = (task_context_t *) object;

                    *pMinimum = task->min_selected_objects;
                    *pMaximum = task->max_selected_objects;

                } else {
                    LOG_ERROR("The handle given is not for a task context.\n");
                    rc = EINVAL;
                }
            } else {
                LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
                rc = EINVAL;
            }

        } else {
            if (pMinimum == NULL) {
                LOG_ERROR("The address for the minimum value cannot be NULL.\n");
            }
            if (pMaximum == NULL) {
                LOG_ERROR("The address for the maximum value cannot be NULL.\n");
            }
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * int is_in_list(ADDRESS object
 *                TAG     object_tag
 *                uint    object_size
 *                ADDRESS object_handle
 *                ADDRESS parameters)
 *
 * Description:
 *      This is a helper function for is_acceptable_object() below.
 *      The parameters are structured so that it can be called by ForEachItem().
 *      This function is called for each object in an acceptable_objects list.
 *      It simply checks to see if the object passed in the parameters is in the
 *      acceptable_objects list.  If the object is found this function returns
 *      the non-zero value DLIST_OBJECT_ALREADY_IN_LIST.  A non-zero value will
 *      terminate the ForEachItem() processor, which is what we want because the
 *      search is over; the object was found in the list.
 *
 * Entry:
 *      object        - object from the acceptable_objects list
 *      object_tag    - object's tag (not used)
 *      object_size   - size of the object (not used)
 *      object_handle - handle of the object (not used)
 *      parameters    - pointer to an object that should be in the list
 *
 * Exit:
 *      DLIST_SUCCESS                 - Object was not found in the list
 *      DLIST_OBJECT_ALREADY_IN_LIST  - Object was found in the list.
 */
static int is_in_list(ADDRESS object,
                      TAG     object_tag,
                      uint    object_size,
                      ADDRESS object_handle,
                      ADDRESS parameters) {

    int rc = DLIST_SUCCESS;
    storage_object_t * obj = (storage_object_t *) object;
    storage_object_t * search_object = (storage_object_t *) parameters;

    LOG_PROC_ENTRY();

    if (obj == search_object) {
        /* Use the error code DLIST_OBJECT_ALREADY_IN_LIST to indicate that
         * the object was found in the list.  A non-zero error code will
         * stop ForEachItem from further processing the list, which is what
         * we want since we found the object.
         */
        rc = DLIST_OBJECT_ALREADY_IN_LIST;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * BOOLEAN is_acceptable_object(ADDRESS   object,
 *                              TAG       object_tag,
 *                              uint      object_size,
 *                              ADDRESS   object_handle,
 *                              ADDRESS   parameters,
 *                              BOOLEAN * free_memory,
 *                              uint    * error))
 *
 * Description:
 *      This is a helper function for validate_selected_objects() below.
 *      The parameters are structured so that it can be called by PruneList()
 *      to process a list of selected_objects.  This function is called for
 *      each object in the selected_objects list.  It checks to see if the
 *      object is in the acceptable_objects list.  If not, it creates a
 *      declined_object_t for the object and puts the declined_object_t on
 *      the declined_object list (one of the fields in the parameter block)
 *      and returns TRUE so that Prune list will remove the object from the
 *      selected_objects list.
 *
 * Entry:
 *      object        - object from the selected_objects list
 *      object_tag    - object's tag (not used)
 *      object_size   - size of the object (not used)
 *      object_handle - handle of the object (not used)
 *      parameters    - pointer to a is_acc_obj_parms_t
 *      free_memory   - pointer to BOOLEAN that indicates if the object
 *                      memory should be freed if the object is removed
 *                      from the list
 *      error         - pointer to where to put the error code
 *
 * Exit:
 *      FALSE  - Leave the object in the list
 *      TRUE   - Remove the object from the list
 *      *error - Error code
 */
typedef struct is_acc_obj_parms_s {
    dlist_t acceptable_objects;
    dlist_t declined_objects;
} is_acc_obj_parms_t;


static BOOLEAN is_acceptable_object(ADDRESS   object,
                                    TAG       object_tag,
                                    uint      object_size,
                                    ADDRESS   object_handle,
                                    ADDRESS   parameters,
                                    BOOLEAN * free_memory,
                                    uint    * error) {
    BOOLEAN result = FALSE;
    int rc = DLIST_SUCCESS;
    storage_object_t * obj = (storage_object_t *) object;
    is_acc_obj_parms_t * parms = (is_acc_obj_parms_t *) parameters;

    LOG_PROC_ENTRY();

    /* Never free the item memory. */
    *free_memory = FALSE;

    /* Check if the object is in the acceptable_objects list. */
    rc = ForEachItem(parms->acceptable_objects,
                     is_in_list,
                     object,
                     TRUE);

    /*
     * is_in_list() will return DLIST_OBJECT_ALREADY_IN_LIST if it finds
     * the object in the list, which is what we want.  So, in this case
     * an error code of DLIST_SUCCESS means the object was not found in
     * the acceptable_object list, which is bad.  If the object is not
     * found in the acceptable_object list, put it in the declined_object
     * list and return TRUE so that PruneList will remove the object from
     * the selected_objects list.
     */
    if (rc == DLIST_SUCCESS) {
        declined_object_t * declined_object = malloc(sizeof(declined_object_t));

        if (declined_object != NULL) {
            ADDRESS trash;

            declined_object->reason = EINVAL;
            declined_object->object = obj;

            rc = InsertItem(parms->declined_objects,
                            sizeof(declined_object_t),
                            declined_object,
                            DECLINED_OBJECT_TAG,
                            NULL,
                            AppendToList,
                            FALSE,
                            &trash);

            if (rc == DLIST_SUCCESS) {
                /*
                 * Return TRUE so that the object is removed from the
                 * selected_objects list.
                 */
                result = TRUE;

            } else {
                LOG_CRITICAL("Error %d inserting declined object into declined_objects list.\n", rc);
            }

        } else {
            LOG_CRITICAL("Error allocating memory for a declined object.\n");
        }
    }

    /*
     * If the object was found in the acceptable_objects list, reset the error
     * code so that PruneList will not abort.  We want to process the whole
     * selected_objects list, so only catastrophic errors should be allowed to
     * abort PruneList.
     */
    if (rc == DLIST_OBJECT_ALREADY_IN_LIST) {
        rc = DLIST_SUCCESS;
    }

    *error = rc;

    LOG_PROC_EXIT_BOOLEAN_INT(result, *error);
    return result;

}

/*
 * int validate_selected_objects(task_context_t *, dlist_t)
 *
 * Description:
 *      This is a helper function for evms_set_selected_objects() below.
 *      It makes sure that all the objects in the selected_objects list in
 *      the task context appear in the acceptable_objects list in the task
 *      context.  Any selected objects that do not appear in the
 *      acceptable_objects list are removed from the selected_objects list.
 *      A declined_object_t is created for the removed object and is placed
 *      on the declined_objects list.
 *
 * Entry:
 *      task             - a pointer to a task context
 *      declined_objects - a dlist for declined objects
 *
 * Exit:
 *      0        - All selected objects are in the acceptable_objects list
 *      EINVAL   - One or more selected objects is not in the acceptable_objects
 *                 list
 *      error    - some other error occurred during processing
 */
static int validate_selected_objects(task_context_t * task, dlist_t declined_objects) {

    int rc = 0;
    is_acc_obj_parms_t * parms = malloc(sizeof(is_acc_obj_parms_t));

    LOG_PROC_ENTRY();

    if (parms != NULL) {
        parms->acceptable_objects = task->acceptable_objects;
        parms->declined_objects = declined_objects;

        rc = PruneList(task->selected_objects,
                       is_acceptable_object,
                       parms);

        if (rc == DLIST_SUCCESS) {

            /*
             * The return code will be 0 even if selected objects were removed
             * from the selected_objects list and put on the declined_objects
             * list.  To determine if any selected objects were declined, check
             * if there are any objects on the declined_objects list.
             */
            uint count;

            rc = GetListSize(declined_objects, &count);

            if (rc == DLIST_SUCCESS) {
                if (count != 0) {
                    rc = EINVAL;
                }

            } else {
                LOG_CRITICAL("Error %d getting the size of the declined_objects list.\n", rc);
            }
        }

        free(parms);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * int remove_declined_object(ADDRESS object
 *                            TAG     object_tag
 *                            uint    object_size
 *                            ADDRESS object_handle
 *                            ADDRESS parameters)
 *
 * Description:
 *      This is a helper function for evms_set_selected_objects() below.
 *      The parameters are structured so that it can be called by ForEachItem().
 *      This function is called for each object in an declined_objects list.
 *      It simply makes sure that the object in the delined_object_t is not in
 *      the list given in the parameters.
 *
 * Entry:
 *      object        - object from the acceptable_objects list
 *      object_tag    - object's tag
 *      object_size   - size of the object (not used)
 *      object_handle - handle of the object (not used)
 *      parameters    - pointer to an object that should be in the list
 *
 * Exit:
 *      DLIST_SUCCESS
 */
static int remove_declined_object(ADDRESS object,
                                  TAG     object_tag,
                                  uint    object_size,
                                  ADDRESS object_handle,
                                  ADDRESS parameters) {

    int rc = DLIST_SUCCESS;
    declined_object_t * declined_object = (declined_object_t *) object;
    dlist_t list = (dlist_t) parameters;

    LOG_PROC_ENTRY();

    /*
     * Safety check.  This function should only be called on a list of declined
     * objects, but we'll check, just in case.
     */
    if (object_tag == DECLINED_OBJECT_TAG) {
        DeleteObject(list, declined_object->object);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   int evms_set_selected_objects(task_handle_t,
 *                                 handle_array_t *,
 *                                 declined_handle_array_t **,
 *                                 task_effect_t)
 *
 *   Description:
 *      This routine allows the application to associate/set the choice of
 *      objects from the initial acceptable objects. If the choices
 *      are not acceptable to the backend, the handles of the declined
 *      objects and reason codes are returned in the declined_list.
 *
 *   Entry:
 *      handle               - handle to task context created by evms_create_task()
 *      selected_object_list - address of handle_array_t containing selected
 *                             objects
 *      declined_list        - address of pointer to a declined_handle_array_t
 *                             containing any declined handles and reason codes
 *      effect               - flags indicating side effects from this command,
 *                             e.g., object list(s) changed, options changed
 *
 *   Exit:
 *      On success, returns a return code of 0 and possibly
 *      **declined_list and *effect are updated.
 *      On error, returns an error code < 0
 *
 */
int evms_set_selected_objects(task_handle_t               handle,
                              handle_array_t            * selected_object_list,
                              declined_handle_array_t * * declined_list,
                              task_effect_t             * effect) {
    int rc;
    void * object;
    object_type_t type;
    dlist_t declined_objects;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {

        /* The caller must provide an selected_object_list. */
        if (selected_object_list != NULL) {
            rc = translate_handle(handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {

                if (type == TASK_TAG) {
                    task_context_t * task = (task_context_t *) object;

                    /* Reset the lists in the context record */
                    DestroyList(&(task->selected_objects), FALSE);
                    task->selected_objects = CreateList();

                    if (task->selected_objects) {
                        make_dlist(selected_object_list, task->selected_objects);
                        declined_objects = CreateList();

                        if (declined_objects != NULL) {

                            /*
                             * Do some pre-validation for the plug-in.
                             * Make sure all of the selected objects appear in
                             * the acceptable_objects list.
                             */
                            rc = validate_selected_objects(task, declined_objects);

                            if (rc == 0) {
                                switch (GetPluginType(task->plugin->id)) {
                                    case EVMS_DEVICE_MANAGER:
                                    case EVMS_SEGMENT_MANAGER:
                                    case EVMS_REGION_MANAGER:
                                    case EVMS_FEATURE:
                                    case EVMS_ASSOCIATIVE_FEATURE:
                                        rc = task->plugin->functions.plugin->set_objects(task, declined_objects,effect);
                                        break;

                                    case EVMS_FILESYSTEM_INTERFACE_MODULE:
                                        rc = task->plugin->functions.fsim->set_volumes(task, declined_objects, effect);
                                        break;

                                    default:
                                        rc = ENOSYS;
                                }

                                /*
                                 * Remove any declined objects from the
                                 * selected_objects list in the task context,
                                 * just in case the plug-in didn't do it.
                                 */
                                ForEachItem(declined_objects,
                                            remove_declined_object,
                                            task->selected_objects,
                                            TRUE);
                            }

                            /* Does the user want a declined handle array? */
                            if (declined_list != NULL) {

                                /*
                                 * Make a declined handle array only if we have
                                 * declined objects.
                                 */
                                uint count = 0;

                                GetListSize(declined_objects, &count);

                                if (count != 0) {
                                    int err;

                                    err = make_user_declined_handle_array(declined_objects, declined_list);

                                    /*
                                     * If we don't already have a bad error
                                     * code to return, return any error from
                                     * make_user_declined_handle_array().
                                     */
                                    if (rc == 0) {
                                        rc = err;
                                    }

                                } else {
                                    *declined_list = NULL;
                                }
                            }

                            DestroyList(&declined_objects,FALSE);

                        } else {
                            LOG_CRITICAL("Error allocating memory for the declined objects list.\n");
                            rc = ENOMEM;
                        }

                    } else {
                        LOG_CRITICAL("Error allocating memory for the new selected objects list in the task context.\n");
                        rc = ENOMEM;
                    }

                } else {
                    LOG_ERROR("The handle given is not for a task context.\n");
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("translate_handle() could not find the task context for handle %d.\n", handle);
                rc = EINVAL;
            }

        } else {
            LOG_ERROR("The pointer to the source list cannot be NULL.\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

