/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: container.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>
#include "container.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "internalAPI.h"
#include "object.h"
#include "message.h"


/*
 * Ask the plug-in if the object can be added to the container.
 */
int evms_can_add_to_container(object_handle_t object_handle,
                              object_handle_t container_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(object_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if ((type == DISK) ||
                (type == SEGMENT) ||
                (type == REGION) ||
                (type == EVMS_OBJECT)) {

                storage_object_t * obj = (storage_object_t *) object;

                if (obj->consuming_container == NULL) {
                    if (obj->consuming_container->plugin->container_functions != NULL) {

                        rc = translate_handle(container_handle,
                                              &object,
                                              &type);

                        if (rc == HANDLE_MANAGER_NO_ERROR) {
                            if (type == CONTAINER) {
                                storage_container_t * con = (storage_container_t *) object;

                                if (con->plugin->container_functions != NULL) {
                                    rc = con->plugin->container_functions->can_add_object(obj, con);

                                } else {
                                    LOG_ERROR("The plug-in %s for container %s does not support containers.\n", con->plugin->short_name, con->name);
                                    rc = EINVAL;
                                }

                            } else {
                                LOG_ERROR("Handle %d is not for a container.\n", container_handle);
                            }
                        }

                    } else {
                        LOG_ERROR("The plug-in %s for object %s does not support containers.\n", obj->plugin->short_name, obj->name);
                        rc = EINVAL;
                    }

                } else {
                    LOG_ERROR("Object %s is already in container %s.\n", obj->name, obj->consuming_container->name);
                }

            } else {
                LOG_ERROR("Handle %d is for a thing of type %d which cannot belong in a container.\n", object_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Ask the plug-in if the object can be removed from its container.
 */
int evms_can_remove_from_container(object_handle_t object_handle) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(object_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if ((type == DISK) ||
                (type == SEGMENT) ||
                (type == REGION) ||
                (type == EVMS_OBJECT)) {

                storage_object_t * obj = (storage_object_t *) object;

                if (obj->consuming_container != NULL) {
                    if (obj->consuming_container->plugin->container_functions != NULL) {

                        rc = obj->consuming_container->plugin->container_functions->can_remove_object(obj);

                    } else {
                        LOG_ERROR("The plug-in %s for container %s does not support containers.\n", obj->plugin->short_name, obj->consuming_container->name);
                        rc = EINVAL;
                    }

                } else {
                    LOG_ERROR("Object %s is not in a container.\n", obj->name);
                    rc = EINVAL;
                }

            } else {
                LOG_ERROR("Handle %d is for a thing of type %d which cannot belong in a container.\n", object_handle);
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int isa_valid_container_input_object(ADDRESS object,
                                            TAG     object_tag,
                                            uint    object_size,
                                            ADDRESS object_handle,
                                            ADDRESS parameters) {
    int rc = 0;

    storage_object_t * obj = (storage_object_t *) object;
    plugin_record_t * plugin = (plugin_record_t *) parameters;

    LOG_PROC_ENTRY();

    /* The object must not be part of a volume. */
    if (obj->volume == NULL) {

        /* The object must not have any parents. */
        if (obj->parent_objects != NULL) {
            uint count;

            rc = GetListSize(obj->parent_objects,&count);
            if (rc == 0) {
                if (count != 0) {
                    LOG_ERROR("Object %s is not a valid input object.  It has parent objects.\n", obj->name);
                    rc = EINVAL;
                }
            }
        }

        if (rc == 0) {
            /* The object must not insist on being a top object. */
            if (!(obj->flags & SOFLAG_MUST_BE_TOP)) {

                /*
                 * If the object is part of a container, the container must be
                 * managed by the creating plug-in.
                 */
                if (obj->consuming_container != NULL) {
                    if (obj->consuming_container->plugin == plugin) {
                        LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
                    } else {
                        LOG_ERROR("Object %s is not a valid input object.\n", obj->name);
                        LOG_ERROR("  It is currently part of container %s which is managed by plug-in %s.\n", obj->consuming_container->name, obj->consuming_container->plugin->short_name);
                        LOG_ERROR("  Plug-in %s is not the same as the creating plug-in, %s.\n", obj->consuming_container->plugin->short_name, plugin->short_name);
                        rc = EINVAL;
                    }

                } else {
                    LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
                }

            } else {
                LOG_ERROR("Object %s is not a valid input object.  It insists it must be a top level object.\n", obj->name);
                rc = EINVAL;
            }
        }

    } else {
        LOG_ERROR("Object %s is not a valid input object.  It is part of a volume.\n", obj->name);
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_create_container(plugin_handle_t   plugin_handle,
                          handle_array_t  * input_objects,
                          option_array_t  * options,
                          object_handle_t * new_container_handle) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(plugin_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == PLUGIN) {
                plugin_record_t * pPlugRec = (plugin_record_t *) object;

                /* Make sure the plug-in does containers. */
                if (pPlugRec->container_functions != NULL) {

                    dlist_t object_list = CreateList();

                    if (object_list != NULL) {
                        rc = make_dlist(input_objects, object_list);

                        if (rc == 0) {
                            /*
                             * Make sure that the input object list contains
                             * top level objects and none of them insist on
                             * being a top object.
                             */
                            rc = ForEachItem(object_list,
                                             isa_valid_container_input_object,
                                             pPlugRec,
                                             TRUE);

                            if (rc == 0) {
                                storage_container_t * new_container;

                                rc = pPlugRec->container_functions->create_container(object_list, options, &new_container);

                                if (rc == 0) {
                                    rc = create_handle(new_container,
                                                       CONTAINER,
                                                       &new_container->app_handle);
                                    if (rc == 0) {
                                        *new_container_handle = new_container->app_handle;

                                    } else {
                                        LOG_WARNING("Error %d creating a handle for container %s.\n", rc, new_container->name);
                                    }
                                }
                            }
                        }

                        DestroyList(&object_list, FALSE);

                    } else {
                        rc = ENOMEM;
                    }
                }

            } else {
                rc = EINVAL;
            }

        } else {
            rc = EINVAL;
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_get_container_list returns a pointer to a handle_array_t with handles for
 * storage containers, optionally filtering on the region manager that manages
 * the container.  If the object handle for the region manager is 0, handles for
 * all of the storage containers will be returned.
 */
int evms_get_container_list(object_handle_t    plugin_handle,
                            handle_array_t * * container_handle_list) {
    int rc = 0;
    void * object = NULL;
    object_type_t type;
    plugin_record_t * pPlugRec = NULL;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        if (plugin_handle != 0) {
            /* Translate the handle for the region manager to make sure it is valid */
            /* and to get the plugin_record_t for the region manager. */
            rc = translate_handle(plugin_handle,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {
                if (type == PLUGIN) {
                    pPlugRec = (plugin_record_t *) object;
                } else {
                    rc = EINVAL;
                }
            }
        }

        if (rc == 0) {
            dlist_t container_list;

            /* Call the internal version of GetContainerList.  "region_manager" */
            /* will be NULL if the caller did not specify a region manager, */
            /* else it will be a pointer to the region manager's plugin_record_t. */
            rc = engine_get_container_list(pPlugRec, &container_list);

            if (rc == 0) {
                rc = make_user_handle_array(container_list, container_handle_list);

                /* We are finished with the list that was returned by */
                /* Engine_GetContainerList. */
                DestroyList(&container_list, FALSE);
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

