/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: common.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>

#include "common.h"
#include "engine.h"
#include "discover.h"
#include "handlemgr.h"
#include "volume.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"


BOOLEAN is_top_object(storage_object_t * obj) {

    BOOLEAN result = FALSE;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Examining object %s.\n", obj->name);

    /* The object must not be part of a volume. */
    if (obj->volume == NULL) {

        /* The object must not be part of a container. */
        if (obj->consuming_container == NULL) {

            /* The object must have no parent objects. */
            if (obj->parent_objects != NULL) {
                uint count = 0;

                GetListSize(obj->parent_objects, &count);
                if (count == 0) {
                    result = TRUE;

                } else {
                    LOG_DEBUG("Object %s has parent object(s).\n", obj->name);
                }

            } else {
                result = TRUE;
            }

        } else {
            LOG_DEBUG("Object %s is part of container %s.\n", obj->name, obj->consuming_container->name);
        }

    } else {
        LOG_DEBUG("Object %s is part of volume %s.\n", obj->name, obj->volume->name);
    }

    LOG_PROC_EXIT_BOOLEAN(result);
    return result;
}


int evms_can_remove(object_handle_t thing) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case VOLUME:
                    {
                        logical_volume_t * volume = (logical_volume_t *) object;

                        /* Check if the volume is mounted. */
                        if (!is_volume_mounted(volume)) {

                            /*
                             * Ask the object below the volume if it can handle
                             * having the volume removed.
                             */
                            if (volume->object->flags & SOFLAG_MUST_BE_VOLUME) {
                                /*
                                 * It's an object that must be a volume.  The
                                 * object must go down with the volume.  The
                                 * volume can be removed if the object can be
                                 * deleted.
                                 */
                                rc = volume->object->plugin->functions.plugin->can_delete(volume->object);

                            } else {
                                rc = volume->object->plugin->functions.plugin->can_set_volume(volume->object, FALSE);
                            }

                        } else {
                            LOG_ERROR("Volume %s cannot be removed because it is mounted.\n", volume->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                case EVMS_OBJECT:
                case REGION:
                case SEGMENT:
                case DISK:
                    {
                        storage_object_t * storage_object = (storage_object_t *) object;

                        /*
                         * Make sure the object is a top level object.  We
                         * can't remove objects in the middle of a feature
                         * stack.  A top level object has no parent.
                         */
                        if (is_top_object(storage_object)) {

                            /*
                             * Ask the feature if the object can be
                             * removed.
                             */
                            rc = storage_object->plugin->functions.plugin->can_delete(storage_object);

                        } else {
                            /* The object is not a top level object. */
                            LOG_ERROR("Object %s cannot be removed because it is not a top level object.\n", storage_object->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                default:
                    /* We can't remove a thing of this type. */
                    LOG_ERROR("A thing of type %lX cannot be removed.\n", type);
                    rc = EINVAL;
                    break;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int can_destroy_object(ADDRESS Object,
                              TAG     ObjectTag,
                              uint    ObjectSize,
                              ADDRESS ObjectHandle,
                              ADDRESS Parameters) {
    int rc = 0;

    storage_object_t * obj = (storage_object_t *) Object;

    LOG_PROC_ENTRY();

    /* Disks cannot be destroyed, they must be removed.  Return 0 if this is a
     * DISK so that the can_destroy_object() will not fail if this is a DISK.
     * The disk will not be destroyed on a evms_destroy(), but anything stacked
     * on top of the disk will.
     */

    if (obj->object_type != DISK) {
        rc = obj->plugin->functions.plugin->can_delete(obj);

        if (rc == 0) {
            /*
             * If the object is not produced by a container
             * do a recursive check to see if all
             * of its children can be deleted.
             */
            if (obj->producing_container == NULL) {
                rc = ForEachItem(obj->child_objects,
                                 can_destroy_object,
                                 NULL,
                                 TRUE);
            }
        }
    }

    /* If the object is a DISK, we return 0 so that the recursive check doesn't
     * fail and the recursion ends.
     * can_destroy_object() might be called initially on a DISK (e.g., an EVMS
     * volume is made from a DISK and the volume is being destroyed).  In that
     * case, can_destroy_object() will return 0, but a subsequent call to
     * evms_destroy() will destroy the volume but leave the disk.
     */

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_can_destroy(object_handle_t thing) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case VOLUME:
                    {
                        logical_volume_t * volume = (logical_volume_t *) object;

                        /* Check if the volume is mounted. */
                        if (!is_volume_mounted(volume)) {

                            rc = can_destroy_object(volume->object,
                                                    get_tag_for_object(volume->object),
                                                    sizeof(storage_object_t),
                                                    NULL,
                                                    NULL);

                        } else {
                            LOG_ERROR("Volume %s cannot be destroyed because it is mounted.\n", volume->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                case CONTAINER:
                    {
                        storage_container_t * container = (storage_container_t *) object;

                        rc = container->plugin->container_functions->can_delete_container(container);
                    }
                    break;

                case EVMS_OBJECT:
                case REGION:
                case SEGMENT:
                    {
                        storage_object_t * obj = object;

                        if (is_top_object(obj)) {

                            rc = can_destroy_object(obj,
                                                    get_tag_for_object(obj),
                                                    sizeof(storage_object_t),
                                                    NULL,
                                                    NULL);

                        } else {
                            LOG_ERROR("Object %s cannot be destroyed because it is not a top level object.\n", obj->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                case DISK:
                    {
                        storage_object_t * obj = object;

                        LOG_ERROR("Disk %s cannot be destroyed because disks cannot be destroyed.  Disks must be removed.\n", obj->name);
                        rc = EINVAL;
                    }
                    break;

                default:
                    rc = EINVAL;
                    break;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_can_move(object_handle_t source, object_handle_t target) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(source,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case DISK:
                case CONTAINER:
                    rc = EINVAL;
                    break;

                case SEGMENT:
                case REGION:
                case EVMS_OBJECT:
                    {
                        storage_object_t * obj = object;

                        rc = obj->plugin->functions.plugin->can_move(obj);
                    }
                    break;


                case VOLUME:
                    {
                        logical_volume_t * volume = (logical_volume_t *) object;

                        rc = volume->object->plugin->functions.plugin->can_move(volume->object);
                    }
                    break;

                default:
                    rc = EINVAL;
                    break;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_can_set_info will determine if a things's information can be set.
 */
int evms_can_set_info(object_handle_t thing_handle) {

    int rc = 0;
    object_type_t type;
    void  * thing = NULL;
    int count = -1;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        task_context_t * task = calloc(1, sizeof(task_context_t));

        if (task != NULL) {
            task->action = EVMS_Task_Set_Info;

            rc = translate_handle(thing_handle, &thing, &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {

                /*
                 * Check if the plug-in supports setting of object information
                 * by getting the option count for EVMS_Task_Set_Info.  If the
                 * option count is 0 or less, then the plug-in does not support
                 * the setting of object information.
                 */
                switch (type) {
                    case DISK:
                    case SEGMENT:
                    case REGION:
                    case EVMS_OBJECT:
                        {
                            storage_object_t * obj = (storage_object_t *) thing;
                            task->object = obj;
                            count = obj->plugin->functions.plugin->get_option_count(task);

                            if (count <= 0) {
                                LOG_DEBUG("Plug-in %s does not have any descriptors for setting object info.\n", obj->plugin->short_name);
                                rc = ENOSYS;
                            }
                        }
                        break;

                    case CONTAINER:
                        {
                            storage_container_t * con = (storage_container_t *) thing;
                            task->container = con;
                            count = con->plugin->functions.plugin->get_option_count(task);

                            if (count <= 0) {
                                LOG_DEBUG("Plug-in %s does not have any descriptors for setting container info.\n", con->plugin->short_name);
                                rc = ENOSYS;
                            }
                        }
                        break;

                    case VOLUME:
                        {
                            logical_volume_t * vol = (logical_volume_t *) thing;
                            task->volume = vol;
                            if (vol->file_system_manager != NULL) {
                                rc = vol->file_system_manager->functions.fsim->get_option_count(task);

                                if (count <= 0) {
                                    LOG_DEBUG("Plug-in %s does not have any descriptors for setting volume info.\n", vol->file_system_manager->short_name);
                                    rc = ENOSYS;
                                }

                            } else {
                                LOG_ERROR("Volume %s does not have a file system interface module.\n", vol->name);
                                rc = ENOSYS;
                            }
                        }
                        break;

                    default:
                        LOG_ERROR("Cannot set info on a thing of type %d.\n", type);
                        rc = EINVAL;
                        break;
                }
            }

            free(task);

        } else {
            LOG_CRITICAL("Error allocating memory for a task context.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static inline int get_plugin_extended_info(plugin_record_t         * plugin,
                                           char                    * descriptor_name,
                                           extended_info_array_t * * info) {
    int rc = 0;

    LOG_PROC_ENTRY();

    switch (GetPluginType(plugin->id)) {
        case EVMS_DEVICE_MANAGER:
        case EVMS_SEGMENT_MANAGER:
        case EVMS_REGION_MANAGER:
        case EVMS_FEATURE:
        case EVMS_ASSOCIATIVE_FEATURE:

            rc = plugin->functions.plugin->get_plugin_info(descriptor_name, info);
            break;

        case EVMS_FILESYSTEM_INTERFACE_MODULE:
            rc = plugin->functions.fsim->get_plugin_info(descriptor_name, info);
            break;

        default:
            LOG_ERROR("We don't get info for %d plug-in types.\n", GetPluginType(plugin->id) );
            rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static void free_extended_info_object_contents(void * object) {

    int i, j;
    extended_info_array_t * eia = (extended_info_array_t *) object;
    extended_info_t       * ei;

    LOG_PROC_ENTRY();

    for (i = 0; i < eia->count; i++) {
        ei = &eia->info[i];

        if (ei->name != NULL) {
            free(ei->name);
        }
        if (ei->title != NULL) {
            free(ei->title);
        }
        if (ei->desc != NULL) {
            free(ei->desc);
        }

        switch (ei->collection_type) {
            case EVMS_Collection_List:
                if (ei->collection.list != NULL) {
                    if (ei->type == EVMS_Type_String) {
                        for (j = 0; j < ei->collection.list->count; j++) {
                            if (ei->collection.list->value[j].s != NULL) {
                                free(ei->collection.list->value[j].s);
                            }
                        }
                    }
                    free(ei->collection.list);

                } else {
                    LOG_WARNING("Collection says it has a list but the list pointer is NULL.\n");
                }
                break;

            case EVMS_Collection_Range:
                if (ei->collection.range != NULL) {
                    free((void *) ei->collection.range);

                } else {
                    LOG_WARNING("Collection says it has a range but the range pointer is NULL.\n");
                }

            default:
                break;
        }

        if ((ei->group.group_number != 0) && (ei->group.group_name != NULL)) {
            free(ei->group.group_name);
        }
    }

    LOG_PROC_EXIT_VOID();
}


int evms_get_extended_info(object_handle_t thing, char * descriptor_name, extended_info_array_t * * user_info) {
    int rc = 0;
    void * object;
    object_type_t type;
    extended_info_array_t * info = NULL;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        LOG_DEBUG("Get extended info for handle %d.\n", thing);
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case PLUGIN:{
                        plugin_record_t * plugin = (plugin_record_t *) object;

                        LOG_DEBUG("Handle %d maps to plug-in %s.\n", thing, plugin->short_name);
                        rc = get_plugin_extended_info(plugin, descriptor_name, &info);
                    }
                    break;

                case DISK:
                case SEGMENT:
                case REGION:
                case EVMS_OBJECT:{
                        storage_object_t * obj = (storage_object_t *) object;

                        LOG_DEBUG("Handle %d maps to storage object %s.\n", thing, obj->name);
                        rc = obj->plugin->functions.plugin->get_info(obj, descriptor_name, &info);
                    }
                    break;


                case CONTAINER:{
                        storage_container_t * container = (storage_container_t *) object;

                        LOG_DEBUG("Handle %d maps to container %s.\n", thing, container->name);
                        rc = container->plugin->container_functions->get_container_info(container, descriptor_name, &info);
                    }
                    break;

                case VOLUME:{
                        logical_volume_t * volume = (logical_volume_t *) object;

                        LOG_DEBUG("Handle %d maps to volume %s.\n", thing, volume->name);
                        if (volume->file_system_manager != NULL) {
                            rc = volume->file_system_manager->functions.fsim->get_volume_info(volume, descriptor_name, &info);

                        } else {
                            /*
                             * The volume doesn't have an FSIM.
                             * Return an empty info array.
                             */
                            info = calloc(1, sizeof(extended_info_array_t));

                            if (info == NULL) {
                                rc = ENOMEM;
                            }
                        }
                    }
                    break;

                default:
                    LOG_DEBUG("Handle %d maps to unknown object type %d.\n", thing, type);
                    rc = EINVAL;
                    break;
            }

            if ((rc == 0) && (info != NULL)) {

                /*
                 * Copy the extended info that was returned from the plug-in
                 * into an application memory object suitable for returning
                 * to the API caller.
                 */
                if (info->count == 0) {

                    *user_info = alloc_app_struct(sizeof(info->count), NULL);
                    if (*user_info != NULL) {
                        /* Fill in the structure returned to the app. */
                        (*user_info)->count = 0;

                        /*
                         * Free the structure that was returned by the plug-in.
                         */
                        free(info);

                    } else {
                        rc = ENOMEM;
                    }

                } else {
                    /* The extended info array has at least one entry. */
                    uint size = sizeof(extended_info_array_t) + (sizeof(extended_info_t) * (info->count - 1));

                    *user_info = alloc_app_struct(size, free_extended_info_object_contents);

                    if (*user_info != NULL) {
                        /* Fill in the structure returned to the app. */
                        memcpy(*user_info, info, size);

                        /*
                         * Free the structure that was returned by the plug-in.
                         */
                        free(info);

                    } else {
                        rc = ENOMEM;
                    }
                }
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_set_info will set information, specified in options, for a thing.
 */
int evms_set_info(engine_handle_t  thing_handle,
                  option_array_t * options) {

    int rc = 0;
    object_type_t type;
    void *        thing;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(thing_handle, &thing, &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {

            switch (type) {
                case DISK:
                case SEGMENT:
                case REGION:
                case EVMS_OBJECT:
                    {
                        storage_object_t * obj = (storage_object_t *) thing;
                        char * old_object_name[sizeof(obj->name)] = {'\0'};

                        rc = engine_can_rename(obj);

                        if (rc == 0) {
                            /* Check if it's part of a compatibility volume. */
                            if ((obj->volume != NULL) &&
                                (obj->object_type != EVMS_OBJECT) &&
                                (obj->feature_header == NULL)) {

                                /*
                                 * Check if it's the top object of the
                                 * compatibility volume.
                                 */
                                uint count = 0;

                                GetListSize(obj->parent_objects, &count);

                                if (count == 0) {
                                    /*
                                     * This is the top object of a compatibility
                                     * volume.  If the object's name changes,
                                     * the volume name must change as well.
                                     * save the current volume name so that
                                     * we can check if it changed during the
                                     * call to the plug-in's set_info().
                                     */
                                    memcpy(old_object_name, obj->name, sizeof(obj->name));
                                }
                            }

                            rc = obj->plugin->functions.plugin->set_info(obj, options);

                            /*
                             * Check if we need to handle a possible name
                             * change.
                             */
                            if (old_object_name[0] != '\0') {
                                if (memcmp(old_object_name, obj->name, sizeof(obj->name)) != 0) {

                                    char * pVolName = obj->name;
                                    ADDRESS trash;

                                    /*
                                     * The object is a top level object of a
                                     * compatibility volume and its name has
                                     * changed.  Delete the old volume and
                                     * create a new one with the new name.
                                     */

                                    /* Unregister the current volume name. */
                                    engine_unregister_name(obj->volume->name);

                                    /* Remove the volume from the VolumeList. */
                                    DeleteObject(VolumeList, obj->volume);

                                    if (!(obj->volume->flags & VOLFLAG_NEW)) {
                                        /* Put it on the HardVolumeDeleteList. */
                                        rc = InsertObject(HardVolumeDeleteList,
                                                          sizeof(logical_volume_t),
                                                          obj->volume,
                                                          VOLUME_TAG,
                                                          NULL,
                                                          AppendToList,
                                                          FALSE,
                                                          &trash);

                                        if (rc == DLIST_SUCCESS) {
                                            /*
                                             * Make sure the volume is not on
                                             * the SoftDeleteList from some
                                             * prior operation.  It can't be
                                             * both hard deleted and soft
                                             * deleted.
                                             */
                                            DeleteObject(SoftVolumeDeleteList, obj->volume);

                                        } else {
                                            LOG_WARNING("Error code %d when putting volume $s on the HardVolumeDeleteList.\n", rc, obj->volume->name);
                                            rc = 0;
                                        }
                                    }

                                    /*
                                     * Build a new volume name for the object.
                                     */
                                    if (strncmp(pVolName, EVMS_DEV_NODE_PATH, strlen(EVMS_DEV_NODE_PATH)) != 0) {
                                        pVolName = malloc(strlen(pVolName) + strlen(EVMS_DEV_NODE_PATH) + 1);
                                        if (pVolName != NULL) {
                                            strcpy(pVolName, EVMS_DEV_NODE_PATH);
                                            strcat(pVolName, obj->name);
                                        } else {
                                            LOG_CRITICAL("Could not get memory for building a volume name for object %s.\n", obj->name);
                                            rc = ENOMEM;
                                        }
                                    }

                                    if (rc == 0) {

                                        /*
                                         * Make a new volume patterned after the
                                         * old one.
                                         */
                                        rc = make_volume(obj,
                                                         pVolName,
                                                         obj->volume->minor_number,
                                                         obj->volume->flags,
                                                         obj->volume->serial_number);

                                        if (rc == 0) {
                                            /* Mark the volume new and dirty. */
                                            obj->volume->flags |= (VOLFLAG_NEW | VOLFLAG_DIRTY);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    break;

                case CONTAINER:
                    {
                        storage_container_t * con = (storage_container_t *) thing;
                        rc = con->plugin->container_functions->set_container_info(con, options);
                    }
                    break;

                case VOLUME:
                    {
                        logical_volume_t * vol = (logical_volume_t *) thing;
                        if (vol->file_system_manager != NULL) {
                            rc = vol->file_system_manager->functions.fsim->set_volume_info(vol, options);

                        } else {
                            LOG_DEBUG("Volume %s does not have a file system interface module.\n", vol->name);
                            rc = ENOSYS;
                        }
                    }
                    break;

                default:
                    LOG_ERROR("Cannot set info on a thing of type %d.\n", type);
                    rc = EINVAL;
                    break;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Mark dirty the feature header hanging off of this object.
 * This function is a service function for set_feature_header_dirty() below.
 * This function has its parameters structures so that it can be called by
 * the ForEachItem() dlist processor.
 */
static int dirty_feature_header(ADDRESS object,
                                TAG     object_tag,
                                uint    object_size,
                                ADDRESS object_handle,
                                ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;

    LOG_PROC_ENTRY();

    if (obj->feature_header != NULL) {
        obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Remove this object's feature headers and add the feature header sectors to
 * the kill list.
 */
int remove_feature_headers(ADDRESS object,
                           TAG     object_tag,
                           uint    object_size,
                           ADDRESS object_handle,
                           ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;

    LOG_PROC_ENTRY();

    free(obj->feature_header);
    obj->feature_header = NULL;

    /* Wipe out both feature headers. */
    obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (FEATURE_HEADER_SECTORS * 2), FEATURE_HEADER_SECTORS * 2);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Mark this object's feature header(s) dirty.  The feature header(s) for this
 * object is/are in this object's child objects.
 * This function has its parameters structures so that it can be called by
 * the ForEachItem() dlist processor.
 */
int set_feature_header_dirty(ADDRESS object,
                             TAG     object_tag,
                             uint    object_size,
                             ADDRESS object_handle,
                             ADDRESS parameters) {
    int rc = 0;
    storage_object_t * obj = (storage_object_t *) object;

    LOG_PROC_ENTRY();

    /* Only EVMS objects have feature headers. */
    if (obj->object_type == EVMS_OBJECT) {
        rc = ForEachItem(obj->child_objects,
                         dirty_feature_header,
                         NULL,
                         TRUE);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int do_volume_remove_warnings(logical_volume_t * vol) {

    int rc = 0;
    BOOLEAN do_unmkfs = FALSE;

    LOG_PROC_ENTRY();

    /*
     * If the volume is new then it really doesn't have an FSIM associated with
     * it and it's not already a compatibility volume so there are no warnings
     * to post.
     */
    if (vol->flags & VOLFLAG_NEW) {
        LOG_PROC_EXIT_INT(rc);
        return rc;
    }

    /*
     * The volume has an FSIM, ask the user if unmkfs should be run on the
     * volume.
     */
    if (vol->file_system_manager != NULL) {

        /* Check if the FSIM can unmkfs. */
        rc = vol->file_system_manager->functions.fsim->can_unmkfs(vol);

        if (rc == 0) {
            /*
             * Only need to ask the user to run unmkfs if the current FSIM is
             * the original FSIM, it isn't already marked for unmkfs, and the
             * volume is not read-only.
             */
            if ((vol->file_system_manager == vol->original_fsim) &&
                !(vol->flags & (VOLFLAG_UNMKFS | VOLFLAG_READ_ONLY))) {

                char * choices[] = {"Run unmkfs", "Do not run unmkfs", "Cancel remove", NULL};
                int answer = 0;     /* Default is "Run unmkfs". */

                engine_user_message(&answer, choices,
                                    "Volume \"%s\" has the %s file system on it.  "
                                    "Do you want to run unmkfs to remove the file system before the volume is removed?\n",
                                    vol->name, vol->file_system_manager->short_name);
                switch (answer) {
                    case 0:
                        /*
                         * Turn on the VOLFLAG_UNMKFS flag so that unmkfs will
                         * be run at commit time.  If the volume's VOLFLAG_MKFS
                         * is set we don't care.  The volume will be deleted and
                         * so won't have the chance to run mkfs after redicovery
                         * when it is gone.
                         */
                        vol->flags |= VOLFLAG_UNMKFS;

                        /* Mark that this volume has no FSIM. */
                        vol->file_system_manager = NULL;

                        do_unmkfs = TRUE;

                        break;

                    case 1:
                        /* Nothing to do. */
                        break;

                    case 2:
                        rc = EINTR;
                    default:
                        break;

                }
            }

        } else {
            /*
             * The FSIM does not support unmkfs on this volume. Ask the user
             * if we should continue with the remove or abort.
             */
            char * choices[] = {"Continue", "Abort", NULL};
            int answer = 0;     /* Default is "Continue". */

            engine_user_message(&answer, choices,
                                "Volume \"%s\" has the %s file system on it.  "
                                "The %s File System Interface Module plug-in cannot remove the file sytem from the volume.  "
                                "The file system will be left on the volume.  "
                                "Do you want continue with the remove?\n",
                                vol->name, vol->file_system_manager->short_name, vol->file_system_manager->short_name);

            if (answer != 0) {
                /* An answer other than "Continue" is considered an abort. */
                rc = EINTR;
            }
        }

    } else {
        /*
         * The volume does not have an FSIM assigned to it.  If it did not
         * originally have and FSIM assigned to it, it may have a file system on
         * it for which we have no FSIM.  If the volume is not marked read-only
         * ask the user if we should run a crude "unmkfs" that wipes out the
         * first 1KB of the disk.
         */
        if ((vol->original_fsim == NULL) &&
            !(vol->flags & VOLFLAG_READ_ONLY)) {
            char * choices[] = {"Write zeros", "Do not write zeros", "Cancel remove", NULL};
            int answer = 0;     /* Default is "Write zeroes". */

            sector_count_t one_meg_in_sectors = (1024 * 1024) >> EVMS_VSECTOR_SIZE_SHIFT;

            engine_user_message(&answer, choices,
                                "Volume \"%s\" does not have a File System Interface Module (FSIM) assigned to it.  "
                                "The volume may have a file system on it, but none of the installed FSIM plug-ins recognizes it.  "
                                "Do you want to write zeros to the %sdisable any file system that may be on the volume?\n",
                                vol->name,
                                (vol->object->size > one_meg_in_sectors) ? "first 1MB of the volume to potentially " : "volume to ");
            switch (answer) {
                case 0:
                    /* Add the kill sectors. */
                    rc = vol->object->plugin->functions.plugin->add_sectors_to_kill_list(vol->object, 0, min(vol->object->size, one_meg_in_sectors));

                    do_unmkfs = TRUE;
                    break;

                case 1:
                    /* Nothing to do. */
                    break;

                case 2:
                    rc = EINTR;
                default:
                    break;

            }
        }
    }

    if (rc == 0) {
        /*
         * If this is a compatibility volume and we did not do an unmkfs (the
     * real kind or the poor man's kind of writing 1MB), warn the user that
     * the last few sectors will have meta data written to them.
         */
        if ((vol->flags & VOLFLAG_COMPATIBILITY) && !do_unmkfs) {
            char * choices[] = {"Continue", "Abort", NULL};
            int answer = 0;     /* Default is "Continue". */

            engine_user_message(&answer, choices,
                                "Volume \"%s\" is not an EVMS volume.  "
                                "Removing a non-EVMS volume requires writing %d bytes of meta data at the end of the volume. "
                                "The meta data will overwrite any data that may be at the end of the volume.  "
                                "Do you want continue with the remove?\n",
                                vol->name, (FEATURE_HEADER_SECTORS * 2) << EVMS_VSECTOR_SIZE_SHIFT);

            if (answer != 0) {
                /* An answer other than "Continue" is considered an abort. */
                rc = EINTR;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int remove_volume(logical_volume_t * volume) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /* Check if the volume is mounted. */
    if (!is_volume_mounted(volume)) {
        storage_object_t * obj = volume->object;

        /*
         * Ask the object below the volume if it can handle having the volume
         * removed.
         */
        if (obj->flags & SOFLAG_MUST_BE_VOLUME) {
            /*
             * It's an object that must be a volume.  The object must go down
             * with the volume.  The volume can be removed if the object can be
             * deleted.
             */
            rc = obj->plugin->functions.plugin->can_delete(obj);
        } else {
            rc = obj->plugin->functions.plugin->can_set_volume(obj, FALSE);
        }

        if (rc == 0) {
            /*
             * Display any warning message(s) for the removal of this volume.
             * do_volume_remove_warning() can return EINTR if the user
             * selects to cancel the removal.
             */
            rc = do_volume_remove_warnings(volume);

            if (rc == 0) {
                /*
                 * If the object has a feature header attached,
                 * it is one for making the object into an EVMS
                 * volume.  Remove the feature header and kill
                 * the sectors so that it won't be rediscovered.
                 */
                if (obj->feature_header != NULL) {
                    rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (FEATURE_HEADER_SECTORS * 2), FEATURE_HEADER_SECTORS * 2);

                    if (rc == 0) {
                        free(obj->feature_header);
                        obj->feature_header = NULL;

                    } else {
                        LOG_WARNING("When trying to wipe out volume feature header on volume %s, add_sectors_to_kill_list() returned error code %d.\n", volume->name, rc);
                    }
                }

                if (rc == 0) {

                    /*
                     * If the volume is not new then put it on the list of
                     * volumes to be deleted by the kernel runtime.
                     */
                    if (!(volume->flags & VOLFLAG_NEW)) {
                        ADDRESS handle;

                        LOG_DEBUG("Volume exists.  Put it on the delete list.\n");
                        rc = InsertObject(HardVolumeDeleteList,
                                          sizeof(logical_volume_t),
                                          volume,
                                          VOLUME_TAG,
                                          NULL,
                                          AppendToList,
                                          FALSE,
                                          &handle);

                        if (rc == DLIST_SUCCESS) {
                            /*
                             * Make sure the volume is not on the SoftDeleteList
                             * from some prior operation.  It can't be both
                             * hard deleted and soft deleted.
                             */
                            DeleteObject(SoftVolumeDeleteList, volume);
                        }

                        /*
                         * If the volume's object is associated with another object,
                         * put the other object's volume into this volume's
                         * associated_volume field so that it will be soft deleted
                         * and rediscovered at commit time.
                         */
                        if (obj->associated_object != NULL) {
                            volume->associated_volume = obj->associated_object->volume;
                        }
                    }

                    if (rc == 0) {
                        /* Remove the volume from the master VolumeList. */
                        rc = DeleteObject(VolumeList, volume);

                        if (rc == DLIST_SUCCESS) {

                            /* Destroy the volume handle. */
                            destroy_handle(volume->app_handle);

                            /* Unregister the volume's name. */
                            engine_unregister_name(volume->name);

                            /*
                             * Clear out the volume pointer on this object and on all
                             * the child objects in the tree.  Notify all the objects
                             * that the volume has been removed.
                             */
                            set_volume_in_object(obj,
                                                 get_tag_for_object(obj),
                                                 sizeof(storage_object_t),
                                                 NULL,
                                                 NULL);    /* volume pointer */

                            /*
                             * If this is a volume we created but didn't commit yet
                             * (indicated by the VOLFLAG_NEW flag) then just throw
                             * away the volume structure.
                             */
                            if (volume->flags & VOLFLAG_NEW) {
                                LOG_DEBUG("Volume is new, so just toss it out.\n");
                                free(volume);
                            }

                        } else {
                            LOG_WARNING("Error code %d when deleting volume %s from the VolumeList.\n", volume->name);
                        }

                    } else {
                        LOG_WARNING("Error code %d when inserting volume %s into the HardVolumeDeleteList.\n", volume->name);
                    }
                }
            }

        } else {
            LOG_ERROR("Object %s does not allow volume %s to be removed.  Reason code is %d.\n", obj->name, volume->name, rc);
        }

    } else {
        LOG_ERROR("Volume %s cannot be destroyed because it is mounted.\n", volume->name);
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * sync_compatibility_name will make sure that a compatibility volume has the
 * correct name, i.e., the EVMS_DEV_NODE_PATH prefix followed by the object
 * name.  In the case where a segment is deleted or added to the disk, the
 * compatibility names of the objects may shift and hence the name of the
 * volume may change.
 */
int sync_compatibility_volume_names(void) {
    int rc = 0;
    logical_volume_t * vol;

    LOG_PROC_ENTRY();

    GoToStartOfList(VolumeList);

    rc = GetObject(VolumeList,
                   sizeof(logical_volume_t),
                   VOLUME_TAG,
                   NULL,
                   FALSE,
                   (ADDRESS *) &vol);

    while ((rc == 0) && vol != NULL) {
        /* Only update compatibility volumes. */
        if (vol->flags & VOLFLAG_COMPATIBILITY) {
            storage_object_t * comp_obj = vol->object;

            while ((rc == 0) && (comp_obj != NULL) &&
                   (GetPluginType(comp_obj->plugin->id) == EVMS_ASSOCIATIVE_FEATURE)) {
                int size;
                TAG tag;

                rc = BlindGetObject(comp_obj->child_objects,
                                    &size,
                                    &tag,
                                    NULL,
                                    FALSE,
                                    (ADDRESS *) &comp_obj);
            }

            if (rc == 0) {
                char * pch = vol->name + strlen(EVMS_DEV_NODE_PATH);

                if ((strcmp(pch, comp_obj->name) != 0)) {
                    logical_volume_t * rename_vol = NULL;
                    BOOLEAN found = FALSE;

                    engine_user_message(NULL, NULL,
                                        "Volume name %s has shifted to %s%s.\n",
                                        vol->name, EVMS_DEV_NODE_PATH, comp_obj->name);

                    if (!(vol->flags & VOLFLAG_NEW)) {
                        /*
                         * Check if this volume is already in the VolumeRemoveList
                         * by comparing the minor numbers.
                         */
                        GoToStartOfList(VolumeRemoveList);

                        rc = GetObject(VolumeRemoveList,
                                       sizeof(logical_volume_t),
                                       VOLUME_TAG,
                                       NULL,
                                       FALSE,
                                       (ADDRESS *) &rename_vol);

                        while (!found && (rc == DLIST_SUCCESS) && (rename_vol != NULL)) {

                            if (vol->minor_number == rename_vol->minor_number) {
                                found = TRUE;

                            } else {
                                rc = GetNextObject(VolumeRemoveList,
                                                   sizeof(logical_volume_t),
                                                   VOLUME_TAG,
                                                   (ADDRESS *) &rename_vol);
                            }
                        }

                        /* Clear out acceptable return codes. */
                        if ((rc == DLIST_EMPTY) ||
                            (rc = DLIST_END_OF_LIST)) {
                            rc = DLIST_SUCCESS;
                        }

                        if (found) {
                            /*
                             * Check if this volume has returned back to its
                             * original name.  If so, remove it from the
                             * VolumeRemoveList.
                             */
                            char * pch = rename_vol->name + strlen(EVMS_DEV_NODE_PATH);

                            if ((strcmp(pch, comp_obj->name) == 0)) {
                                /* Restore the orignal volume name. */
                                memcpy(vol->name, rename_vol->name, sizeof(vol->name));

                                /* Remove it from the list and free the memory. */
                                DeleteObject(VolumeRemoveList, rename_vol);
                                free(rename_vol);
                            }

                        } else {
                            if (rc == DLIST_SUCCESS) {
                                ADDRESS trash;

                                /*
                                 * The volume was renamed and it is not in the
                                 * VolumeRemoveList.  Use the current volume
                                 * structure for the entry in the
                                 * VolumeRemoveList.  Create a new volume
                                 * structure for the volume with the new name.
                                 * Put the volume on the front of the list so
                                 * that it doesn't get in the way of our
                                 * processing of the list.  We'll sort the
                                 * VolumeList when we're finished.
                                 */
                                rc = InsertObject(VolumeRemoveList,
                                                  sizeof(logical_volume_t),
                                                  vol,
                                                  VOLUME_TAG,
                                                  NULL,
                                                  AppendToList,
                                                  FALSE,&trash);

                                if (rc == DLIST_SUCCESS) {
                                    logical_volume_t * new_volume = (logical_volume_t *) malloc(sizeof(logical_volume_t));

                                    if (new_volume != NULL) {

                                        /* Destroy the old volume's app_handle. */
                                        destroy_handle(vol->app_handle);
                                        vol->app_handle = 0;

                                        /*
                                         * Copy the old volume contents to the
                                         * new volume.
                                         */
                                        *new_volume = *vol;

                                        /* Set the name of the new volume. */
                                        memset(new_volume->name, 0, sizeof(new_volume->name));
                                        strcpy(new_volume->name, EVMS_DEV_NODE_PATH);
                                        strcat(new_volume->name, comp_obj->name);

                                        /* Mark the new volume new. */
                                        new_volume->flags |= VOLFLAG_NEW;

                                        engine_unregister_name(vol->name);
                                        rc = engine_register_name(new_volume->name);

                                        if (rc == 0) {
                                            uint size = sizeof(logical_volume_t);
                                            TAG  tag = VOLUME_TAG;
                                            ADDRESS old_object;

                                            rc = ReplaceObject(VolumeList,
                                                               &size,
                                                               new_volume,
                                                               &tag,
                                                               NULL,
                                                               FALSE,
                                                               &old_object);

                                            /*
                                             * Tell the child objects they are part
                                             * of a new volume.
                                             */
                                            if (rc == DLIST_SUCCESS) {
                                                set_volume_in_object(new_volume->object,
                                                                     get_tag_for_object(new_volume->object),
                                                                     sizeof(storage_object_t),
                                                                     NULL,
                                                                     new_volume);
                                            }
                                        }

                                    } else {
                                        LOG_CRITICAL("Error allocating memory for a new volume structure.\n");
                                        rc = ENOMEM;
                                    }

                                } else {
                                    LOG_WARNING("Error code %d when inserting a volume %s into the VolumeRemoveList.\n", rc, vol->name);
                                }

                            } else {
                                LOG_WARNING("Error code %d when scanning the VolumeRemoveList.\n", rc);
                            }
                        }
                    }
                }
            }
        }

        if (rc == 0) {
            rc = GetNextObject(VolumeList,
                               sizeof(logical_volume_t),
                               VOLUME_TAG,
                               (ADDRESS *) &vol);
        }
    }

    if ((rc == DLIST_EMPTY) ||
        (rc == DLIST_END_OF_LIST)) {
        rc = DLIST_SUCCESS;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_remove(object_handle_t thing) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case VOLUME:
                    {
                        logical_volume_t * volume = (logical_volume_t *) object;
                        storage_object_t * obj = volume->object;

                        LOG_DEBUG("Request to remove volume %s.\n", volume->name);

                        rc = remove_volume(volume);

                        if (rc == 0) {
                            if (obj->flags & SOFLAG_MUST_BE_VOLUME) {
                                /*
                                 * The object must be torn down with the volume.
                                 */
                                dlist_t child_list = CreateList();

                                if (child_list != NULL) {
                                    rc = obj->plugin->functions.plugin->delete(obj, child_list);

                                } else {
                                    LOG_CRITICAL("Error getting memory for a child objects list.\n");
                                    rc = ENOMEM;
                                }
                            }
                        }
                    }
                    break;

                case EVMS_OBJECT:
                case REGION:
                case SEGMENT:
                case DISK:
                    {
                        storage_object_t * obj = (storage_object_t *) object;

                        LOG_DEBUG("Request to remove object %s.\n", obj->name);

                        /*
                         * Make sure the object is a top level object.  We can't
                         * remove objects in the middle of a feature stack or
                         * under a volume has no parent.
                         */
                        if (is_top_object(obj)) {
                            dlist_t child_list = CreateList();

                            if (child_list != NULL) {
                                rc = obj->plugin->functions.plugin->delete(obj, child_list);

                                if (rc == 0) {

                                    /*
                                     * Now that the child objects are top
                                     * objects, they will not contain feature
                                     * headers.  Remove the feature headers from
                                     * the child objects.
                                     */
                                    ForEachItem(child_list,
                                                remove_feature_headers,
                                                NULL,
                                                TRUE);

                                    /*
                                     * Mark the children's child objects'
                                     * feature headers dirty so that the volume
                                     * complete flag will be set correctly on
                                     * commit.
                                     */
                                    ForEachItem(child_list,
                                                set_feature_header_dirty,
                                                NULL,
                                                TRUE);

                                    /*
                                     * Make sure the compatibility volume names
                                     * are in sync.
                                     */
                                    sync_compatibility_volume_names();
                                }

                                DestroyList(&child_list, FALSE);

                            } else {
                                LOG_CRITICAL("Error getting memory for a child objects list.\n");
                                rc = ENOMEM;
                            }

                        } else {
                            LOG_ERROR("Object %s is not a top level object.\n", obj->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                default:
                    LOG_ERROR("A thing with a type of %lX cannot be removed.\n", type);
                    rc = EINVAL;
                    break;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Delete an object and all of its children.
 * This function is a service function for evms_destroy() below.
 * This function has its parameters structured so that it can be called by
 * the DLIST ForEachItem() processor.  In fact, this function uses
 * ForEachItem() to call itself recursively to delete its children.
 */
static int destroy_object(ADDRESS Object,
                          TAG     ObjectTag,
                          uint    ObjectSize,
                          ADDRESS ObjectHandle,
                          ADDRESS Parameters) {
    int rc = 0;

    storage_object_t * obj = (storage_object_t *) Object;

    dlist_t child_list;

    LOG_PROC_ENTRY();

    /* Disks are not deleted on a destroy. */

    if (obj->object_type != DISK) {
        child_list = CreateList();

        if (child_list != NULL) {
            /*
             * Save the producing container so we can check it after the object
             * is deleted.
             */
            storage_container_t * producing_container = obj->producing_container;

            rc = obj->plugin->functions.plugin->delete(obj, child_list);

            if (rc == 0) {
                /*
                 * Now that the child objects are top objects, they will not contain
                 * feature headers.  Remove the feature headers from the child
                 * objects.
                 */
                ForEachItem(child_list,
                            remove_feature_headers,
                            NULL,
                            TRUE);

                /*
                 * If the object was not produced by a container, do a recursive
                 * deletion of its children.
                 */
                if (producing_container == NULL) {
                    rc = ForEachItem(child_list,
                                     destroy_object,
                                     NULL,
                                     TRUE);
                }
            }

        } else {
            LOG_CRITICAL("Error allocating memory for a child object list.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int destroy_volume(logical_volume_t * volume) {

    int rc = 0;
    storage_object_t * obj = volume->object;

    LOG_PROC_ENTRY();

    /*
     * Ask the object below the volume if it can handle
     * being deleted.
     */
    rc = can_destroy_object(obj,
                            get_tag_for_object(obj),
                            sizeof(storage_object_t),
                            NULL,
                            NULL);

    if (rc == 0) {

        rc = remove_volume(volume);

        if (rc == 0) {
            rc = destroy_object(obj,
                                get_tag_for_object(obj),
                                sizeof(storage_object_t),
                                NULL,
                                NULL);
        }

    } else {
        LOG_ERROR("Object %s does not allow volume %s to be removed.  Reason code is %d.\n", obj->name, volume->name, rc);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * evms_destroy() deletes an object and all the child objects in the tree
 * beneath the object.
 */
int evms_destroy(object_handle_t thing) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            switch (type) {
                case VOLUME:
                    {
                        logical_volume_t * volume = (logical_volume_t *) object;

                        LOG_DEBUG("Request to destroy volume %s.\n", volume->name);

                        rc = destroy_volume(volume);

                        if (rc == 0) {
                            /*
                             * Make sure the compatibility volume names
                             * are in sync.
                             */
                            sync_compatibility_volume_names();
                        }
                    }
                    break;

                case EVMS_OBJECT:
                case REGION:
                case SEGMENT:
                    {
                        storage_object_t * obj = (storage_object_t *) object;

                        LOG_DEBUG("Request to destroy storage object %s.\n", obj->name);

                        /*
                         * Make sure the object is a top level object.  We can't
                         * destroy objects in the middle of a feature stack.  A
                         * top level object has no parent.
                         */
                        if (is_top_object(obj)) {

                            rc = destroy_object(obj,
                                                get_tag_for_object(obj),
                                                sizeof(storage_object_t),
                                                NULL,
                                                NULL);
                            if (rc == 0) {
                                /*
                                 * Make sure the compatibility volume names
                                 * are in sync.
                                 */
                                sync_compatibility_volume_names();
                            }

                        } else {
                            /* The object is not a top level object. */

                            LOG_ERROR("Object %s is not a top level object.\n", obj->name);
                            rc = EINVAL;
                        }
                    }
                    break;

                case CONTAINER:
                    {
                        storage_container_t * container = (storage_container_t *) object;
                        dlist_t objects_consumed = CreateList();

                        LOG_DEBUG("Request to destroy container %s.\n", container->name);

                        if (objects_consumed != NULL) {

                            rc = container->plugin->container_functions->delete_container(container, objects_consumed);
                            LOG_DEBUG("Return code from call to %s plug-in's delete_container() is %d.\n", container->plugin->short_name, rc);

                            DestroyList(&objects_consumed, FALSE);

                        } else {
                            LOG_CRITICAL("Error allocating memory for a dlist of consumed objects.\n");
                        }
                    }
                    break;

                case DISK:
                    {
                        storage_object_t * obj = object;

                        LOG_ERROR("Disk %s cannot be destroyed because disks cannot be destroyed.  Disks must be removed.\n", obj->name);
                        rc = EINVAL;
                    }
                    break;

                default:
                    LOG_ERROR("A thing with a type of %lX cannot be destroyed.\n", type);
                    rc = EINVAL;
                    break;
            }
        }

        if (rc == 0) {
            changes_pending = TRUE;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_move(object_handle_t source, object_handle_t target) {
    int rc = ENOSYS;

//  rc = check_engine_write_access();
//
//  if (rc == 0) {
//
//      if (rc == 0) {
//          changes_pending = TRUE;
//      }
//  }

    return rc;
}


int make_dlist(handle_array_t * ha, dlist_t list) {
    int rc = 0;
    int i;
    void * object;
    object_type_t type;
    TAG tag = 0;

    LOG_PROC_ENTRY();

    /* If a null was array passed in, just return the empty dlist_t. */
    if (ha != NULL) {
        for (i = 0; (rc == 0) && (i < ha->count); i++) {
            rc = translate_handle(ha->handle[i],
                                  &object,
                                  &type);
            if (rc == HANDLE_MANAGER_NO_ERROR) {
                u_int size = 0;

                switch (type) {
                    case DISK:
                        tag = DISK_TAG;
                        size = sizeof(storage_object_t);
                        break;
                    case SEGMENT:
                        tag = SEGMENT_TAG;
                        size = sizeof(storage_object_t);
                        break;
                    case REGION:
                        tag = REGION_TAG;
                        size = sizeof(storage_object_t);
                        break;
                    case EVMS_OBJECT:
                        tag = EVMS_OBJECT_TAG;
                        size = sizeof(storage_object_t);
                        break;
                    case CONTAINER:
                        tag = CONTAINER_TAG;
                        size = sizeof(storage_container_t);
                        break;
                    case VOLUME:
                        tag = VOLUME_TAG;
                        size = sizeof(logical_volume_t);
                        break;
                    default:
                        rc = EINVAL;
                }

                if (rc == 0) {
                    ADDRESS handle;

                    rc = InsertObject(list,
                                      size,
                                      object,
                                      tag,
                                      NULL,
                                      AppendToList,
                                      FALSE,
                                      &handle);
                }
            }
        }
    }

    if (rc != 0) {
        DeleteAllItems(list, FALSE);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static void free_info_object_contents(void * object) {

    handle_object_info_t * info = (handle_object_info_t *) object;

    LOG_PROC_ENTRY();

    switch (info->type) {
        case PLUGIN:
            break;

        case DISK:
        case SEGMENT:
        case REGION:
        case EVMS_OBJECT:
            if (info->info.object.parent_objects != NULL) {
                free(info->info.object.parent_objects);
            }

            if (info->info.object.child_objects != NULL) {
                free(info->info.object.child_objects);
            }
            break;

        case CONTAINER:
            if (info->info.container.objects_consumed != NULL) {
                free(info->info.container.objects_consumed);
            }
            if (info->info.container.objects_produced != NULL) {
                free(info->info.container.objects_produced);
            }
            break;

        case VOLUME:
            if (info->info.volume.mount_point != NULL) {
                free(info->info.volume.mount_point);
            }
            break;

        default:
            break;
    }

    LOG_PROC_EXIT_VOID();
}


int evms_get_handle_object_type(object_handle_t handle,
                                object_type_t * type) {
    int rc = 0;
    void * object;

    LOG_PROC_ENTRY();

    *type = 0;

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              type);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 * Make an array of handles (handle_array_t) for the objects in a dlist_t.
 */
int make_handle_array(dlist_t list, handle_array_t * * ha) {
    int rc = 0;
    uint count;
    uint size;

    LOG_PROC_ENTRY();

    rc = GetListSize(list, &count);

    if (rc == 0) {
        LOG_DEBUG("Number of objects in the list:  %d\n", count);
        if (count > 1) {
            size = sizeof(handle_array_t) + ((count -1) * sizeof(object_handle_t));
        } else {
            size = sizeof(handle_array_t);
        }

        *ha = calloc(1, size);
        if (*ha != NULL) {
            rc = ForEachItem(list,
                             make_handle_entry,
                             *ha,
                             TRUE);
        } else {
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_get_info(object_handle_t thing, handle_object_info_t * * user_info) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    *user_info = NULL;

    rc = check_engine_read_access();

    if (rc == 0) {
        handle_object_info_t * info = NULL;

        info = alloc_app_struct(sizeof(handle_object_info_t), free_info_object_contents);

        if (info != NULL) {
            LOG_DEBUG("Get info for handle %d.\n", thing);
            rc = translate_handle(thing,
                                  &object,
                                  &type);

            if (rc == HANDLE_MANAGER_NO_ERROR) {
                switch (type) {
                    case PLUGIN:
                        {
                            plugin_record_t * plugin = (plugin_record_t *) object;

                            LOG_DEBUG("Handle %d maps to plug-in %s.\n", thing, plugin->short_name);
                            info->type = PLUGIN;

                            rc = ensure_app_handle(plugin,
                                                   PLUGIN,
                                                   &plugin->app_handle);
                            if (rc == 0) {
                                info->info.plugin.handle                          = plugin->app_handle;
                                info->info.plugin.id                              = plugin->id;
                                info->info.plugin.version.major                   = plugin->version.major;
                                info->info.plugin.version.minor                   = plugin->version.minor;
                                info->info.plugin.version.patchlevel              = plugin->version.patchlevel;
                                info->info.plugin.required_api_version.major      = plugin->required_api_version.major;
                                info->info.plugin.required_api_version.minor      = plugin->required_api_version.minor;
                                info->info.plugin.required_api_version.patchlevel = plugin->required_api_version.patchlevel;
                                info->info.plugin.short_name                      = plugin->short_name;
                                info->info.plugin.long_name                       = plugin->long_name;
                                info->info.plugin.oem_name                        = plugin->oem_name;
                                info->info.plugin.supports_containers             = (plugin->container_functions != NULL);
                            }
                        }
                        break;

                    case DISK:
                    case SEGMENT:
                    case REGION:
                    case EVMS_OBJECT:
                        {
                            storage_object_t * obj = (storage_object_t *) object;

                            LOG_DEBUG("Handle %d maps to storage object %s.\n", thing, obj->name);
                            info->type = obj->object_type;

                            rc = make_handle_array(obj->parent_objects, &info->info.object.parent_objects);
                            if (rc == 0) {

                                rc = make_handle_array(obj->child_objects, &info->info.object.child_objects);
                                if (rc == 0) {

                                    rc = ensure_app_handle(obj,
                                                           obj->object_type,
                                                           &obj->app_handle);
                                    if (rc == 0) {
                                        info->info.object.handle = obj->app_handle;
                                        info->info.object.object_type = obj->object_type;
                                        info->info.object.data_type = obj->data_type;
                                    }

                                    if (rc == 0) {
                                        if (obj->plugin != NULL) {
                                            rc = ensure_app_handle(obj->plugin,
                                                                   PLUGIN,
                                                                   &obj->plugin->app_handle);
                                            if (rc == 0) {
                                                info->info.object.plugin = obj->plugin->app_handle;
                                            }
                                        } else {
                                            info->info.object.plugin = 0;
                                        }
                                    }

                                    if (rc == 0) {
                                        if (obj->producing_container != NULL) {

                                            rc = ensure_app_handle(obj->producing_container,
                                                                   CONTAINER,
                                                                   &obj->producing_container->app_handle);
                                            if (rc == 0) {
                                                info->info.object.producing_container = obj->producing_container->app_handle;
                                            }
                                        } else {
                                            info->info.object.producing_container = 0;
                                        }
                                    }

                                    if (rc == 0) {
                                        if (obj->consuming_container != NULL) {

                                            rc = ensure_app_handle(obj->consuming_container,
                                                                   CONTAINER,
                                                                   &obj->consuming_container->app_handle);
                                            if (rc == 0) {
                                                info->info.object.consuming_container = obj->consuming_container->app_handle;
                                            }
                                        } else {
                                            info->info.object.consuming_container = 0;
                                        }
                                    }

                                    if (rc == 0) {
                                        if (obj->volume != NULL) {

                                            rc = ensure_app_handle(obj->volume,
                                                                   VOLUME,
                                                                   &obj->volume->app_handle);
                                            if (rc == 0) {
                                                info->info.object.volume = obj->volume->app_handle;
                                            }
                                        } else {
                                            info->info.object.volume = 0;
                                        }
                                    }

                                    if (rc == 0) {
                                        info->info.object.flags = obj->flags;
                                        memcpy(info->info.object.name, obj->name, sizeof(obj->name));
                                        info->info.object.start = obj->start;
                                        info->info.object.size = obj->size;
                                        info->info.object.geometry = obj->geometry;
                                    }
                                }
                            }
                        }
                        break;

                    case CONTAINER:
                        {
                            storage_container_t * con = (storage_container_t *) object;

                            LOG_DEBUG("Handle %d maps to container %s.\n", thing, con->name);
                            info->type = CONTAINER;

                            rc = make_handle_array(con->objects_consumed, &info->info.container.objects_consumed);
                            if (rc == 0) {
                                rc = make_handle_array(con->objects_produced, &info->info.container.objects_produced);
                                if (rc == 0) {

                                    rc = ensure_app_handle(con,
                                                           CONTAINER,
                                                           &con->app_handle);
                                    if (rc == 0) {
                                        info->info.container.handle = con->app_handle;
                                    }

                                    if (rc == 0) {
                                        if (con->plugin != NULL) {
                                            rc = ensure_app_handle(con->plugin,
                                                                   PLUGIN,
                                                                   &con->plugin->app_handle);
                                            if (rc == 0) {
                                                info->info.container.plugin = con->plugin->app_handle;
                                            }
                                        } else {
                                            info->info.container.plugin = 0;
                                        }
                                    }

                                    if (rc == 0) {
                                    }
                                    info->info.container.flags = con->flags;
                                    memcpy(info->info.container.name, con->name, sizeof(con->name));
                                    info->info.container.size = con->size;
                                }
                            }
                        }
                        break;

                    case VOLUME:
                        {
                            logical_volume_t * vol = (logical_volume_t *) object;

                            LOG_DEBUG("Handle %d maps to volume %s.\n", thing, vol->name);
                            info->type = VOLUME;

                            rc = ensure_app_handle(vol,
                                                   VOLUME,
                                                   &vol->app_handle);
                            if (rc == 0) {
                                info->info.volume.handle = vol->app_handle;
                            }

                            if (rc == 0) {
                                if (vol->file_system_manager != NULL) {
                                    rc = ensure_app_handle(vol->file_system_manager,
                                                           PLUGIN,
                                                           &vol->file_system_manager->app_handle);
                                    if (rc == 0) {
                                        info->info.volume.file_system_manager = vol->file_system_manager->app_handle;
                                    }
                                } else {
                                    info->info.volume.file_system_manager = 0;
                                }
                            }

                            if (rc == 0) {
                                if (vol->object != NULL) {
                                    rc = ensure_app_handle(vol->object,
                                                           vol->object->object_type,
                                                           &vol->object->app_handle);
                                    if (rc == 0) {
                                        info->info.volume.object = vol->object->app_handle;
                                    }
                                } else {
                                    info->info.volume.object = 0;
                                }
                            }

                            if (rc == 0) {
                                if (vol->associated_volume != NULL) {
                                    rc = ensure_app_handle(vol->associated_volume,
                                                           VOLUME,
                                                           &vol->associated_volume->app_handle);
                                    if (rc == 0) {
                                        info->info.volume.associated_volume = vol->associated_volume->app_handle;
                                    }
                                } else {
                                    info->info.volume.associated_volume = 0;
                                }
                            }

                            if (rc == 0) {
                                if (vol->mount_point != NULL) {
                                    info->info.volume.mount_point = strdup(vol->mount_point);
                                } else {
                                    info->info.volume.mount_point = NULL;
                                }
                                info->info.volume.fs_size       = vol->fs_size;
                                info->info.volume.min_fs_size   = vol->min_fs_size;
                                info->info.volume.max_fs_size   = vol->max_fs_size;
                                info->info.volume.vol_size      = vol->vol_size;
                                info->info.volume.max_vol_size  = vol->max_vol_size;
                                info->info.volume.minor_number  = vol->minor_number;
                                info->info.volume.serial_number = vol->serial_number;
                                info->info.volume.flags         = vol->flags;
                                memcpy(info->info.volume.name, vol->name, sizeof(vol->name));
                            }
                        }
                        break;

                    default:
                        LOG_DEBUG("Handle %d maps to unknown object type %d.\n", thing, type);
                        rc = EINVAL;
                        break;
                }
            }

            if ((rc != 0) && (info != NULL)) {
                evms_free(info);
                info = NULL;
            }


        } else {
            rc = ENOMEM;
        }

        *user_info = info;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int add_plugin_to_list(ADDRESS Object,
                              TAG Tag,
                              uint size,
                              ADDRESS ObjectHandle,
                              ADDRESS Parameters) {

    int rc = DLIST_SUCCESS;
    storage_object_t * object = (storage_object_t *) Object;
    dlist_t plugin_list = (dlist_t) Parameters;
    ADDRESS handle;

    LOG_PROC_ENTRY();

    /* Only get the plug-in (feature) if this is an EVMS storage object. */
    if (object->object_type == EVMS_OBJECT) {

        /* Make sure the plug-in isn't already on the list. */
        rc = GoToStartOfList(plugin_list);

        if (rc == DLIST_SUCCESS) {
            plugin_record_t * pPlugRec;

            rc = GetObject(plugin_list,
                           sizeof(plugin_record_t),
                           PLUGIN_TAG,
                           NULL,
                           FALSE,
                           (ADDRESS *) &pPlugRec);

            while ((rc == DLIST_SUCCESS) && (pPlugRec != NULL) && (pPlugRec != object->plugin)) {
                rc = GetNextObject(plugin_list,
                                   sizeof(plugin_record_t),
                                   PLUGIN_TAG,
                                   (ADDRESS *) &pPlugRec);
            }

            if ((rc == DLIST_EMPTY) ||
                (rc == DLIST_END_OF_LIST)) {
                /*
                 * This plug-in record is not in the list.  Insert it into the
                 * list.
                 */
                rc = InsertObject(plugin_list,
                                  sizeof(plugin_record_t),
                                  object->plugin,
                                  PLUGIN_TAG,
                                  NULL,
                                  AppendToList,
                                  FALSE,
                                  &handle);
            }
        }

        if (rc == DLIST_SUCCESS) {
            /* Get the feature(s) for the child object(s). */
            rc = ForEachItem(object->child_objects,
                             add_plugin_to_list,
                             plugin_list,
                             TRUE);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_get_feature_list(object_handle_t    thing,
                          handle_array_t * * plugins) {
    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(thing,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if ((type == VOLUME) ||
                (type == EVMS_OBJECT)) {
                dlist_t feature_list = CreateList();

                if (feature_list != NULL) {
                    storage_object_t * storage_object;

                    if (type == VOLUME) {
                        logical_volume_t * volume = (logical_volume_t *) object;

                        storage_object = volume->object;
                    } else {
                        storage_object = (storage_object_t *) object;
                    }

                    rc = add_plugin_to_list(storage_object,
                                            EVMS_OBJECT_TAG,
                                            sizeof(storage_object_t),
                                            NULL,
                                            feature_list);

                    if (rc == DLIST_SUCCESS) {
                        rc = make_user_handle_array(feature_list, plugins);
                    }

                    DestroyList(&feature_list, FALSE);


                } else {
                    rc = ENOMEM;
                }


            } else {
                /* The thing is not a volume or object. */
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int find_object_handle_by_name(dlist_t list, TAG tag, char * name, object_handle_t * object_handle) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = GoToStartOfList(list);

    if (rc == DLIST_SUCCESS) {
        storage_object_t * object;
        BOOLEAN found = FALSE;

        rc = GetObject(list,
                       sizeof(storage_object_t),
                       tag,
                       NULL,
                       FALSE,
                       (ADDRESS *) &object);

        while ((rc == 0) && (object != NULL) && !found) {

            found = (strcmp(object->name, name) == 0);

            if (!found) {
                rc = GetNextObject(list,
                                   sizeof(storage_object_t),
                                   tag,
                                   (ADDRESS *) &object);
            }
        }

        if (found) {
            rc = ensure_app_handle(object,
                                   object->object_type,
                                   &object->app_handle);

            if (rc == 0) {
                *object_handle = object->app_handle;
            }

        } else {
            rc = ENOENT;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

int evms_get_object_handle_for_name(object_type_t type, char * name, object_handle_t * object_handle) {
    int rc = 0;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        switch (type) {
            case DISK:
                rc = find_object_handle_by_name(DiskList, DISK, name, object_handle);
                break;

            case SEGMENT:
                rc = find_object_handle_by_name(SegmentList, SEGMENT, name, object_handle);
                break;

            case REGION:
                rc = find_object_handle_by_name(RegionList, REGION, name, object_handle);
                break;

            case EVMS_OBJECT:
                rc = find_object_handle_by_name(EVMSObjectList, EVMS_OBJECT, name, object_handle);
                break;

            case CONTAINER:
                rc = GoToStartOfList(ContainerList);

                if (rc == DLIST_SUCCESS) {
                    storage_container_t * container;
                    BOOLEAN found = FALSE;

                    rc = GetObject(ContainerList,
                                   sizeof(storage_container_t),
                                   CONTAINER_TAG,
                                   NULL,
                                   FALSE,
                                   (ADDRESS *) &container);

                    while ((rc == 0) && (container != NULL) && !found) {

                        found = (strcmp(container->name, name) == 0);

                        if (!found) {
                            rc = GetNextObject(ContainerList,
                                               sizeof(storage_container_t),
                                               CONTAINER_TAG,
                                               (ADDRESS *) &container);
                        }
                    }

                    if (found) {
                        rc = ensure_app_handle(container,
                                               CONTAINER,
                                               &container->app_handle);
                        if (rc == 0) {
                            *object_handle = container->app_handle;
                        }


                    } else {
                        rc = ENOENT;
                    }
                }
                break;

            case VOLUME:
                rc = GoToStartOfList(VolumeList);

                if (rc == DLIST_SUCCESS) {
                    logical_volume_t * volume;
                    BOOLEAN found = FALSE;

                    rc = GetObject(VolumeList,
                                   sizeof(logical_volume_t),
                                   VOLUME_TAG,
                                   NULL,
                                   FALSE,
                                   (ADDRESS *) &volume);

                    while ((rc == 0) && (volume != NULL) && !found) {

                        found = (strcmp(volume->name, name) == 0);

                        if (!found) {
                            rc = GetNextObject(VolumeList,
                                               sizeof(logical_volume_t),
                                               VOLUME_TAG,
                                               (ADDRESS *) &volume);
                        }
                    }

                    if (found) {
                        rc = ensure_app_handle(volume,
                                               VOLUME,
                                               &volume->app_handle);
                        if (rc == 0) {
                            *object_handle = volume->app_handle;
                        }


                    } else {
                        rc = ENOENT;
                    }
                }
                break;

            case PLUGIN:
                rc = GoToStartOfList(PluginList);

                if (rc == DLIST_SUCCESS) {
                    plugin_record_t * pPlugRec;
                    BOOLEAN found = FALSE;

                    rc = GetObject(PluginList,
                                   sizeof(plugin_record_t),
                                   PLUGIN_TAG,
                                   NULL,
                                   FALSE,
                                   (ADDRESS *) &pPlugRec);

                    while ((rc == 0) && (pPlugRec != NULL) && !found) {

                        found = (strcmp(pPlugRec->short_name, name) == 0);

                        if (!found) {
                            rc = GetNextObject(PluginList,
                                               sizeof(plugin_record_t),
                                               PLUGIN_TAG,
                                               (ADDRESS *) &pPlugRec);
                        }
                    }

                    if (found) {
                        rc = ensure_app_handle(pPlugRec,
                                               PLUGIN,
                                               &pPlugRec->app_handle);
                        if (rc == 0) {
                            *object_handle = pPlugRec->app_handle;
                        }


                    } else {
                        rc = ENOENT;
                    }
                }
                break;

            default:
                rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_direct_plugin_communication(plugin_handle_t plugin_handle,
                                     object_handle_t thing,
                                     BOOLEAN         target_kernel_plugin,
                                     void          * arg) {

    int rc = 0;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_write_access();

    if (rc == 0) {
        rc = translate_handle(plugin_handle,
                              &object,
                              &type);

        if (rc == HANDLE_MANAGER_NO_ERROR) {
            if (type == PLUGIN) {
                plugin_record_t * plugin = (plugin_record_t *) object;

                rc = translate_handle(thing,
                                      &object,
                                      &type);

                if (rc == HANDLE_MANAGER_NO_ERROR) {
                    switch (GetPluginID(plugin->id)) {
                        case EVMS_DEVICE_MANAGER:
                        case EVMS_SEGMENT_MANAGER:
                        case EVMS_REGION_MANAGER:
                        case EVMS_FEATURE:
                        case EVMS_ASSOCIATIVE_FEATURE:{

                                LOG_DEBUG("Direct communication going to the %s feature.\n", plugin->short_name);

                                rc = plugin->functions.plugin->direct_plugin_communication(object, target_kernel_plugin, arg);
                            }
                            break;

                        default:
                            rc = EINVAL;
                            break;
                    }
                }

            } else {
                rc = EINVAL;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

