/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


// Jarl van Katwijk - 2001
//                    2002 februari : mayor update, linked to libmpeg2 and added fifo's
// 
// original based on mpeg2divx 0.8
// 2000/10/23 Ulrich Hecht <uli@emulinks.de>
// licensed under the GNU Public License (GPL) v2
//
// added second audio stream - Rainer Lay 2001
// added DX50 and XVID and corrected DIVX attributes - Michael Deindl, May 2002
// changed matching from fourCC to "privname"        - Michael Deindl, May 2002


/* system */
#include <sys/time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <fstream>
#include <math.h>
#include <errno.h>
#include <pthread.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
/* mpeg3 decoding */
#include "demuxer.hh"
/* drip general */
#include "../src/drip.h"
#include "../src/external.h"
/* filters */
#include "deinterlace.hh"
#include "autoclipper.hh"
#include "pulldown.hh"
#include "scaler.hh"
#include "clipper.hh"
#include "conversion.hh"
#include "plugin-loader.hh"
/* main */
#include "main.hh"
#include "fast_memcpy.hh"
#include "encoder.hh"

using namespace std;

// excessive loggin
//#define DEBUG
#ifdef DEBUG
#warning         -------------------------------------
#warning
#warning
#warning
#warning
#warning
#warning         DEBUGGING WILL PRODUCE HUGE LOG FILES
#warning
#warning
#warning
#warning
#warning
#warning         -------------------------------------
#endif

#define DOUBLED_OFFSET 3

// protos
void set_dimensions_aspect(void);
void allocate_buffers(guchar* buffer);

// types
typedef struct {
    gint in_stream;
    fourcc_t codec;
    gint mp3bitrate;
    int achan;
    int rate;
    long samples;
    gdouble sampsperframe_add;
    gint sampsperframe;
    gdouble samplepos;
    gdouble samplepos_last;
    gshort* audiobuffer;
    WAVEFORMATEX fmt;
    IAviAudioWriteStream* outstream;
    gint volume_adjustment;
} au_t;
 
typedef struct {
    gint rc_reaction_period;
    gint rc_reaction_ratio;
    gint rc_period;
    gint max_key_interval;
    gint obmc;
    gint quality;
    gint max_quantizer;
    gint min_quantizer;
} opendivx_params_t;

// globals   TODO: a little much .. ;(
IAviAudioWriteStream **avifile_audio_outstream = (IAviAudioWriteStream**)malloc(2*sizeof(IAviAudioWriteStream *));
gdouble aspect = 0;
gboolean scale = FALSE;
gdouble framerate;
fourcc_t video_codec=fccDIV3;
GString *video_codev_module = g_string_new("");
BitmapInfo bminfo_in; // Codec struct
CImage* im; // framebuffer image 
CImage* progress_frame_im; // I420->RGB conversion buffer
gint clipleft = 0;
gint clipright = 0;
gint cliptop = 0;
gint clipbottom = 0;
gint max_size = 731906048;
glong frames;
au_t au[AMAX];
gboolean autoclip = FALSE;
gint autoclip_frames = 256;
guchar* framebuffer;
guint8 *clippedbuffer[3];
gboolean Video_Synced = FALSE;
guchar* ToRGBbuffer;
u_int8_t *planes[3];
guchar* extrabuffer = NULL; // ! extrabuffer == NULL -> no setup done yet
guchar** rowptr;
guint8* scaledbuffer[3];
guchar* _RGBbuffer;
gchar* outputfile;
gint divxbitrate=1200;
gint subpicture = -1;
gchar* clutfile = NULL;
gboolean pulldown = FALSE;
GdkPixbuf* gframe;
GdkPixbuf* gscaled;
opendivx_params_t opendivx_params;
gint endframe=-1;
GString* inputfiles[256];
IAviSegWriteFile* avifile;
IAviVideoWriteStream* stream = NULL;
glong todo_frames;
gboolean hard_todo_frame = TRUE;
gint quit;
gboolean audio_encoding;
gint numfiles;
gint file;
gint result;
GString *output;
glong frames_doubled,frames_dropped;
gint argbase=0;
gdouble mp3bitrate=-1;
gboolean deinterlace=FALSE;
gint austream=0;
gint vistream=0;
gint limit=0;
gboolean moreopts=true;
struct stat VOB_file;
gulong total_bytes;
gdouble frames_per_bytes;
glong totalframes=0;
struct timeval tv;
struct timezone tz={0,0};
glong startsec;
glong startusec;
double fps=0;
gint total_channels;
gint assetChannels;
int16_t *input_ptr;
int16_t *output_ptr;
gchar *buffer;
// audio
gchar *audio_buffer;
gint downmix51;
int16_t *input_end;
off_t len;
gint cached_frames = 0;
glong totalframes_countdown = 0;
GString *framefile;
glong writeout_frames_countdown = 10;
glong writeout_frames_countdown_amount = 10;
gboolean audio_output_codec_setup = FALSE;
gboolean audio_buffer_filled;
glong totalframes_buffered = 0;
/* Frame buffer */
typedef struct _plane_t plane_t;
struct _plane_t {
    guint8 *plane;
    gint slot;
    glong framenr;
    gdouble PTS;
    gdouble SCR;
};
/* FB slot */
typedef struct _slot_t slot_t;
struct _slot_t {
    guint8 *plane;
    gboolean free;
};
static gint frame_buffer_index = -1;
static gint frame_buffer_base = 0;
const gint frame_buffer_index_MAX = 64; 
static plane_t frame_buffer[frame_buffer_index_MAX];
static slot_t frame_buffer_slots[frame_buffer_index_MAX];
static guint8 *frame_raw = NULL;
enum {NONE,CODEC,BUFFER,FETCH,ALL};
static gboolean first_frame = TRUE;
guint8* f_buffer_bak[3];
gboolean noise = FALSE;
guint8* deinterlace_buffer;
gdouble SCRframe = 0;



/* Wrapper for SetAttribute of avifile codec object */
void Codec_SetAttribute(const CodecInfo* codecInfo,const gchar* attribute, gint value) {
    gint result;
    #ifndef STANDALONE
    /* Lock */
    drip_lock("Codec SetAttribute");
    #endif
    /* Try to set attribute */
    result = Creators::SetCodecAttr(*codecInfo, attribute, value);
    #ifndef STANDALONE
    /* Unlock */
    drip_unlock("Codec SetAttribute");
    #endif
    /* Handle negative sets */
    if (result==-1) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Could not set attribute %s to value %i. Ignoring, wrong codec or outdated avifile?",attribute,value);
    }
    return;
}


/* Setup audio output codec */
void audio_output_codec(void) {
    /* Check if audio is already available */
    /* NOTE: untill audio_output_codec_setup is TRUE chunks
             of silence will be added to audio channel */
    if ((audio_output_codec_setup == TRUE)) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"audio_output_codec, audio already setup");
        return;
    }
    /* Parse audio info */
    audio_encoding=FALSE;
    for (gint anum=0;anum<AMAX;anum++) {
    /* Output audio codec */
        if (au[anum].in_stream>-1) {
            au[anum].achan = 2;//drip_audio_channels(); //TODO
            au[anum].rate  = drip_audio_samplerate();
            audio_encoding = TRUE;
            switch (au[anum].codec) {
                case 85:         g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio %d: Using MP3 codec",anum);
                                 break;
                case 304:        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio %d: Using ACELP codec",anum);
                                 break;
                default:         g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Audio %d: No codec found for this channel, not encoding.",anum);
                                 return;
            }
        }
    }
    /* Audio channels */
    if (!audio_encoding) 
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"There will be NO audio encoded!");

    /* Output Audio header */
    for (gint anum=0;anum<AMAX;anum++) {
        if (au[anum].in_stream>-1) {
            /* Output Audio header */
            memset(&au[anum].fmt,0,sizeof(WAVEFORMATEX));
            au[anum].fmt.wFormatTag = 1; /* PCM audio id */
            au[anum].fmt.nChannels = au[anum].achan;
            au[anum].fmt.nSamplesPerSec = au[anum].rate;
            au[anum].fmt.wBitsPerSample = 16;
            au[anum].fmt.nBlockAlign = 2;
            au[anum].fmt.nAvgBytesPerSec = au[anum].fmt.nSamplesPerSec*2;
            au[anum].fmt.cbSize = sizeof(WAVEFORMATEX);
            au[anum].sampsperframe_add = ceil(au[anum].rate/drip_framerate());
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio %d: Samples per frames being added = %.3f",anum,au[anum].sampsperframe_add);
            au[anum].sampsperframe = (gint)au[anum].sampsperframe_add + 1;
            au[anum].samplepos = 0;
            au[anum].samplepos_last = 0;
            /* Auto bitrate? */
            if (au[anum].mp3bitrate == -1) {
                switch(au[anum].rate) {
                case 48000: au[anum].mp3bitrate=(au[anum].achan>1?160:80*au[anum].achan); break;
                case 44100: au[anum].mp3bitrate=(au[anum].achan>1?112:56*au[anum].achan); break;
                case 22050: au[anum].mp3bitrate=(au[anum].achan>1?64:32*au[anum].achan);break;
                default:    g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Audio %d: Unknown sample rate %d Hz, specify MP3 bitrate",anum,au[anum].rate);
                            exit(1);
                } 
            }
            if (au[anum].achan>0) {
                #ifndef STANDALONE
                drip_lock("audio_output_codec");
                #endif
                au[anum].outstream=avifile->AddAudioStream(au[anum].codec,&au[anum].fmt,(gint)(au[anum].mp3bitrate*125));
                avifile_audio_outstream[anum] = au[anum].outstream;
                avifile_audio_outstream[anum]->Start(); //au[anum].outstream->Start();
                #ifndef STANDALONE
                drip_unlock("audio_output_codec");
                #endif
                g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Audio %d: MP3 bitrate %ibps, codec %lu",anum,au[anum].mp3bitrate,au[anum].codec);
            }
        }
    }
    audio_output_codec_setup = TRUE;
    return;
}


/* Setup video output codec */
gboolean video_output_codec(void) {
    /* Video header */
    bminfo_in = BitmapInfo(Config.out_width,Config.out_height,AVIFILE_COLOUR_DEPT_OUTPUT);
    bminfo_in.SetSpace(fccI420);
    bminfo_in.biSize = sizeof (BITMAPINFOHEADER);

    /* Various encoding settings */
    gint quality = 1000;
    gint keyfreq = 250;
    frames_doubled = 0; // keep track of extra frames need for syncing
    frames_dropped = 0; // keep track of dropped frames need for syncing

    /* Output codec */
    switch (video_codec) {
        case fccDIVX:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to OpenDIVX");               break;
        case fccXVID:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to XviD");                   break;
        case fccDIV3:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to MPEG4 low motion");       break;
        case fccDIV4:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to MPEG4 high motion");      break;
        case fccDIV5:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to MS-DIV5 (low-motion)");   break;
        case fccDIV6:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to MS-DIV6 (high-motion)");  break;
        case fccDX50:    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Encoding video to DIVX 5.0");               break;
        default:         g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"No DIVX video codec (%li), might not produce valid results..",video_codec);
                         //return FALSE;
    }
    /* Output codec attributes */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"About to initialise avifile, when Drip stops (crashes) here get yourself a working avifile...");
    /* Create output avi stream */
    avifile = CreateSegmentedFile(outputfile,max_size);
    const CodecInfo* codecInfo=0;
    #if (AVIFILE_MAJOR_VERSION == 0) && (AVIFILE_MINOR_VERSION>=7) && (AVIFILE_PATCHLEVEL>=26)
    while (codecInfo=CodecInfo::match(video_codec,CodecInfo::Video,codecInfo,CodecInfo::Encode)) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Found codec: %s, %s", codecInfo->GetName(), codecInfo->GetPrivateName());
        if (0 == *video_codev_module->str) {
            // no codec-module specified: take first one!
            break;
        }
        if (0==strcmp(video_codev_module->str,codecInfo->GetPrivateName())) {
            // we found the right codec!
            break;
        }
    }
    #else
        codecInfo=CodecInfo::match(CodecInfo::Video,video_codev_module->str);
    #endif
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Trying to utilize %s codec",codecInfo->GetPrivateName());
    if (codecInfo == NULL) {
        // FIXME!! Error-Handling: Codec not found
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Codec not found: %X, %s\n", video_codec, video_codev_module->str);
        return FALSE;
    }
    if (0 == strncmp (codecInfo->GetPrivateName(),"divx",4)) { /* MSMPEG4 */
       Codec_SetAttribute(codecInfo, "Crispness", 100);
       Codec_SetAttribute(codecInfo, "KeyFrames", keyfreq);
       Codec_SetAttribute(codecInfo, "BitRate", divxbitrate);
    }
    if (0 == strncmp (codecInfo->GetPrivateName(),"odivx",5)) { //* OpenDIVX, DIVX4, DIVX5 */
        Codec_SetAttribute(codecInfo, "bitrate",divxbitrate*1000);
        Codec_SetAttribute(codecInfo, "quality",5); //opendivx_params.quality); // DIVX:5=best,1=lowest, avifile:1=enable VBR... ????
        Codec_SetAttribute(codecInfo, "rc_period",keyfreq*10);
        Codec_SetAttribute(codecInfo, "rc_reaction_period",(gint)(keyfreq/20)); 
        Codec_SetAttribute(codecInfo, "rc_reaction_ratio",(gint)(keyfreq/10));
        Codec_SetAttribute(codecInfo, "max_key_interval",keyfreq);
        Codec_SetAttribute(codecInfo, "min_quantizer",opendivx_params.min_quantizer);
        Codec_SetAttribute(codecInfo, "max_quantizer",opendivx_params.max_quantizer);
        Codec_SetAttribute(codecInfo, "deinterlace",0);          // not always available
        Codec_SetAttribute(codecInfo, "bidirect",1);             // use B-Frames (not always available)
        Codec_SetAttribute(codecInfo, "obmc",1);                 // not always available
        Codec_SetAttribute(codecInfo, "interlace_mode",0);
        Codec_SetAttribute(codecInfo, "temporal_enable",1);
        // deactivated because avifile treats it as int, but divx4linux docu
        // says temporal_level is a double
        Codec_SetAttribute(codecInfo, "temporal_level",1000);
        Codec_SetAttribute(codecInfo, "spatial_passes",3);
        Codec_SetAttribute(codecInfo, "spatial_level",1000);
        Codec_SetAttribute(codecInfo, "enable_crop",0);
        Codec_SetAttribute(codecInfo, "enable_resize",0);

    }
    if (0 == strcmp (codecInfo->GetPrivateName(), "xvid")) { /* XviD = OpenDIVX, DIVX4 */
        #if (AVIFILE_MAJOR_VERSION == 0) && (AVIFILE_MINOR_VERSION>=7) && (AVIFILE_PATCHLEVEL>=26)
        Codec_SetAttribute(codecInfo, "quality",9000);
        Codec_SetAttribute(codecInfo, "Mode",2);
        Codec_SetAttribute(codecInfo, "rc_bitrate",divxbitrate*1000);
        Codec_SetAttribute(codecInfo, "motion_search",6); // ultra high
        Codec_SetAttribute(codecInfo, "me_pmvfast",0);
        Codec_SetAttribute(codecInfo, "me_epzs",1);
        Codec_SetAttribute(codecInfo, "quant_type",0);
        Codec_SetAttribute(codecInfo, "min_quantizer",2);
        if (divxbitrate>1500) {
            Codec_SetAttribute(codecInfo, "max_quantizer",6);
        } else if (divxbitrate<1501) {
            Codec_SetAttribute(codecInfo, "max_quantizer",12);
        } else if (divxbitrate<1001) {
            Codec_SetAttribute(codecInfo, "max_quantizer",18);
        } else if (divxbitrate<901) {
            Codec_SetAttribute(codecInfo, "max_quantizer",24);
        } else if (divxbitrate<801) {
            Codec_SetAttribute(codecInfo, "max_quantizer",30);
        }
        Codec_SetAttribute(codecInfo, "inter4v",1);
        //Codec_SetAttribute(codecInfo, "diamond_search",1);
        Codec_SetAttribute(codecInfo, "adaptive_quant",1);
        Codec_SetAttribute(codecInfo, "halfpel",1);
        Codec_SetAttribute(codecInfo, "interlacing",0);
        Codec_SetAttribute(codecInfo, "max_key_interval",keyfreq);
        Codec_SetAttribute(codecInfo, "lum_masking",0);
        #else
        Codec_SetAttribute(codecInfo, "rc_bitrate",divxbitrate*1000);
        Codec_SetAttribute(codecInfo, "motion_search",6); // ultra high
        Codec_SetAttribute(codecInfo, "quant_type",0);
        Codec_SetAttribute(codecInfo, "halfpel",1);
        //Codec_SetAttribute(codecInfo, "rc_buffer",10000);
        Codec_SetAttribute(codecInfo, "interlacing",0);
        Codec_SetAttribute(codecInfo, "max_key_interval",keyfreq);
        Codec_SetAttribute(codecInfo, "lum_masking",0);
        Codec_SetAttribute(codecInfo, "rc_period",keyfreq*10);
        Codec_SetAttribute(codecInfo, "rc_reaction_ratio",(gint)(keyfreq/10));
        Codec_SetAttribute(codecInfo, "rc_reaction_period",(gint)(keyfreq/20));
        #endif
    }

    /* Add video stream to output avi */
    #ifndef STANDALONE
    drip_lock("video_output_codec");
    #endif
    #if (AVIFILE_MAJOR_VERSION == 0) && (AVIFILE_MINOR_VERSION>=7) && (AVIFILE_PATCHLEVEL>=26)
    stream = avifile->AddVideoStream(*codecInfo, &(bminfo_in), (guint)(1000000./framerate));
    #else
    stream = avifile->AddVideoStream(video_codec, &(bminfo_in), (guint)(1000000./framerate));
    #endif
    stream->SetQuality(quality);
    stream->SetKeyFrame(keyfreq);
    stream->Start();
    #ifndef STANDALONE
    drip_unlock("video_output_codec");
    #endif
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Avifile initialisation calls passed");
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Video Encoding bitrate %i",divxbitrate);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"DIVX Chunk size %i bytes",max_size);
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"AVI Output File %s",outputfile);

    audio_output_codec();
    fprintf(stderr,"--------------------------------------------------------\n\n\n");

    return TRUE;
}


/* Autoclipping */
void autoclipping(gint frames_per_sample) {
    gint samples = 5; // TODO!
    gint factor = 16;
    /* Anything todo? */
    if (autoclip==FALSE) return;
    if (endframe<100) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Autoclipper disabled because not enough frames (100) in source");
        autoclip = FALSE;
    }
    /* Set clip values to high */
    cliptop = 1000;clipbottom = 1000;clipleft = 1000;clipright = 1000;
    /* Run autoclipper */
    autoclipper_apply(inputfiles[0]->str,endframe,1,samples,frames_per_sample);
    /* Round clipping values to 4 */
    factor = 2;
    cliptop    = ((gint)((gdouble)cliptop/factor)*factor);
    clipbottom = ((gint)((gdouble)clipbottom/factor)*factor);
    clipleft   = ((gint)((gdouble)clipleft/factor)*factor);
    clipright  = ((gint)((gdouble)clipright/factor)*factor);
    /* Any clipping -> set scale to TRUE */
    if ((cliptop!=0)||(clipbottom!=0)||(clipleft!=0)||(clipright!=0)) {
        scale = TRUE;
    };
    Config.in_width = drip_mpeg_width();
    Config.in_height = drip_mpeg_height();
    /* Complete clipping? */
    if ((cliptop>Config.in_height) || (clipbottom>Config.in_height) || (clipleft>Config.in_width) || (clipright>Config.in_width)) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Autoclipping disabled, empty or too less video");
        cliptop    = 0;
        clipbottom = 0;
        clipleft   = 0;
        clipright  = 0;
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Clipping top %i,bottom %i,left %i,right %i",cliptop,clipbottom,clipleft,clipright);
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Video clipped to %ix%i",Config.in_width-(clipleft+clipright),Config.in_height-(cliptop+clipbottom));
    }
    return;
}

/* Clip frame */
void clip_frame_i420(guint8 *src[3], guint8 *dest[3],gint xoff,gint yoff,gint w, gint h,gint w_in,gint h_in) {
    clipper_apply(src,dest,xoff,yoff,w,h,w_in,h_in);
    return;
}

/* Scale I420 frame */
void scale_frame_i420(guint8* source[3], guchar* dest,gint swidth,gint sheight,gint dwidth,gint dheight, gint cleft, gint cright, gint ctop, gint cbottom) {
    scaler_apply(swidth,sheight,dwidth,dheight,source,dest);
    return;
}


/* Encode frame : here a video frame is either put into or fetched from the frame fifo.
                  the buffer takes care of sync as well by dropping and doubling frames */
gdouble encode_frame(IAviVideoWriteStream *stream,CImage *frame,gdouble SCRmpeg,gboolean fetching,gboolean probing,gint PTSaudio) {
    gint i,slot;
    gint writeout_frame = NONE;
    gboolean result = TRUE;
    static guint WH32 = Config.out_width*Config.out_height*3/2;
    static gdouble SCRvideo = -1;
    static gdouble SCRvideo_frame = (gdouble)TimeStamp_Frame() / (gdouble)drip_PTS_correction();
    static gint DOUBLED = 0;
    static gdouble SCRcalc = 0;
    static gint slots_free = frame_buffer_index_MAX;

    /* Allocate buffer */
    if (frame_raw == NULL) {
        SCRvideo = (gdouble)PTSvideo;
        SCRframe = (gdouble)PTSvideo;
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Allocating frame buffer for %ux%u with dept %i, SCRvideo = %f",Config.out_width,Config.out_height,frame_buffer_index_MAX,SCRvideo);
        for (i=0;i<frame_buffer_index_MAX;i++) {
            frame_buffer_slots[i].plane = (guint8*)malloc(WH32);
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Allocated plane %i at %p",i,frame_buffer_slots[i].plane);
            #endif
            frame_buffer_slots[i].free = TRUE;
        }
        //#ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: SCRvideo_frame = %f",SCRvideo_frame);
        //#endif
        /* Locate raw buffer */
        if (scale == TRUE) {
            frame_raw = (guchar*)scaledbuffer[0];
        } else {
            frame_raw = framebuffer;
        }
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Allocated frame buffer");
    }

    /* Offset video later as audio? */
    if (Video_Synced == FALSE && PTSvideo > PTSaudio && PTSaudio > 0) {
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Adjusted SCRframe with %li to sync with audio (needed to add extra frame doubling), SCRvideo = %f, PTSaudio = %li",(glong)(PTSvideo-PTSaudio),PTSvideo,PTSaudio);
        #endif
        Video_Synced = TRUE;
        SCRvideo = (gdouble)PTSvideo;
        SCRframe = (gdouble)PTSaudio;
        #ifdef DEBUG
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Adjusted SCRframe to sync with audio , SCRframe = %f",SCRframe);
        #endif
    }

    /* Only probing buffer? */
    if (probing==TRUE) {
        if (frame_buffer_index>0) {
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Probing returned TRUE, index=%i, SCR=%g",frame_buffer_index,SCRvideo);
            #endif
            return SCRvideo;
        } else {
            #ifdef DEBUG
            g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Probing returned FALSE");
            #endif
            return FALSE;
        }
    }
    /* Fetching frame from buffer? */
    if (fetching==TRUE) {
        if (audio_buffer_filled==TRUE) {
            writeout_frame = FETCH;
        }
    } else {
        writeout_frame = BUFFER;
    }

    /* Execute action */
    switch (writeout_frame) {
        case CODEC:  /* Feed frame & audio to codec */
                     stream->AddFrame(frame);
                     SCRvideo += SCRvideo_frame;
                     /* Frame counter */
                     totalframes++;
                     break;
        case BUFFER: /* Add frame to FIFO buffer */
                     if (first_frame == TRUE) {
                         /* DONT buffer very 1st frame */
                         first_frame = FALSE;
                         break;
                     }
                     if (slots_free == 0) {
                         /* No slot left in buffer .. should not reach this due to silence adding.. */
                         result = FALSE;
                         g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Cant add new frame to frame buffer, no free buffer space left");
                         g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Frame buffer out of space, selected audio not available?");
                     } else {
                         /* Find a free slot */
                         slot = -1;
                         for (i=0;i<frame_buffer_index_MAX;i++) {
                             if (frame_buffer_slots[i].free == TRUE) {
                                 slot = i;
                                 i = frame_buffer_index_MAX;
                             }
                         }
                         if (slots_free < 3) { // 2 free for frame doubling
                             /* Last free slots: add extra silence to audio buffer */
                             g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Audio missing, added silence (PTS %i)",(gint)frame_buffer[frame_buffer_base].SCR);
                             add_silence_to_audio_buffer((gint)frame_buffer[frame_buffer_base].SCR,TRUE);
                         } else {
                             /* -- Add frame to buffer -- */
                             /* Does the FIFO buffer needs to be rotated? */
                             if ((frame_buffer_index+frame_buffer_base)>=frame_buffer_index_MAX) {
                                 for (i=0;i<frame_buffer_index;i++) {
                                     fast_memcpy(&frame_buffer[i],&frame_buffer[i+frame_buffer_base],sizeof(plane_t));
                                 }
                                 frame_buffer_base = 0; /* Reset base */
                             }
                             /* New PTS value, or do we need to calculate? */
                             if (SCRmpeg == 0) {
                                 SCRcalc += SCRvideo_frame;
                                 SCRmpeg = SCRcalc;
                             } else {
                                 if (SCRmpeg<(SCRframe-500000)) {
                                     g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Buffering, PTS overflow, SCRmpeg = %li, SCRframe = %li",SCRmpeg,SCRframe);
                                     /* PTS overflow, reset SCRframe */
                                     SCRcalc = SCRmpeg;
                                     SCRframe = SCRmpeg;
                                     PTSvideo = -1;
                                     //add_silence_to_audio_buffer((gint)frame_buffer[frame_buffer_base+frame_buffer_index-1].SCR,FALSE);
                                 } else if (SCRframe<(SCRmpeg-500000)) {
                                     /* PTS overflow, reset SCRframe */
                                     g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Buffering, SCR overflow, resetted calculated SCR  (SCRframe=%li, SCRmpeg=%li, SCRcalc=%li, SCRframe=%li, BufferedFrame=%li",SCRframe,SCRmpeg,SCRcalc,SCRframe,frame_buffer[frame_buffer_base+frame_buffer_index-1].SCR);
                                     SCRcalc = SCRmpeg;
                                     SCRframe = SCRmpeg;
                                     PTSvideo = -1; // have the demoxer reset the audio/vide pts's
                                    // add_silence_to_audio_buffer((gint)frame_buffer[frame_buffer_base+frame_buffer_index-1].SCR,FALSE);
                                 }
                             }
                             /* Add new frame to buffer */
                             frame_buffer_slots[slot].free = FALSE; /* Lock this slot */
                             slots_free--; // decrease amount of free slots
                             frame_buffer[frame_buffer_index+frame_buffer_base].plane = frame_buffer_slots[slot].plane;
                             fast_memcpy(frame_buffer[frame_buffer_index+frame_buffer_base].plane,frame_raw,WH32);
                             frame_buffer[frame_buffer_index+frame_buffer_base].slot = slot;
                             frame_buffer[frame_buffer_index+frame_buffer_base].framenr = totalframes_buffered;
                             frame_buffer[frame_buffer_index+frame_buffer_base].PTS = SCRmpeg;
                             frame_buffer[frame_buffer_index+frame_buffer_base].SCR = SCRframe;
                             #ifdef DEBUG
                             g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Buffering: Frames buffered = %li, PTS=%f, SCRmpeg=%f, SCRcalc=%f, SCRframe=%f, slots free = %i",totalframes_buffered,frame_buffer[frame_buffer_index+frame_buffer_base].PTS,frame_buffer[frame_buffer_index+frame_buffer_base].SCR,SCRcalc,SCRframe,slots_free);
                             #endif
                             frame_buffer_index++; /* Increase amount of frames in buffer */
                             totalframes_buffered++;
                             SCRframe += SCRvideo_frame;
                             DOUBLED--;
                             /* Is buffer PTS one frame behind SCR? */
                             if (((frame_buffer[frame_buffer_index+frame_buffer_base-1].PTS - frame_buffer[frame_buffer_index+frame_buffer_base-1].SCR) >= SCRvideo_frame) && (DOUBLED<1)) {
                                 /* Yes, double frame */
                                 #ifdef DEBUG
                                 g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Buffering: Doubled frame, PTS=%g/%g, bufferSCR=%g, SCRvideo_frame=%g, DOUBLED=%i",PTSframe,frame_buffer[frame_buffer_index+frame_buffer_base-1].PTS,frame_buffer[frame_buffer_index+frame_buffer_base-1].SCR,SCRvideo_frame,DOUBLED);
                                 #endif
                                 DOUBLED = DOUBLED_OFFSET;   // Reset double counter.. dont double all needed frames from the same frame
                                 frames_doubled++;
                                 encode_frame(stream,frame,0,FALSE,FALSE,PTSaudio); // double this frame
                                 todo_frames++;  // adjust amount of frames tobe encoded
                             } 
                             #ifdef DEBUG
                             else {
                                 g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Buffering: **NOT*** Doubled frame, PTS=%g/%g, bufferSCR=%g, SCRvideo_frame=%g, DOUBLED=%i",PTSframe,frame_buffer[frame_buffer_index+frame_buffer_base-1].PTS,frame_buffer[frame_buffer_index+frame_buffer_base-1].SCR,SCRvideo_frame,DOUBLED);
                             }
                             #endif
                         }
                     }
                     break;
        case FETCH:  if (frame_buffer_index>0) {
                         /* Drop frames behind (SCR - SCRframe) */
                         while ((totalframes,frame_buffer[frame_buffer_base].PTS < (frame_buffer[frame_buffer_base].SCR - SCRframe) && (frame_buffer_index>0))) {
                             /* Update base & index */
                             #ifdef DEBUG
                             g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Fetched frame %li dropped (PTS=%i,SCR=%i)",frame_buffer[frame_buffer_base].framenr,frame_buffer[frame_buffer_base].PTS,frame_buffer[frame_buffer_base].SCR);
                             #endif
                             frame_buffer_base++;
                             frame_buffer_index--;
                             frames_dropped++;
                         }

                         if (frame_buffer_index>0) {
                             /* Feed frame from fifo buffer to codec */
                             fast_memcpy(frame_raw,frame_buffer[frame_buffer_base].plane,WH32);
                             stream->AddFrame(frame);
                             SCRvideo += SCRvideo_frame;
                             /* Frame counter */
                             totalframes++;
                             /* Free slot that contains frame */
                             frame_buffer_slots[frame_buffer[frame_buffer_base].slot].free = TRUE;
                                 /* Display sequence number of fetched frame */
                             #ifdef DEBUG
                             g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Fetched frame %li, base=%i, slot %i freed, slot pointer=%p, frame=%li,  PTS=%f,  SCR=%f, slots free = %i",frame_buffer[frame_buffer_base].framenr,frame_buffer_base,frame_buffer[frame_buffer_base].slot,frame_buffer[frame_buffer_base].plane,totalframes,frame_buffer[frame_buffer_base].PTS,frame_buffer[frame_buffer_base].SCR,slots_free);
                             #endif
                             /* Update base & index */
                             frame_buffer_base++;
                             frame_buffer_index--;
                             /* Buffer reached end -> rotate buffer */
                             if ((frame_buffer_index+frame_buffer_base)>=frame_buffer_index_MAX) {
                                 for (i=0;i<frame_buffer_index;i++) {
                                     fast_memcpy(&frame_buffer[i],&frame_buffer[i+frame_buffer_base],sizeof(plane_t));
                                 }
                                 frame_buffer_base = 0; //Reset base
                                 #ifdef DEBUG
                                 g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Rotated frame buffer");
                                 #endif
                             }
                             slots_free++; // increase free slots
                         } else {
                             result = FALSE;
                             g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: No frame in buffer to fetch");
                         }
                     }
                     break;
        default:     result = FALSE;
                     g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: frame buffer UNKNOWN action, index=%i, frame=%li",frame_buffer_index,totalframes);
                     break;
    }
    /* Clean & Exit */
    if (result==FALSE) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"FB: Error occured, index=%i, frame=%li",frame_buffer_index,totalframes);
        return -1;
    }
    return SCRvideo;
}
#undef DEBUG

/* Allocate buffers */
void allocate_buffers(void) {
    framebuffer = (guchar*)malloc(Config.in_width*Config.in_height*3);
    /* Allocate video buffers */
    extrabuffer = (guchar*)malloc(Config.in_width*BYTES_PER_COLOUR_OUTPOUT+4);
    rowptr = new (guchar*)[Config.in_height];
    for (gint i=0; i<Config.in_height; i++) {
        if(i==Config.in_height-1) {
            rowptr[i]=extrabuffer;
        } else {
            rowptr[i]=framebuffer+(Config.in_width*(Config.in_height-1)*BYTES_PER_COLOUR_OUTPOUT)-(Config.in_width*BYTES_PER_COLOUR_OUTPOUT*i);
        }
    }
    /* Fill colour planes pointers */
    planes[0] = framebuffer;
    planes[1] = planes[0] + Config.in_width * Config.in_height;
    planes[2] = planes[1] + (Config.in_width * Config.in_height)/4;
    /* Alloc clipped buffer */
    clippedbuffer[0] = (guint8*)malloc(Config.in_width*Config.in_height);
    clippedbuffer[1] = (guint8*)malloc(Config.in_width*Config.in_height/4);
    clippedbuffer[2] = (guint8*)malloc(Config.in_width*Config.in_height/4);
    scaledbuffer[0] = new unsigned char[labs(bminfo_in.biHeight*bminfo_in.biWidth*BYTES_PER_COLOUR_OUTPOUT)];
    scaledbuffer[1] = (guint8*)((glong)scaledbuffer[0]+Config.out_width*Config.out_height);
    scaledbuffer[2] = (guint8*)((glong)scaledbuffer[1]+Config.out_width*Config.out_height/4);
    _RGBbuffer = (guchar*)malloc(labs(bminfo_in.biHeight*bminfo_in.biWidth*3));
    deinterlace_buffer = (guint8*)malloc(Config.in_width * Config.in_height * 3);
    im = new CImage(&bminfo_in,scaledbuffer[0], false);
    ToRGBbuffer = (guchar*)malloc(Config.in_width * Config.in_height * 3);
    progress_frame_im =  new CImage(&bminfo_in,ToRGBbuffer,false);
    /* Bakup f_buffer */
    f_buffer_bak[0] = (guint8*)malloc(Config.in_width*Config.in_height);
    f_buffer_bak[1] = (guint8*)malloc(Config.in_width*Config.in_height/4);
    f_buffer_bak[2] = (guint8*)malloc(Config.in_width*Config.in_height/4);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Allocated buffers for %ix%i frame (%ix%i)",Config.in_width,Config.in_height,bminfo_in.biWidth,bminfo_in.biHeight);
    return;
}

/* Set video dimensions to requested based on input size, and adjust for aspect */
void set_dimensions_aspect(void) {
    frames=-1;//frames=mpeg3_video_frames(file, vistream);
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Input video reads %dx%d %.2ffps",Config.in_width,Config.in_height,framerate);
    /* Automatic width\height */
    gboolean correct_aspect = FALSE;
    if(Config.set_height==-1 && Config.set_width==-1) { // All auto
        Config.out_width=Config.in_width-(clipleft+clipright);
        Config.out_height=Config.in_height-(cliptop+clipbottom);
        correct_aspect = TRUE;
    }
    if(Config.set_width==-1 && Config.set_height!=-1) { // Width auto
        Config.out_height = Config.set_height;
        Config.out_width  = Config.in_width-(clipleft+clipright);
        correct_aspect = TRUE;
    }
    if(Config.set_height==-1 && Config.set_width!=-1) { // Height auto
        Config.out_width  = Config.set_width;
        Config.out_height = Config.in_height-(cliptop+clipbottom);
        correct_aspect = TRUE;
    }

    /* Aspect scaling */
    if (correct_aspect==TRUE) {
        gdouble ow = Config.out_width;
        gdouble oh = Config.out_height;
        gdouble as = (ow/oh);
        gdouble correct;
        if (aspect==0) {
            /* No aspect given, correct based on sizes we got so far, no very correct... */
            if (as>2.06388888) {
                /* 2.35:1 aspect */
                aspect = (gdouble)2.35/(gdouble)1;
            }
            if ((as>1.5555555) && (as<2.06388889)) {
                /* 16:9 aspect */
                aspect = (gdouble)16/(gdouble)9;
            }
            if (as<1.5555556) {
                /* 4:3 aspect */
                aspect = (gdouble)4/(gdouble)3;
            }
        }
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Using DVD aspect %.3f:1",aspect);
        correct = aspect / ((gdouble)Config.in_width/(gdouble)Config.in_height);
        Config.out_height = (gint)(oh / correct);
        Config.out_height = (gint)(Config.out_height / ((gdouble)Config.in_width/(gdouble)Config.out_width));
    }
    /* Adjust Width & Height to a multipe of 16 for better encoding */
    Config.out_width = ((gint)Config.out_width/16)*16;
    Config.out_height= ((gint)Config.out_height/16)*16;
    if ((Config.out_height!=Config.in_height)||(Config.out_width!=Config.in_width)) {  /* Scaling needed? */
        scale = TRUE;
    } else {
        scale = FALSE;
    }
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Output video scaled to %ix%i",Config.out_width,Config.out_height);
    return;
}


/* Display dripencoder CLI mode usage and exit() */
void display_help(gchar* progname) {
    printf("\nUsage: %s [OPTION]... [input MPEG]... [output AVI]\n",progname);
    printf("\nOptions:\n");
    printf("  -b\tDivX ;-) bitrate (default 1200)\n");
    printf("  -f\tuse fast-motion codec (default low-motion)\n");
    printf("  -d\tuse DIVX4 encoding codec\n");
    printf("  -dq\tDIVX4 quality\n");
    printf("  -dminq\tDIVX4 min quantizer\n");
    printf("  -dmaxq\tDIVX4 max quantizer\n");
    printf("  -c\tuse avifile video codec (for use with Drip GUI)\n");
    printf("  -au\tuse avifile audio codec (for use with Drip GUI)\n");
    printf("  -m\tuse avifile plugin for a codec (default: let avifile select a plugin)\n");
    if (AMAX>1)
        printf("  -au2\tuse avifile audio codec (for use with Drip GUI) for audio channel 2\n");
    printf("  -a\tMP3 bitrate (kBit, default auto)\n");
    if (AMAX>1)
       printf("  -a2\tMP3 bitrate (kBit, default auto) for audio channel 2\n");
    printf("  -av\tAudio volume normalisation (0=off, 1=on)\n");
    if (AMAX>1)
       printf("  -av2\tAudio volume normalisation for audio channel 2\n");
    printf("  -e\tnumber of frames to encode (default all)\n");
    printf("  -E\tdo not honor amount of frames, encode untill no frame left\n");
    printf("  -ct\tnumber of lines to clip from top (default none)\n");
    printf("  -cb\tnumber of lines to clip from bottom (default none)\n");
    printf("  -cl\tnumber of lines to clip from left (default none)\n");
    printf("  -cr\tnumber of lines to clip from right (default none)\n");
    printf("  -ac\tauto clip (default off, disables other clipping options)\n");
    printf("  -acf\tnumber of video frames skipped with each autoclip sample (default 256)\n");
    printf("  -w\toutput width (-1 = input width)\n");
    printf("  -h\toutput height (-1 = auto, when width=-1 too both are same as input)\n");
    printf("  -s\tadd subpictures to result, also needs a CLUT file\n");
    printf("  -r\taspect ratio\n");
    printf("  -ip\tdeinterlace PAL video stream (default off)\n");
    //printf("  -in\tdeinterlace (reverse 3:2 pulldown) NTSC video stream (default off) !! Not yet working !!\n");
    printf("  -as\taudio stream number (default 0)\n");
    if (AMAX>1)
        printf("  -as2\taudio stream number (default disabled) for audio channel 2\n");
    printf("  -vs\tvideo stream number (default 0)\n");
    printf("  -l\tlimit output to size in kb, open new output when limit is reached\n");
    printf("\nNOTE: input files should be standalone MPEG2 files, dvd .VOB files\n");
    printf("      and cached drip files are not valid mpeg2 files and must be read\n");
    printf("      in concationation mode: file1.vob:file2.vob:file3.vob\n\n");
    printf("\n%s %s by Jarl van Katwijk (http://drip.sourceforge.net)\n",APPNAME,VERSION);
    exit(0);
    return;
}


/* Setup encoding environment */
gboolean init_environment(void) {
    /* Fill video stream values */
    Config.in_width     = drip_mpeg_width();
    Config.in_height    = drip_mpeg_height();
    framerate = drip_framerate();
    /* Init filters */
    set_dimensions_aspect();
    if (video_output_codec()==FALSE) return FALSE;
    allocate_buffers();
    /* Init scaler */
    scaler_init(Config.in_width-(clipleft+clipright),Config.in_height-(cliptop+clipbottom),Config.out_width,Config.out_height);
    /* Init subpicture */
    if (subpicture>-1) {
        #ifndef STANDALONE
        drip_lock("spu_init");
        #endif
        spu_init(subpicture,Config.in_width,Config.in_height,cliptop,clipbottom,clipleft,clipright,framerate,clutfile,DRIP_CB_LD);
        #ifndef STANDALONE
        drip_unlock("spu_init");
        #endif
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Initialized subpicture %i",subpicture);
    }
    /* Init reverse 3:2 pulldown */
    if (pulldown) {
        pulldown_init(Config.in_width,Config.in_height,false,0);  //TODO: odd frame & offset should not be hardcoded
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Initialized reverse 3:2 pulldown");
    }
    #ifndef STANDALONE
    I420toRGB24_init();
    RGB24toI420_init();
    #endif
    /* Initialise plugins */
    for (gint i=0;i<filters_index;i++) {
        switch ( filters[i].module_phase ) {
            case PRE    : (*filters[i].module_init_function)(Config.in_width,Config.in_height,0,0,Config.in_width,Config.in_height,framerate);
                          break;
            case NORMAL : (*filters[i].module_init_function)(Config.out_width,Config.out_height,0,0,Config.out_width,Config.out_height,framerate);
                          break;
            case POST   : (*filters[i].module_init_function)(Config.out_width,Config.out_height,0,0,Config.out_width,Config.out_height,framerate);
                          break;
            default     : break;
        }
    }
    /* Define SCRframe (value by which SCRdrip is increased every video frame */
    SCRframe = TimeStamp_Frame();
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Encoding environment setup");
    return TRUE;
}


/* Apply all filters and encode a frame with avifile */
void process_frame(guint8 *f_buffer[3],uint32_t pts,gint PTSaudio) {
    static guint8** current_buffer_planar_list;
    guint8* current_buffer_planar[3];  // pointers to most recent processed frame 
    guint8* current_buffer;

    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"PROCESS_FRAME: PTS=%li",pts);
    #endif

    /* Still need to setup encoding environment? */
    if (extrabuffer==NULL) {
        if (init_environment()==FALSE) return;
        current_buffer_planar_list = (guint8**)malloc(sizeof(guint8*)*3);
    }

    /* Our buffer to use */
    fast_memcpy(f_buffer_bak[0],f_buffer[0],Config.in_width*Config.in_height);
    fast_memcpy(f_buffer_bak[1],f_buffer[1],Config.in_width*Config.in_height/4);
    fast_memcpy(f_buffer_bak[2],f_buffer[2],Config.in_width*Config.in_height/4);
    current_buffer = f_buffer_bak[0];
    current_buffer_planar[0] = f_buffer_bak[0];
    current_buffer_planar[1] = f_buffer_bak[1];
    current_buffer_planar[2] = f_buffer_bak[2];

    /* Paused? */
    while ( encoding_paused == TRUE ) {
        sleep(1);
    }

    /* Stop? */
    if ( stopped == TRUE ) {
        stopped = FALSE;
        #ifdef STANDALONE
        printf("\tstopped.\n");
        #endif
        todo_frames = totalframes;
        quit = TRUE;
    }

    /* --- Plugin Filters, type PRE VIDEO --- */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE & filters[i].module_type==VIDEO && filters[i].module_phase==PRE) {
            current_buffer_planar_list[0] = current_buffer_planar[0];
            current_buffer_planar_list[1] = current_buffer_planar[1];
            current_buffer_planar_list[2] = current_buffer_planar[2];

            current_buffer_planar_list = (*filters[i].module_apply_function)(current_buffer_planar_list,Config.in_width*Config.in_height,(gulong)SCRframe,(glong)pts);

            current_buffer_planar[0] = current_buffer_planar_list[0];
            current_buffer_planar[1] = current_buffer_planar_list[1];
            current_buffer_planar[2] = current_buffer_planar_list[2];
        }
    }

    /* +++ Deinterlace PAL +++ */
    if ( deinterlace == TRUE ) {
        deinterlace_apply(deinterlace_buffer,current_buffer_planar[0],Config.in_width,Config.in_height);
        current_buffer = deinterlace_buffer;
        current_buffer_planar[0] = deinterlace_buffer;
    }

    /* +++ Deinterlace NTSC (reverse 3:2 pulldown) +++ */
    /* !!!! TODO : finish, this is incomplete code !!! */
    /* !!!!        Needs audio encoding to work 1st !! */
    if ( pulldown == TRUE ) {
        static guchar* tmpbuffer = inverse_pulldown(current_buffer_planar[0],Config.in_width,Config.in_height,totalframes);
        if (tmpbuffer!=current_buffer_planar[0] && tmpbuffer) {
            current_buffer = tmpbuffer;
            current_buffer_planar[0] = tmpbuffer;
            current_buffer_planar[1] = tmpbuffer+Config.in_width*Config.in_height;
            current_buffer_planar[2] = tmpbuffer+Config.in_width*Config.in_height*5/4;
        }
    }

    /* +++ Subpicture +++ */
    if ( subpicture > -1 ) {
        spu_overlay(current_buffer_planar,Config.in_width,Config.in_height,clipbottom);
    }

    /* +++ Scaling  / Clipping +++ */
    if ( scale == TRUE ) {
        clip_frame_i420(current_buffer_planar, clippedbuffer,clipleft,cliptop,Config.in_width-(clipleft+clipright), Config.in_height-(cliptop+clipbottom),Config.in_width,Config.in_height);
        scale_frame_i420(clippedbuffer,(guchar*)scaledbuffer[0],Config.in_width-(clipleft+clipright), Config.in_height-(cliptop+clipbottom), Config.out_width, Config.out_height, clipleft, clipright, cliptop, clipbottom);
        current_buffer = scaledbuffer[0];
        current_buffer_planar[0] = scaledbuffer[0];
        current_buffer_planar[1] = scaledbuffer[1];
        current_buffer_planar[2] = scaledbuffer[2];
    }

    /* --- Plugin Filters, type NORMAL VIDEO --- */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE & filters[i].module_type==VIDEO && filters[i].module_phase==NORMAL) {
            current_buffer_planar_list = (*filters[i].module_apply_function)(current_buffer_planar,Config.out_width*Config.out_height,(gulong)SCRframe,(glong)pts);
            current_buffer_planar[0] = scaledbuffer[0];
            current_buffer_planar[1] = scaledbuffer[1];
            current_buffer_planar[2] = scaledbuffer[2];
        }
    }

    /* --- Plugin Filters, type POST VIDEO --- */
    for (gint i=0;i<filters_index;i++) {
        if (filters[i].active==TRUE && filters[i].module_type==VIDEO && filters[i].module_phase==POST) {
            current_buffer_planar_list = (*filters[i].module_apply_function)(current_buffer_planar,Config.out_width*Config.out_height,(gulong)SCRframe,(glong)pts);
            current_buffer_planar[0] = scaledbuffer[0];
            current_buffer_planar[1] = scaledbuffer[1];
            current_buffer_planar[2] = scaledbuffer[2];
        }
    }

    /* === HANDLE PROCESSED FRAME === */
    if ( current_buffer != scaledbuffer[0] ) {
        fast_memcpy(scaledbuffer[0],current_buffer_planar[0],Config.out_width*Config.out_height);
        fast_memcpy(scaledbuffer[1],current_buffer_planar[1],Config.out_width*Config.out_height/4);
        fast_memcpy(scaledbuffer[2],current_buffer_planar[2],Config.out_width*Config.out_height/4);
    }

    /* Store current frame to frame buffer */
    encode_frame(stream,im,(gdouble)pts,FALSE,FALSE,PTSaudio);
    /* Add audio if available.. this will also add a frame from the frame buffer when audio is around */
    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Add audio if available from main encoding loop");
    #endif
    result = TRUE;
    while (result == TRUE) {
        result = output_audio(totalframes);
    }

    /* -- FEEDBACK -- */
    /* Notify GUI with progress every ?100? frames */
    if ((totalframes_countdown--)==0) {
        totalframes_countdown = 49;
        #ifndef STANDALONE
        gui_progress(totalframes,todo_frames);//,j,numfiles);
        #else
        fprintf(stderr,"\rEncoding frame %i of %i ",totalframes,todo_frames);
        /* Display encoding info */
        gettimeofday(&tv,&tz);
        fps=totalframes/((tv.tv_sec+tv.tv_usec/1000000.0)-(startsec+startusec/1000000.0));
        fprintf(stderr,"at %.2gfps  |",fps);
        gint i;
        gint done = (gint)((gdouble)40*((gdouble)totalframes/(gdouble)todo_frames))%40;
        for (i=0;i<done;i++) {
            fprintf(stderr,"=");
        }
        for (i=done;i<40;i++) {
            fprintf(stderr,"-");
        }
        fprintf(stderr,"| ");
        #endif
    }
    #ifndef STANDALONE
    /* Writeout current frame, at every 500 frames minimal */
    if ( (writeout_frames_countdown--) == 0 ) {
        /* Next writeout */
        writeout_frames_countdown = writeout_frames_countdown_amount;
        if (writeout_frames_countdown_amount<1000) {
            writeout_frames_countdown_amount+=25;
        } else {
            writeout_frames_countdown_amount = 999;
        }
        /* Convert current frame to RGB */
        I420toRGB24((guint8*)ToRGBbuffer,(guchar*)scaledbuffer[0],bminfo_in.biWidth,labs(bminfo_in.biHeight));
        /* Writeout frame */
        gui_frame(ToRGBbuffer,totalframes,bminfo_in.biWidth,labs(bminfo_in.biHeight));
    }
    #endif

    /* Reached endframe (and honor amount of frames) ? */
    if ((totalframes > (todo_frames-1)) && (hard_todo_frame) ) {
        quit = TRUE;
    }
    return;
}


/*
 * Main encoding function
 */
gpointer encode(gpointer data) {
    arg_parmS *arg_parm = (arg_parmS*)data;
    gint argc = arg_parm->argc;
    gchar **argv = arg_parm->argv;
    GString *eta = g_string_new("");
    //time_t Time;// = time(NULL);
    gulong seconds;
    gulong minutes;
    gulong hours;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Starting encoding process");
    /* Renice */
    if ((arg_parm->nice<=19) && (arg_parm->nice>=-20)) {
        result = nice(arg_parm->nice);
        if (result!=0) {
            g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Renice failed for value %i, %s",arg_parm->nice,strerror(errno));
        }
    }
    /* Check params */
    if(argc < 4) {
        display_help(argv[0]);
    }
    /* Fill audio structure for both channels */
    for (gint anum=0;anum<AMAX;anum++) {
        au[anum].in_stream = -1;
        au[anum].codec = 0x55;   // Default to MP3 audio 
        au[anum].mp3bitrate = -1;
        au[anum].achan = 0;
        au[anum].rate = 0;
        au[anum].samples = 0;
        au[anum].outstream = NULL;
        au[anum].volume_adjustment = 1; 
    }
    /* Set audio channel 0 to default */
    au[0].in_stream = 0; 
    au[1].in_stream = -1;
    Config.anum = 1; // Default to 1 audio channel
    autoclip = FALSE;
    /* Fill codec structure */
    opendivx_params.rc_period=0;
    opendivx_params.rc_reaction_period=0;
    opendivx_params.rc_reaction_ratio=0;
    opendivx_params.max_key_interval=0;
    opendivx_params.obmc=0;
    opendivx_params.quality=5;
    opendivx_params.max_quantizer=12;
    opendivx_params.min_quantizer=2;
    /* Parse parameters */
    moreopts = TRUE;
    argbase = 0;
    while(moreopts) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Handling parameter %s",argv[argbase+1]);
        moreopts=FALSE;
        if(strcmp(argv[argbase+1],"-cl")==0) {
            clipleft=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-cr")==0) {
            clipright=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-b")==0) {
            divxbitrate=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-f")==0) {
            video_codec=fccDIV4;
            argbase++;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-d")==0) {
            video_codec=fccDIVX;//mmioFOURCC('d', 'i', 'v', '3');//1482049860;
            argbase++;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-dq")==0) {
            opendivx_params.quality=atoi(argv[argbase+2]);
            argbase++;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-dminq")==0) {
            opendivx_params.min_quantizer=atoi(argv[argbase+2]);
            argbase++;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-dmaxq")==0) {
            opendivx_params.max_quantizer=atoi(argv[argbase+2]);
            argbase++;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-a")==0) {
            au[0].mp3bitrate=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-a2")==0) {
            au[1].mp3bitrate=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-av")==0) {
            au[0].volume_adjustment=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-av2")==0) {
            au[1].volume_adjustment=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-r")==0) {
            aspect=atof(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-e")==0) {
            endframe=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-E")==0) {
            hard_todo_frame = FALSE;
            argbase+=1;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-c")==0) {
            video_codec=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
	if(strcmp(argv[argbase+1],"-m")==0) {
            #ifndef STANDALONE
            drip_lock("encode1");
            #endif
            video_codev_module=g_string_assign(video_codev_module,argv[argbase+2]);
            #ifndef STANDALONE
            drip_unlock("encode1");
            #endif
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-au")==0) {
            au[0].codec=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if (AMAX>1)
            if(strcmp(argv[argbase+1],"-au2")==0) {
            au[1].codec=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=TRUE;
        }
        if(strcmp(argv[argbase+1],"-ct")==0) {
            cliptop=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-cb")==0) {
            clipbottom=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-ip")==0) {
            deinterlace = TRUE;
            argbase+=1;
            moreopts=true;
            g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Activated PAL deinterlace filter");
        }
        //        if(strcmp(argv[argbase+1],"-in")==0) {
        //            pulldown = TRUE;
        //            argbase+=1;
        //            moreopts=true;
        //            g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Activated NTSC reverse 3:2 pulldown. NOT YET READY!");
        //        }
        if(strcmp(argv[argbase+1],"-ac")==0) {
            autoclip = TRUE;
            argbase+=1;
            moreopts=true;
        }

        if(strcmp(argv[argbase+1],"-acf")==0) {
            autoclip_frames = atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-w")==0) {
            Config.set_width=atoi(argv[argbase+2]);
            scale = TRUE;
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-h")==0) {
            Config.set_height=atoi(argv[argbase+2]);
            scale = TRUE;
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-as")==0) {
            au[0].in_stream=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }
        if (AMAX>1) {
            if(strcmp(argv[argbase+1],"-as2")==0) {
                au[1].in_stream=atoi(argv[argbase+2]);
                argbase+=2;
                moreopts=true;
                Config.anum = 2;
            } 
        }
        if(strcmp(argv[argbase+1],"-vs")==0) {
            vistream=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-s")==0) {
            subpicture=atoi(argv[argbase+2]);
            clutfile= argv[argbase+3];
            argbase+=3;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-l")==0) {
            limit=1;
            max_size=atoi(argv[argbase+2]);
            argbase+=2;
            moreopts=true;
        }

        if(strcmp(argv[argbase+1],"-A")==0) {
            argbase+=1;
            moreopts=true;
        }
        if(strcmp(argv[argbase+1],"-F")==0) {
            argbase+=1;
            moreopts=true;
        }
    }
    /* Chunk size */
    if (max_size<50000) {
        /* Stop polling loops */
        encoding = TRUE;
        usleep(15000);
        #ifndef STANDALONE
        gui_ready();
        #endif
        encoding = FALSE;
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Max size too small, use at least 50000kb");
        return 0;
    } else {
        /* Adjust max size so max_size_new + header =~ max_size */
        max_size=max_size-50000;
    }
    /* Input files */
    numfiles=argc-argbase-2;
    for(gint i=argbase+1; i<argc-1; i++) {
        #ifndef STANDALONE
        drip_lock("encode2");
        #endif
        inputfiles[i-argbase-1] = g_string_new(argv[i]);
        #ifndef STANDALONE
        drip_unlock("encode2");
        #endif
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Input %d out %s",i-argbase-1,inputfiles[i-argbase-1]->str);
    }
    if ((inputfiles[0] == NULL) || (inputfiles[0]->str == NULL)) {
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Input or Output filename missing in parameters");
    }
    outputfile=argv[argc-1];
    if (outputfile == NULL) {
        g_log(DRIP_LD,G_LOG_LEVEL_ERROR,"Output filename missing in parameters");
    }
    /* Todo frames */
    todo_frames = endframe;
    /* Various varis' */
    #ifndef STANDALONE
    drip_lock("encode3");
    #endif
    output = g_string_new("");
    framefile = g_string_new("");
    g_string_sprintf(framefile,"%s/.drip/frame.raw",getenv("HOME"));
    #ifndef STANDALONE
    drip_unlock("encode3");
    #endif
    frame_buffer_index = 0;
    frame_buffer_base = 0;
    /* Init filters */
    autoclipping(autoclip_frames);
    /* Notify user begin status, and fetch start time */
    #ifndef STANDALONE
    gui_progress(0,todo_frames);//,j,numfiles);
    #else
    fprintf(stderr,"\rEncoding frame %i of %i",0,todo_frames);
    #endif 
    /* Inint time */
    gettimeofday(&tv,&tz); /* Get start time for FPS */
    startsec=tv.tv_sec;
    startusec=tv.tv_usec;
    totalframes = 0;

    /* 
     * Encoding starts
     */

    /* Open mpeg stream (video+audio) */
    demuxing = TRUE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opening %s",inputfiles[0]->str);
    file = drip_mpeg_open(inputfiles[0]->str,process_frame);
    if (!file) {
        /* Error, not able to open stream properly */
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Unable to open %s, possibly not a plain MPEG2 or valid IFO file",inputfiles[0]->str);
        encoding = TRUE;
        #ifndef STANDALONE
        gui_ready();
        #endif
        encoding = FALSE;
        return 0;
    }
    encoding = TRUE;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Opening input AV stream from %s",inputfiles[0]->str);
    if (!file) {
        /* Error, not able to open input stream properly */
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Unable to open %s, possibly not a plain MPEG2 or valid IFO file",inputfiles[0]->str);
        encoding = TRUE;
        #ifndef STANDALONE
        gui_ready();
        #endif
        /* Close streams */
        stream->Stop();
        delete avifile;
        drip_mpeg_close(FALSE);
        /* Encoding done */
        encoding = FALSE;
    }
    /* Reset total frames counter */
    totalframes_countdown = 49;

    /* -- Start demuxer (decoding + filtering + encoding process) -- */
    quit = FALSE;
    gint **in_streams = (gint**)malloc(2*sizeof(gint*));
    in_streams[0] = (gint*)malloc(sizeof(gint));        *in_streams[0] = au[0].in_stream;
    in_streams[1] = (gint*)malloc(sizeof(gint));        *in_streams[1] = au[1].in_stream;
    drip_mpeg_start_demuxing(in_streams);
    /* ------------------------------------------------------------- */


    /* Stop & close encoding */
    #ifndef STANDALONE
    drip_lock("Destructing video encoding codec");
    #endif
    if (stream != NULL) stream->Stop();
    #ifndef STANDALONE
    drip_unlock("Destructing video encoding codec");
    drip_lock("Destructing encoder object");
    #endif
    delete avifile;
    #ifndef STANDALONE
    drip_unlock("Destructing encoder object");
    //drip_lock("Destructing mpeg decoder");
    #endif
    drip_mpeg_close(FALSE);
    #ifndef STANDALONE
    //drip_unlock("Destructing mpeg decoder");
    #endif
    /* Display encoding info */
    gettimeofday(&tv,&tz);
    fps=totalframes/((tv.tv_sec+tv.tv_usec/1000000.0)-(startsec+startusec/1000000.0));
    seconds = (gulong)(tv.tv_sec-startsec);
    hours = (guint)(seconds/3600);
    seconds -= 3600*hours;
    minutes = (guint)(seconds/60);
    seconds -= 60*minutes;
    #ifndef STANDALONE
    drip_lock("encode4");
    #endif
    eta = g_string_new("");
    if (hours>0) {
        g_string_sprintf(eta,"%li hours %li minutes",hours,minutes);
    } else {
        g_string_sprintf(eta,"%li minutes %li seconds",minutes,seconds);
    }
    #ifndef STANDALONE
    drip_unlock("encode4");
    fprintf(stderr,"\n");
    #endif
    if (frames_doubled > 0) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"To keep video in sync %li frames were added",frames_doubled);
    }
    if (frames_dropped > 0) {
        g_log(DRIP_LD,G_LOG_LEVEL_INFO,"To keep video in sync %li frames were dropped",frames_dropped);
    }
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Finished, %li frames in %s at %.2g fps",totalframes,eta->str,fps);
    #ifndef STANDALONE
    drip_lock("encode5");
    #endif
    g_string_free(output,TRUE);
    for (gint i=0;i<frame_buffer_index_MAX;i++) {
        free(frame_buffer_slots[i].plane);
    }
    #ifndef STANDALONE
    drip_unlock("encode5");
    #endif

    /* Notify GUI encoding has finished */
    #ifndef STANDALONE
    gui_ready();
    #endif
    /* Clean & Exit */
    free(framebuffer);framebuffer = NULL;
    free(extrabuffer);extrabuffer = NULL;
    delete(rowptr);
    free(clippedbuffer[0]);
    free(clippedbuffer[1]);
    free(clippedbuffer[2]);
    //free(f_buffer_bak[0]);
    //free(f_buffer_bak[1]);
    //free(f_buffer_bak[2]);
    delete(scaledbuffer[0]);
    free(_RGBbuffer);
    delete(im);
    if (pulldown==TRUE) pulldown_cleanup();
    delete(progress_frame_im);
    free(ToRGBbuffer);
    //free(deinterlace_buffer);  .. leaking
    #ifndef STANDALONE
    drip_lock("encode6");
    #endif
    g_string_free(eta,TRUE);
    #ifndef STANDALONE
    drip_unlock("encode6");
    #endif
    ///* Deallocate frame buffer */
    //for (gint i=0;i<frame_buffer_index_MAX;i++) {
    //    free(frame_buffer[i].plane);
    //}
    frame_raw = NULL;
    frame_buffer_index = -1;
    frame_buffer_base = 0;
    writeout_frames_countdown_amount = 10;
    writeout_frames_countdown = 10;
    /* Destruct plugins (NOT unload them) */
    #ifdef STANDALONE
    close_plugins();
    #endif
    /* Encoding done */
    encoding = FALSE;
    audio_output_codec_setup = FALSE;
    first_frame = TRUE;
    #ifdef STANDALONE
    return NULL;
    #else
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Exiting from encoder.cpp");
    return NULL;//pthread_exit(0);
    #endif
}

