package imagedir_base;
use strict;
use error;
use diskless;
1;

# Get a ordered list of options that the user may set
sub getoptiondescn
{
	my $self = shift;
	my $n = shift;
	my $i = 0;
	return $self->getoptiondesc("nfsserver")  if ($n == $i); $i++;
	return $self->getoptiondesc("nfsimagedir")  if ($n == $i); $i++;
	return $self->getoptiondesc("nfshostsdir")  if ($n == $i); $i++;
	return $self->getoptiondesc("nfshomedir") if ($n == $i); $i++;
	return undef;
}

# Get description for user settable option $id
sub getoptiondesc
{
	my $self = shift;
	my $id = shift;

	return
	{
		id => "nfsserver",
		prompt => "NFS server address",
		default => "192.168.87.129",
		valid => '^.*$',
		help => 
"This specifies the address of the central NFS server where to mount
partitions from on the clients. It is recommended that this
contain the IP address, although this is not required.
",
	} if ($id eq "nfsserver");

	return
	{
		id => "nfsimagedir",
		prompt => "NFS image directory mount point",
		default => "/var/lib/diskless/default/root",
		valid => '^.*$',
		help => 
"This options specifies the path which will be used by the clients
to mount the NFS image directory.
",
	} if ($id eq "nfsimagedir");

	return
	{
		id => "nfshostsdir",
		prompt => "Hosts' directories mount point",
		default => "/var/lib/diskless/default",
		valid => '^.*$',
		help => 
"This options specifies the path which will be used by the clients
to their private read-write directories.  The IP address
will be appended to the end of this path.
",
	} if ($id eq "nfshostsdir");

	return
	{
		id => "nfshomedir",
		prompt => "NFS exported home directories",
		default => "/home",
		valid => '^.*$',
		help => 
"This specifies where /home on the clients should be mounted from your
NFS server. Examples are /home, /tftpboot/home, etc. This directory must
be exported on your NFS server via /etc/exports, and will be mounted as
/home. Set this value to \"none\", if you don't want to mount /home (in
this case, /home will be an empty read-only directory).
",
	} if ($id eq "nfshomedir");

	# no option $id exists
	return undef;
}

# Get value of user settable option
sub getoption
{
	my $self=shift;
	my $option=shift;

	if (defined($self->getoptiondesc($option)))
	{
		return($self->{$option});
	}
	else
	{
		return undef;
	}
}

# Get the REAL value, no matter where it may be stored.
# (in this case, it is the same thing)
sub getoptionvalue
{
	my $self=shift;
	my $option=shift;
	return($self->getoption($option));
}

# Set the value of user settable option.
sub setoption
{
	my $self=shift;
	my $option=shift;
	my $value=shift;

	if (defined($self->getoptiondesc($option)))
	{
		$self->{$option}=$value;
		return 1;
	}
	else
	{
		return 0;
	}
}

#------------------------------------------
# get non-usable settable option that is frequently required
sub get_dirname_usrtemplate
{
        my $self=shift;
        return $self->{"imagedir"}."/usr/lib/diskless-image/template";
}

sub get_filename_usrtemplaterules
{
        my $self=shift;
        return $self->{"imagedir"}."/usr/lib/diskless-image/rules-template";
}

sub get_dirname_imagedir
{
        my $self=shift;
        return $self->{"imagedir"};
}

sub get_filename_imagedirrules
{
        my $self=shift;
        return $self->{"imagedir"}."/usr/lib/diskless-image/rules-image";
}

sub get_filename_config
{
        my $self=shift;
        return $self->{"imagedir"}."/etc/diskless-image/config";
}

sub get_filename_configm4
{
        my $self=shift;
        return $self->{"imagedir"}."/etc/diskless-image/config.m4";
}

sub get_dirname_host
{
        my $self=shift;
	my $ip=shift;
        return $self->{"nfshostsdir"}."/$ip";
}


# create a new object
sub new {
        my $proto = shift;
        my $class = ref($proto) || $proto;
        my $self = {
		imagedir => undef,
        };
        bless($self,$class);
        return $self;
}

# Get object, if it exists
sub get
{
	my $class=$_[0];
	my $imagedir=$_[1];
	my $error=$_[2];

	die if (!ref($error) or defined ($$error));

	my $rc;

	if ( ! -d $imagedir )
	{
		$$error="error"->new("error"->notfound,
			"Group directory $imagedir does not exist");
		return undef;
	}

	if ( ! -d "$imagedir/etc/diskless-image" )
	{
		$$error="error"->new("error"->general,
			"Not an NFS-root image directory: $imagedir/etc/diskless-image doesn't exist or isn't a directory");
		return undef;
	}

	my $self=$class->new;
	$self->{"imagedir"} = $imagedir;

	my $config = $self->get_filename_config;
	if ( ! -f $config )
	{
		return $self;
	}

	$rc = open(FILE,$config);
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot open $config for reading: $!");
		return undef;
	}
	while (<FILE>)
	{
		if (!(/^([A-Za-z_0-9]+)=(.*)/))
		{
			$$error="error"->new("error"->general,
				"$config contains an invalid entry");
			return undef;
		}
		if ( $self->setoption($1,$2) )
		{
			# All done
		}
	}

	$rc = close(FILE);
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot close $config: $!");
		return undef;
	}
	return $self;
}

# Save values of user-settable options
sub save
{
	my $self=$_[0];
	my $error=$_[1];

	die if (!ref($error) or defined ($$error));

	my $imagedir = $self->{"imagedir"};
	my $masterdir = $self->{"masterdir"};
	my $nameserver = $self->{"nameserver"};
	my $domain = $self->{"domain"};

	my $config = $self->get_filename_config;
	my $configm4 = $self->get_filename_configm4;
	my $rc = open(FILE,">$config");
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot open $config for writing: $!");
		return 0;
	}

	$rc = open(FILEM4,">$configm4");
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot open $configm4 for writing: $!");
		return 0;
	}
	print FILEM4 "divert(-1)dnl\n";

	my $n=0;
	my $optiondesc = $self->getoptiondescn($n);
	while (defined($optiondesc))
	{
		my $value = $self->getoption($optiondesc->{"id"});
		if (defined($value))
		{
			my $id = $optiondesc->{"id"};
			print FILE $id,"=",$value,"\n";
			print FILEM4 "define(<[image_$id]>,<[$value]>)\n";
		}

		$n++;
		$optiondesc = $self->getoptiondescn($n);
	}

	$rc=close(FILE);
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot close $config: $!");
		return 0;
	}

	print FILEM4 "divert(0)dnl\n";
	$rc=close(FILEM4);
	if (!$rc)
	{
		$$error="error"->new("error"->general,
			"Cannot close $configm4: $!");
		return 0;
	}

	return 1;
}
