/* cfe.c -- the main part of the console font editor
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <config.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <curses.h>
#include <unistd.h>
#include "defines.h"
#include "font.h"
#include "scr.h"
#include "undo.h"
#include "keyb.h"

struct CFE_FONT font[2] = {
    {0,0,0,0,0,F_RAW,FONT_HEIGHT,0,0},
    {0,0,0,0,0,F_RAW,FONT_HEIGHT,0,0}
};

struct CFE_GLYPH glyph[2];

/*------------------------------------------------------------------*/

int main(int argc, char **argv)
{
    char *delim;
    int opt, height, res;
    int is_opt_ascii = 0, is_opt_bw = 0, is_opt_height = 0;
    int is_opt_offset[2] = {0, 0};
    sigset_t sigset;
    
    while ((opt = getopt(argc, argv, "abho:pPs:v")) != EOF) {
        switch (opt) {
        case 'h':
            printf("Console font editor %s\n"
               "Usage: cfe [-abhv] [-oN | -o[N1]:[N2]] [-p | -P] [-sH] file1 [file2]\n"
                   " -a            ASCII line drawing\n"
                   " -b            force black and white display\n"
                   " -h            give this help\n"
                   " -o N          set font offset to position N (default is 0)\n"
                   " -o [N1]:[N2]  set different (optional) offsets for file1 and file2\n"
                   " -p            create .psf font file1 with 256 characters\n"
                   " -P            create .psf font file1 with 512 characters\n"
                   " -s H          set font size to 8xH (default is %d)\n"
                   " -v            display version number\n",
                   VERSION, FONT_HEIGHT);
            return 0;
        case 'a':
            is_opt_ascii = 1;
            break;
        case 'b':
            is_opt_bw = 1;
            break;
        case 'o':
            if ((delim = strchr(optarg, ':')) != 0) {
                *delim = '\0';
                if (*optarg != '\0') {
                    is_opt_offset[0] = 1;
                    font[0].offset = atoi(optarg);
                }
                if (*(delim + 1) != '\0') {
                    is_opt_offset[1] = 1;
                    font[1].offset = atoi(delim + 1);
                }
                *delim = ':';
            } else {
                is_opt_offset[0] = is_opt_offset[1] = 1;
                font[0].offset = font[1].offset = atoi(optarg);
            }
            break;
        case 'p':
            font[0].type = F_PSF;
            font[0].num_of_chars = 256;
            break;
        case 'P':
            font[0].type = F_PSF;
            font[0].num_of_chars = 512;
            break;
        case 's':
            is_opt_height = 1;
            height = font[0].height = font[1].height = atoi(optarg);
            if (height < MIN_FONT_HEIGHT || height > MAX_FONT_HEIGHT) {
                fprintf(stderr, "%s: illegal font height\n", PACKAGE);
                return 1;
            }
            break;
        case 'v':
            printf("cfe-%s\n", VERSION);
            return 0;
        case '?':
            return 1;
        }
    }

    if (optind == argc) {
        fprintf(stderr, "For help, type: cfe -h\n");
        return 1;
    }
    font[0].name = argv[optind];

    optind++;

    if (optind == argc)
        font[1].name = 0;
    else
        font[1].name = argv[optind];

    /* Load the first file (that can be modified) and probably
     * the second one. If the first file doesn't exist
     * it will be created.
     */
    res = load_font_file(&font[0], is_opt_offset[0], is_opt_height);
    if (res == -1) {
        if (!create_font_file(&font[0]))
            return 1;
    } else if (!res) {
        return 1;
    }
    
    if (font[1].name) {
        res = load_font_file(&font[1], is_opt_offset[1], is_opt_height);
        if (res == -1) {
            perror(font[1].name);
            return 1;
        } else if (!res) {
            return 1;
        }
        if (font[0].height != font[1].height) {
            fprintf(stderr, "%s & %s: fonts have different heights\n",
                    font[0].name, font[1].name);
            return 1;
        }
    }

    glyph[0].data = font[0].data;
    glyph[1].data = font[1].data;
    height = font[0].height;
   
    /* Signal handling */

    sigfillset(&sigset);
    sigdelset(&sigset, SIGINT);
    sigdelset(&sigset, SIGTERM);
    sigdelset(&sigset, SIGHUP);
    sigdelset(&sigset, SIGWINCH);
    sigprocmask(SIG_BLOCK, &sigset, 0);

    /* Undo initialization */
    if (!init_undo(&font[0])) {
        perror(PACKAGE);
        return 1;
    }

    /* Console initialization */
    cfe_init_scr(is_opt_ascii, is_opt_bw);
    
#ifdef NCURSES_MOUSE_VERSION            
    /* Mouse initialization */
    mousemask(BUTTON1_CLICKED, 0);
    mouseinterval(100);
#endif  /* NCURSES_MOUSE_VERSION */ 

    /* Initial drawing */
    if (chk_win_size(1)) {
        draw_frame();
        move(5, 2);
    }
    refresh();

    /* Main keyboard loop */
    keyboard_loop();

    return 0;
}


/* EOF */
