/*
 * suite.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2004-09-09 20:00:10 +0200 (Do, 09 Sep 2004) $
 * $LastChangedRevision: 624 $
 */

#include <config.h>

#include "frontend.h"
#include "install.h"
#include "suite.h"
#include "suite_action.h"
#include "suite_config.h"

#include <debian-installer.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>

struct suite_config *suite = NULL;
const char *arch = NULL;
const char *flavour = NULL;
const char *suite_name = NULL;
static const char **include;
static unsigned int include_size;
const char *configdir = NULL;

static void suite_init_check_action (suite_config_action *action)
{
  di_slist_node *node;

  if (action->flavour.head)
    for (node = action->flavour.head; node; node = node->next)
    {
      if (!strcasecmp (flavour, node->data))
      {
        action->activate = true;
        break;
      }
    }
  else
    action->activate = true;
}

static void suite_init_check_section (void *key __attribute__ ((unused)), void *data, void *user_data)
{
  suite_config *config = user_data;
  suite_config_section *section = data;
  di_slist_node *node1, *node2;

  if (section->flavour.head)
    for (node1 = section->flavour.head; node1; node1 = node1->next)
    {
      if (!config->flavour_valid &&
          !strcasecmp (flavour, node1->data))
        config->flavour_valid = true;
      if (!strcasecmp ("any", node1->data) ||
          !strcasecmp (flavour, node1->data))
      {
        section->activate = true;
        break;
      }
    }
  else
    section->activate = true;

  if (section->activate)
    for (node1 = section->packages.head; node1; node1 = node1->next)
    {
      suite_config_packages *packages = node1->data;

      if (packages->arch.head)
      {
        for (node2 = packages->arch.head; node2; node2 = node2->next)
          if (!strcasecmp ("any", node2->data) ||
              !strcasecmp (arch, node2->data))
          {
            packages->activate = true;
            break;
          }
      }
      else
        packages->activate = true;
    }
}

static int suite_check (void);

int suite_init (const char *_suite_name, const char *_arch, const char *_flavour, const char **_include, unsigned int _include_size, const char *_configdir)
{
  arch = _arch;
  flavour = _flavour;
  suite_name = _suite_name;
  include = _include;
  include_size = _include_size;
  configdir = _configdir;

  if (suite_config_init (_suite_name))
    return 1;
  if (suite)
    return suite_check ();
  return 0;
}
  
int suite_init_second (const char *_suite_name)
{
  if (suite_config_init_second (_suite_name))
    return 1;

  suite_name = _suite_name;

  return suite_check ();
}

static int suite_check (void)
{
  di_slist_node *node;

  if (!strncmp ("minimal", flavour, sizeof ("minimal") - 1))
    suite->flavour_valid = true;

  for (node = suite->actions.head; node; node = node->next)
    suite_init_check_action (node->data);
  di_hash_table_foreach (suite->sections, suite_init_check_section, suite);

  if (!suite->flavour_valid)
    log_text (DI_LOG_LEVEL_ERROR, "Unknown flavour %s", flavour);

  return 0;
}

int suite_install (di_packages *packages, di_packages_allocator *allocator, di_slist *install)
{
  return suite_action (packages, allocator, install);
}

struct suite_packages_list_user_data
{
  di_slist list;
  di_packages *packages;
  bool select_priority_required;
};

static void suite_packages_list_add (di_packages *packages, const char *name, di_slist *list)
{
  di_package *p;
  p = di_packages_get_package (packages, name, 0);
  if (p)
    di_slist_append (list, p);
  else
    log_text (DI_LOG_LEVEL_WARNING, "can't find package: %s", name);
}

static void suite_packages_list_sections (void *key __attribute__ ((unused)), void *data, void *_user_data)
{
  suite_config_section *section = data;
  struct suite_packages_list_user_data *user_data = _user_data;
  di_slist_node *node1, *node2;

  if (section->activate)
    for (node1 = section->packages.head; node1; node1 = node1->next)
    {
      suite_config_packages *packages = node1->data;

      if (packages->activate)
        for (node2 = packages->packages.head; node2; node2 = node2->next)
        {
          const char *name = node2->data;
          if (!strcmp (name, "priority-required"))
            user_data->select_priority_required = true;
          else
            suite_packages_list_add (user_data->packages, name, &user_data->list);
        }
    }
}

static void suite_packages_list_packages (void *key __attribute__ ((unused)), void *data, void *_user_data)
{
  di_package *p = data;
  struct suite_packages_list_user_data *user_data = _user_data;

  if (p->essential || (user_data->select_priority_required && p->priority == di_package_priority_required))
    /* These packages will automatically be installed */
    di_slist_append (&user_data->list, p);
}

di_slist *suite_packages_list (di_packages *packages, di_packages_allocator *allocator)
{
  di_slist *install;
  struct suite_packages_list_user_data user_data = { { NULL, NULL }, packages, false };
  unsigned int i;

  di_hash_table_foreach (suite->sections, suite_packages_list_sections, &user_data);
  di_hash_table_foreach (packages->table, suite_packages_list_packages, &user_data);

  for (i = 0; i < include_size; i++)
    suite_packages_list_add (packages, include[i], &user_data.list);

  install = di_packages_resolve_dependencies (packages, &user_data.list, allocator);

  di_slist_destroy (&user_data.list, NULL);

  return install;
}

