//  bbpal -- The Blackbox Slit Pal -- for Blackbox
//  Copyright (c) 1999-2000 by Charles Nusbaum, nusbaucj@wooster.edu
//  Base code from John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)

#include "bbpal.hh"
#include "../version.h"

ToolWindow::ToolWindow(int argc, char **argv, struct CMDOPTIONS *options) : 
Basewindow(argc, argv, options) {
	XrmInitialize();

	iargc = argc;
	iargv = argv;

	resource = new Resource(this);
	MakeWindow(False);

	palPtr = 0;
	if (!palName)
		palName = resource->pal.name;

	timer = new BTimer(getCurrentScreenInfo()->getBaseDisplay(), this);
	timer->setTimeout(1000 * resource->pal.delay);
	timer->start();

	LoadPalArray();
	Redraw();
	eventLoop();
}

ToolWindow::~ToolWindow() {
	XUnmapWindow(getXDisplay(), framewin);

	/* destroy pixmaps */
	if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
	if (pixmap.label) getImageControl()->removeImage(pixmap.label);
	
	/* destroy windows */
	XDestroyWindow(getXDisplay(), framewin);

	/* destroy pal pixmaps */
	for (int i = 0 ; i < resource->pal.numImages ; i++)
		XFreePixmap(getXDisplay(), palXpmArray[i]);
	
}

void ToolWindow::reconfigure(void) {
	/* destroy pixmaps */
	getImageControl()->removeImage(pixmap.frame);

	resource->Reload();
	MakeWindow(True);
	XClearWindow(getXDisplay(), framewin);
	Redraw();
}

void ToolWindow::MakeWindow(bool reconfigure) {
	XSetWindowAttributes attrib;
	XWMHints wmhints;
	XClassHint classhints;
	XTextProperty windowname;

	unsigned long create_mask = CWBackPixmap|CWBorderPixel|
		CWOverrideRedirect|CWCursor|CWEventMask; 
	
	frame.width = 52;
//	frame.width = resource->pal.width + 2 * resource->frame.bevelWidth;
//	frame.height = resource->pal.height + 2 * resource->frame.bevelWidth;
	frame.height = 52;
//	frame.x = resource->position.x;
//	frame.y = resource->position.y;
	frame.x = 0;
	frame.y = 0;
	
	label.width = 48;
	label.height = 48;
	label.x = 2;
	label.y = 2;

	if (resource->position.mask & XNegative) {
		frame.x = getCurrentScreenInfo()->getWidth() + 
			resource->position.x - frame.width;
	}

	if (resource->position.mask & YNegative) {
		frame.y = getCurrentScreenInfo()->getHeight() + 
			resource->position.y - frame.height;
	}
	
	if (withdrawn) {
		attrib.override_redirect = False;
		wmhints.initial_state = WithdrawnState;
  	} else {
		attrib.override_redirect = False;
	  	wmhints.initial_state = NormalState;
	}

	attrib.background_pixmap = ParentRelative;

	pixmap.frame = getImageControl()->renderImage(frame.width, 
		frame.height, &resource->frame.texture);
			  								
	pixmap.label = getImageControl()->renderImage(resource->pal.width, 
		resource->pal.height, &resource->label.texture);
											
	attrib.cursor = getSessionCursor();
	attrib.event_mask = ButtonPressMask|ButtonReleaseMask| 
		ExposureMask|FocusChangeMask|StructureNotifyMask|
		SubstructureRedirectMask;

	if (!reconfigure) {
		framewin = XCreateWindow(getXDisplay(),
			getCurrentScreenInfo()->getRootWindow(), 
			frame.x, frame.y, frame.width, frame.height, 
			0, getCurrentScreenInfo()->getDepth(), 
			InputOutput,
			getCurrentScreenInfo()->getVisual(), 
			create_mask, &attrib);
		labelwin = XCreateWindow(getXDisplay(),
			framewin, label.x, label.y, label.width,
			label.height, 0, getCurrentScreenInfo()->getDepth(),
			InputOutput, getCurrentScreenInfo()->getVisual(),
			create_mask, &attrib);
   	} else if (!withdrawn) {
	    	XMoveResizeWindow(getXDisplay(), framewin, frame.x, 
			frame.y, frame.width, frame.height);
		XMoveResizeWindow(getXDisplay(), labelwin, label.x,
			label.y, label.width, label.height);
	} else {
    		XResizeWindow(getXDisplay(), framewin, 
			frame.width, frame.height);
		XResizeWindow(getXDisplay(), labelwin,
			label.width, label.height);
  	}

	char *name = BBTOOL;
	XSizeHints sizehints;

	wmhints.flags = StateHint;

	classhints.res_name = BBTOOL;
	classhints.res_class = "bbtools";

	sizehints.x = frame.x; //getResource()->position.x;
	sizehints.y = frame.y; //getResource()->position.y;
	sizehints.max_width = sizehints.min_width = frame.width;
	sizehints.max_height = sizehints.min_height = frame.height;
	sizehints.flags = USPosition|PMinSize|PMaxSize;

	XStringListToTextProperty(&name, 1, &windowname);
	XSetWMProperties(getXDisplay(), framewin, &windowname, NULL, 
		getArgv(), getArgc(), &sizehints, &wmhints, &classhints);
	Atom wmproto[2];
	wmproto[0]=wm_delete_window;
	wmproto[1]=getNETStructureMessagesAtom();
	XSetWMProtocols(getXDisplay(), framewin, wmproto, 2);

	if (!decorated && !withdrawn) {
		NetHints net_hints;
		net_hints.decoration = NETDecorNone;
		net_hints.attrib = NETAttribOmnipresent;
		net_hints.flags = NETAttribDecoration|NETAttribOmnipresent;
		XChangeProperty(getXDisplay(), framewin, getNETHintsAtom(),
			getNETHintsAtom(), 32, PropModeReplace,
			(unsigned char *) &net_hints,  PropNetHintsElements);
	}

	if (!shape) {
		XSetWindowBackgroundPixmap(getXDisplay(), framewin, 
			pixmap.frame);
	}
		XSetWindowBackgroundPixmap(getXDisplay(), labelwin,
			pixmap.label);

	if (!withdrawn && resource->report.auto_raise) {
		XRaiseWindow(getXDisplay(), framewin);
		lower = False;
	} else lower = True;

	XClearWindow(getXDisplay(), framewin);
	XMapWindow(getXDisplay(), framewin);
	XMapSubwindows(getXDisplay(), framewin);
}

void ToolWindow::CheckConfig() {
	struct stat file_status;

	if (stat(config_filename,&file_status) != 0) {
		fprintf(stderr, "Could not open config file %s\n",
		resource->style.conf_filename);
	} else if (file_status.st_mtime != resource->style.mtime) {
		resource->style.mtime = file_status.st_mtime;
		reconfigure();
	}
}

void ToolWindow::LoadPalArray() {
	xpmAttributes.valuemask = XpmCloseness;
	xpmAttributes.closeness = 65536;

	for (int i = 0 ; i < resource->pal.numImages ; i++)
	{	sprintf(currentPal, "%s/%s%d.xpm", resource->pal.directory, 
			palName, i + 1);
		XpmReadFileToPixmap(getXDisplay(), labelwin, currentPal, 
			&palXpmArray[i], 0, &xpmAttributes);
	}

}

void ToolWindow::ChangePal(bool redraw) {

	if (palPtr == resource->pal.numImages)
		palPtr = 1;
	else
		palPtr++;

	XSetWindowBackgroundPixmap(getXDisplay(), labelwin, 
		palXpmArray[palPtr - 1]);

	Redraw();

//	XFreePixmap(getXDisplay(), palpixmap);
//	XpmFreeAttributes(&xpmAttributes);
}
  
void ToolWindow::timeout() {
	ChangePal(True);
}

void ToolWindow::Redraw() { 
	XClearWindow(getXDisplay(), framewin); 
	XClearWindow(getXDisplay(), labelwin); 
}

void ToolWindow::process_event(XEvent *Event) {
	switch (Event->type) {
		case PropertyNotify: {
		}
		break;
		case ClientMessage: {
			if ((unsigned) Event->xclient.data.l[0] == wm_delete_window)
				shutdown();
		}
		break;
		case ButtonPress: {
		}
		break;
		case ButtonRelease: {
		}
		break;
		case MotionNotify: {
		}
		break;
		case ConfigureNotify: {
		}
		break;
		case Expose: {
			ChangePal(True);
		}
		break;
	}
}
