------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--             G N A T C H E C K . G L O B A L _ S T A T E . C G            --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2005-2006, AdaCore                     --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the interface routines for building and analyzing
--  the call graph

with Asis;
with Gnatcheck.Rules; use Gnatcheck.Rules;

package Gnatcheck.Global_State.CG is

   procedure Add_CG_Info
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State);
   --  This procedure is supposed to be called during the element traversal
   --  as a part of pre-operation for each element being visited. It adds to
   --  the global structure the information corresponding to the given Element.
   --  It also updates the auxiliary data structures used for the call graph
   --  construction (scope stack?)
   --
   --  For each piece of the stored call graph information this procedure
   --  should store the full "trace" of the information needed to build a full
   --  cal graph. That is, if we process a subprogram call, we store the
   --  declaration of the called subprogram, and we also check for the source
   --  containing the body, and if it is not in the set of sources to be
   --  processed by gnatcheck, we try to locate this source and add to the
   --  source table
   --
   --  We need Control and State parameters because this procedure may start
   --  its own subtraversals.

   procedure Complete_CG_Info (El : Asis.Element);
   --  This procedure is supposed to be called during the element traversal
   --  as a part of post-operation for each element being visited. It completes
   --  the information stored in the global structure for the given Element.
   --  It also updates the auxiliary data structures used for the call graph
   --  construction (scope stack?)

   procedure Transitive_Closure;
   --  This procedure should be called when all the "local" call information is
   --  already stored in the global data structures. It performs the transitive
   --  closure of the call graph. After the call to this procedure, for
   --  each entity (call graph node) we have a full set of all calls/callers.
   --
   --  Call graph transitive closure is a time-consuming operation, At the
   --  moment we use the following approach:
   --
   --  - First, we build the call graph as a part of ASIS traversing of all the
   --    units. As a result we get a representation of the call graph as a
   --    table of nodes, and for each node we have a set of direct calls
   --
   --  - Then, for each node, we compute a set of nodes that are called by this
   --    node, directly or indirectly. This is done by workpile algorithm.
   --
   --  - As soon as a set of nodes called by the given node is computed, we
   --    fill in the corresponding row of Gnatcheck.Global_State.Matrix
   --
   --  - In the very end we have Gnatcheck.Global_State.Matrix as representing
   --    the transitive closure of the gnarph connectivity matrix.

   procedure Set_Used_Entities;
   --  This procedure is supposed to be called after the call to
   --  Transitive_Closure. It traverses the call graph and sets Is_Used flags
   --  for the nodes. (A node used (called) by unused nodes only is itself
   --  considered as unused.)

   procedure Set_Priorities;
   --  This procedure is supposed to be called when all the priorities defined
   --  by pragmas are already computed. It sets the priorities for all the
   --  other nodes using the following rules:
   --
   --  * each task object has the priority of the corresponding task
   --    definition (represented by A_Task_Type or A_Task_Object node in the
   --    call graph);
   --
   --  * each protected operation has the priority of the corresponding
   --    protected definition;
   --
   --  * for all the other entities, the priority is taked from statically
   --    enclosing scope.

   procedure Set_Side_Effect;
   --  This procedure is supposed to be called after the call to
   --  Transitive_Closure. It traverses the call graph and for the nodes for
   --  that the side effect status is not defined yet because of the calls to
   --  subprograms for that their side effect was not known at the moment of
   --  creating the corresponding call graph nodes, tries to define their side
   --  effect. (What a terrible wording)

   function Is_Called (N : GS_Node_Id) return Boolean;
   --  Checks if N is called by some other node

end Gnatcheck.Global_State.CG;
