/*
   ArrayUtil, program to check and configure a SmartArray controller
   Copyright (C) 1999  Hugo Trippaers

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
   $Header: /home/cvs/cvsroot/array-util/discover.c,v 1.13 2001/08/30 12:49:09 spark Exp $
 */

#include <unistd.h>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <ida_ioctl.h>
#include "array_util.h"
#include "utils.h"

#define ISSET( var, bit) ((var & bit) == bit)


/* structure to pass to setinfo and unsetinfo */
struct gen_info 
{
  GString *info;
  int ctrl;
  int logdrv;
  int bus;
  int physdrv;
};


int select_ida (const struct dirent *);
int interrogate_controller (GtkWidget *, int);
int interrogate_logical (GtkWidget *, int, int);
void boardid2str (unsigned long, char *);
int discover_physdisks (GtkWidget *, int, int, int);
static void setinfo (GtkWidget *, struct gen_info *);
static void unsetinfo (GtkWidget *, struct gen_info *);
int blink(int, int, __u8 [256]);


/* there is no device for the controller itself
   so i use the first disk for getting controller info
 */
char *controllers[] =
{
  "/dev/ida/c0d0",
  "/dev/ida/c1d0",
  "/dev/ida/c2d0",
  "/dev/ida/c3d0",
  "/dev/ida/c4d0",
  "/dev/ida/c5d0",
  "/dev/ida/c6d0",
  "/dev/ida/c7d0"
};

/* raid level descriptions */
char *raidlvlstr[] =
{
  "No Fault Tolerance",
  "RAID 4",
  "RAID 1 (Mirroring)",
  "RAID 5",
  "RAID 5+1",
  "Unknown RAID code (5)",
  "Unknown RAID code (6)"
};



/* keep track of the drives that are used by a logical disk */
__u16 drives_in_use[8];


int
discover_controllers (GtkWidget * tree)
{
  GtkWidget *nocontlabel;

  int cntr;
  int foundone = 0;

  for (cntr = 0; cntr < 8; cntr++)
    {
      /* does this device exist ? */
      if ((access (controllers[cntr], R_OK | F_OK)) == 0)
	{
	  /* it does :) */
	  if (interrogate_controller (tree, cntr))
	    {
	      foundone = 1;
#ifdef DEBUG
	      fprintf (stderr, "DEBUG: %s is a existing controller\n",
		       controllers[cntr]);
#endif
	    }
	}
      else
	{
#ifdef DEBUG
	  /* it doesn't */
	  fprintf (stderr, "Device %s could not be opened\n", controllers[cntr]);
	  perror ("reason");
#endif
	}
    }

  if (!foundone)
    {
      /* display the 'no controller' label */
      nocontlabel = mygtk_tree_item_with_label_and_pixmap ("No Controllers Found",
							   icon_error_image,
							 icon_error_bitmap);
      gtk_tree_insert (GTK_TREE (tree), GTK_WIDGET (nocontlabel), 1);

      gtk_widget_show (nocontlabel);
    }
  return 1;
}

int
interrogate_controller (GtkWidget * tree, int contrnum)
{
  int devicefd;
  ida_ioctl_t io;
  char buffer[30];
  GtkWidget *controller;
  GtkWidget *ctrl_subtree;
  GtkWidget *nologdrv;
  GtkWidget *unasgndrv;
  GtkWidget *unasgndrvtree;
  struct gen_info *controllerinfo;
  int foundone = 0;
  int cntr;
  __u16 unused_drives;


  devicefd = open (controllers[contrnum], O_RDONLY);
  /* no checks, did that before */

  /* clear io */
  memset (&io, 0, sizeof (io));

  io.cmd = ID_CTLR;

  if (ioctl (devicefd, IDAPASSTHRU, &io) < 0)
    {
#ifdef DEBUG
      /* usually this means that the file exists but the major device is
       * is not found in the kernel. So there is no controller here..
       */
      perror ("ioctl");
#endif
      return 0;
    }

#ifdef DEBUG
  fprintf (stderr, "DEBUG: nr_drvs=%d\n", io.c.id_ctlr.nr_drvs);
  for (cntr = 0; cntr < 8; cntr++)
    {
      fprintf (stderr, "DEBUG: big_drv_present_map = 0x%04X\n", io.c.id_ctlr.big_drv_present_map[cntr]);
      fprintf (stderr, "DEBUG: big_ext_drv_map = 0x%04X\n", io.c.id_ctlr.big_ext_drv_map[cntr]);
      fprintf (stderr, "DEBUG: big_non_disk_map = 0x%04X\n", io.c.id_ctlr.big_non_disk_map[cntr]);
    }
#endif

  boardid2str (io.c.id_ctlr.board_id, buffer);

  controller = mygtk_tree_item_with_label_and_pixmap (buffer,
						      icon_ctrl_image,
						      icon_ctrl_bitmap);
  gtk_tree_append (GTK_TREE (tree), GTK_WIDGET (controller));
  gtk_widget_show (controller);

  ctrl_subtree = gtk_tree_new ();
  gtk_tree_item_set_subtree (GTK_TREE_ITEM (controller),
			     ctrl_subtree);

  /* now i have to associate some info with this controller */
  strcat ( buffer, "\n");
  controllerinfo = (struct gen_info *)malloc(sizeof(struct gen_info));
  controllerinfo->ctrl = contrnum;
  controllerinfo->logdrv = -1;
  controllerinfo->bus = -1;
  controllerinfo->physdrv = -1;
  controllerinfo->info = g_string_new (buffer);
  g_string_sprintfa (controllerinfo->info, "\tFirmware Revision : %c%c%c%c\n", 
		    io.c.id_ctlr.firm_rev[0],
		    io.c.id_ctlr.firm_rev[1],
		    io.c.id_ctlr.firm_rev[2],
		    io.c.id_ctlr.firm_rev[3]);
  g_string_sprintfa (controllerinfo->info, "\tRom Revision      : %c%c%c%c\n",
		    io.c.id_ctlr.rom_rev[0],
		    io.c.id_ctlr.rom_rev[1],
		    io.c.id_ctlr.rom_rev[2],
		    io.c.id_ctlr.rom_rev[3]);

  gtk_signal_connect (GTK_OBJECT (controller), "select",
		      GTK_SIGNAL_FUNC (setinfo), controllerinfo);
  gtk_signal_connect (GTK_OBJECT (controller), "deselect",
		      GTK_SIGNAL_FUNC (unsetinfo), controllerinfo);

  ctrls_found[ctrls_found_num].ctrl_object = controller;
  strncpy (ctrls_found[ctrls_found_num].ctrl_devicename, 
	   controllers[contrnum], 20);
  ctrls_found[ctrls_found_num].num_logd_found = 0;

  for (cntr = 0; cntr < io.c.id_ctlr.nr_drvs; cntr++)
    {
      if (interrogate_logical (ctrl_subtree, devicefd, cntr))
	{
	  foundone = 1;
	}
    }

  ctrls_found_num++;

  if (!foundone)
    {
      nologdrv = mygtk_tree_item_with_label_and_pixmap ("No Logical Drives Found",
							icon_error_image,
							icon_error_bitmap);
      gtk_tree_append (GTK_TREE (ctrl_subtree), GTK_WIDGET (nologdrv));
      gtk_widget_show (nologdrv);
    }



  unused_drives = io.c.id_ctlr.big_drv_present_map[0] ^ drives_in_use[0];
  if (unused_drives != 0)
    {
      unasgndrv = mygtk_tree_item_with_label_and_pixmap ("Unassigned Drives",
							 icon_logdrv_image,
							 icon_logdrv_bitmap);
      gtk_tree_append (GTK_TREE (ctrl_subtree), unasgndrv);
      gtk_widget_show (GTK_WIDGET (unasgndrv));
      unasgndrvtree = gtk_tree_new ();
      gtk_tree_item_set_subtree (GTK_TREE_ITEM (unasgndrv),
				 unasgndrvtree);

      for (cntr = 0; cntr < 16; cntr++)
	{
	  if (ISSET (unused_drives, 1 << cntr))
	    {
	      discover_physdisks (unasgndrvtree, devicefd, 0, cntr);
	    }
	}
    }

  close (devicefd);
  return 1;
}

int
interrogate_logical (GtkWidget * tree, int devicefd, int unit_nr)
{
  ida_ioctl_t io;
  ida_ioctl_t io2;

  char buffer[16];
  GtkWidget *logdrive;
  GtkWidget *disk_subtree;
  struct gen_info *controllerinfo;
  int cntr, bus;

#ifdef DEBUG
  printf ("DEBUG: interrogating unit %d\n", unit_nr);
#endif

  memset (&io, 0, sizeof (io));

  io.cmd = ID_LOG_DRV;
  io.unit = unit_nr | UNITVALID;

  if (ioctl (devicefd, IDAPASSTHRU, &io) < 0)
    {
      perror ("ID_LOG_DRV ioctl");
      return 0;
    }

  memset (&io2, 0, sizeof (io2));

  io2.cmd = SENSE_CONFIG;
  io2.unit = unit_nr | UNITVALID;

  if (ioctl (devicefd, IDAPASSTHRU, &io2) < 0)
    {
      perror ("SENSE_CONFIG ioctl");
      return 0;
    }

  sprintf (buffer, "Logical Drive %d", unit_nr);

  logdrive = mygtk_tree_item_with_label_and_pixmap (buffer,
						    icon_logdrv_image,
						    icon_logdrv_bitmap);
  gtk_tree_append (GTK_TREE (tree), logdrive);
  gtk_widget_show (GTK_WIDGET (logdrive));

  ctrls_found[ctrls_found_num].logd_found[unit_nr].logd_object = logdrive;
  ctrls_found[ctrls_found_num].logd_found[unit_nr].laststatus = -1;
  ctrls_found[ctrls_found_num].logd_found[unit_nr].lastpvalue = 0.0;
  ctrls_found[ctrls_found_num].num_logd_found++;

  /* now i have to associate some info with this controller */
  controllerinfo = (struct gen_info *)malloc(sizeof(struct gen_info));
  controllerinfo->ctrl = -1;
  controllerinfo->logdrv = unit_nr;
  controllerinfo->bus = -1;
  controllerinfo->physdrv = -1;
  controllerinfo->info = g_string_new (NULL);

  gtk_signal_connect (GTK_OBJECT (logdrive), "select",
		      GTK_SIGNAL_FUNC (setinfo), controllerinfo);
  gtk_signal_connect (GTK_OBJECT (logdrive), "deselect",
		      GTK_SIGNAL_FUNC (unsetinfo), controllerinfo);

  g_string_sprintfa (controllerinfo->info, "Logical Unit %d:\n", unit_nr);
  g_string_sprintfa (controllerinfo->info, "\tSize:  %2.2fGb\n",
		     (double) io.c.id_log_drv.nr_blks *
		     (int) io.c.id_log_drv.blk_size / 1024 / 1024 / 1024);
  g_string_sprintfa (controllerinfo->info , "\tRAID:  %s\n",
		     raidlvlstr[io.c.id_log_drv.fault_tol]);
  g_string_sprintfa (controllerinfo->info , "\tDisabled in Bios:  %s\n",
		     io.c.id_log_drv.bios_disable ? "True" : "False");

  disk_subtree = gtk_tree_new ();
  gtk_tree_item_set_subtree (GTK_TREE_ITEM (logdrive),
			     disk_subtree);

  


  /* big_drv_map[0] contains a bitmap of the positions of the physical disk */
  /* printf ("big_drv_map[%d] = 0x%04x\n", cntr, io2.c.config.big_drv_map[cntr]); */

  /* clear drives_in_use */
  memset(drives_in_use, 0, sizeof(drives_in_use));

  for (bus = 0; bus<8; bus++) {
    /* array can have multiple busses */  
    
    drives_in_use[bus] |= io2.c.config.big_drv_map[bus];
    
    for (cntr = 0; cntr < 16; cntr++)
      {
	if (ISSET (io2.c.config.big_drv_map[bus], 1 << cntr))
	  {
	    discover_physdisks (disk_subtree, devicefd, bus, cntr);
	  }
      }
    
  }
  
  return 1;
}


void
boardid2str (unsigned long board_id, char *name)
{

  switch (board_id)
    {

    case 0x0040110E:		/* IDA */
      strcpy (name, "Compaq IDA");
      break;
    case 0x0140110E:		/* IDA-2 */
      strcpy (name, "Compaq IDA-2");
      break;
    case 0x1040110E:		/* IAES */
      strcpy (name, "Compaq IAES");
      break;
    case 0x2040110E:		/* SMART */
      strcpy (name, "Compaq SMART");
      break;
    case 0x3040110E:		/* SMART-2/E */
      strcpy (name, "Compaq SMART-2/E");
      break;
    case 0x40300E11:		/* SMART-2/P or SMART-2DH */
      strcpy (name, "Compaq SMART-2/P (2DH)");
      break;
    case 0x40310E11:		/* SMART-2SL */
      strcpy (name, "Compaq SMART-2SL");
      break;
    case 0x40320E11:		/* SMART-3200 */
      strcpy (name, "Compaq SMART-3200");
      break;
    case 0x40330E11:		/* SMART-3100ES */
      strcpy (name, "Compaq SMART-3100ES");
      break;
    case 0x40340E11:		/* SMART-221 */
      strcpy (name, "Compaq SMART-221");
      break;
    case 0x40400E11:		/* Integrated Array */
      strcpy (name, "Compaq Integrated Array");
      break;
    case 0x40500E11:		/* Smart Array 4200 */
      strcpy (name, "Compaq Smart Array 4200");
      break;
    case 0x40510E11:		/* Smart Array 4250ES */
      strcpy (name, "Compaq Smart Array 4250ES");
      break;
    case 0x40580E11:		/* Smart Array 431 */
      strcpy (name, "Compaq Smart Array 431");
      break;
    default:
      /*
       * Well, its a SMART-2 or better, don't know which
       * kind.
       */
      strcpy (name, "Unknown Controller Type");
    }
}

int
discover_physdisks (GtkWidget * tree, int devicefd, int bus, int unit)
{

  GtkWidget *disklabel;
  char buffer[40];
  struct gen_info *physdrvinfo;
  ida_ioctl_t io;


  memset (&io, 0, sizeof (io));

  io.cmd = ID_PHYS_DRV;
  io.blk = (0x80 | (bus << 4) | (unit)) << 24;

  if (ioctl (devicefd, IDAPASSTHRU, &io) < 0)
    {
      perror ("ID_PHYS_DRV ioctl");
      return 0;
    }

  strncpy (buffer, (char *) io.c.id_phys_drv.drv_model, 40);
  buffer[40] = '\0';		/* just in case... */

#ifdef DEBUG
  fprintf (stderr, "DEBUG: drive model = %s %d/%d\n", buffer,
	   (short) io.c.id_phys_drv.scsi_bus,
	   (short) io.c.id_phys_drv.scsi_id);
#endif

  disklabel = mygtk_tree_item_with_label_and_pixmap (buffer,
						     icon_physdrv_image,
						     icon_physdrv_bitmap);
  gtk_tree_append (GTK_TREE (tree), disklabel);
  gtk_widget_show (GTK_WIDGET (disklabel));

  /* physcal drive info */
  physdrvinfo = (struct gen_info *)malloc(sizeof(struct gen_info));
  physdrvinfo->ctrl = -1;
  physdrvinfo->logdrv = -1;
  physdrvinfo->bus = bus;
  physdrvinfo->physdrv = unit;
  physdrvinfo->info = g_string_new (NULL);

  g_string_sprintfa (physdrvinfo->info, "Physical Disk:\n");
  g_string_sprintfa (physdrvinfo->info, "\tSize:  %2.2fGb\n",
		     (double) io.c.id_phys_drv.nr_blks *
		     (int) io.c.id_phys_drv.blk_size / 1024 / 1024 / 1024);
  g_string_sprintfa (physdrvinfo->info, "\tSCSI Bus: %d\n", io.c.id_phys_drv.scsi_bus);
  g_string_sprintfa (physdrvinfo->info, "\tSCSI Id : %d\n", io.c.id_phys_drv.scsi_id);


  gtk_signal_connect (GTK_OBJECT (disklabel), "select",
		      GTK_SIGNAL_FUNC (setinfo), physdrvinfo);
  gtk_signal_connect (GTK_OBJECT (disklabel), "deselect",
		      GTK_SIGNAL_FUNC (unsetinfo), physdrvinfo);

  return 1;

}

static void 
setinfo (GtkWidget * item, struct gen_info *geninfo)
{
  int fd;
  __u8 drives[256];
  int cntr;
  
  /* set info in text box */
  gtk_text_freeze (GTK_TEXT (infotxt));
  gtk_text_insert (GTK_TEXT (infotxt), infofont, NULL, NULL,
		   geninfo->info->str, geninfo->info->len);
  gtk_text_thaw (GTK_TEXT (infotxt));
  /* blink the drive */
  if (geninfo->ctrl >= 0) {
    /* it's a controller so blink all drives */
    fd = open (controllers[geninfo->ctrl], O_RDONLY);
    /* no check (yet) */
    for (cntr=0; cntr<256; cntr++) {
      drives[cntr] = 0x1;
    }
#ifdef DEBUG
    fprintf(stderr, "blink called...\n");
#endif
    blink (fd, 300, drives);
    close(fd);
  }
  else if (geninfo->logdrv >= 0) {
    /* it's a logical drive blink drives in use for this logical drive */
  }
  else {
    /* it's a physical drive blink it.. */
  }
  
}

static void 
unsetinfo (GtkWidget * item, struct gen_info *geninfo)
{
  int fd;
  int cntr;
  __u8 drives[256];

  gtk_text_freeze (GTK_TEXT (infotxt));
  gtk_text_set_point (GTK_TEXT (infotxt), 0);
  gtk_text_forward_delete (GTK_TEXT (infotxt),
			   gtk_text_get_length (GTK_TEXT (infotxt)));
  gtk_text_thaw (GTK_TEXT (infotxt));
  /* stop blinking the drive */
  if (geninfo->ctrl >= 0) {
    /* it's a controller so stop blinking all drives */
    fd = open (controllers[geninfo->ctrl], O_RDONLY);
    /* no check (yet) */
    for (cntr=0; cntr<256; cntr++) {
      drives[cntr] = 0;
    }
#ifdef DEBUG
    fprintf(stderr, "(un)blink called...\n");
#endif
    blink (fd, 300, drives);
    close(fd);
  }
  else if (geninfo->logdrv >= 0) {
    /* it's a logical drive blink drives in use for this logical drive */
  }
  else {
    /* it's a physical drive blink it.. */
  }

}

gint status_check (gpointer data) 
{
  int devicefd;
  int ctrl_cntr;
  int logd_cntr;
  ida_ioctl_t io, io2;
  int status, nr_blks, blks_tr;
  gfloat pvalue; /* progress value in % */
  gchar *message, *statusmsg;
  
  
  for ( ctrl_cntr=0;
	ctrl_cntr <  ctrls_found_num;
	ctrl_cntr++) {
    
    devicefd = open (controllers[ctrl_cntr], O_RDONLY);
    
    for ( logd_cntr=0;
	  logd_cntr < ctrls_found[ctrl_cntr].num_logd_found;
	  logd_cntr++) {
      
        memset (&io, 0, sizeof (io));

	io.cmd = SENSE_LOG_DRV_STAT;
	io.unit = logd_cntr  | UNITVALID;
	
	if (ioctl (devicefd, IDAPASSTHRU, &io) < 0)
	  {
	    perror ("SENSE_LOG_DRV_STAT ioctl");
	    return 0;
	  }

	status=io.c.sense_log_drv_stat.status;
	
	if ((status == 3) || (status == 5) || (status == 7)) {
	  memset (&io2, 0, sizeof (io));
	  
	  io2.cmd = ID_LOG_DRV;
	  io2.unit = logd_cntr  | UNITVALID;
	  
	  if (ioctl (devicefd, IDAPASSTHRU, &io2) < 0)
	    {
	      perror ("ID_LOG_DRV ioctl");
	      /* return 0;   no return this isn't fatal for now */
	    }
	  else 
	    {
	      nr_blks = io2.c.id_log_drv.nr_blks;
	      blks_tr = io.c.sense_log_drv_stat.blks_to_recover;
	      	  
	      pvalue = ((gfloat)(nr_blks - blks_tr)/(gfloat)nr_blks) * 100;

	    }
	}
	else {
	  pvalue = 0.0;
	}


	if (ctrls_found[ctrl_cntr].logd_found[logd_cntr].laststatus !=
            status) {
           /* Status change */
  	   message = (gchar *)malloc(2048);
	   statusmsg = (gchar *)malloc(2048);

	   sprintf(statusmsg, statusstr[status], ctrl_cntr, logd_cntr, pvalue);
           insert_event((GtkWidget *)GTK_CLIST(data), status, statusmsg);	
       }
       else if (pvalue - ctrls_found[ctrl_cntr].logd_found[logd_cntr].lastpvalue > 25) {
	   /* Status change */
           message = (gchar *)malloc(2048);
           statusmsg = (gchar *)malloc(2048);

           sprintf(statusmsg, statusstr[status], ctrl_cntr, logd_cntr, pvalue);
           insert_event((GtkWidget *)GTK_CLIST(data), status, statusmsg);
       }  
       ctrls_found[ctrl_cntr].logd_found[logd_cntr].laststatus = status;
       ctrls_found[ctrl_cntr].logd_found[logd_cntr].lastpvalue = pvalue;
    }
    
    close (devicefd);
  }
  
  return TRUE;
  
}

int blink(int cntlr, int duration, __u8	blink[256])
{
  ida_ioctl_t io;
  int cntr;

  memset (&io, 0, sizeof(io));
  
  io.cmd = BLINK_DRV_LEDS;
  io.c.blink_drv_leds.blink_duration = duration;
  for (cntr = 0; cntr < 256; cntr++) {
    io.c.blink_drv_leds.blink[cntr] = blink[cntr];
  }
  
  if (ioctl (cntlr, IDAPASSTHRU, &io) < 0){
#ifdef DEBUG
    perror ("ioctl");
    return 0;
#endif
  }
  
  return 1;
}

  


