/*
 * ARCload (c) 2005 Stanislaw Skowronek
 */

#include <arclib/arc.h>
#include <arclib/stddef.h>
#include <arclib/stdio.h>
#include <arclib/stdlib.h>
#include <arclib/types.h>
#include <arclib/string.h>
#include <arclib/spb.h>

/* ARCS system -> label translation */
static struct {
	char *name;
	char *label;
} archs[]={
	{"SGI-IP22", "ip22"},
	{"SGI-IP28", "ip28"},
	{"SGI-IP30", "ip30"},
	{"SGI-IP32", "ip32"},
	{NULL, NULL}
};

/* CPU PRId -> label translation */
static struct {
	unsigned short ident;
	char *label;
} cpus[]= {
	{0x0004, "r4000"},
	{0x0009, "r10000"},
	{0x000e, "r12000"},
	{0x000f, "r14000"},
	{0x0020, "r4600"},
	{0x0021, "r4700"},
	{0x0023, "r5000"},
	{0x0027, "rm7000"},
	{0x0028, "rm5200"},
	{0, NULL}
};

/* IP27-specific stuff */

#ifdef M64

#define LOCAL_HUB_NI_STATUS_REV_ID *((volatile unsigned long *)0x9200000001600000UL)
unsigned get_nasid(void)
{
	return (LOCAL_HUB_NI_STATUS_REV_ID>>8)&0x1FF;
}

#define NODE_UNCAC_BASE(nasid)	(0x9600000000000000UL|(((unsigned long)(nasid))<<32))
#define KLDIR_BASE(nasid)	((kldir_t *)(NODE_UNCAC_BASE(nasid)|0x2000))
#define KLI_KLCONFIG		1
#define KLDIR_MAGIC		0x434d5f53505f5357
#define KLCFGINFO_MAGIC		0xbeedbabe

typedef struct kldir_s {
	unsigned long	magic;		/* Indicates validity of entry      */
	unsigned long	offset;		/* Offset from start of node space  */
	unsigned long	pointer;	/* Pointer to area in some cases    */
	unsigned long	size;		/* Size in bytes 		    */
	unsigned long	count;		/* Repeat count if array, 1 if not  */
	unsigned long	stride;		/* Stride if array, 0 if not        */
	char		rsvd[16];	/* Pad entry to 0x40 bytes          */
} kldir_t;

typedef struct klconf_s {
	unsigned long	ch_magic;	/* set this to KLCFGINFO_MAGIC */
	unsigned int	ch_version;    /* structure version number */
	unsigned int	ch_malloc_hdr_off; /* offset of ch_malloc_hdr */
	unsigned int	ch_cons_off;       /* offset of ch_cons */
	unsigned int	ch_board_info;	/* the link list of boards */
	unsigned int	ch_cons_info;	/* address info of the console */
	/* the rest doesn't matter for ARCLoad */
} klconf_t;

static klconf_t *find_klconf(void)
{
	unsigned nasid=get_nasid();
	if(KLDIR_BASE(nasid)[KLI_KLCONFIG].magic!=KLDIR_MAGIC)
		return 0;
	return (klconf_t *)((KLDIR_BASE(nasid)[KLI_KLCONFIG].offset)|NODE_UNCAC_BASE(nasid));
}

#endif

/* actual detection code - system */

#ifdef M64

static char *archwalkklconf(void)
{
	klconf_t *klconf=find_klconf();
	if(!klconf)
		return NULL;
	if(klconf->ch_magic!=KLCFGINFO_MAGIC)
		return NULL;
/* FIXME */
	return "ip27";
}

#endif

static char *archnametolabel(char *name)
{
	int i;
	for(i=0;archs[i].label;i++) {
		if(!strcmp(name,archs[i].name))
			return archs[i].label;
	}
	return NULL;
}

static char *subdetectarch(COMPONENT *root)
{
	COMPONENT *index;
	char *ret;
	if(!root)
		return NULL;
	if(root->Class==SystemClass && root->Type==ARC)
		return archnametolabel(root->Identifier);
	for(index=root;index;index=ArcGetPeer(index)) {
		ret=subdetectarch(ArcGetChild(index));
		if(ret)
			return ret;
	}
	return NULL;
}
static char *detectarch(void)
{
#ifdef M64
	if(!ArcGetChild(NULL))
		return archwalkklconf();
#endif
	return subdetectarch(ArcGetChild(NULL));
}

/* actual detection code - CPU */

static char *cpupridtolabel(unsigned prid)
{
	int i;
	prid = (prid >> 8) & 0xFFFF;
	for(i=0;cpus[i].label;i++) {
		if(cpus[i].ident==prid)
			return cpus[i].label;
	}
	return NULL;
}

#define read_prid()							\
({ int __res;								\
		__asm__ __volatile__(					\
			"mfc0\t%0, $15\n\t"				\
			: "=r" (__res));				\
	__res;								\
})

static char *detectcpu(void)
{
	return cpupridtolabel(read_prid());
}

/* actual detection code - console */

static char *detectconsole(void)
{
	char *var=ArcGetEnvironmentVariable("Console");
	if(!var)
		return NULL;
	if(var[0]=='g')
		return "video";
	if(var[0]=='d')
		return "serial";
	return NULL;
}

static char *detectserial(void)
{
	char *var=ArcGetEnvironmentVariable("Console");
	if(var && var[0]=='d') {
		if(var[1]=='2')
			return "port2";
		return "port1";
	}
	return NULL;
}

static char baudrate[32];

static char *detectbaud(void)
{
	char *var=ArcGetEnvironmentVariable("Console");
	if(var && var[0]=='d') {
		var=ArcGetEnvironmentVariable("dbaud");
		if(!var)
			return "baud=9600";
		sprintf(baudrate,"baud=%s",var);
		return baudrate;
	}
	return NULL;
}

struct sgi_pvector {
	LONG	ioctl;
	LONG	get_nvram_tab;
	LONG	load_abs;
	LONG	invoke_abs;
	LONG	exec_abs;
	LONG	fs_register;
	LONG	fs_unregister;
	LONG	signal;
	VOID *	(*get_promgfx)(void);
};

struct sgi_promgfx {
	LONG	gfxvector;
	LONG	gfxaddr;
	LONG	unknown1[8];
	LONG	width;
	LONG	height;
	LONG	unknown2[10];
	LONG	state;
};

static char screensize[32];

static char *detectscreen(void)
{
	char *var=ArcGetEnvironmentVariable("Console");
	struct sgi_pvector *pv=SystemParameterBlock->PrivateVector;
	struct sgi_promgfx *pg;
	if(!var || var[0]!='g')
		return NULL;
	if(SystemParameterBlock->PrivateVectorLength<sizeof(struct sgi_pvector))
		return NULL;
	pg=pv->get_promgfx();
	if(!pg)
		return NULL;
	sprintf(screensize,"video=%ux%u",pg->width,pg->height);
	return screensize;
}

/* actual detection code - SMP */

static int subdetectsmp(COMPONENT *root)
{
	COMPONENT *index;
	int ret=0;
	if(!root)
		return 0;
	if(root->Class==ProcessorClass && root->Type==CPU)
		ret++;
	for(index=root;index;index=ArcGetPeer(index))
		ret+=subdetectsmp(ArcGetChild(index));
	return ret;
}

static char cpunum[32];
static int cpucount=0;

static char *detectsmp(void)
{
	if(!cpucount)
		cpucount=subdetectsmp(ArcGetChild(NULL));
	if(!cpucount)
		return "smp";
	if(cpucount>1)
		return "smp";
	return "up";
}

static char *detectcpucount(void)
{
	if(!cpucount)
		cpucount=subdetectsmp(ArcGetChild(NULL));
	if(!cpucount)
		return NULL;
	sprintf(cpunum,"ncpu=%u",cpucount);
	return cpunum;
}

/* actual detection code - integration */

#define ADD_LABEL(str)							\
	do {								\
		char *_tmp=str;						\
		if(_tmp) {						\
			if(*label_count>=n_label) {			\
				printf("Error: N_LABEL exceeded.\r\n"); \
				return; 				\
			}						\
			label[*label_count]=_tmp;			\
			(*label_count)++;				\
		}							\
	} while(0)

void detectsystem(char **label, int n_label, int *label_count)
{
	ADD_LABEL(detectarch());
	ADD_LABEL(detectcpu());
	ADD_LABEL(detectconsole());
	ADD_LABEL(detectserial());
	ADD_LABEL(detectbaud());
	ADD_LABEL(detectscreen());
	ADD_LABEL(detectsmp());
	ADD_LABEL(detectcpucount());
}
