/***************************************************************************
                          netimap4.h  -  IMAPv4 protocol
                             -------------------
    begin                : Wed Feb  7 12:23:00 EET 2001
    copyright            : (C) 2001 by theKompany (www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#ifndef NETIMAP4_H
#define NETIMAP4_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <vector>
#include <imap4uidjar.h>
#include <stlutil.h>
#include <netprotocol.h>

#define TAG_LENGTH      8
#define NO_DELIMITER    "NIL"

typedef struct IMAPIncomingMessage
{
	string message, uid, account;
	unsigned int rcvtime;
	unsigned flags;
	unsigned long size;
} IMAP_MESSAGE;

typedef list<IMAP_MESSAGE> MessagesList;
typedef MessagesList::const_iterator MessagesListIterator;

typedef struct MboxElement
{
  string name;
  unsigned int state;
  string delimiter;
} MBOX_ELEMENT;

typedef list<MBOX_ELEMENT> MboxList;
typedef MboxList::const_iterator MboxListIterator;

class NetIMAP4 : protected NetProtocol, protected STLUtil
{
  public:
		/** The server answer about accepting the command.*/
		enum COMMAND_REZULT {OK=0, NO, BAD, ERROR};
		/** The server state.*/
		enum SERVER_STATE {NON_AUTHENTICATED=0, AUTHENTICATED, SELECTED, LOGOUTED};
		/** The protocol commands.*/
		enum IMAP4_COMMANDS { NO_COMMAND=-1,
		            NOOP=0, CAPABILITY, LOGOUT, LOGIN, AUTHENTICATE, SELECT,
		            EXAMINE, CREATE, DELETE, RENAME, SUBSCRIBE, UNSUBSCRIBE,
		            LIST, LSUB, STATUS, APPEND, CHECK, CLOSE, EXPUNGE, SEARCH,
		            FETCH, STORE, COPY, UID };
		/** The protocol flags.*/
		enum IMAP4_FLAGS {NO_FLAGS=0, Seen=1, Answered=2, Flagged=4, Draft=8, Recent=16, Deleted=32, Other=64};
		
		/** The protocol access types.*/
		enum IMAP4_ACCESS { READ_WRITE=true, READ_ONLY=false};
		
		/** The folders and mailboxes list state.*/
		enum IMAP4_MAILBOX_STATE { NoState=0, NoInferiors=1, NoSelect=2, Marked=4, UnMarked=8};
		
		/** Set the connection speed category.*/
		NetIMAP4(const char *category);
		~NetIMAP4();
		
		/** Connect to the IMAP server.*/
		bool connect(const string &host, int port=143);
		/** Get the mailbox name.*/
		string mailbox() const;
		/** Get the server state.*/
		unsigned state() const;
		/** Get the last error.*/
		string error() const;
		/** Check if the client gets the end of response.*/
		bool checkEndOfResponse(const string&) const;
		/** Process the server answer.*/
		bool process(string&);
		
		/**
		  * Parse the server answer for the command status.
		  * TODO: It will generate an exception
		  */
		int commandResult(string&);
		/** Get the command tag.*/
		string getTag() const;
		/** Get the command tag and increment the tag number.*/
		string commandTag();
		
		//Capability
		/** Set the authenticate attribute.*/
		void setAuth(const string &auth);
		/** Get the authenticate attribute.*/
		string auth() const;
		
		// Flags methods.
		/** Set permanent flags.*/
		void setPermanentFlags(string& flags);
		/** Get the permanent flags.*/
		unsigned permanentFlags() const;
		/** Set the flags.*/
		void setFlags(string &flags);
		/** Get the flags.*/
		unsigned flags() const;
		/** Set the access flag.*/
		void setAccess(bool access=true);
		/** Get the access flag.*/
		bool access() const;
		
		// Info attributes methods
		/** Set the UIDVALIDITY.*/
		void setUidValidity(unsigned long uidValidity);
		/** Get the UIDVALIDITY.*/
		unsigned long uidValidity() const;
		/** Set the messages number.*/
		void setMessagesNumber(unsigned long messagesNumber);
		/** Get the messages number.*/
		unsigned long messagesNumber() const;
		/** Set the unseen message number.*/
		void setUnseen(unsigned long unseen);
		/** Get the unseen message number.*/
		unsigned long unseen() const;
		/** Set the recent messages number.*/
		void setRecent(unsigned long recent);
		/** Get the recent messages number.*/
		unsigned long recent() const;
		/** Set the next message uid.*/
		void setUidNext(unsigned long uidNext);
		/** Get the next message uid.*/
		unsigned long uidNext() const;
		
		// ------------ IMAPv4 Commands ----------------
		/**
		  * Capability.
		  * Ask the server what authenticate does it support.
		  */
		bool capability();
		
		/**
		  * Noop.
		  * Just a command for keeping alive the connection,
		  * but the server can send the last changes like an
		  * answer.
		  */
		bool noop();
		
		/**
		  * Login.
		  * It has 2 parameters:
		  *   - user name
		  *   - password
		  * The server will be in authenticated state.
		  */
		bool login(const string&, const string&);
		
		/**
		  * Logout.
		  * It close the connection and the server will in
		  * the logout state.
		  */
		bool logout();
		
		/**
		  * Close.
		  * It close the connection and the server will in
		  * the logout state.
		  */
		bool close();
		
		/**
		  * Expunge.
		  * Delete all marked messages.
		  */
		bool expunge();
		
		/**
		  * Status.
		  * Get the actual status of the IMAP folder (mailbox).
		  */
		bool status(string fields="");
		
		/**
		  * Select.
		  * It has one parameter:
		  *   - mailbox name
		  * It select a mailbox and the server will send some
		  * information about it.
		  * The server will be in the selected state.
		  */
    bool select(const string &mboxName="INBOX");

    /**
      * Uid.
      * UID SEARCH ALL and UID SEARCH NEW
      */
    bool uid(string subCommand);

    /**
      * FETCH.
      */
    bool fetch(unsigned long message, string subCommand);

    /**
      * APPEND.
      * Add a new mail to the mailbox.
      */
    bool append(const string &message, const string &mboxName="INBOX");

    /**
      * CREATE.
      * Create a new mailbox.
      */
    bool create(const string &mboxName);

    /**
      * DELETE.
      * Delete a mailbox.
      * It doesn't make unsubscribe!
      */
    bool delete_(const string &mboxName);

    /**
      * RENAME.
      * Rename the mailbox.
      * It doesn't change the subscribed name.
      * The INBOX rename will move all the mails into the new mailbox and
      * INBOX is still available.
      */
    bool rename(const string &newName, const string &oldName="INBOX");

    /**
      * SUBSCRIBE.
      * Subscribe a mailbox.
      */
    bool subscribe(const string &mailbox);

    /**
      * UNSUBSCRIBE.
      * Unsubscribe a mailbox.
      */
    bool unsubscribe(const string &mailbox);

    /**
      * LIST.
      * Return a list of folders and mailboxes.
      * A path = "" means 'use the selected mailbox by default'
      * The folders parameter can have wildchars.
      */
    bool list(const string &path="\"\"", const string &folders="*");

    /**
      * LSUB.
      * Return a list of subscribed mailboxes.
      * A path = "" means 'use the selected mailbox by default'
      * The folders parameter can have wildchars.
      */
    bool lsub(const string &path="\"\"", const string &folders="*");


    // --------- IMAPv4 parser methods -------------
    /**
      * Find the next word and return the first position after it.
      */
    unsigned parseNextWord(string &strWord, string &line, unsigned uPos=0) const;

    /**
      * Parse the untaged lines.
      */
    bool parseLine(string &line);

    /**
      * Parse the line result.
      */
    bool parseLineResult(string &line, bool bTaggedLine=false);

    /**
      * Parse the flags (permanent or not).
      */
    unsigned parseFlags(string &flags);

    /**
      * Parse the capability answer.
      */
    void parseCapability(string &capability);

    /**
      * Parse the FETCH answer.
      */
    bool parseFetch(string &line);

    /**
      * Parse the SEARCH answer.
      */
    void parseSearch(string &line);

    /**
      * Parse the STATUS answer.
      */
    void parseStatus(string &line);

    /**
      * Parse the LIST/LSUB answer.
      */
    void parseList(string &line);

    /** Get the first word from line like an unsigned long.*/
    unsigned long parseUnsignedLong(string &line);

    /** Debug method.*/
    void showStatus();

    /** Clear.*/
    void clear();

    /** The UID list result of the UID SEARCH command.*/
    UIDMap searchList;
    /** The unsigned long vector result of the SEARCH command.*/
    ULongVect searchVect;
    /** The list which contains the fetched messages.*/
    MessagesList fetchMessages;

    /** The mailboxes list as a result of the LIST/LSUB command.*/
    MboxList mboxList;

  private:
		/** The server state.*/
		unsigned _state;
		/** The mailbox name.*/
		string _mailbox;
		/** The tag name.*/
		string _tagName;
		/** The tag number.*/
		unsigned _tagNo;
		/** The last error text.*/
		string imap_error;
		/** The current command.*/
		int _command;
		/**
		  * The current sub command.
		  * See: UID FETCH and UID SEARCH
		  */
		int _subCommand;
		/** Authenticate protocol.*/
		string _auth;
		
		// Flags
		/** Permanent flags.*/
		unsigned _permanentFlags;
		/** Flags.*/
		unsigned _flags;
		/** Access flag.*/
		bool _access;
		
		//Info
		unsigned long _uidValidity;
		unsigned long _messagesNumber;
		unsigned long _unseen;
		unsigned long _recent;
		unsigned long _uidNext;
};

#endif




