/***************************************************************************
                          netimap4.cpp  -  IMAPv4 protocol
                             -------------------
    begin                : Wed Feb  7 12:23:00 EET 2001
    copyright            : (C) 2001 by theKompany <www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#include <netimap4.h>
#include <stdio.h>
#include <assert.h>

NetIMAP4::NetIMAP4(const char *category)
	: NetProtocol(category)
{
  _mailbox="";
  _tagNo=0;
  _tagName="AETH";//from Aethera :)
  _state=NetIMAP4::NON_AUTHENTICATED;
  imap_error="";
  _permanentFlags=0;
  _flags=0;
  _access=true;// Read and Write
  _uidValidity=0;
  _messagesNumber=0;
  _unseen=0;
  _recent=0;
  _uidNext=0;
  _command=NetIMAP4::NO_COMMAND;
  _subCommand=NetIMAP4::NO_COMMAND;
}

NetIMAP4::~NetIMAP4()
{
	// clear all the lists
	searchList.clear();
	searchVect.clear();
	mboxList.clear();
}


bool NetIMAP4::connect(const string &host, int port)
{
	string rep;
	
	// connect
	NetProtocol::connect(host, port);
	
	// check connection
	if(!NetProtocol::isConnected())
	{
		imap_error=netsock.error();
		return false;
	}
	
	// set line mode
	setMode(Read_One_Line);
	
	// server greeting message
	(*this)>>rep;

	if( NetIMAP4::OK != commandResult(rep) )
	{
		imap_error=rep.empty()?string("no response from server"):rep;
		
		printf("\ncommand result --- disconect\n");
		fflush(stdout);
		
		disconnect();
		return false;
	}
	
	_state=NetIMAP4::NON_AUTHENTICATED;
	return true;
}

unsigned NetIMAP4::state() const
{
  return _state;
}

string NetIMAP4::mailbox() const
{
  return _mailbox;
}

string NetIMAP4::getTag() const
{
  string command;
  char szNo[5];// "9999" + "\0"

  // Ex: tag = "AETH" + "0001"
  command=_tagName;
  assert( _tagNo<9999 );
  sprintf(szNo,"%.4u",_tagNo);
  command+=szNo;

  // increment the command number
  return command;
}

string NetIMAP4::commandTag()
{
  // increment the tag number
  _tagNo++;
  return getTag();
}

bool NetIMAP4::checkEndOfResponse(const string &answer) const
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word
  string tag, result;

  if( answer.size()<3 )
    return true;

  // check if it is finished with \r\n
  if( answer.substr(answer.length()-2, 2) != "\r\n" )
    return true;

  // Check the last line for the corresponding tag
  unsigned uFind=answer.rfind("\r\n", answer.length()-3, 0);
  if( answer.npos==uFind )
    result=answer;
  else
    result=answer.substr(uFind+2);
  //Get the tag.
  tag=getTag();
  // pass over the tag
  uPos=parseNextWord(strWord, result);
  result.erase(0,uPos);
  // check the tag
  if( !strWord.compare(tag) )
    return false;
  // check if it is a greetings message
//  if( oneLine && !strWord.compare('*') )
//    return false;
  return true;
}

int NetIMAP4::commandResult(string &answer)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word
  string tag,result="";

//    printf("\n------------- command result --------------------\n");
//    fflush(stdout);

  // check the greetings message from the server
  if( _state==NetIMAP4::NON_AUTHENTICATED &&
      _command==NetIMAP4::NO_COMMAND )
  {
    if( answer.npos!=answer.find("OK") )
      return NetIMAP4::OK;
    else
      return NetIMAP4::ERROR;
  }

  // Check the last line for the result
  unsigned uFind=answer.rfind("\r\n");
  if( answer.npos==uFind )
    result=answer;
  else
    result=answer.substr(uFind+2);
  //Get the tag.
  tag=getTag();
  // pass over the tag
  uPos=parseNextWord(strWord, result);
  result.erase(0,uPos);
  if( strWord.compare(tag) )
  {
    imap_error="An unknown tag or a wrong answer from server :(";
    return NetIMAP4::ERROR;
  }
  else
  {
    // check the result
    uPos=parseNextWord(strWord, result);
    result.erase(0,uPos);
    // convert it into an uppercase string
    toUpper(strWord);

    if( !strWord.compare("OK") )
    {
      if( parseLineResult(result, true) )
	      return NetIMAP4::OK;
      else
        return NetIMAP4::ERROR;
    }
    else if( !strWord.compare("NO") )
    {
      // set the error
      imap_error=result;
      printf("\nIMAP ERROR: %s\n", result.c_str());
      fflush(stdout);
      return NetIMAP4::NO;
    }
    else if( !strWord.compare("BAD") )
    {
      // set the error
      imap_error=result;
      printf("\nIMAP ERROR: %s\n", result.c_str());
      fflush(stdout);
      return NetIMAP4::BAD;
    }
  }

  return NetIMAP4::ERROR;
}

string NetIMAP4::error() const
{
  return imap_error;
}

bool NetIMAP4::capability()
{
  string buffer;

  _command=NetIMAP4::CAPABILITY;

  setMode(IMAP4_Read_Line);
  buffer=commandTag()+" CAPABILITY"+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;

  printf("\nCAPABILITY:\n%s\n", buffer.c_str());
  fflush(stdout);

  if( NetIMAP4::OK!=commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nCAPABILITY\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::noop()
{
  string buffer;

  _command=NetIMAP4::NOOP;

  setMode(IMAP4_Read_Line);
  buffer=commandTag()+" NOOP "+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK!=commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nNOOP\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::login(const string &userName, const string &passwd)
{
  string buffer;

  _command=NetIMAP4::LOGIN;

  if( _state!=NetIMAP4::NON_AUTHENTICATED )
  {
    imap_error="login called in an wrong server state";
    return false;
  }

  setMode(IMAP4_Read_Line);
  buffer=commandTag()+" LOGIN "+userName+" "+passwd+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;

  if( NetIMAP4::OK!=commandResult(buffer) )
    return false;

  _state=NetIMAP4::AUTHENTICATED;

  if( !process(buffer) )
  {
    printf("\nLOGIN\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::logout()
{
  string buffer;

  _command=NetIMAP4::LOGOUT;

  buffer=commandTag()+" LOGOUT"+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  _state=NetIMAP4::LOGOUTED;
  _mailbox="";

  if( !process(buffer) )
  {
    printf("\nLOGOUT\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::close()
{
  string buffer;

  _command=NetIMAP4::CLOSE;

  buffer=commandTag()+" CLOSE"+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  _state=NetIMAP4::AUTHENTICATED;
  _mailbox="";

  if( !process(buffer) )
  {
    printf("\nCLOSE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::select(const string &mboxName)
{
  string buffer;

  _command=NetIMAP4::SELECT;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="select called in an wrong server state";
    return false;
  }

  // set the mailbox
  _mailbox=mboxName;

  buffer=commandTag()+" SELECT "+_mailbox+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;

  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  _state=NetIMAP4::SELECTED;
  if( !process(buffer) )
  {
    printf("\nSELECT\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::status(string fields)
{
  string buffer;

  _command=NetIMAP4::STATUS;

  if( !fields.size() )
    buffer=commandTag()+" STATUS "+_mailbox+" (MESSAGES RECENT UNSEEN UIDNEXT UIDVALIDITY)\r\n";
  else
    buffer=commandTag()+" STATUS "+_mailbox+" ("+fields+")\r\n";

  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;
  if( !process(buffer) )
  {
    printf("\nCLOSE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::uid(string subCommand)
{
  string buffer;

  _command=NetIMAP4::UID;

  if( _state!=NetIMAP4::SELECTED )
  {
    imap_error="uid called in an wrong server state";
    return false;
  }

//    printf("\n------------- UID --------------------\n");
//    fflush(stdout);

  buffer=commandTag()+" UID "+subCommand+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nUID\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::fetch(unsigned long messageNr, string subCommand)
{
  string buffer;

  _command=NetIMAP4::FETCH;

  if( _state!=NetIMAP4::SELECTED )
  {
    imap_error="fetch called in an wrong server state";
    return false;
  }

  char szMessage[20];
  sprintf(szMessage, "%lu", messageNr);

  buffer=commandTag()+" FETCH "+szMessage+" "+subCommand+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nFETCH\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::expunge()
{
  string buffer;

  _command=NetIMAP4::EXPUNGE;

  if( _state!=NetIMAP4::SELECTED )
  {
    imap_error="expunge called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" EXPUNGE\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nEXPUNGE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::append(const string &message, const string &mboxName="INBOX")
{
  string buffer;
  char messageSize[100];

  _command=NetIMAP4::APPEND;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="append called in an wrong server state";
    return false;
  }

  sprintf(messageSize,"%u", message.size());
  buffer=commandTag()+" APPEND "+mboxName+" (\\SEEN) {"+messageSize+"}\r\n"+message+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;

  printf("\n Append=%s\n", buffer.c_str());
  fflush(stdout);

  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nAPPEND\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::create(const string &mboxName)
{
  string buffer;

  _command=NetIMAP4::CREATE;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="create called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" CREATE "+mboxName+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nCREATE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::delete_(const string &mboxName)
{
  string buffer;

  _command=NetIMAP4::DELETE;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="delete called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" DELETE "+mboxName+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nDELETE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::rename(const string &mboxNewName, const string &mboxOldName)
{
  string buffer;

  _command=NetIMAP4::RENAME;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="rename called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" RENAME "+mboxNewName+" "+mboxOldName+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nRENAME\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::subscribe(const string &mboxName)
{
  string buffer;

  _command=NetIMAP4::SUBSCRIBE;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="subscribe called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" SUBSCRIBE "+mboxName+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nSUBSCRIBE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::unsubscribe(const string &mboxName)
{
  string buffer;

  _command=NetIMAP4::UNSUBSCRIBE;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="unsubscribe called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" UNSUBSCRIBE "+mboxName+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  if( !process(buffer) )
  {
    printf("\nUNSUBSCRIBE\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::list(const string &path, const string &folders)
{
  string buffer;

  _command=NetIMAP4::LIST;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="list called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" LIST "+path+" "+folders+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  // clear the mailboxes list
  mboxList.clear();

  if( !process(buffer) )
  {
    printf("\nLIST\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::lsub(const string &path, const string &folders)
{
  string buffer;

  _command=NetIMAP4::LSUB;

  if( _state==NetIMAP4::NON_AUTHENTICATED || _state==NetIMAP4::LOGOUTED )
  {
    imap_error="lsub called in an wrong server state";
    return false;
  }

  buffer=commandTag()+" LSUB "+path+" "+folders+"\r\n";
  (*this) << buffer;
  buffer="";
  (*this) >> buffer;
  if( NetIMAP4::OK != commandResult(buffer) )
    return false;

  // clear the mailboxes list
  mboxList.clear();

  if( !process(buffer) )
  {
    printf("\nLSUB\n Error: %s\n", error().c_str());
    fflush(stdout);
  }

  return true;
}

bool NetIMAP4::process(string &answer)
{
  unsigned linesNo=0, i=0;
  bool bRet=true;
	/** The server answer.*/
	vector<string> allLines;

	if( _command==NetIMAP4::FETCH || _subCommand==NetIMAP4::FETCH )
    bRet=parseFetch(answer);
	else
	{
    // Split the answer in lines (delimited by \r\n)
    allLines=split("\r\n",answer);

    linesNo=allLines.size();
    // process all the lines without the last one
    for(vector<string>::iterator it=allLines.begin(); it!=allLines.end() && i<linesNo-1 && bRet; it++)
    {
      i++;
      if( (*it).data()[0]=='*' )
        bRet=parseLine((*it));
// TODO
//      else
//        printf("\nUnimplemented feature! :(\n");
    }
	}

  return bRet;
}

unsigned NetIMAP4::parseNextWord(string &strWord, string &line, unsigned uPos) const
{
  unsigned i;

  skipSpaces(line, uPos);
  for(i=uPos; i<line.size(); i++)
    if( line.data()[i]==' ' || line.data()[i]=='\t' ||
        line.data()[i]=='{' || line.data()[i]=='}'  ||
        line.data()[i]=='(' || line.data()[i]==')'  ||
        line.data()[i]=='[' || line.data()[i]==']' )
      break;

  if( i==uPos )
    strWord="";
  else
    strWord=line.substr(uPos, i-uPos);

  return i;
}

bool NetIMAP4::parseLine(string &line)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word

  imap_error="";

  uPos=parseNextWord(strWord, line); // passing over "*"
  line.erase(0, uPos);
  uPos=parseNextWord(strWord, line); // get the first word
  line.erase(0, uPos);
  // convert it into an upper case string
  toUpper( strWord );

  switch( strWord.data()[0] )
  {
    case 'B' :
      if( !strWord.compare("BAD") )
        parseLineResult(line);
      else if( !strWord.compare("BYE") )
      {
        parseLineResult(line);
        _state = NetIMAP4::LOGOUTED;
      }
      else
        imap_error="wrong answer";
      break;

    case 'C' :
      if( !strWord.compare("CAPABILITY") )
        parseCapability(line);
      else
        imap_error="wrong answer";
      break;

    case 'F' :
      if( !strWord.compare("FLAGS") )
        setFlags(line);
      else
        imap_error="wrong answer";
      break;

    case 'L' :
      if( !strWord.compare("LIST") )
        parseList(line);
      else if( !strWord.compare("LSUB") )
        parseList(line);
      else
        imap_error="wrong answer";
      break;

    case 'N' :
      if( !strWord.compare("NO") )
         parseLineResult(line);
      else
        imap_error="wrong answer";
      break;

    case 'O' :
      if( !strWord.compare("OK") )
          parseLineResult(line);
      else
        imap_error="wrong answer";
      break;

    case 'P' :
      if( !strWord.compare("PREAUTH") )
      {
        parseLineResult(line);
        _state=NON_AUTHENTICATED;
      }
      else
        imap_error="wrong answer";
      break;

    case 'S' :
      if( !strWord.compare("SEARCH") )
        parseSearch(line);
      else if( !strWord.compare("STATUS") )
        parseStatus(line);
      else
        imap_error="wrong answer";
      break;

    // It has to begin with a number
    default:
      {
        unsigned long ulNumber=0;
        sscanf(strWord.c_str(), "%lu", &ulNumber);

        uPos=parseNextWord(strWord, line);
        line.erase(0, uPos);
        switch( strWord.data()[0] )
        {
          case 'E' :
            if( !strWord.compare("EXISTS") )
              setMessagesNumber(ulNumber);
            else if( !strWord.compare("EXPUNGE") )
              printf("\nExpunge the deleted mails!\n");
            else
              imap_error="wrong answer";
            break;

//          case 'F' :
//            if( !strWord.compare("FETCH") )
//              parseFetch(ulNumber, line);
//            else
//              imap_error="wrong answer";
//            break;

          case 'R' :
            if( !strWord.compare("RECENT") )
              setRecent(ulNumber);
            else
              imap_error="wrong answer";
            break;

          case 'S' :
            if( !strWord.compare("STORE") )
              printf("\nSTORE: %s", line.c_str());
            else
              imap_error="wrong answer";
            break;

          default:
            imap_error="wrong answer";
            break;
        }
      }
      break;
  }

  if( !imap_error.compare("wrong answer") )
    return false;
  return true;
}


bool NetIMAP4::parseLineResult(string &result, bool bTaggedLine)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word

  //reset the imap error message
  imap_error="";
  //reset the subcommand
  _subCommand=NetIMAP4::NO_COMMAND;

  skipSpaces(result);
  if(result.data()[0] == '[')
  {
    uPos=parseNextWord(strWord, result,1);
    result.erase(0,uPos);
    // convert it into an upper case string
    toUpper( strWord );

    switch( strWord.data()[0] )
    {
      case 'A' :
        if( !strWord.compare("ALERT") )
          printf("imapParser::parseLineResult - %s %s\n",strWord.c_str(), result.c_str());
          // TODO: send a warning message to the client
        else
          imap_error="wrong answer";
        break;

      case 'N' :
        if( !strWord.compare("NEWNAME") )
          printf("imapParser::parseLineResult - %s %s\n",strWord.c_str(), result.c_str());
        else
          imap_error="wrong answer";
        break;

      case 'P' :
        if( !strWord.compare("PARSE") )
          printf("imapParser::parseLineResult - %s %s\n",strWord.c_str(), result.c_str());
        else if( !strWord.compare("PERMANENTFLAGS") )
        {
          printf("imapParser::parseLineResult - %s %s\n",strWord.c_str(), result.c_str());
          setPermanentFlags(result);
        }
        else
          imap_error="wrong answer";
        break;

      case 'R' :
        if( !strWord.compare("READ-ONLY") )
        {
          setAccess(NetIMAP4::READ_ONLY);
        }
        else if( !strWord.compare("READ-WRITE") )
        {
          setAccess(NetIMAP4::READ_WRITE);
        }
        else
          imap_error="wrong answer";
        break;

      case 'T' :
        if( !strWord.compare("TRYCREATE") )
          printf("imapParser::parseLineResult - %s %s\n",strWord.c_str(), result.c_str());
        else
          imap_error="wrong answer";
        break;

      case 'U' :
        if( !strWord.compare("UIDVALIDITY") )
        {
          setUidValidity( parseUnsignedLong(result) );
        }
        else if( !strWord.compare("UNSEEN") )
        {
          setUnseen( parseUnsignedLong(result) );
        }
        else if( !strWord.compare("UIDNEXT") )
        {
          setUidNext( parseUnsignedLong(result) );
        }
        else
          imap_error="wrong answer";
        break;

    }
    skipSpaces(result);
    if( result.data()[0] == ']')
      result.erase(0,1);
  }

  if( bTaggedLine )
  {

    // Check if we need to call the fetch parser
    if( _command==NetIMAP4::UID )
    {
      toUpper( result );
      if( result.find("FETCH")!=result.npos )
        _subCommand=NetIMAP4::FETCH;
    }

// I commented these lines because some servers
// doesn't give a right answer.
// Also the IMAP RFC isn't clear in this direction.
#if 0
    uPos=parseNextWord(strWord, result);
    result.erase(0,uPos);
    // convert it into an upper case string
    toUpper( strWord );

    switch( _command )
    {
      case NetIMAP4::NOOP:
        if( strWord.compare("NOOP") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::CAPABILITY:
        if( strWord.compare("CAPABILITY") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::LOGOUT:
        if( strWord.compare("LOGOUT") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::LOGIN:
        if( strWord.compare("LOGIN") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::AUTHENTICATE:
        if( strWord.compare("AUTHENTICATE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::SELECT:
        if( strWord.compare("SELECT") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::EXAMINE:
        if( strWord.compare("EXAMINE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::CREATE:
        if( strWord.compare("CREATE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::DELETE:
        if( strWord.compare("DELETE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::RENAME:
        if( strWord.compare("RENAME") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::SUBSCRIBE:
        if( strWord.compare("SUBSCRIBE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::UNSUBSCRIBE:
        if( strWord.compare("UNSUBSCRIBE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::LIST:
        if( strWord.compare("LIST") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::LSUB:
        if( strWord.compare("LSUB") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::STATUS:
        if( strWord.compare("STATUS") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::APPEND:
        if( strWord.compare("APPEND") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::CHECK:
        if( strWord.compare("CHECK") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::CLOSE:
        if( strWord.compare("CLOSE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::EXPUNGE:
        if( strWord.compare("EXPUNGE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::SEARCH:
        if( strWord.compare("SEARCH") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::FETCH:
        if( strWord.compare("FETCH") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::STORE:
        if( strWord.compare("STORE") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::COPY:
        if( strWord.compare("COPY") )
          imap_error="wrong answer";
        break;
      case NetIMAP4::UID:
        if( strWord.compare("UID") )
          imap_error="wrong answer";
        uPos=parseNextWord(strWord, result);
        result.erase(0,uPos);
        if( !strWord.compare("FETCH") )
          _subCommand=NetIMAP4::FETCH;
        break;
    }
#endif

  }
  if( !imap_error.compare("wrong answer") )
    return false;
  return true;
}


void NetIMAP4::parseCapability(string &capability)
{
  vector<string> words;

  words=split(' ',capability);
  for(vector<string>::iterator it=words.begin(); it!=words.end(); it++)
  {
    if( 5<(*it).size() && !(*it).compare("AUTH=",0,5) )
      setAuth( (*it).substr(5) );
  }
}

bool NetIMAP4::parseFetch(string &line)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word
  unsigned long ulNumber=0;
  IMAP_MESSAGE fetchMessage;

  while( line.size() )
  {
    // init the fetch buffer
    fetchMessage.message="";

    uPos=parseNextWord(strWord, line);
    // see if it is the last line, we shouldn't parse it
    // This is the only place where we can return true !
    if( !strWord.compare(getTag()) )
      return true;

    // passing over "*"
    line.erase(0, uPos);
    if( strWord.data()[0]!='*' )
      break;

    // get the message number
    uPos=parseNextWord(strWord, line);
    line.erase(0, uPos);
    sscanf(strWord.c_str(), "%lu", &ulNumber);

    // get the FETCH
    uPos=parseNextWord(strWord, line);
    line.erase(0, uPos);
    // convert it into an upper case string
    toUpper( strWord );
    if( strWord.compare("FETCH") )
      break;

    uPos=parseNextWord(strWord, line);
    line.erase(0, uPos);

    // it has to begin with '(' and to end with ')'
    skipSpaces(line);
    if(line.data()[0] == '(')
    {
      line.erase(0,1);
      skipSpaces(line);

      while( line.size() && line.data()[0]!=')' )
      {
        if(line.data()[0] != '(')
        {
          uPos=parseNextWord(strWord, line);
          line.erase(0, uPos);
          // convert it into an upper case string
          toUpper( strWord );
          switch( strWord.data()[0] )
          {
            case 'B' :
              if( !strWord.compare("BODY") )
              {
                // BODY[HEADER] {1234} \r\n
                // Return-Path ....
                // Remove [HEADER]
                skipSpaces(line);
                if(line.data()[0] == '[')
                  line.erase(0, line.find(']')+1);

                // Set the size of the message.
                skipSpaces(line);
                if(line.data()[0] == '{')
                {
                  line.erase(0,1);
                  fetchMessage.size=parseUnsignedLong(line);
                  line.erase(0, line.find('}')+1);
                  // Read the message
                  // Remove the \r\n chars
                  int iFind=line.find("\r\n");
                  line.erase(0, iFind+2);
                  fetchMessage.message=line.substr(0, fetchMessage.size);
                  line.erase(0, fetchMessage.size);
                }
                else
                  imap_error="wrong answer";
              }
              else if( !strWord.compare("BODYSTRUCTURE") )
                printf("\nparseFetch - BODYSTRUCTURE - %s\n", strWord.c_str());
              break;

            case 'E' :
              if( !strWord.compare("ENVELOPE") )
                printf("\nparseFetch - ENVELOPE - %s\n", strWord.c_str());
              break;

            case 'F' :
              if( !strWord.compare("FLAGS") )
                fetchMessage.flags=parseFlags(line);
              break;

            case 'I' :
              if( !strWord.compare("INTERNALDATE") )
              {
                skipSpaces(line);
                if( line.data()[0]=='"' )
                {
                  unsigned uFind=line.find('"', 1);
                  if( uFind!=line.npos )
                  {
                    string date=line.substr(1, uFind-1);
                    line.erase(0,uFind+1);
                  }
                  else
                    imap_error="wrong answer";
                }
              }
              break;

            case 'R' :
              if( !strWord.compare("RFC822.SIZE") )
                printf("\nparseFetch - RFC822.SIZE - %s\n", strWord.c_str());
              else if( strWord.npos != strWord.find("RFC822") )
                printf("\nparseFetch - RFC822 - %s\n", strWord.c_str());
              break;

            case 'U' :
              if( !strWord.compare("UID") )
              {
                uPos=parseNextWord(strWord, line);
                line.erase(0, uPos);
                if( strWord.size() )
                  fetchMessage.uid=strWord;
              }
              break;

            default:
                printf("\nparseFetch - default - %s\n", strWord.c_str());
              break;
          }
        }
        else
          printf("\nNot available yet! :(\n");
      }
      // remove the )\r\n chars
      if( line.size()>=3 )
        line.erase(0,3);
      else
        line="";
      // Add the last message into the list
      if( fetchMessage.message.size() )
        fetchMessages.push_back( fetchMessage );
    }
    else
      break;
    skipSpaces(line);
  }

  return false;
}

void NetIMAP4::parseStatus(string &line)
{
// INBOX (MESSAGES 1234 RECENT 0 UNSEEN 12 UIDNEXT 12345 UIDVALIDITY 123456)

  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word

  // parse the mailbox name
  uPos=parseNextWord(strWord, line);
  line.erase(0, uPos);

  // it has to begin with '(' and to end with ')'
  skipSpaces(line);
  if(line.data()[0] == '(')
  {
    line.erase(0,1);
    skipSpaces(line);

    while( line.size() && line.data()[0]!=')' )
    {
      if(line.data()[0] != '(')
      {
        uPos=parseNextWord(strWord, line);
        line.erase(0, uPos);
        if( !strWord.compare("MESSAGES") )
          setMessagesNumber( parseUnsignedLong(line) );
        else if( !strWord.compare("RECENT") )
          setRecent( parseUnsignedLong(line) );
        else if( !strWord.compare("UIDVALIDITY") )
          setUidValidity( parseUnsignedLong(line) );
        else if( !strWord.compare("UNSEEN") )
          setUnseen( parseUnsignedLong(line) );
        else if( !strWord.compare("UIDNEXT") )
          setUidNext( parseUnsignedLong(line) );
      }
      else
        printf("\nNot available yet! :(\n");
    }
  }
  else
    imap_error="wrong answer";
}

unsigned long NetIMAP4::parseUnsignedLong(string &line)
{
  unsigned long number;
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word

  uPos=parseNextWord(strWord, line);
  line.erase(0,uPos);
  sscanf(strWord.c_str(),"%lu", &number);

  return number;
}

void NetIMAP4::parseSearch(string &line)
{
  if( _command==NetIMAP4::UID )
  {
    UIDINFO uidInfo;
    string strWord;   // the current word
    unsigned uPos=0;  // the first position after the current word

    uidInfo.info="null";
    uidInfo.rcvTime=0;
    searchList.clear();

    while( line.size() && line.data()[0]!=')' )
    {
      uPos=parseNextWord(strWord, line);
      line.erase(0,uPos);
      if( strWord.size() )
        searchList.insert(UIDValuePair(strWord,uidInfo));
    }
  }
  else if( _command==NetIMAP4::SEARCH )
  {
    //TODO: the normal way is to use a vector for the message number

    printf("\n parseSearch: not implemented yet :(\n");
    fflush(stdout);

  }
}

void NetIMAP4::parseList(string &line)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word
  MBOX_ELEMENT mbox;

  // Init the mbox element
  mbox.name="";
  mbox.delimiter="";
  mbox.state=NetIMAP4::NoState;

  // it has to begin with '('
  skipSpaces(line);
  if(line.data()[0] == '(')
  {
    line.erase(0,1);
    skipSpaces(line);

    // parse the mailbox state state
    while( line.size() && line.data()[0]!=')' )
    {
      uPos=parseNextWord(strWord, line);
      line.erase(0,uPos);

      // convert it into an upper case string
      toUpper(strWord);

      if( !strWord.compare("\\NOSELECT") )
        mbox.state|=NetIMAP4::NoSelect;
      else if( !strWord.compare("\\NOINFERIORS") )
        mbox.state|=NetIMAP4::NoInferiors;
      else if( !strWord.compare("\\MARKED") )
        mbox.state|=NetIMAP4::Marked;
      else if( !strWord.compare("\\UNMARKED") )
        mbox.state|=NetIMAP4::UnMarked;
      else
      {
        printf("\n parseList: unknown mailbox state\n");
        fflush(stdout);
      }
    }

    // remove the ) from the line
    if( line.size() )
      line.erase(0,1);
    else
      return;

    // parse the delimiter
    uPos=parseNextWord(strWord, line);
    line.erase(0,uPos);
    mbox.delimiter=strWord;

    // parse the mailbox name
    uPos=parseNextWord(strWord, line);
    line.erase(0,uPos);
    mbox.name=strWord;

    // insert a new element in the mailboxes list
    mboxList.push_back(mbox);
  }
}

//
//  F L A G S
//
unsigned NetIMAP4::permanentFlags() const
{
  return _permanentFlags;
}

unsigned NetIMAP4::flags() const
{
  return _flags;
}

void NetIMAP4::setPermanentFlags(string &flags)
{
  _permanentFlags=parseFlags(flags);
}

void NetIMAP4::setFlags(string &flags)
{
  _flags=parseFlags(flags);
}

unsigned NetIMAP4::parseFlags(string &flags)
{
  string strWord;   // the current word
  unsigned uPos=0;  // the first position after the current word
  unsigned uFlags=0;

//  printf("\n--------------- parseFlags -----------\n");
//  fflush(stdout);

  skipSpaces(flags);
  if(flags.data()[0] == '(')
    flags.erase(0,1);

  while( flags.size() && flags.data()[0]!=')' )
  {
    uPos=parseNextWord(strWord,flags);
    flags.erase(0,uPos);
    // convert it into an upper case string
    toUpper( strWord );

    switch( strWord.data()[1] )
    {
      case 'A':
        if( !strWord.compare("\\ANSWERED") )
          uFlags |= NetIMAP4::Answered;
        break;
      case 'D':
        if( !strWord.compare("\\DELETED") )
          uFlags |= NetIMAP4::Deleted;
        else if( !strWord.compare("\\DRAFT") )
          uFlags |= NetIMAP4::Draft;
        break;
      case 'F':
        if( !strWord.compare("\\FLAGGED") )
          uFlags |= NetIMAP4::Flagged;
        break;
      case 'R':
        if( !strWord.compare("\\RECENT") )
          uFlags |= NetIMAP4::Recent;
        break;
      case 'S':
        if( !strWord.compare("\\SEEN") )
          uFlags |= NetIMAP4::Seen;
        break;
      case '*':
        if( !strWord.compare("\\*") )
          uFlags |= NetIMAP4::Other;
        break;
    }
  }
  if( flags[0]==')' )
    flags.erase(0,1);

  return uFlags;
}

void NetIMAP4::setAccess(bool access)
{
  _access=access;
}

bool NetIMAP4::access() const
{
  return _access;
}

// Info. attributes
void NetIMAP4::setUidValidity(unsigned long uidValidity)
{
  _uidValidity=uidValidity;
}

unsigned long NetIMAP4::uidValidity() const
{
  return _uidValidity;
}

void NetIMAP4::setMessagesNumber(unsigned long messagesNumber)
{
  _messagesNumber=messagesNumber;
}

unsigned long NetIMAP4::messagesNumber() const
{
  return _messagesNumber;
}

void NetIMAP4::setUnseen(unsigned long unseen)
{
  _unseen=unseen;
}

unsigned long NetIMAP4::unseen() const
{
  return _unseen;
}

void NetIMAP4::setRecent(unsigned long recent)
{
  _recent=recent;
}

unsigned long NetIMAP4::recent() const
{
  return _recent;
}

void NetIMAP4::setUidNext(unsigned long uidNext)
{
  _uidNext=uidNext;
}

unsigned long NetIMAP4::uidNext() const
{
  return _uidNext;
}

void NetIMAP4::setAuth(const string &auth)
{
  _auth=auth;
}

string NetIMAP4::auth() const
{
  return _auth;
}

void NetIMAP4::showStatus()
{
  printf("\nUid Validity = %lu", _uidValidity);
  printf("\nMessages number = %lu", _messagesNumber);
  printf("\nUnseen = %lu", _unseen);
  printf("\nRecent = %lu", _recent);
  printf("\nUid Next = %lu\n", _uidNext);
  fflush(stdout);
}

void NetIMAP4::clear()
{
  searchList.clear();
  searchVect.clear();
  fetchMessages.clear();
  mboxList.clear();
}














