/* InvariantFactoryDropHandler.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.dnd;

import java.text.MessageFormat;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JComponent;
import javax.swing.JOptionPane;

import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryManager;
import org.grinvin.gui.components.AcceptsInvariant;
import org.grinvin.gui.components.AcceptsInvariant.Reason;
import org.grinvin.invariants.IllegalInvariantComputerException;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantFactory;
import org.grinvin.invariants.InvariantManager;

/**
 * Allows an invariant factory to be dropped onto objects that support
 * {@link AcceptsInvariant}. After dropping, the user is asked
 * to present parameters for the invariant.
 */
public class InvariantFactoryDropHandler extends AbstractDropHandler {
    
    //
    private static final ResourceBundle BUNDLE = ResourceBundle.getBundle("org.grinvin.gui.dnd.resources");
    
    // cannot be instantiated by clients
    private InvariantFactoryDropHandler() {
        super(InvariantFactory.class, false);
    }
    
    /**
     * The unique shared object of this type.
     */
    private static final InvariantFactoryDropHandler DROP_HANDLER = new InvariantFactoryDropHandler();
    
    public static InvariantFactoryDropHandler getInstance() {
        return DROP_HANDLER;
    }
    
    public boolean acceptDrop(JComponent target, Object object, int seqNr) {
        if(!target.isEnabled())
            return seqNr != 0;
        if (seqNr > 0)
            return false;
        if (target instanceof AcceptsInvariant && object instanceof InvariantFactory) {
            try {
                InvariantFactory factory = (InvariantFactory)object;
                if(((AcceptsInvariant)target).acceptsInvariant(factory.getType()) == Reason.SUCCESS) {
                    if (FactoryManager.configureFactory(factory)) {
                        Invariant inv = InvariantManager.getInstance().getInvariantForFactory(factory);
                        switch (((AcceptsInvariant)target).acceptsInvariant(inv)) {
                            case SUCCESS:
                                ((AcceptsInvariant)target).addInvariant(inv);
                                return true;
                            case DUPLICATE:
                                JOptionPane.showMessageDialog(target, MessageFormat.format(BUNDLE.getString("droperror.invariant.duplicate"), inv), BUNDLE.getString("droperror.title"), JOptionPane.DEFAULT_OPTION);
                                return true; //continue with adding remaining invariants
                            case WRONG_TYPE:
                                JOptionPane.showMessageDialog(target, MessageFormat.format(BUNDLE.getString("droperror.invariant.wrong_type"), inv), BUNDLE.getString("droperror.title"), JOptionPane.DEFAULT_OPTION);
                                return true; //continue with adding remaining invariants
                            default:
                                return false;
                        }
                    } else {
                        return false;
                    }
                } else if(((AcceptsInvariant)target).acceptsInvariant(factory.getType()) == Reason.WRONG_TYPE) {
                    JOptionPane.showMessageDialog(target, MessageFormat.format(BUNDLE.getString("droperror.invariantfactory.wrong_type"), factory), BUNDLE.getString("droperror.title"), JOptionPane.DEFAULT_OPTION);
                    return true; //continue with adding remaining invariants
                } else {
                    // DUPLICATE has no meaning here
                    return false;
                }
            } catch (FactoryException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            } catch (IllegalInvariantComputerException ex) {
                Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create invariant from factory", ex);
                return false;
            }
        } else {
            return false;
        }
    }
    
}
