/* Transformation.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;


/**
 * Coordinate transformation from n-dimensional 
 * real coordinates (in <i>source</i> space)
 * to 2-dimensional real coordinates (in <i>destination</i> space).<p>
 * Additionally functions as a model in a model-view-controller triple. 
 */
public interface Transformation {
    
    /**
     * Transform the given set of coordinates from 
     * source space to destination space.
     * @param src Coordinates in the n-dimensin
     * @param dest Array into which the resulting coordinates shall be stored
     * @return a reference to <tt>dest</tt>.
     */
    public double[] transform(double[] src, double[] dest);
    
    /**
     * Is this transformation invertible?
     */
    public boolean isInvertible ();
    
    /**
     * Perform an inverse transformation from destination space back to
     * source space.<p>
     * This implementation throws an exception.
     * @param x X-coordinate in destination space
     * @param y Y-coordinate in destination space
     * @param result Array into which the resulting source space coordinates
     * should be stored
     * @throws IllegalStateException when the transformation is not invertible.
     */
    public void inverseTransform (double x, double y, double[] result) throws IllegalStateException;
    
    /**
     * Return the dimension of the source space.
     */
    public int getDimension ();
    

    /**
     * Register a transformation listener with this model.
     */
    void addTransformationListener(TransformationListener l);

    /**
     * Unregister a previously registered transformation listener.
     */
    void removeTransformationListener(TransformationListener l);
}
