#ifndef _FDISK_H_
#define _FDISK_H_

#include "config.h"

struct fdisk_partition;

struct fdisk_disk {
  char *name;
  unsigned size; /* not yet used */
  char *partition_format; /* not yet used */

  struct fdisk_partition *partitions;
  struct fdisk_partition *last_partition;

  struct fdisk_disk *next;
};


struct fdisk_partition {
  char *name;
  char *mount_point;
  int type; /* type code, see PTYPE_* macros below */
  unsigned size; /* in kByte */
  int in_use;

  struct fdisk_disk *disk;

  struct fdisk_partition *next;
  struct fdisk_partition *next_by_disk;
  struct fdisk_partition *next_by_type;
  struct fdisk_partition *next_in_use;
};

struct fdisk_disk *fdisk_disks;
struct fdisk_partition *fdisk_partitions;
#define FSTYPE_EXT2	0
#define FSTYPE_SWAP	1
#define FSTYPE_MSDOS	2
#define FSTYPE_MINIX	3
#define FSTYPE_AFFS	4
#define FSTYPE_HFS	5
#define FSTYPE_NFS	6
#define FSTYPE_EXTPART	7
#define FSTYPE_UNKNOWN	8
#define FSTYPE_MAX	9
struct fdisk_partition *fdisk_partitions_by_type[FSTYPE_MAX];
/* points to the list of partitions, index is FSTYPE_* */
struct fdisk_partition *mounted_partitions;
struct fdisk_partition *swapon_partitions;

char *fstype_name[FSTYPE_MAX];

char *
sysname(int type);

void
fdisk_reread();

struct fdisk_disk *
fdisk_find_disk(const char *name);

struct fdisk_partition *
fdisk_find_partition_by_name(const char *name);

struct fdisk_partition *
fdisk_find_partition_by_mntpoint(const char *mntpoint);

struct fdisk_disk *
fdisk_add_disk(const char *name, unsigned long size);

int
fdisk_fstype_of(unsigned int ptype);

char *
fdisk_fstype_name_of(unsigned int ptype);

struct fdisk_partition *
fdisk_add_partition(const char *name, int minor, unsigned int type,
		    unsigned long size);

int
sread(int fd, unsigned long sector, void *buffer);

#if HAVE_MSDOS_PARTITION
int parse_msdos_partition(char *device, int fd);
#endif
#if HAVE_OSF_PARTITION
int parse_osf_partition(char *device, int fd);
#endif
#if HAVE_SUN_PARTITION
int parse_sun_partition(char *device, int fd);
#endif
#if HAVE_AMIGA_PARTITION
int parse_amiga_partition(char *device, int fd);
#endif
#if HAVE_ATARI_PARTITION
int parse_atari_partition(char *device, int fd);
#endif
#if HAVE_MAC_PARTITION
int parse_mac_partition(char *device, int fd);
#endif

/* -------------------- partition type numbers -------------------- */
/*
 * currently used prefixes:
 *  0x000000xx          : PC
 *  0x41xxxxxx ("A")    : Atari
 *  0x425344xx ("BSD")  : BSD
 *  0x4d4143xx ("MAC")  : Apple
 *  0x4f5346xx ("OSF")  : OSF
 *  0x534fxxxx ("SO")   : Solaris/x86
 *  0x53554exx ("SUN")  : Sun
 *  0x8xxxxxxx (msb set): Amiga
 *  0x4C4F50xx ("LOP")  : Loop mounted
 */


/* undefined type -- for all machines */
#define PTYPE_UNKNOWN		0x00000100

/* PC */
#define PTYPE_PREFIX_MSDOS	0x00000000
#define PTYPE_EMPTY		0
#define PTYPE_DOS12		0x01
#define PTYPE_DOS16S		0x04
#define PTYPE_DOSEXT		0x05
#define PTYPE_DOS16L		0x06
#define PTYPE_WFAT32		0x0b
#define PTYPE_WFAT32L		0x0c
#define PTYPE_WEXT		0x0f
#define PTYPE_MINIX_DR		0x41
#define PTYPE_SWAP_DR		0x42
#define PTYPE_LINUX_DR		0x43
#define PTYPE_MINIX		0x81
#define PTYPE_LSWAP		0x82
#define PTYPE_LINUX		0x83
#define PTYPE_LINUXEXT		0x85

/* BSD disklabel (contained in MSDOS-partition); uses codes 0x425344xx
 * ("BSD"), where xx is the 8 bit type */
#define PTYPE_PREFIX_BSD	0x42534400
#define PTYPE_BSD_SWAP		(PTYPE_PREFIX_BSD|0x01)
#define PTYPE_BSD_V6		(PTYPE_PREFIX_BSD|0x02)
#define PTYPE_BSD_V7		(PTYPE_PREFIX_BSD|0x03)
#define PTYPE_BSD_SYSV		(PTYPE_PREFIX_BSD|0x04)
#define PTYPE_BSD_V71K		(PTYPE_PREFIX_BSD|0x05)
#define PTYPE_BSD_V8		(PTYPE_PREFIX_BSD|0x06)
#define PTYPE_BSD_BSDFFS	(PTYPE_PREFIX_BSD|0x07)
#define PTYPE_BSD_MSDOS		(PTYPE_PREFIX_BSD|0x08)
#define PTYPE_BSD_BSDLFS	(PTYPE_PREFIX_BSD|0x09)
#define PTYPE_BSD_OTHER		(PTYPE_PREFIX_BSD|0x0a)
#define PTYPE_BSD_HPFS		(PTYPE_PREFIX_BSD|0x0b)
#define PTYPE_BSD_ISOFS		(PTYPE_PREFIX_BSD|0x0c)
#define PTYPE_BSD_BOOT		(PTYPE_PREFIX_BSD|0x0d)
#define PTYPE_BSD_ADOS		(PTYPE_PREFIX_BSD|0x0e)
#define PTYPE_BSD_HFS		(PTYPE_PREFIX_BSD|0x0f)

/* Solaris/x86 disklabel (contained in MSDOS-partition); uses codes 0x534fxxxx
 * ("SO"), where xxxx is the 16 bit type */
#define PTYPE_PREFIX_SOLARIS_X86	0x534f0000
#define PTYPE_SOLARIS_X86	PTYPE_PREFIX_SOLARIS_X86

/* OSF (alpha); uses codes 0x4f5346xx ("OSF"), where xx is the 8 bit type */
/* is basically the same as BSD disklabel, but fstype 8 is ext2 instead of
 * msdos, and type 1 is ok as Linux swap. */
#define PTYPE_PREFIX_OSF	0x4f534600
#define PTYPE_OSF_SWAP		(PTYPE_PREFIX_OSF|0x01)
#define PTYPE_OSF_V6		(PTYPE_PREFIX_OSF|0x02)
#define PTYPE_OSF_V7		(PTYPE_PREFIX_OSF|0x03)
#define PTYPE_OSF_SYSV		(PTYPE_PREFIX_OSF|0x04)
#define PTYPE_OSF_V71K		(PTYPE_PREFIX_OSF|0x05)
#define PTYPE_OSF_V8		(PTYPE_PREFIX_OSF|0x06)
#define PTYPE_OSF_BSDFFS	(PTYPE_PREFIX_OSF|0x07)
#define PTYPE_OSF_EXT2		(PTYPE_PREFIX_OSF|0x08)
#define PTYPE_OSF_BSDLFS	(PTYPE_PREFIX_OSF|0x09)
#define PTYPE_OSF_OTHER		(PTYPE_PREFIX_OSF|0x0a)
#define PTYPE_OSF_HPFS		(PTYPE_PREFIX_OSF|0x0b)
#define PTYPE_OSF_ISOFS		(PTYPE_PREFIX_OSF|0x0c)
#define PTYPE_OSF_BOOT		(PTYPE_PREFIX_OSF|0x0d)
#define PTYPE_OSF_ADOS		(PTYPE_PREFIX_OSF|0x0e)
#define PTYPE_OSF_HFS		(PTYPE_PREFIX_OSF|0x0f)

/* Sun; uses codes 0x53554exx ("SUN"), where xx is the 8 bit type */
#define PTYPE_PREFIX_SUN	0x53554e00
#define PTYPE_SUN_EMPTY		(PTYPE_PREFIX_SUN|0x00)
#define PTYPE_SUN_BOOT		(PTYPE_PREFIX_SUN|0x01)
#define PTYPE_SUN_S_ROOT	(PTYPE_PREFIX_SUN|0x02)
#define PTYPE_SUN_S_SWAP	(PTYPE_PREFIX_SUN|0x03)
#define PTYPE_SUN_S_USR		(PTYPE_PREFIX_SUN|0x04)
#define PTYPE_SUN_WHOLE		(PTYPE_PREFIX_SUN|0x05)
#define PTYPE_SUN_S_STAND	(PTYPE_PREFIX_SUN|0x06)
#define PTYPE_SUN_S_VAR		(PTYPE_PREFIX_SUN|0x07)
#define PTYPE_SUN_S_HOME	(PTYPE_PREFIX_SUN|0x08)
#define PTYPE_SUN_L_MINIX	(PTYPE_PREFIX_SUN|0x81)
#define PTYPE_SUN_L_SWAP	(PTYPE_PREFIX_SUN|0x82)
#define PTYPE_SUN_L_NATIVE	(PTYPE_PREFIX_SUN|0x83)

/* Amiga; uses codes 0x80000000..0xffffffff, i.e. all with highest bit set;
 * low 31 bits are type (really 32 bit, but highest bit obviously never used */
#define PTYPE_PREFIX_AMIGA	0x80000000
#define PTYPE_AMIGA_BOOT	(PTYPE_PREFIX_AMIGA|0x424F4F55)	 /* BOOU */
#define PTYPE_AMIGA_OFS		(PTYPE_PREFIX_AMIGA|0x444F5300)	 /* DOS\0 */
#define PTYPE_AMIGA_FFS		(PTYPE_PREFIX_AMIGA|0x444F5301)	 /* DOS\1 */
#define PTYPE_AMIGA_OFS_INTL	(PTYPE_PREFIX_AMIGA|0x444F5302)	 /* DOS\2 */
#define PTYPE_AMIGA_FFS_INTL	(PTYPE_PREFIX_AMIGA|0x444F5303)	 /* DOS\3 */
#define PTYPE_AMIGA_OFS_DC	(PTYPE_PREFIX_AMIGA|0x444F5304)	 /* DOS\4 */
#define PTYPE_AMIGA_FFS_DC	(PTYPE_PREFIX_AMIGA|0x444F5305)	 /* DOS\5 */
#define PTYPE_AMIGA_MUFS	(PTYPE_PREFIX_AMIGA|0x6d754653)	 /* muFS */
#define PTYPE_AMIGA_MU_OFS	(PTYPE_PREFIX_AMIGA|0x6d754600)	 /* muF\0 */
#define PTYPE_AMIGA_MU_FFS	(PTYPE_PREFIX_AMIGA|0x6d754601)	 /* muF\1 */
#define PTYPE_AMIGA_MU_OFS_INTL (PTYPE_PREFIX_AMIGA|0x6d754602)	 /* muF\2 */
#define PTYPE_AMIGA_MU_FFS_INTL (PTYPE_PREFIX_AMIGA|0x6d754603)	 /* muF\3 */
#define PTYPE_AMIGA_MU_OFS_DC	(PTYPE_PREFIX_AMIGA|0x6d754604)	 /* muF\4 */
#define PTYPE_AMIGA_MU_FFS_DC	(PTYPE_PREFIX_AMIGA|0x6d754605)	 /* muF\5 */
#define PTYPE_AMIGA_LINUX	(PTYPE_PREFIX_AMIGA|0x4c4e5800)	 /* LNX\0 */
#define PTYPE_AMIGA_EXT2	(PTYPE_PREFIX_AMIGA|0x45585432)	 /* EXT2 */
#define PTYPE_AMIGA_SWAP	(PTYPE_PREFIX_AMIGA|0x53574150)	 /* SWAP */
#define PTYPE_AMIGA_SWP		(PTYPE_PREFIX_AMIGA|0x53575000)	 /* SWP\0 */
#define PTYPE_AMIGA_MINIX	(PTYPE_PREFIX_AMIGA|0x4d4e5800)	 /* MNX\0 */
#define PTYPE_AMIGA_AMIX0	(PTYPE_PREFIX_AMIGA|0x554e4900)	 /* UNI\0 */
#define PTYPE_AMIGA_AMIX1	(PTYPE_PREFIX_AMIGA|0x554e4901)	 /* UNI\1 */
#define PTYPE_AMIGA_NETBSD_ROOT	(PTYPE_PREFIX_AMIGA|0x4e425207)	 /* NBR\7 */
#define PTYPE_AMIGA_NETBSD_SWAP	(PTYPE_PREFIX_AMIGA|0x4e425301)	 /* NBS\1 */
#define PTYPE_AMIGA_NETBSD_OTH	(PTYPE_PREFIX_AMIGA|0x4e425507)	 /* NBU\7 */
#define PTYPE_AMIGA_PFS0	(PTYPE_PREFIX_AMIGA|0x50465300)	 /* PFS\0 */
#define PTYPE_AMIGA_PFS1	(PTYPE_PREFIX_AMIGA|0x50465301)	 /* PFS\1 */
#define PTYPE_AMIGA_AFS0	(PTYPE_PREFIX_AMIGA|0x41465300)	 /* AFS\0 */
#define PTYPE_AMIGA_AFS1	(PTYPE_PREFIX_AMIGA|0x41465301)	 /* AFS\1 */

/* Atari; uses codes 0x41xxxxxx, where xxxxxx is the three-letter ID */
#define PTYPE_PREFIX_ATARI	0x41000000
#define PTYPE_GEMDOSS		(PTYPE_PREFIX_ATARI|0x47454d)	 /* GEM */
#define PTYPE_GEMDOSL		(PTYPE_PREFIX_ATARI|0x42474d)	 /* BGM */
#define PTYPE_ATARI_LINUX	(PTYPE_PREFIX_ATARI|0x4c4e58)	 /* LNX */
#define PTYPE_ATARI_SWAP	(PTYPE_PREFIX_ATARI|0x535750)	 /* SWP */
#define PTYPE_ATARI_MINIX1	(PTYPE_PREFIX_ATARI|0x4d4958)	 /* MIX */
#define PTYPE_ATARI_MINIX2	(PTYPE_PREFIX_ATARI|0x4d4e58)	 /* MNX */
#define PTYPE_ATARI_HFS		(PTYPE_PREFIX_ATARI|0x4d4143)	 /* MAC */
#define PTYPE_ATARI_SYSV	(PTYPE_PREFIX_ATARI|0x554e58)	 /* UNX */
#define PTYPE_ATARI_RAW		(PTYPE_PREFIX_ATARI|0x524157)	 /* RAW */
#define PTYPE_ATARI_EXTD	(PTYPE_PREFIX_ATARI|0x58474d)	 /* XGM */

/* Mac; uses codes 0x4d4143xx ("MAC"), where xx is a synthetic number. On Mac,
 * partition types are no codes, but strings, that are translated to this
 * numbers. */
#define PTYPE_PREFIX_MAC	0x4d414300
#define PTYPE_MAC_PMAP		(PTYPE_PREFIX_MAC|0x01)
#define PTYPE_MAC_DRIVER	(PTYPE_PREFIX_MAC|0x02)
#define PTYPE_MAC_DRIVER43	(PTYPE_PREFIX_MAC|0x03)
#define PTYPE_MAC_HFS		(PTYPE_PREFIX_MAC|0x04)
#define PTYPE_MAC_MFS		(PTYPE_PREFIX_MAC|0x05)
#define PTYPE_MAC_SCRATCH	(PTYPE_PREFIX_MAC|0x06)
#define PTYPE_MAC_PRODOS	(PTYPE_PREFIX_MAC|0x07)
#define PTYPE_MAC_FREE		(PTYPE_PREFIX_MAC|0x08)
#define PTYPE_MAC_SWAP		(PTYPE_PREFIX_MAC|0x09)
#define PTYPE_MAC_AUX		(PTYPE_PREFIX_MAC|0x0a)
#define PTYPE_MAC_MSDOS		(PTYPE_PREFIX_MAC|0x0b)
#define PTYPE_MAC_MINIX		(PTYPE_PREFIX_MAC|0x0c)
#define PTYPE_MAC_AFFS		(PTYPE_PREFIX_MAC|0x0d)
#define PTYPE_MAC_EXT2		(PTYPE_PREFIX_MAC|0x0e)

/* Fake partition types for loop mounts */
#define PTYPE_PREFIX_LOOP	0x4c4f5000
#define PTYPE_LOOP_EXT2		(PTYPE_PREFIX_LOOP|FSTYPE_EXT2)
#define PTYPE_LOOP_SWAP		(PTYPE_PREFIX_LOOP|FSTYPE_SWAP)
#define PTYPE_LOOP_MSDOS	(PTYPE_PREFIX_LOOP|FSTYPE_MSDOS)
#define PTYPE_LOOP_MINIX	(PTYPE_PREFIX_LOOP|FSTYPE_MINIX)
#define PTYPE_LOOP_AFFS		(PTYPE_PREFIX_LOOP|FSTYPE_AFFS)
#define PTYPE_LOOP_HFS		(PTYPE_PREFIX_LOOP|FSTYPE_HFS)

/* Fake partition type for NFS mounts */
#define PTYPE_NFS               0x4e465300 /* "NFS\0" */

#endif
