/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "approx.h"
#include "expr.h"
#include "misc.h"
#include "str.h"
#include "lexer.h"
#include "lexerP.h"
#include "var.h"
#include "vector.h"
#include "cases.h"

/* I can't think of any really good reason to disable debugging for
   this module. */
/*#undef DEBUGGING */
#define DEBUGGING 1
#include "debug-print.h"

/* COMPUTE and IF transformation. */
typedef struct
  {
    trns_header h;

    /* Destination.  (Used only during parsing.) */
    variable *v;		/* Destvar, if dest isn't a vector elem. */
    int created;		/* Whether we created the destvar (used only during
				   parsing). */

    /* Destination.  (Used during execution.) */
    vector *vec;		/* Destination vector, if dest is a vector elem. */
    int fv;			/* `value' index of destination variable. */
    int width;			/* Target variable width (string vars only). */

    /* Expressions. */
    struct expression *vec_elem;		/* Destination vector element expr. */
    struct expression *target;			/* Target expression. */
    struct expression *test;			/* Test expression (IF only). */
  }
compute_trns;

static int type_check (compute_trns *,
		       int (*func_tab[4]) (any_trns *, ccase *));
static compute_trns *new_trns (void);
static void delete_trns (compute_trns *);
static void free_trns (any_trns *);
static int parse_var_or_vec (compute_trns *);

/* COMPUTE. */

static int compute_num (any_trns *, ccase *);
static int compute_str (any_trns *, ccase *);
static int compute_num_vec (any_trns *, ccase *);
static int compute_str_vec (any_trns *, ccase *);

int
cmd_compute (void)
{
  /* Table of functions to process data. */
  static int (*func_tab[4]) (any_trns *, ccase *) =
  {
    compute_num,
    compute_str,
    compute_num_vec,
    compute_str_vec,
  };

  /* Transformation being constructed. */
  compute_trns *c;

  match_id (COMPUTE);

  c = new_trns ();
  if (!parse_var_or_vec (c))
    goto fail;

  force_match ('=');

  c->target = parse_expression (PXP_NONE);
  if (!c->target)
    goto fail;

  if (!type_check (c, func_tab))
    goto fail;

  /* Goofy behavior, but compatible: Turn off LEAVE on the destvar. */
  if (c->v && c->v->left && c->v->name[0] != '#')
    {
      devector (c->v);
      c->v->left = 0;
      envector (c->v);
    }

  add_transformation ((any_trns *) c);

  return 1;
fail:
  delete_trns (c);
  return 0;
}

static int
compute_num (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;
  evaluate_expression (t->target, c, &c->data[t->fv]);
  return -1;
}

static int
compute_num_vec (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  /* Index into the vector. */
  value index;

  /* Rounded index value. */
  int rindx;

  evaluate_expression (t->vec_elem, c, &index);
  rindx = floor (index.f + EPSILON);
  if (index.f == SYSMIS || rindx < 1 || rindx > t->vec->nv)
    {
      if (index.f == SYSMIS)
	msg (SW, _("When executing COMPUTE: SYSMIS is not a valid value as "
	     "an index into vector %s."), t->vec->name);
      else
	msg (SW, _("When executing COMPUTE: %g is not a valid value as "
	     "an index into vector %s."), index.f, t->vec->name);
      return -1;
    }
  evaluate_expression (t->target, c, &c->data[t->vec->v[rindx - 1]->fv]);
  return -1;
}

static int
compute_str (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  /* Temporary storage for string expression return value. */
  value v;

  evaluate_expression (t->target, c, &v);
  strbarepadlencpy (c->data[t->fv].s, &v.c[1], t->width,
		    ' ', v.c[0]);
  return -1;
}

static int
compute_str_vec (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  /* Temporary storage for string expression return value. */
  value v;

  /* Index into the vector. */
  value index;

  /* Rounded index value. */
  int rindx;

  /* Variable reference by indexed vector. */
  variable *vr;

  evaluate_expression (t->vec_elem, c, &index);
  rindx = floor (index.f + EPSILON);
  if (index.f == SYSMIS || rindx < 1 || rindx > t->vec->nv)
    {
      if (index.f == SYSMIS)
	msg (SW, _("When executing COMPUTE: SYSMIS is not a valid value as "
	     "an index into vector %s."), t->vec->name);
      else
	msg (SW, _("When executing COMPUTE: %g is not a valid value as "
	     "an index into vector %s."), index.f, t->vec->name);
      return -1;
    }

  evaluate_expression (t->target, c, &v);
  vr = t->vec->v[rindx - 1];
  strbarepadlencpy (c->data[vr->fv].s, &v.c[1], vr->width, ' ', v.c[0]);
  return -1;
}

/* IF. */

static int if_num (any_trns *, ccase *);
static int if_str (any_trns *, ccase *);
static int if_num_vec (any_trns *, ccase *);
static int if_str_vec (any_trns *, ccase *);

int
cmd_if (void)
{
  /* Table of functions to process data. */
  static int (*func_tab[4]) (any_trns *, ccase *) =
  {
    if_num,
    if_str,
    if_num_vec,
    if_str_vec,
  };

  /* Transformation being constructed. */
  compute_trns *c;

  match_id (IF);
  c = new_trns ();

  /* Test expression. */
  c->test = parse_expression (PXP_BOOLEAN);
  if (!c->test)
    goto fail;

  /* Target variable. */
  if (!parse_var_or_vec (c))
    goto fail;

  /* Target expression. */
  force_match ('=');

  c->target = parse_expression (PXP_NONE);
  if (!c->target)
    goto fail;

  /* Set up destination. */
  if (!type_check (c, func_tab))
    goto fail;

  add_transformation ((any_trns *) c);

  return 1;
fail:
  delete_trns (c);
  return 0;
}

static int
if_num (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  if (evaluate_expression (t->test, c, NULL) == 1.0)
    evaluate_expression (t->target, c, &c->data[t->fv]);
  return -1;
}

static int
if_str (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  if (evaluate_expression (t->test, c, NULL) == 1.0)
    {
      value v;

      evaluate_expression (t->target, c, &v);
      strbarepadlencpy (c->data[t->fv].s, &v.c[1], t->width,
			' ', v.c[0]);
    }
  return -1;
}

static int
if_num_vec (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  if (evaluate_expression (t->test, c, NULL) == 1.0)
    {
      /* Index into the vector. */
      value index;

      /* Rounded index value. */
      int rindx;

      evaluate_expression (t->vec_elem, c, &index);
      rindx = floor (index.f + EPSILON);
      if (index.f == SYSMIS || rindx < 1 || rindx > t->vec->nv)
	{
	  if (index.f == SYSMIS)
	    msg (SW, _("When executing COMPUTE: SYSMIS is not a valid value as "
		 "an index into vector %s."), t->vec->name);
	  else
	    msg (SW, _("When executing COMPUTE: %g is not a valid value as "
		 "an index into vector %s."), index.f, t->vec->name);
	  return -1;
	}
      evaluate_expression (t->target, c,
			   &c->data[t->vec->v[rindx]->fv]);
    }
  return -1;
}

static int
if_str_vec (any_trns * _t, ccase * c)
{
  compute_trns *t = (compute_trns *) _t;

  if (evaluate_expression (t->test, c, NULL) == 1.0)
    {
      /* Index into the vector. */
      value index;

      /* Rounded index value. */
      int rindx;

      /* Temporary storage for result of target expression. */
      value v2;

      /* Variable reference by indexed vector. */
      variable *vr;

      evaluate_expression (t->vec_elem, c, &index);
      rindx = floor (index.f + EPSILON);
      if (index.f == SYSMIS || rindx < 1 || rindx > t->vec->nv)
	{
	  if (index.f == SYSMIS)
	    msg (SW, _("When executing COMPUTE: SYSMIS is not a valid value as "
		 "an index into vector %s."), t->vec->name);
	  else
	    msg (SW, _("When executing COMPUTE: %g is not a valid value as "
		 "an index into vector %s."), index.f, t->vec->name);
	  return -1;
	}
      evaluate_expression (t->target, c, &v2);
      vr = t->vec->v[rindx - 1];
      strbarepadlencpy (c->data[vr->fv].s, &v2.c[1], vr->width, ' ', v2.c[0]);
    }
  return -1;
}

/* Code common to COMPUTE and IF. */

/* Checks for type mismatches on transformation C.  Also checks for
   command terminator, sets the case-handling proc from the array
   passed. */
static int
type_check (compute_trns * c, int (*proc_list[4]) (any_trns *, ccase *))
{
  /* Type of destination variable/vector. */
  int dest_type;

  if (c->v)
    dest_type = c->v->type;
  else
    dest_type = c->vec->v[0]->type;

  c->h.proc = proc_list[(dest_type == ALPHA) + ((c->vec != NULL) << 1)];

  if (c->created && c->target->type == EX_STRING)
    return msg (SE, _("String variables must be declared with STRING "
		"before they may be used on COMPUTE."));
  if (c->target->type == EX_STRING && dest_type == NUMERIC)
    return msg (SE, _("Cannot assign a string value to a numeric variable."));
  if ((c->target->type == EX_NUMERIC || c->target->type == EX_BOOLEAN)
      && dest_type == ALPHA)
    return msg (SE, _("Cannot assign a numeric value to a string variable."));
  if (token != '.')
    return syntax_error (_("expecting end of command"));
  return 1;
}

/* Returns a new compute_trns after initializing its fields. */
static compute_trns *
new_trns (void)
{
  compute_trns *c = xmalloc (sizeof (compute_trns));
  c->h.proc = NULL;
  c->h.free = free_trns;
  c->v = NULL;
  c->created = 0;
  c->vec = NULL;
  c->fv = 0;
  c->width = 0;
  c->vec_elem = NULL;
  c->target = NULL;
  c->test = NULL;
  return c;
}

/* Deletes all the fields in C, the variable C->v if we created it,
   and C itself. */
static void
delete_trns (compute_trns * c)
{
  free_trns ((any_trns *) c);
  if (c->created)
    delete_variable (&default_dict, c->v);
  free (c);
}

/* Deletes all the fields in C. */
static void
free_trns (any_trns * _t)
{
  compute_trns *t = (compute_trns *) _t;

  free_expression (t->vec_elem);
  free_expression (t->target);
  free_expression (t->test);
}

/* Parses a variable name or a vector element into C.  If the
   variable does not exist, it is created.  Returns success. */
static int
parse_var_or_vec (compute_trns * c)
{
  force_id ();
  if (lookahead () == '(')
    {
      /* Vector element. */
      c->vec = find_vector (tokstr);
      if (!c->vec)
	return msg (SE, _("There is no vector named %s."), tokstr);
      get_token ();
      force_match ('(');
      c->vec_elem = parse_expression (PXP_NUMERIC);
      if (!c->vec_elem)
	return 0;
      force_match (')');
    }
  else
    {
      /* Variable name. */
      c->v = find_variable (tokstr);
      if (!c->v)
	{
	  c->v = force_create_variable (&default_dict, tokstr, NUMERIC, 0);
	  envector (c->v);
	  c->created = 1;
	}
      c->fv = c->v->fv;
      c->width = c->v->width;
      get_token ();
    }
  return 1;
}

/* EVALUATE. */

#if GLOBAL_DEBUGGING
int
cmd_evaluate (void)
{
  expression *expr;

  match_id (EVALUATE);
  expr = parse_expression (PXP_DUMP);
  if (!expr)
    return 0;
  free_expression (expr);
  if (token != '.')
    return msg (SE, _("Extra characters after expression."));
  return 1;
}
#endif
