// ForthCompiler.cpp
//
// FORTH compiler to generate FORTH Byte Code (FBC) from expressions
//   or programs
//
// Copyright (c) 1998--1999 Krishna Myneni, Creative Consulting for
//   Research and Education
//
// This software is provided under the terms of the General Public License.
//
// Revisions:
// 	9-12-1998
//	9-15-1998 added SP@, RP@, +!
//      9-16-1998 added -ROT, PICK, ROLL, A@ 
//	9-18-1998 error checking for incomplete structures at end of definition
//	10-6-1998 added ?DUP
//	10-14-1998 fixed COUNT
//	10-19-1998 added 0<, 0=, 0>, TRUE, FALSE, INVERT
//      02-09-1999 added EXECUTE, ' (tick)
//      03-01-1999 added OPEN, LSEEK, CLOSE, READ, WRITE
//      03-02-1999 added IOCTL
//      03-03-1999 added USLEEP
//      03-07-1999 added FILL, CMOVE
//      03-27-1999 added +LOOP, UNLOOP
//      03-31-1999 added CMOVE>, KEY
//      05-06-1999 added FLOOR, FROUND
//      05-24-1999 added FATAN2, LSHIFT, RSHIFT
//      05-27-1999 added ACCEPT
//      05-29-1999 added QUIT, BASE, BINARY, DECIMAL, HEX, U<, U.
//      06-02-1999 added */, */MOD, NUMBER?
//      06-05-1999 added CHAR (ASCII)
//      06-09-1999 function IsInt now calls Forth's NUMBER? 
//      06-16-1999 fixed to allow multiple LEAVEs within single DO-LOOP
//
#include <fstream.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "fbc.h"
#include "ForthCompiler.h"

extern int debug;
extern int ForthVM (vector<byte>*, istream&, ostream&, int**, byte**);
extern vector<DictionaryEntry> Dictionary;
extern vector<DictionaryEntry>::iterator LocateWord (char*);
extern void RemoveLastWord();
extern vector<char*> StringTable;

extern "C" int* GlobalSp;
extern "C" byte* GlobalTp;
extern "C" int* JumpTable;
extern "C" int Base;
extern "C" int State;  // TRUE = compile, FALSE = interpret
extern "C" int To_In; 
extern "C" char TIB[];  // contains current line of input
extern "C" int C_numberquery();
extern "C" int L_abort();

void strupr (char*);

// stacks for keeping track of nested control structures

vector<int> ifstack;	// stack for if-then constructs
vector<int> beginstack;	// stack for begin ... constructs
vector<int> whilestack;	// stack for while jump holders
vector<int> dostack;    // stack for do loops
vector<int> leavestack; // stack for leave jumps

char* WordNames[] =
    {
        "WORDS", "ALLOT",
        "@", "!", "A@",
        "C@", "C!",
        "W@", "W!",
        "F@", "F!",
        "DF@", "DF!",
        "SF@", "SF!",
	"SP@", "RP@",
        ">R", "R>", "R@", "?DUP",
        "DUP", "DROP", "SWAP",
        "OVER", "ROT", "-ROT", 
	"NIP", "TUCK", "PICK", "ROLL",
        "2DUP", "2DROP", "2SWAP",
        "2OVER", "2ROT",
        "DEPTH",
	"BASE", "BINARY", "DECIMAL", "HEX",
        "1+", "1-", "2+", "2-", 
	"2*", "2/",
        "LOOP", "+LOOP", "UNLOOP", 
	"I", "J",
        "EXIT", "QUIT", "ABORT",
        "EXECUTE", "CALL", 
	"\x22", "COUNT", "NUMBER?",
        ".", "U.", "F.", ".\x22", ".S",
        "CR", "SPACES", "EMIT", "TYPE",
	"CHAR", "ASCII",
	"KEY", "ACCEPT",
        "=", "<>", "<", ">", "<=", ">=",
	"U<",
	"0<", "0=", "0>", "FALSE", "TRUE",
        "AND", "OR", "XOR", "NOT", "INVERT",
	"LSHIFT", "RSHIFT",
        "+", "-", "*", "/",
	"MOD", "/MOD", 
	"*/", "*/MOD", "+!",
        "ABS", "NEGATE", "MIN", "MAX",
	"OPEN", "LSEEK", "CLOSE", 
	"READ", "WRITE", "IOCTL",
	"USLEEP", "FILL", "CMOVE", "CMOVE>",
        "FDUP", "FDROP", "FSWAP",
        "FOVER", "FROT",
        "F=", "F<>", "F<", "F>", "F<=", "F>=",
        "F+", "F-", "F*", "F/", "F**", "FSQRT",
        "FABS", "FNEGATE",
	"FLOOR", "FROUND",
	"FMIN", "FMAX",
        "FSIN", "FCOS", "FTAN",
        "FACOS", "FASIN", "FATAN",
	"FATAN2",
        "FLOG", "FLN", "FEXP",
        "DEG>RAD", "RAD>DEG",
        "S>F", "F>S"
    };

byte WordCodes[] =
    {
        OP_WORDS, OP_ALLOT,
        OP_FETCH, OP_STORE, OP_AFETCH,
        OP_CFETCH, OP_CSTORE,
        OP_WFETCH, OP_WSTORE,
        OP_DFFETCH, OP_DFSTORE,
        OP_DFFETCH, OP_DFSTORE,
        OP_SFFETCH, OP_SFSTORE,
	OP_SPFETCH, OP_RPFETCH,
        OP_PUSH, OP_POP, OP_RFETCH, OP_QUERYDUP,
        OP_DUP, OP_DROP, OP_SWAP,
        OP_OVER, OP_ROT, OP_MINUSROT, 
	OP_NIP, OP_TUCK, OP_PICK, OP_ROLL,
        OP_2DUP, OP_2DROP, OP_2SWAP,
        OP_2OVER, OP_2ROT,
        OP_DEPTH,
	OP_BASE, OP_BINARY, OP_DECIMAL, OP_HEX,
        OP_INC, OP_DEC, OP_TWOPLUS, OP_TWOMINUS,
	OP_TWOSTAR, OP_TWODIV,
        OP_LOOP, OP_PLUSLOOP, OP_UNLOOP,
	OP_I, OP_J,
        OP_RET, OP_QUIT, OP_ABORT,
        OP_EXECUTE, OP_CALL, 
	OP_QUOTE, OP_COUNT, OP_NUMBERQUERY,
        OP_DOT, OP_UDOT, OP_FDOT, OP_DOTQUOTE, OP_DOTS,
        OP_CR, OP_SPACES, OP_EMIT, OP_TYPE,
	OP_CHAR, OP_CHAR,
	OP_KEY, OP_ACCEPT,
        OP_EQ, OP_NE, OP_LT, OP_GT, OP_LE, OP_GE,
	OP_ULT,
	OP_ZEROLT, OP_ZEROEQ, OP_ZEROGT, OP_FALSE, OP_TRUE,
        OP_AND, OP_OR, OP_XOR, OP_NOT, OP_NOT,
	OP_LSHIFT, OP_RSHIFT,
        OP_ADD, OP_SUB, OP_MUL, OP_DIV, 
	OP_MOD, OP_SLASHMOD,
	OP_STARSLASH, OP_STARSLASHMOD, OP_PLUSSTORE,
        OP_ABS, OP_NEG, OP_MIN, OP_MAX,
	OP_OPEN, OP_LSEEK, OP_CLOSE, 
	OP_READ, OP_WRITE, OP_IOCTL,
	OP_USLEEP, OP_FILL, OP_CMOVE, OP_CMOVEFROM,
        OP_2DUP, OP_2DROP, OP_2SWAP,
        OP_2OVER, OP_2ROT,
        OP_FEQ, OP_FNE, OP_FLT, OP_FGT, OP_FLE, OP_FGE,
        OP_FADD, OP_FSUB, OP_FMUL, OP_FDIV, OP_FPOW, OP_FSQRT,
        OP_FABS, OP_FNEG,
	OP_FLOOR, OP_FROUND,
	OP_FMIN, OP_FMAX,
        OP_FSIN, OP_FCOS, OP_FTAN,
        OP_FACOS, OP_FASIN, OP_FATAN,
	OP_FATAN2,
        OP_FLOG, OP_FLN, OP_FEXP,
        OP_DEGTORAD, OP_RADTODEG,
        OP_STOF, OP_FTOS
    };

char* C_ErrorMessages[] =
{
	"",
	"",
	"End of definition with no beginning",
	"End of string",	
	"ELSE without matchin IF",
	"THEN without matching IF",
	"No matching BEGIN", 
        "Not allowed inside colon definition",
	"Error opening file",
	"Incomplete IF...THEN structure",
	"Incomplete BEGIN structure",
	"Unknown word",
	"No matching DO",
	"Incomplete DO loop"
};


//---------------------------------------------------------------

char* ExtractName (char* str, char* name)
{
// Starting at ptr str, extract the non delimiter text into
//   a buffer starting at name with null terminator appended
//   at the end. Return a pointer to the next position in
//   str.

    char* delim = "\n\r\t ";
    char *pStr = str, *pName = name;

    while (strchr(delim, *pStr)) ++pStr;
    if (*pStr)
    {
        while (strchr(delim, *pStr) == NULL)
        {
            *pName = *pStr;
            ++pName;
            ++pStr;
        }
    }
    *pName = 0;
    return pStr;
}
//---------------------------------------------------------------
	
int IsForthWord (char* name, DictionaryEntry* pE)
{
// Locate and Return a copy of the dictionary entry
//   with the specified name.  Return True if found,
//   False otherwise. A copy of the entry is returned
//   in *pE.

    vector<DictionaryEntry>::iterator i = LocateWord (name);

    if (i)
    {
        *pE = *i;
        return TRUE;
    }
    else
        return FALSE;
}
//---------------------------------------------------------------

int IsFloat (char* token, double* p)
{
// Check the string token to see if it is an LMI style floating point
//   number; if so set the value of *p and return True, otherwise
//   return False.

    char *pStr = token;

//    cout << "\nIsFloat: token = " << token;

    if (strchr(pStr, 'E'))
    {
        while ((isdigit(*pStr)) || (*pStr == '-')
          || (*pStr == 'E') || (*pStr == '+') || (*pStr == '.'))
        {
            ++pStr;
//            cout << ' ' << ((int) *pStr);
        }
        if (*pStr == 0)
        {
            // LMI Forth style

            --pStr;
            if (*pStr == 'E') *pStr = '\0';
            *p = atof(token);
            return TRUE;
        }
    }

    return FALSE;
}

int IsInt (char* token, int* p)
{
// Check the string token to see if it is an integer number;
//   if so set the value of *p and return True, otherwise return False.

  char s[256];
  *s = (unsigned char) strlen(token);
  strcpy (s+1, token);
  *GlobalSp-- = (int) s;
  *GlobalTp-- = OP_ADDR;
  
  int err = C_numberquery();
  if (err)
    {
      // stack has probably become corrupted -- call abort

      cout << "Stack error during compilation.\n";
      L_abort();
      return FALSE;
    }

  ++GlobalSp; ++GlobalTp;
  int b = *GlobalSp++; ++GlobalTp;
  ++GlobalSp; ++GlobalTp;
  *p = *GlobalSp;

  return b;
}
//---------------------------------------------------------------

void OutputForthByteCode (vector<byte>* pFBC, ostream& OutStream)
{
// Output opcode vector to an output stream for use in
//   debugging the compiler.

    int i, n = pFBC->size();
    byte* bp = pFBC->begin();

    OutStream << "\nOpcodes:\n";
    for (i = 0; i < n; i++)
    {
        OutStream << ((int) *bp) << ' ';
        if (((i + 1) % 8) == 0) OutStream << '\n';
        ++bp;
    }
    OutStream << '\n';
    return;
}
//---------------------------------------------------------------

int ForthCompiler (vector<byte>* pOpCodes,
    istream& SourceStream, ostream& OutStream, int* pLc)
{
// The FORTH Compiler
//
// Reads and compile the source statements from the input stream
//   into a vector of FORTH Byte Codes.
//
// Return value:
//
//  0   no error
//  other --- see ForthCompiler.h

  int ecode = 0, opcount = 0;
  char s[256], token[256], *cp, *begin_string, *end_string;
  double fval;
  int i, j, ival, *sp;
  vector<byte>::iterator ib1, ib2;
  vector<int>::iterator iI;
  DictionaryEntry d;
  vector<DictionaryEntry>::iterator id;
  byte opval, *fp, *ip, *bp, *tp;

  State = FALSE;
  static int linecount = 0;
  static DictionaryEntry NewWord;

  fp = (byte *) &fval;
  ip = (byte *) &ival;

  if (! State) linecount = 0;

  while (TRUE)
    {
      // Read each line and parse

      SourceStream.getline(TIB, 255);
      if (debug) OutStream << TIB << '\n';

      if (SourceStream.fail())
	{
	  if (State)
	    {
	      ecode = E_C_ENDOFSTREAM;  // reached end of stream before end of definition
	      break;
	    }
	  pOpCodes->push_back(OP_RET);
	  break;    // end of stream reached
	}
      ++linecount;
      cp = TIB;
      while (*cp && (cp < (TIB + 255)))
	{
	  if (*cp == ' ' || *cp == '\t')
	    ++cp;
	  else if (*cp == '\\' && (*(cp + 1) == ' ' ||
				   *(cp+1) == '\0' || *(cp+1) == '\t'))
	    break;  // ignore rest of the line
	  else if (*cp == ':')
	    {
	      if (pOpCodes->size())
		{
		  // Execute the code outside of a definition

		  pOpCodes->push_back(OP_RET);
		  ival = ForthVM (pOpCodes, cin, OutStream, &sp, &tp);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  if (ival) 
		    {
		      OutStream << "\nVM error: " << ival;
		      goto endcompile;
		    }
		}

	      State = TRUE;
	      ++cp;
	      cp = ExtractName (cp, token);
	      strupr(token);
	      NewWord.WordName = new char [strlen(token) + 1];
	      strcpy (NewWord.WordName, token);
	      NewWord.WordCode = OP_DEFINITION;
	      NewWord.MemPtr = NULL;
	    }
	  else if (*cp == ';')
	    {
	      pOpCodes->push_back(OP_RET);

	      if (State)
		{
		  // Check for incomplete control structures
		    
		  if (ifstack.size())
		    {
		      ecode = E_C_INCOMPLETEIF;
		      ifstack.erase(ifstack.begin(), ifstack.end());
		      goto endcompile;
		    }
		  if (beginstack.size() || whilestack.size())
		    {
		      ecode = E_C_INCOMPLETEBEGIN;
		      beginstack.erase(beginstack.begin(),beginstack.end());
		      whilestack.erase(whilestack.begin(),whilestack.end());
		      goto endcompile;
		    }
		  if (dostack.size() || leavestack.size())
		    {
		      ecode = E_C_INCOMPLETELOOP;
		      dostack.erase(dostack.begin(), dostack.end());
		      leavestack.erase(leavestack.begin(), leavestack.end());
		      goto endcompile;
		    }

		  // Add a new entry into the dictionary

		  if (debug) OutputForthByteCode (pOpCodes, OutStream);
 		    
		  NewWord.MemPtr = new byte[pOpCodes->size()];
		  byte* dest = (byte*) NewWord.MemPtr;
		  bp = pOpCodes->begin();
		  while (bp < pOpCodes->end()) *dest++ = *bp++;
		  Dictionary.push_back(NewWord);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  State = FALSE;
		}
	      else
		{
		  ecode = E_C_ENDOFDEF;
		  goto endcompile;
		}
	      ++cp;
	    }
	  else if (*cp == '(')
	    {
	      while ((cp < (TIB + 255)) && *cp != ')') ++cp;
	      if (*cp == ')') ++cp;
	    }
	  else
	    {
	      cp = ExtractName (cp, token);
	      strupr(token);

	      if (IsForthWord(token, &d))
		{
		  pOpCodes->push_back(d.WordCode);
		    
		  if (d.WordCode == OP_DEFINITION)
		    {
		      pOpCodes->pop_back();
		      pOpCodes->push_back(OP_ADDR);
		      bp = (byte*) &d.MemPtr;
		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(*(bp + i));
		      pOpCodes->push_back(OP_EXECUTE);
		    }
		  else if (d.WordCode == OP_ADDR)
		    {
		      // push address into the byte code vector

		      bp = (byte*) &d.MemPtr;

		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_IVAL)
		    {
		      // push value into the byte code vector

		      bp = (byte*) d.MemPtr;			
		      for (i = 0; i < sizeof(int); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_FVAL)
		    {
		      // push float value into the vector

		      bp = (byte*) d.MemPtr;
		      for (i = 0; i < sizeof(double); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_CHAR)
		    {
		      pOpCodes->pop_back();
		      pOpCodes->push_back(OP_IVAL);
		      while (*cp == ' ') ++cp;
		      if (*cp) 
			{
			  ival = (int) *cp ;
			  for (i = 0; i < sizeof(int); i++)
			    pOpCodes->push_back(*(ip + i));
			  ++cp;
			}
		    }
		  else if ((d.WordCode == OP_QUOTE) ||
			   (d.WordCode == OP_DOTQUOTE))
		    {
		      pOpCodes->pop_back();
		      pOpCodes->push_back(OP_ADDR);
		      begin_string = cp + 1;
		      end_string = strchr(begin_string, '"');
		      if (end_string == NULL)
			{
			  ecode = E_C_ENDOFSTRING;
			  goto endcompile;
			}
		      cp = end_string + 1;
		      ival = (int) (end_string - begin_string);
		      d.MemPtr = new char[ival + 2];
		      *((byte*)d.MemPtr) = (byte) ival;
		      strncpy((char*) d.MemPtr+1, begin_string, ival);
		      ((char*)d.MemPtr)[ival+1] = '\0';
		      StringTable.push_back((char*) d.MemPtr);
		      bp = (byte*) &d.MemPtr;
		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(*(bp + i));

		      d.MemPtr = NULL;

		      if (d.WordCode == OP_DOTQUOTE)
			{
			  pOpCodes->push_back(OP_COUNT);
			  pOpCodes->push_back(OP_TYPE);
			}
		    }
		  else if ((d.WordCode == OP_BINARY) ||
			   (d.WordCode == OP_DECIMAL) ||
			   (d.WordCode == OP_HEX) ||
			   (d.WordCode == OP_ALLOT))
		    {
		      if (State == FALSE)
			{
			  pOpCodes->push_back(OP_RET);
			  if (debug) OutputForthByteCode (pOpCodes, OutStream);
			  ival = ForthVM (pOpCodes, cin, OutStream, &sp, &tp);
			  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
			  if (ival) goto endcompile;
			}
		    }
		  else if (d.WordCode == OP_UNLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		    }
		  else if (d.WordCode == OP_LOOP || d.WordCode == OP_PLUSLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		      if (leavestack.size())
			{
			  i = dostack[dostack.size() - 1];
			  do
			    {
			      j = leavestack[leavestack.size() - 1];
			      if (j > i)
				{
				  ival = pOpCodes->size() - j + 1;
				  ib1 = pOpCodes->begin() + j;
				  *ib1++ = *ip;       // write the relative jump count
				  *ib1++ = *(ip + 1);
				  *ib1++ = *(ip + 2);
				  *ib1 = *(ip + 3);
				  leavestack.pop_back();
				}
			    } while ((j > i) && (leavestack.size())) ;
			}
		      dostack.pop_back();
		    }
		  else
		    {
		      ;
		    }
		}
	      else if (IsInt(token, &ival))
		{
		  pOpCodes->push_back(OP_IVAL);
		  for (i = 0; i < sizeof(int); i++)
		    pOpCodes->push_back(*(ip + i)); // store in proper order
		}
	      else if (IsFloat(token, &fval))
		{
		  pOpCodes->push_back(OP_FVAL);
		  for (i = 0; i < sizeof(double); i++)
		    pOpCodes->push_back(*(fp + i)); // store in proper order
		}
	      else if (strcmp(token, "'") == 0)
		{
		  pOpCodes->push_back(OP_ADDR);
		  // get address of next token
		  ++cp;
		  cp = ExtractName (cp, token);
		  strupr(token);
		  if (IsForthWord (token, &d))
		    {
		      if (d.WordCode == OP_DEFINITION)
			{
			  bp = (byte*) &d.MemPtr;
			  for (i = 0; i < sizeof(byte*); i++)
			    pOpCodes->push_back(*(bp + i));
			}
		      else
			{
			  for (i = 0; i < sizeof(byte*); i++)
			    pOpCodes->push_back(0);
			}
		    }
		  else
		    {
		      for (i = 0; i < sizeof(byte*); i++)
			pOpCodes->push_back(0);
		    }
		}
	      else if (strcmp(token, "DO") == 0)
		{
		  // Generate instructions to begin a loop

		  pOpCodes->push_back(OP_PUSH);
		  pOpCodes->push_back(OP_PUSH);
		  pOpCodes->push_back(OP_PUSHIP);

		  dostack.push_back(pOpCodes->size());
		}
	      else if (strcmp(token, "LEAVE") == 0)
		{
		  // Jump out of the current loop

		  if (dostack.empty())
		    {
		      ecode = E_C_NODO;
		      goto endcompile;
		    }
		  pOpCodes->push_back(OP_UNLOOP);
		  pOpCodes->push_back(OP_JMP);
		  leavestack.push_back(pOpCodes->size());
		  for (i = 0; i < sizeof(int); i++) pOpCodes->push_back(0);
		}
	      else if (strcmp(token, "BEGIN") == 0)
		{
		  beginstack.push_back(pOpCodes->size());
		}
	      else if (strcmp(token, "WHILE") == 0)
		{
		  // Build the begin ... while ... repeat structure
			
		  if (beginstack.empty())
		    {
		      ecode = E_C_NOBEGIN;
		      goto endcompile;
		    }
		  pOpCodes->push_back(OP_JZ);
		  whilestack.push_back(pOpCodes->size());
		  for (i = 0; i < sizeof(int); i++) pOpCodes->push_back(0);		
		}
	      else if ((strcmp(token, "UNTIL") == 0) ||
		       (strcmp(token, "REPEAT") == 0) ||
		       (strcmp(token, "AGAIN") == 0))
		{
		  // Complete begin ... until block

		  if (beginstack.empty())
		    {
		      ecode = E_C_NOBEGIN;  // no matching BEGIN
		      goto endcompile;
		    }
		  i = beginstack[beginstack.size()-1];
		  beginstack.pop_back();

		  if (strcmp(token, "REPEAT") == 0)
		    {
		      if (whilestack.size())
			{
			  j = whilestack[whilestack.size()-1];
			  if (j > i)
			    {
			      whilestack.pop_back();
			      ival = pOpCodes->size() - j + 6;
			      ib1 = pOpCodes->begin() + j;
			      *ib1++ = *ip;       // write the relative jump count
			      *ib1++ = *(ip + 1);
			      *ib1++ = *(ip + 2);
			      *ib1 = *(ip + 3);
			    }
			}
		    }

		  ival = i - pOpCodes->size();

		  if (strcmp(token, "UNTIL") == 0)
		    pOpCodes->push_back(OP_JZ);
		  else
		    pOpCodes->push_back(OP_JMP);

		  for (i = 0; i < sizeof(int); i++) pOpCodes->push_back(0);
		  ib1 = pOpCodes->end() - sizeof(int);
		  *ib1++ = *ip;       // write the relative jump count
		  *ib1++ = *(ip + 1);
		  *ib1++ = *(ip + 2);
		  *ib1 = *(ip + 3);
		}
	      else if ((strcmp(token, "VARIABLE") == 0) ||
		       (strcmp(token, "FVARIABLE") == 0) ||
		       (strcmp(token, "CREATE") == 0))
		{
		  // Add a word of type addr to the dictionary

		  NewWord.WordCode = OP_ADDR;

		  if (*token == 'V')
		    NewWord.MemPtr = new int[1];
		  else if (*token == 'F')
		    NewWord.MemPtr = new double[1];
		  else
		    NewWord.MemPtr = NULL;

		  cp = ExtractName(cp, token);
		  strupr(token);
		  NewWord.WordName = new char[strlen(token)+1];
		  strcpy(NewWord.WordName, token);

		  Dictionary.push_back(NewWord);
		}
	      else if ((strcmp(token, "CONSTANT") == 0) ||
		       (strcmp(token, "FCONSTANT") == 0))
		{
		  // Add a word of type ival or fval to the dictionary
		    
		  if (State)
		    {
		      ecode = E_C_NOTINDEF;
		      goto endcompile;
		    }

		  pOpCodes->push_back(OP_ADDR);

		  if (*token == 'C')
		    {
		      NewWord.WordCode = OP_IVAL;
		      NewWord.MemPtr = new int[1];
		    }
		  else if (*token == 'F')
		    {
		      NewWord.WordCode = OP_FVAL;
		      NewWord.MemPtr = new double[1];
		    } 
		  else
		    NewWord.MemPtr = NULL;
		
		  bp = (byte*) &NewWord.MemPtr;
		  for (i = 0; i < sizeof(byte*); i++)
		    pOpCodes->push_back(*(bp + i));
		
		  if (*token == 'F')
		    pOpCodes->push_back(OP_DFSTORE);
		  else
		    pOpCodes->push_back(OP_STORE);

		  cp = ExtractName(cp, token);
		  strupr(token);
		  NewWord.WordName = new char[strlen(token)+1];
		  strcpy(NewWord.WordName, token);
		  Dictionary.push_back(NewWord);

		  // execute immediately
                    
		  pOpCodes->push_back(OP_RET);
		  ival = ForthVM (pOpCodes, cin, OutStream, &sp, &tp);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  if (ival) goto endcompile;
		}
	      else if (strcmp(token, "IF") == 0)
		{
		  // Generate start of an if-then or if-else-then block

		  pOpCodes->push_back(OP_JZ);
		  ifstack.push_back(pOpCodes->size());

		  for (i = 0; i < sizeof(int); i++)
		    pOpCodes->push_back(0); // placeholder for jump count
		}
	      else if (strcmp(token, "ELSE") == 0)
		{
		  // Build the if-else-then block

		  pOpCodes->push_back(OP_JMP);

		  for (i = 0; i < sizeof(int); i++)
		    pOpCodes->push_back(0); // placeholder for jump count

		  if (ifstack.empty())
		    {
		      ecode = E_C_ELSENOIF;  // ELSE without matching IF
		      goto endcompile;
		    }
		  i = ifstack[ifstack.size()-1];
		  ifstack.pop_back();
		  ifstack.push_back(pOpCodes->size() - sizeof(int));
		  ival = pOpCodes->size() - i + 1;
		  ib1 = pOpCodes->begin() + i;;
		  *ib1++ = *ip;       // write the relative jump count
		  *ib1++ = *(ip + 1);
		  *ib1++ = *(ip + 2);
		  *ib1 = *(ip + 3);
		}
	      else if (strcmp(token, "THEN") == 0)
		{
		  // Complete the if-then or if-else-then block

		  if (ifstack.empty())
		    {
		      ecode = E_C_THENNOIF;  // THEN without matching IF or IF-ELSE
		      goto endcompile;
		    }
		  i = ifstack[ifstack.size()-1];
		  ifstack.pop_back();
		  ival = (int) (pOpCodes->size() - i) + 1;
		  ib1 = pOpCodes->begin() + i;
		  *ib1++ = *ip;       // write the relative jump count
		  *ib1++ = *(ip + 1);
		  *ib1++ = *(ip + 2);
		  *ib1 = *(ip + 3);
		}
	      else if (strcmp(token, "INCLUDE") == 0)
		{
		  if (State)
		    {
		      ecode = E_C_NOTINDEF;
		      goto endcompile;
		    }
		  ++cp;
		  cp = ExtractName (cp, token);
		  strcpy (s, cp);  // save remaining part of input line in TIB
		  if (!strchr(token, '.')) strcat(token, ".4th");
		  ifstream f(token);
		  if (f.fail())
		    {
		      OutStream << '\n' << token << '\n';
		      ecode = E_C_OPENFILE;
		      goto endcompile;
		    }
		  ecode = ForthCompiler(pOpCodes, f, OutStream, &linecount);
		  f.close();
		  if (ecode) goto endcompile;
		    
		  // Execute the code immediately

		  ival = ForthVM (pOpCodes, cin, OutStream, &sp, &tp);
		  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		  if (ival) goto endcompile;
		  strcpy(TIB, s);  // restore TIB with remaining input line
		  cp = TIB;  // restore ptr
		}
	      else if (strcmp(token, "FORGET") == 0)
		{
		  if (State)
		    {
		      ecode = E_C_NOTINDEF;
		      goto endcompile;
		    }
		  ++cp;
		  cp = ExtractName (cp, token);
		  strupr(token);
		  id = LocateWord (token);
		  if (id)
		    {
		      while (Dictionary.end() > id) 
			RemoveLastWord();
		    }
		  else
		    {
		      ecode = E_C_UNKNOWNWORD;
		      goto endcompile;
		    }
		}
	      else
		{
		  OutStream << '\n' << token << '\n';
		  ecode = E_C_UNKNOWNWORD;  // unknown keyword
		  goto endcompile;
		}
	    }
	}
    }

endcompile:
    
  if (ecode != E_C_ENDOFSTREAM)
    { 
      State = FALSE;
    }
  *pLc = linecount;
  return ecode;
}

void strupr (char* p)
{
// convert string to upper case

  while (*p) {*p = toupper(*p); ++p;}
}
