/*
 * rtl_time.h
 *
 * RTL architecture-independent clock support
 *
 * Copyright (C) 1999 Michael Barabanov
 *
 */

#ifndef __RTL_TIME__
#define __RTL_TIME__

#define NSECS_PER_SEC 1000000000
#define HRTIME_INFINITY 0x7fffFfffFfffFfffLL

#include <rtl_conf.h>

#ifdef CONFIG_RTL_USE_V1_API
#define HRTICKS_PER_SEC 1193180
#else
#define HRTICKS_PER_SEC 1000000000
#endif

#ifdef __KERNEL__
/* for timespec */
#include <linux/time.h>
#endif

#include <arch/rtl_time.h>


#ifdef __KERNEL__


typedef void (* clock_irq_handler_t)(struct pt_regs *r);

enum { RTL_CLOCK_MODE_UNINITIALIZED = 1, RTL_CLOCK_MODE_ONESHOT,
RTL_CLOCK_MODE_PERIODIC};

struct rtl_clock {
	int (*init) (struct rtl_clock *);
	void (*uninit) (struct rtl_clock *);
	hrtime_t (*gethrtime)(struct rtl_clock *);
	int (*sethrtime)(struct rtl_clock *, hrtime_t t);

	int (*settimer)(struct rtl_clock *, hrtime_t interval);
	int (*settimermode)(struct rtl_clock *, int mode);
	clock_irq_handler_t handler;
	int mode;
	hrtime_t resolution;
	hrtime_t value; /* only makes sense for periodic clocks */
	hrtime_t delta;
	struct rtl_clock_arch arch;
};

typedef struct rtl_clock *clockid_t;

extern struct rtl_clock RTL_CLOCK_DEFAULTS;

extern int rtl_setclockhandler (clockid_t h, clock_irq_handler_t fn);
extern int rtl_unsetclockhandler (clockid_t h);

extern int rtl_clockadjust (clockid_t clock_id, hrtime_t delta);

/* scheduler interface */
static inline hrtime_t clock_gethrtime (clockid_t clock_id)
{
	return clock_id->gethrtime (clock_id);
}

extern clockid_t rtl_getbestclock (unsigned int cpu);


/* end of scheduler interface */


extern int rtl_init_standard_clocks(void);
extern void rtl_cleanup_standard_clocks(void);

#endif /* __KERNEL__ */

/* convenience functions */
#define timespec_add(t1, t2) { \
	(t1)->tv_nsec += (t2)->tv_nsec;  \
	(t1)->tv_sec += (t2)->tv_sec; \
	if ((t1)->tv_nsec >= NSECS_PER_SEC) { \
		(t1)->tv_nsec -= NSECS_PER_SEC; \
		(t1)->tv_sec++; \
	} else if ((t1)->tv_nsec < 0) { \
		(t1)->tv_nsec += NSECS_PER_SEC; \
		(t1)->tv_sec--; \
	} \
}

#define timespec_sub(t1, t2) { \
	(t1)->tv_nsec -= (t2)->tv_nsec;  \
	(t1)->tv_sec -= (t2)->tv_sec; \
	if ((t1)->tv_nsec < 0) { \
		(t1)->tv_nsec += NSECS_PER_SEC; \
		(t1)->tv_sec--; \
	} else if ((t1)->tv_nsec >= NSECS_PER_SEC) { \
		(t1)->tv_nsec -= NSECS_PER_SEC; \
		(t1)->tv_sec++; \
	} \
}

#define timespec_nz(t) ((t)->tv_sec != 0 || (t)->tv_nsec != 0)

#define timespec_lt(t1, t2) ((t1)->tv_sec < (t2)->tv_sec || ((t1)->tv_sec == (t2)->tv_sec && (t1)->tv_nsec < (t2)->tv_nsec))

#define timespec_gt(t1, t2) (timespec_lt(t2, t1))

#define timespec_ge(t1, t2) (!timespec_lt(t1, t2))

#define timespec_le(t1, t2) (!timespec_gt(t1, t2))

#define timespec_eq(t1, t2) ((t1)->tv_sec == (t2)->tv_sec && (t1)->tv_nsec == (t2)->tv_nsec)

#endif

