/*
 *   stdinxout - cross-connect two programs' stdin and stdout
 *   Copyright (C) 1999 Tommi Virtanen <tv@havoc.fi>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <wait.h>

#define PROGNAME "stdinxout"
#define USAGE "usage: " PROGNAME " prog1 [args1..] -- prog2 [args2..]"
#define READ 0
#define WRITE 1

static void /*@exits@*/ fail(char *s) {
  (void) fprintf(stderr,"%s: failure: %s\n",PROGNAME,s);
  exit(100);
}

static void /*@exits@*/ defer(char *s) {
  (void) fprintf(stderr,"%s: deferral: %s\n",PROGNAME,s);
  exit(111);
}

static void /*@exits@*/ usage() {
  (void) fprintf(stderr, "%s\n", USAGE);
  exit(111);
}

int main(int argc, char **argv) {
  pid_t pid_prog1, pid_prog2, pid;
  int to1[2] = {-1, -1};
  int to2[2] = {-1, -1};
  int separator=1;
  if (argc<4)
    usage();
  while ((argv[separator][0]!='-' && argv[separator][0]!='\0')
        || (argv[separator][1]!='-' && argv[separator][0]!='\0')
        || argv[separator][2]!='\0') {
    separator++;
    if (separator>=argc) {
      fail(USAGE);
    }
  }
  if (separator<2) {
    fail(USAGE);
  }

  if (pipe(to1)==-1) { defer("pipe to prog1 failed."); }
  if (pipe(to2)==-1) { defer("pipe to prog2 failed."); }
  pid_prog1=fork();
  if (pid_prog1==-1) { defer("fork failed."); }
  if (pid_prog1==0) { /* child */
    (void) close(to1[WRITE]);
    (void) close(to2[READ]);
    if (dup2(to1[READ],0)==-1)
      defer("dup2 failed on stdin for prog1.");
    if (dup2(to2[WRITE],1)==-1)
      defer("dup2 failed on stdout for prog1.");
    argv[separator]=NULL;
    (void) execvp(argv[1], argv+1);
    defer("exec prog1 failed");
  }
  else { /* parent */
    pid_prog2=fork();
    if (pid_prog2==-1) { defer("fork failed."); }
    if (pid_prog2==0) { /* child */
      (void) close(to2[WRITE]);
      (void) close(to1[READ]);
      if (dup2(to2[READ],0)==-1)
        defer("dup2 failed on stdin for prog2.");
      if (dup2(to1[WRITE],1)==-1)
        defer("dup2 failed on stdout for prog2.");
      (void) execvp(argv[separator+1], argv+separator+1);
      defer("exec prog2 failed");
    }
    else { /* parent */
      pid=wait(NULL);
      if (pid==-1) {
        defer("fail in first wait");
      }
      else if (pid==pid_prog1) {
        if (close(to2[WRITE])!=0)
          defer("cannot close pipe to program 2");
      }
      else if (pid==pid_prog2) {
        if (close(to1[WRITE])!=0)
          defer("cannot close pipe to program 1");
      }
      pid=wait(NULL);
      if (pid==-1) {
        defer("fail in second wait");
      }
      else if (pid==pid_prog1) {
        if (close(to2[WRITE])!=0)
          defer("cannot close pipe to program 2");
      }
      else if (pid==pid_prog2) {
        if (close(to1[WRITE])!=0)
          defer("cannot close pipe to program 1");
      }
      exit(0);
    }
  }
  /*@notreached@*/ exit(1);
}
