/*
 * (C) Copyright Keith Visco 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */
package com.kvisco.xsl;

import org.w3c.dom.*;

import com.kvisco.util.List;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * A representation of an Attribute Value Template
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class AttributeValueTemplate {

    /**
     * Token indicating end of a StringExpr
    **/
    private static final String END_EXPR = "}";
    
    /**
     * Token indicating start of a StringExpr
    **/
    private static final String START_EXPR = "{";
    
    /**
     * the components of this AttributeValueTemplate
    **/
    private List components;

      //----------------/
     //- Constructors -/
    //----------------/

    /**
     * Creates an empty AttributeValueTemplate
    **/
    public AttributeValueTemplate() {
        super();
        components = new List();
    } //-- AttributeValueTemplate
    
    /**
     * Creates an AttributeValueTemplate using the given String
     * @param attValue the String to create the AttributeValueTemplate from
     * @exception InvalidExprException
    **/
    public AttributeValueTemplate(String attValue) 
        throws InvalidExprException {
        this();
        if (attValue != null) parse(attValue);
    } //-- AttributeValueTemplate

      //------------------/
     //- Public Methods -/
    //------------------/
    
    /**
     * Adds the given Expr to this AttributeValueTemplate.
     * @param Expr the Expr to add to this AttributeValueTemplate
    **/
    public void addExpr(Expr expr) {
        components.add(expr);
    } //-- addExpr

    /**
     * Returns the value of this AttributeValueTemplate when evaluated using
     * the given context Node.
     * @param context the Node to evaluate this AttributeValueTemplate with
     * @param rpState the ProcessorState for accessing the current processing
     * environment
    **/ 
    public String evaluate(Node context, ProcessorState ps) {
        StringBuffer sb = new StringBuffer();
        try {
            for (int i = 0; i < components.size(); i++) {
                Expr expr = (Expr)components.get(i);
                StringResult sr = null;
                sr = StringExpr.toStringResult(expr.evaluate(context, ps));
                sb.append(sr.toString());
            }
        }
        catch(InvalidExprException iee) {};
        return sb.toString();
    } //-- getValue

    /**
     * Removes the given Expr from this AttributeValueTemplate
     * @param expr the Expr to remove
    **/
    public void removeExpr(Expr expr) {
        components.remove(expr);
    } //-- removeExpr
    
    /**
     * Returns the String representation of this AttributeValueTemplate
     * @return the String representation of this AttributeValueTemplate
    **/
    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < components.size(); i++) {
            Expr expr = (Expr)components.get(i);
            sb.append( expr.toString() );
        }
        return sb.toString();
    } //-- toString
    
      //-------------------/
     //- Private Methods -/
    //-------------------/
    
    /**
     * Parses the given String into this AttributeValueTemplate
     * @exception InvalidExprException
    **/
    private  void parse
        (String pattern) throws InvalidExprException
    {
        
        char[] chars = pattern.toCharArray();
        int cc = 0;
        StringBuffer buffer = new StringBuffer();
        boolean inExpr      = false;
        boolean inLiteral   = false;
        char endLiteral = '"';
        char prevCh = '\0';

        while ( cc < chars.length ) {
            char ch = chars[cc++];
            // if in literal just add ch to buffer
            if ( inLiteral && (ch != endLiteral) ) {
                    buffer.append(ch);
                    prevCh = ch;
                    continue;
            }
            switch ( ch ) {
                case '\'' :
                case '"' :
                    buffer.append(ch);
                    if (inLiteral) inLiteral = false;
                    else {
                        inLiteral = true;
                        endLiteral = ch;
                    }
                    break;
                case  '{' :
                    // Ignore case where we find two { without a }
                    if (!inExpr) {
                        //-- clear buffer
                        if ( buffer.length() > 0) {
                            addExpr(new PrimaryExpr(buffer.toString()));
                            buffer.setLength(0);
                        }
                        inExpr = true;
                    }
                    else if (prevCh == ch) {
                        inExpr = false;
                        buffer.append(ch);
                    }
                    else {
                        buffer.append(ch); //-- simply append '{'
                        ch = '\0';
                    }
                    break;
                case '}':
                    if (inExpr) {
                        inExpr = false;
                        addExpr(ExpressionParser.createExpr(buffer.toString()));
                        buffer.setLength(0);
                        //-- change in case another '}' follows
                        ch = '\0';
                    }
                    else if (prevCh != ch) {
                        if ( buffer.length() > 0) buffer.append('}');
                        else addExpr(new PrimaryExpr(END_EXPR));
                    }
                    else ch = '\0';
                    break;
                default:
                    buffer.append(ch);
                    break;
            }
            prevCh = ch;
        }
        if ( buffer.length() > 0) {
            if ( inExpr ) {
                //-- error
                StringBuffer errMsg = new StringBuffer();
                errMsg.append("#error evaluating AttributeValueTemplate. ");
                errMsg.append("Missing '}' after: ");
                errMsg.append(buffer.toString());
                addExpr(new PrimaryExpr(errMsg.toString()));
            }
            else addExpr(new PrimaryExpr(buffer.toString()));
        }

    }
} //-- AttributeValueTemplate
