/*
 * dnode.c - Pyramid DC/OSx and Reliant UNIX node functions for lsof
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.10 98/03/06 08:28:24 abe Exp $";
#endif


#include "lsof.h"
#include <sys/fs/namenode.h>
#include <sys/fs/ufs_inode.h>
#include <sys/fs/xnamnode.h>


_PROTOTYPE(static int examine_stream,(KA_T na, KA_T vs, struct queue *q, char *mn, char *sn, KA_T *sqp));
_PROTOTYPE(static struct l_dev * findstrdev,(dev_t *dev));
_PROTOTYPE(static char isvlocked,(struct vnode *va));
_PROTOTYPE(static int readlino,(int fx, struct vnode *v, struct l_ino *i, struct vfs *kv));


static struct protos {
	char *module;			/* stream module name */
	char *proto;			/* TCP/IP protocol name */
} Protos[] = {
	{ "tcp",	"TCP"	},
	{ "udp",	"UDP"	},
	{ "ip",		"IP"	},
	{ "icmp",	"ICMP"	}
};
#define	NPROTOS	(sizeof(Protos)/sizeof(struct protos))


/*
 * examine_stream() - examine stream
 */

static int
examine_stream(na, vs, q, mn, sn, sqp)
	KA_T na;			/* vnode kernel space address */
	KA_T vs;			/* stream head's stdata kernel
					 * address */
	struct queue *q;		/* queue structure buffer */
	char *mn;			/* module name receiver */
	char *sn;			/* special module name */
	KA_T *sqp;			/* special module's q_ptr */
{
	struct module_info mi;
	KA_T qp;
	struct qinit qi;
	struct stdata sd;
	char tbuf[32];
/*
 * Read stream's head.
 */
	if (!vs || readstdata(vs, &sd)) {
	    (void) sprintf(Namech, "vnode@%s; can't read stream head@%s",
		print_kptr(na, tbuf), print_kptr(vs, (char *)NULL));
	    enter_nm(Namech);
	    return(1);
	}
	if (!sd.sd_wrq) {
	    enter_nm("no stream write queue");
	    return(1);
	}
/*
 * Examine the write queue.
 */
	for (qp = (KA_T)sd.sd_wrq, *mn = '\0'; qp; qp = (KA_T)q->q_next) {

	/*
	 * Read stream queue entry.
	 */
	    if (kread(qp, (char *)q, sizeof(struct queue))) {
		(void) sprintf(Namech,
		    "vnode@%s; can't read stream queue@%s",
			print_kptr(na, tbuf), print_kptr(qp, (char *)NULL));
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read queue's information structure.
	 */
	    if (!q->q_qinfo || readstqinit((KA_T)q->q_qinfo, &qi)) {
		(void) sprintf(Namech, "vnode@%s; can't read qinit@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)q->q_qinfo, (char *)NULL));
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read module information structure.
	 */
	    if (!qi.qi_minfo || readstmin((KA_T)qi.qi_minfo, &mi)) {
		(void) sprintf(Namech, "vnode@%s; can't read module info@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)qi.qi_minfo, (char *)NULL));
		enter_nm(Namech);
		return(1);
	    }
	/*
	 * Read module name.
	 */
	    if (!mi.mi_idname || kread((KA_T)mi.mi_idname, mn, STRNML-1)) {
		(void) sprintf(Namech, "vnode@%s; can't read module name@%s",
		    print_kptr(na,tbuf),
		    print_kptr((KA_T)mi.mi_idname, (char *)NULL));
		enter_nm(Namech);
		return(1);
	    }
	    *(mn + STRNML - 1) = '\0';
	/*
	 * Save the q_ptr of the first special module.
	 */
	    if (!sn || *sqp || !q->q_ptr)
		continue;
	    if (strcmp(mn, sn) == 0)
		*sqp = (KA_T)q->q_ptr;
	}
	return(0);
}


/*
 * findstrdev() - look up stream device by device number
 */

static struct l_dev *
findstrdev(dev)
	dev_t *dev;			/* device */
{
	struct clone *c;
	struct l_dev *dp;
/*
 * Search device table for match.
 */

#if	defined(HASDCACHE)

findstrdev_again:

#endif	/* defined(HASDCACHE) */

	if ((dp = lkupdev(dev, 0, 0)))
	    return(dp);
/*
 * Search for clone.
 */
	if (Clone) {
	    for (c = Clone; c; c = c->next) {
		if (major(*dev) == minor(Devtp[c->dx].rdev)) {

#if	defined(HASDCACHE)
		    if (DCunsafe && !Devtp[c->dx].v && !vfy_dev(&Devtp[c->dx]))
			goto findstrdev_again;
#endif	/* defined(HASDCACHE) */

		    return(&Devtp[c->dx]);
		}
	    }
	}
	return((struct l_dev *)NULL);
}


/*
 * isvlocked() - is a vnode locked
 */

static char
isvlocked(va)
	struct vnode *va;		/* local vnode address */
{
	struct filock f;
	KA_T flf, flp;
	int l;

	if (!(flf = (KA_T)va->v_filocks))
	    return(' ');
	flp = flf;
	do {
	    if (kread(flp, (char *)&f, sizeof(f)))
		break;
	    if (f.set.l_sysid || f.set.l_pid != (pid_t)Lp->pid)
		continue;
	    if (f.set.l_whence == 0 && f.set.l_start == 0
	    &&  (f.set.l_len == 0 || f.set.l_len == 0x7fffffff))
		l = 1;
	    else
		l = 0;
	    switch (f.set.l_type & (F_RDLCK | F_WRLCK)) {
	    case F_RDLCK:
		return((l) ? 'R' : 'r');
	    case F_WRLCK:
		return((l) ? 'W' : 'w');
	    case (F_RDLCK + F_WRLCK):
		return('u');
	    default:
		return(' ');
	    }
	} while ((flp = (KA_T)f.next) && flp != flf);
	return(' ');
}


/*
 * process_node() - process node
 */

void
process_node(na)
	KA_T na;			/* vnode kernel space address */
{
	char *cp, tbuf[32];
	dev_t dev;
	struct l_dev *dp;
	int ds = 0;
	struct fifonode f;
	int fx;
	struct l_ino i;
	int j, k;
	struct vfs kv;
	int kvs = 0;
	struct module_info mi;
	char mn[STRNML];
	int ni = 0;
	int px;
	struct queue q;
	struct rnode r;
	struct snode rs;
	struct vnode rv, v;
	struct snode s;
	unsigned char sd = 1;
	struct so_so so;
	KA_T sqp = (KA_T)NULL;
	struct ttyfsnode {		/* This should be defined in a Pyramid
					 * /usr/include header file, but there
					 * doesn't appear to be one. */
	    struct vnode *t_realvp;
	    struct vnode *t_ttyvp;
	} tn;
	static KA_T tvop = (KA_T)0;
	static int tvs = 0;
	char *ty;
	enum vtype type;
	struct sockaddr_un ua;
	struct l_vfs *lvfs;

#if	defined(HASPROCFS)
	struct as as;
	struct proc p;
	struct procfsid *pfi;
	struct pid pids;
	struct prnode pr;
#endif	/* defined(HASPROCFS) */

#if	defined(HASOBJFS)
	struct objnode ob;
#endif	/* defined(HASOBJFS) */


/*
 * Get ttyfs_vnodeops address.
 */
	if (!tvs) {
	    if (get_Nl_value("ttyfs", Drive_Nl, &tvop) < 0 || !tvop)
		tvop = (KA_T)0;
	    tvs = 1;
	}
/*
 * Read the vnode.
 */
	if (!na) {
	    enter_nm("no vnode address");
	    return;
	}
	if (readvnode(na, &v)) {
            enter_nm(Namech);
            return;
        }

# if    defined(HASNCACHE)
	Lf->na = na;
# endif /* defined(HASNCACHE) */

/*
 * Determine the vnode type.
 */
	if (v.v_vfsp
	&&  !kread((KA_T)v.v_vfsp, (char *)&kv, sizeof(kv))) {
	    kvs = 1;
	/*
	 * Check the file system type.
	 */
	    fx = kv.vfs_fstype;
	    if (fx > 0 && fx <= Fsinfomax) {
		if (strcmp(Fsinfo[fx-1], "fifofs") == 0)
		    Ntype = N_FIFO;
		else if (strcmp(Fsinfo[fx-1], "nfs") == 0)
		    Ntype = N_NFS;

#if	defined(HASPROCFS)
		else if (strcmp(Fsinfo[fx-1], "proc") == 0)
		    Ntype = N_PROC;
#endif	/* defined(HASPROCFS) */

#if	defined(HASOBJFS)
		else if (strcmp(Fsinfo[fx-1], "objfs") == 0)
		    Ntype = N_OBJF;
#endif	/* defined(HASOBJFS) */

	    }
	} else
	    fx = Fsinfomax;
	if (Ntype == N_REGLR) {
	    if (v.v_type == VFIFO)
		Ntype = N_FIFO;
	    else if (v.v_stream) {
		Ntype = N_STREAM;
		Lf->is_stream = 1;
	    }
	}
/*
 * Determine the lock state.
 */
	Lf->lock = isvlocked(&v);
/*
 * Establish the local virtual file system structure.
 */
	if (!v.v_vfsp)
	    lvfs = (struct l_vfs *)NULL;
	else {
	    if (!(lvfs = readvfs((KA_T)v.v_vfsp,
				 kvs ? &kv : (struct vfs *)NULL)))
	    {
		(void) sprintf(Namech, "vnode@%s; can't read vfs@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_vfsp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Read the fifonode, inode, pipenode, prnode, rnode, or snode.
 */
	switch (Ntype) {
	case N_FIFO:
	    if (!v.v_data || readfifonode((KA_T)v.v_data, &f)) {
		(void) sprintf(Namech, "vnode@%s; can't read fifonode@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (f.fn_realvp) {
		if (readvnode((KA_T)f.fn_realvp, &rv)) {
		    (void) sprintf(Namech,
			"vnode @%s; fifonode@%s; can't read real vnode@%s",
			print_kptr(na, tbuf),
			print_kptr((KA_T)v.v_data, (char *)NULL),
			print_kptr((KA_T)f.fn_realvp, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		if (!rv.v_data) {
		    f.fn_realvp = (struct vnode *)NULL;
		    ni = 1;
		} else if (readlino(fx, &rv,
				    &i, kvs ? &kv : (struct vfs *)NULL))
		{
		    (void) sprintf(Namech,
			"vnode @%s; fifonode@%s; can't read inode@%s",
			print_kptr(na, tbuf),
			print_kptr((KA_T)v.v_data, (char *)NULL),
			print_kptr((KA_T)rv.v_data, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
	    } else
		ni = 1;
	    break;
	case N_NFS:
	    if (!v.v_data || readrnode((KA_T)v.v_data, &r)) {
		(void) sprintf(Namech, "vnode@%s; can't read rnode@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    break;

#if	defined(HASOBJFS)
	case N_OBJF:
	    if (!v.v_data
	    ||  kread((KA_T)v.v_data, (char *)&ob, sizeof(ob))) {
		(void) sprintf(Namech, "vnode@%s; can't read objnode@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    break;
#endif	/* defined(HASOBJFS) */

#if	defined(HASPROCFS)
	case N_PROC:
	    ni = 1;
	    if (!v.v_data || kread((KA_T)v.v_data, (char *)&pr, sizeof(pr))) {
		(void) sprintf(Namech, "vnode@%s; can't read prnode@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (pr.pr_proc == NULL) {
		sd = 0;
		if (v.v_type == VDIR)
		    i.number = PR_ROOTINO;
		else
		    i.number = 0L;
		break;
	    }
	    if (kread((KA_T)pr.pr_proc, (char *)&p, sizeof(p))) {
		(void) sprintf(Namech,
		    "vnode@%s; prnode@%s; can't read proc@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL),
		    print_kptr((KA_T)pr.pr_proc, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (!p.p_pidp || kread((KA_T)p.p_pidp, (char *)&pids, sizeof(pids)))
	    {
		(void) sprintf(Namech,
		    "vnode@%s; proc struct@%s; can't read pid@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)pr.pr_proc, (char *)NULL),
		    print_kptr((KA_T)p.p_pidp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (!p.p_as ||  kread((KA_T)p.p_as, (char *)&as, sizeof(as)))
		sd = 0;
	    else
		i.size = as.a_size;
	    (void) sprintf(Namech, "/%s/%d", HASPROCFS, pids.pid_id);
	    i.number = (long)pids.pid_id + PR_INOBIAS;
	    enter_nm(Namech);
	    break;
#endif	/* defined(HASPROCFS) */

	case N_STREAM:
	    if (v.v_stream) {

	    /*
	     * Get the queue pointer and module name at the end of the stream.
	     * The module name identifies socket streams.
	     */
		if (examine_stream(na, (KA_T)v.v_stream, &q, mn, "sockmod",
			&sqp))
		    return;
		for (px = 0; px < NPROTOS; px++) {
		    if (strcmp(mn, Protos[px].module) == 0) {
			process_socket(Protos[px].proto, &q);
			return;
		    }
		}
	    /*
	     * If this stream has a "sockmod" module with a non-NULL q_ptr,
	     * try to use it to read an so_so structure.
	     */
		if (sqp && kread(sqp, (char *)&so, sizeof(so)) == 0)
		    break;
		sqp = (KA_T)NULL;
		(void) strcpy(Namech, "STR");
		j = strlen(Namech);
		if (v.v_type == VCHR) {
	    /*
	     * If this is a VCHR stream, look up the device name and record it.
	     */
		    if ((dp = findstrdev((dev_t *)&v.v_rdev))) {
			Namech[j++] = ':';
			k = strlen(dp->name);
			if ((j + k) <= (MAXPATHLEN - 1)) {
			    (void) strcpy(&Namech[j], dp->name);
			    j += k;
			    if ((cp = strrchr(Namech, '/'))
			    &&  *(cp + 1) == '\0')
			    {
				*cp = '\0';
				j--;
			    }
			}
		    }
		}
	    /*
	     * Follow the "STR" and possibly the device name with "->" and
	     * the module name.
	     */
		if ((j + 2) <= (MAXPATHLEN - 1)) {
		    (void) strcpy(&Namech[j], "->");
		    j += 2;
		}
		if (mn[0]) {
		    if ((j + strlen(mn)) <= (MAXPATHLEN - 1))
			(void) strcpy(&Namech[j], mn);
		} else {
		    if ((j + strlen("none")) <= (MAXPATHLEN - 1))
			(void) strcpy(&Namech[j], "none");
	        }
	    }
	    break;
	case N_REGLR:
	default:

	/*
	 * Follow a VCHR vnode to its snode, then to its real vnode, finally
	 * to its inode.
	 */
	    if (v.v_type == VCHR) {
		if (v.v_data && v.v_op
		&&  tvop && tvop == (KA_T)v.v_op) {

		/*
		 * If this is a ttyfs VCHR vnode, read the ttyfsnode to get
		 * the real vnode pointer, and use its v_data pointer to
		 * read the snode.
		 */
		    if (kread((KA_T)v.v_data, (char *)&tn, sizeof(tn))) {
			(void) sprintf(Namech,
			    "vnode@%s; can't read ttyfsnode@%s",
			    print_kptr(na, tbuf),
			    print_kptr((KA_T)v.v_data, (char *)NULL));
			(void) enter_nm(Namech);
			return;
		    }
		    if (!tn.t_realvp || readvnode((KA_T)tn.t_realvp, &rv)) {
			(void) sprintf(Namech,
			    "vnode@%s; ttyfsnode@%s; can't read real vnode@%s",
			    print_kptr(na, tbuf),
			    print_kptr((KA_T)v.v_data, (char *)NULL),
			    print_kptr((KA_T)tn.t_realvp, (char *)NULL));
			(void) enter_nm(Namech);
			return;
		    }
		    v.v_data = rv.v_data;
		}
		if (!v.v_data || readsnode((KA_T)v.v_data, &s)) {
		    (void) sprintf(Namech, "vnode@%s; can't read snode@%s",
			print_kptr(na, tbuf),
			print_kptr((KA_T)v.v_data, (char *)NULL));
		    enter_nm(Namech);
		    return;
		}
		if (s.s_realvp) {
		    if (readvnode((KA_T)s.s_realvp, &rv)) {
			(void) sprintf(Namech,
			    "vnode@%s; snode@%s; can't read real vnode@%s",
			    print_kptr(na, tbuf),
			    print_kptr((KA_T)v.v_data, (char *)NULL),
			    print_kptr((KA_T)s.s_realvp, (char *)NULL));
			enter_nm(Namech);
			return;
		    }
		    if (!rv.v_data || readlino(fx, &rv, &i,
			kvs ? &kv : (struct vfs *)NULL))
		    {
			(void) sprintf(Namech,
			    "vnode@%s; snode@%s; can't read inode@%s",
			    print_kptr(na, tbuf),
			    print_kptr((KA_T)v.v_data, (char *)NULL),
			    print_kptr((KA_T)rv.v_data, (char *)NULL));
			enter_nm(Namech);
			return;
		     }
		}
	    /*
	     * If there's no real vnode, look for a common vnode and a
	     * common snode.
	     */
		else if (s.s_commonvp) {
		    if (readvnode((KA_T)s.s_commonvp, &rv)) {
			(void) sprintf(Namech,
			    "vnode@%s; snode@%s; can't read common vnode@%s",
			    print_kptr(na, tbuf),
			    print_kptr((KA_T)v.v_data, (char *)NULL),
			    print_kptr((KA_T)s.s_commonvp, (char *)NULL));
			enter_nm(Namech);
			return;
		    }
		    if (!rv.v_data || readsnode((KA_T)rv.v_data, &rs)) {
			(void) sprintf(Namech, "vnode@%s; can't read snode@%s",
			    print_kptr((KA_T)s.s_commonvp, tbuf),
			    print_kptr((KA_T)rv.v_data, (char *)NULL));
			enter_nm(Namech);
			return;
		    }
		    ni = 1;
		} else
		    ni = 1;
		break;
	    }
	    if (!v.v_data) {
		(void) sprintf(Namech, "vnode@%s; no further information",
		    print_kptr(na, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	/*
	 * Read inode information.
	 */
	    if (readlino(fx, &v, &i, kvs ? &kv : (struct vfs *)NULL)) {
		(void) sprintf(Namech,
		    "vnode@%s; can't read successor node@%s",
		    print_kptr(na, tbuf),
		    print_kptr((KA_T)v.v_data, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	}
/*
 * Get device and type for printing.
 */
	switch (Ntype) {
	case N_NFS:
	    dev = r.r_attr.va_fsid;
	    ds = 1;
	    break;

#if	defined(HASOBJFS)
	case N_OBJF:
	    dev = ob.o_info.oi_dev;
	    ds = 1;
	    break;
#endif	/* defined(HASOBJFS) */

#if	defined(HASPROCFS)
	case N_PROC:
	    dev = kv.vfs_dev;
	    ds = 1;
	    break;
#endif	/* defined(HASPROCFS) */

	case N_STREAM:
	    if (sqp) {
		dev = so.lux_dev.addr.tu_addr.dev;
		ds = 1;
		break;
	    }
	    /* fall through */
	default:
	    if (v.v_type == VCHR) {
		dev = v.v_rdev;
		ds = 1;
	    } else if (!ni) {
		dev = i.dev;
		ds = 1;
	    }
	}
	type = v.v_type;
	if (lvfs && ds && !lvfs->dir)
	    (void) completevfs(lvfs, &dev);
/*
 * Obtain the inode number.
 */
	switch (Ntype) {
	case N_NFS:
	    Lf->inode = (unsigned long)r.r_attr.va_nodeid;
	    Lf->inp_ty = 1;
	    break;

#if	defined(HASOBJFS)
	case N_OBJF:
	    Lf->inode = (unsigned long)ob.o_inum;
	    Lf->inp_ty = 1;
	    break;
#endif	/* defined(HASOBJFS) */

#if	defined(HASPROCFS)
	case N_PROC:
	    Lf->inode = (unsigned long)i.number;
	    Lf->inp_ty = 1;
	    break;
#endif	/* defined(HASPROCFS) */

	case N_FIFO:
	    if (!f.fn_realvp) {
		enter_dev_ch(print_kptr((KA_T)v.v_data, (char *)NULL));
		if (f.fn_ino) {
		    Lf->inode = (unsigned long)f.fn_ino;
		    Lf->inp_ty = 1;
		}
		if (f.fn_flag & ISPIPE)
		    (void) strcpy(Namech, "PIPE");
		if (f.fn_mate)
		    (void) sprintf(endnm(), "->%s",
			print_kptr((KA_T)f.fn_mate, (char *)NULL));
		break;
	    }
	    /* fall through */
	case N_REGLR:
	    if (!ni) {
		Lf->inode = (unsigned long)i.number;
		Lf->inp_ty = 1;
	    }
	    break;
	case N_STREAM:
	    if (sqp) {
		Lf->inode = (unsigned long)so.lux_dev.addr.tu_addr.ino;
		Lf->inp_ty = 1;
	    }
	}
/*
 * Obtain the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {
	    switch (Ntype) {
	    case N_FIFO:
	    case N_STREAM:
		if (!Fsize)
		    Lf->off_def = 1;
		break;
	    case N_NFS:
		Lf->sz = (SZOFFTYPE)r.r_attr.va_size;
		Lf->sz_def = sd;
		break;

#if	defined(HASOBJFS)
	    case N_OBJF:
		Lf->sz = (SZOFFTYPE)ob.o_size;
		Lf->sz_def = sd;
		break;
#endif	/* defined(HASOBJFS) */

#if	defined(HASPROCFS)
	    case N_PROC:
		Lf->sz = (SZOFFTYPE)i.size;
		Lf->sz_def = sd;
		break;
#endif	/* defined(HASPROCFS) */

	    case N_REGLR:
		if (type == VREG || type == VDIR) {
		    if (!ni) {
			Lf->sz = (SZOFFTYPE)i.size;
			Lf->sz_def = sd;
		    }
		} else if (type == VCHR && !Fsize)
		    Lf->off_def = 1;
		break;
	    }
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (lvfs) {
	    Lf->fsdir = lvfs->dir;
	    Lf->fsdev = lvfs->fsname;
	}
/*
 * Format the vnode type, and possibly the device name.
 */
	switch (type) {
	case VNON:
	    ty ="VNON";
	    break;
	case VREG:
	case VDIR:
	    ty = (type == VREG) ? "VREG" : "VDIR";
	    Lf->dev_def = ds;
	    Lf->dev = dev;
	    break;
	case VBLK:
	    ty = "VBLK";
	    Lf->dev = dev;
	    Lf->dev_def = ds;
	    Ntype = N_BLK;
	    break;
	case VCHR:
	    Lf->dev = dev;
	    Lf->dev_def = ds;
	    ty = "VCHR";
	    if (Lf->is_stream == 0)
		Ntype = N_CHR;
	    break;
	case VLNK:
	    ty = "VLNK";
	    break;

#if	defined(VSOCK)
	case VSOCK:
	    ty = "SOCK";
	    break;
#endif	/* VSOCK */

	case VBAD:
	    ty = "VBAD";
	    break;
	case VFIFO:
	    if (!Lf->dev_ch || Lf->dev_ch[0] == '\0') {
		Lf->dev = dev;
		Lf->dev_def = ds;
	    }
	    ty = "FIFO";
	    break;
	default:
	    if (type > 9999)
		(void) sprintf(Lf->type, "*%03d", type % 1000);
	    else
		(void) sprintf(Lf->type, "%4d", type);
	    (void) strcpy(Namech, "unknown type");
	    ty = NULL;
	}
	if (ty)
	    (void) strcpy(Lf->type, ty);
	Lf->ntype = Ntype;
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (!Lf->inp_ty && Lf->ntype == N_BLK && Lf->dev_def)
	    find_bl_ino();
/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (!Lf->inp_ty && type == VCHR && Lf->dev_def)
	    find_ch_ino();

#if	defined(RELIANTV)
/*
 * If this is a Reliant UNIX VCHR file, missing an inode number, see if it's
 * an "mipc" file.
 */
	if (!Lf->inp_ty && Lf->dev_def && type == VCHR && !Namech[0]
	&&  HaveMipcMaj && (int)major(Lf->dev) == MipcMaj)
	{
	    dev_t md;
	    int ns;

	    Lf->inode = mipc_index(minor(Lf->dev)) & MIPC_INDEX_MASK;
	    Lf->inp_ty = 1;
	    ns = mipc_ns(minor(Lf->dev));
	    md = makedevice(MipcMaj, ns);
	    if ((dp = lkupdev(&md, 0, 0)))
		(void) sprintf(Namech, "%s (ns %d)", dp->name, ns);
	    else
		(void) sprintf(Namech, "MIPC (ns %d)", ns);
	}
#endif	/* defined(RELIANTV) */

/*
 * If this is a stream with a "sockmod" module whose q_ptr leads to an
 * so_so structure, assume it's a UNIX domain socket and try to get
 * the path.  Clear the is_stream status.
 */
	if (Ntype == N_STREAM && sqp) {
	    if (Funix)
		Lf->sf |= SELUNX;
	    (void) strcpy(Lf->type, "unix");
	    if (!Namech[0]
	    &&  so.laddr.buf && so.laddr.len == sizeof(ua)
	    &&  !kread((KA_T)so.laddr.buf, (char *)&ua, sizeof(ua))) {
		ua.sun_path[sizeof(ua.sun_path) - 1] = '\0';
		(void) strcpy(Namech, ua.sun_path);
	    }
	    Lf->is_stream = 0;
	}
/*
 * Test for specified file.
 */

#if	defined(HASPROCFS)
	if (Ntype == N_PROC) {
	    if (Procsrch) {
		Procfind = 1;
		Lf->sf |= SELNM;
	    } else {
		for (pfi = Procfsid; pfi; pfi = pfi->next) {
		    if ((pfi->pid && pfi->pid == pids.pid_id)

# if	defined(HASPINODEN)
		    ||  (Lf->inp_ty == 1 && Lf->inode == pfi->inode)
# endif	/* defined(HASPINODEN) */

		    ) {
			pfi->f = 1;
			if (!Namech[0]) {
			    (void) strncpy(Namech, pfi->nm, MAXPATHLEN - 1);
			    Namech[MAXPATHLEN - 1] = '\0';
			}
			Lf->sf |= SELNM;
			break;
		    }
		}
	    }
	} else
#endif	/* defined(HASPROCFS) */

	{
	    if (Sfile && is_file_named(NULL, type))
		Lf->sf |= SELNM;
	}
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}


/*
 * readlino() - read local inode information
 */

static int
readlino(fx, v, i, kv)
	int fx;				/* file system index */
	struct vnode *v;		/* vnode pointing to inode */
	struct l_ino *i;		/* local inode */
	struct vfs *kv;			/* local copy of kernel vfs structure */
{
	struct namenode nm;
	struct inode sn;
	struct xnamnode xn;

	if (fx < 1 || fx > Fsinfomax || v->v_data == NULL)
	    return(1);
	if (strcmp(Fsinfo[fx-1], "fifofs") == 0
	||  strcmp(Fsinfo[fx-1], "ufs") == 0) {
	    if (kread((KA_T)v->v_data, (char *)&sn, sizeof(sn)))
		return(1);
	    i->dev = sn.i_dev;
	    i->rdev = v->v_rdev;
	    i->number = sn.i_number;
	    i->size = sn.i_size;
	    return(0);
	} else if (!strcmp(Fsinfo[fx-1], "s5"))
	    return(reads5lino(v, i));
 	else if (!strcmp(Fsinfo[fx-1], "rxfs") || !strcmp(Fsinfo[fx-1], "vxfs"))
 	    return(readvxfslino(v, i));
	else if (!strcmp(Fsinfo[fx-1], "namefs")) {
	    if (kread((KA_T)v->v_data, (char *)&nm, sizeof(nm)))
		return(1);
	    i->dev = nm.nm_vattr.va_fsid;
	    i->rdev = nm.nm_vattr.va_rdev;
	    i->number = nm.nm_vattr.va_nodeid;
	    i->size = nm.nm_vattr.va_size;
	    return(0);
	} else if (!strcmp(Fsinfo[fx-1], "tfs"))
	    return(readtfslino(v, i, kv));
	else if (!strcmp(Fsinfo[fx-1], "xnamfs")
	     ||  !strcmp(Fsinfo[fx-1], "XENIX"))
	{
	    if (kread((KA_T)v->v_data, (char *)&xn, sizeof(xn)))
		return(1);
	    i->dev = xn.x_dev;
	    i->rdev = xn.x_fsid;
	    i->size = xn.x_size;
	    return(0);
	}
	return(1);
}
