/*
 * mknbi.c  -  MaKe NetBoot Image for Menu Generation Language
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mknbi.h"
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifndef _MKNBI_H_MGL_
#error Included wrong header file
#endif


char *progname;
char mglname[MAXPATHLEN];	/* Name of mgl source file */
char outname[MAXPATHLEN];	/* Name of output file */
int outfile;			/* File handle for output file */
int is186 = TRUE;		/* Use 186+ optimizations */
int is386 = TRUE;		/* Use 386+ optimizations */
int debug = 0;			/* Include debugging info */
int verbose = 0;		/* Verbosity level; currently just 0, 1 or 2 */
int noheader = 0;		/* Don't include a boot image header */

int cur_rec_num = -1;		/* Number of current load record */
struct load_header header;	/* Load header */
struct load_record *cur_rec;	/* Pointer to current load record */



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(recnum, src, size)
int recnum;
char *src;
long size;
{
  unsigned long l;
  size_t isize;
  char *buf;

  if (cur_rec_num != recnum) {
    fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
	    progname);
    exit(EXIT_MGL_MISORDER);
  }
  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  if ((buf = malloc(isize)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  memset(buf, 0, isize);
  memcpy(buf, src, size);
  if (write(outfile, buf, isize) != isize) {
	perror(outname);
	exit(EXIT_WRITE);
  }
  free(buf);
  l = get_long(cur_rec->ilength) + isize;
  assign(cur_rec->ilength.low, htot(low_word(l)));
  assign(cur_rec->ilength.high, htot(high_word(l)));
  l = get_long(cur_rec->mlength) + isize;
  assign(cur_rec->mlength.low, htot(low_word(l)));
  assign(cur_rec->mlength.high, htot(high_word(l)));
}



/*
 * Copy a certain number of bytes from the compiled image file into the
 * output file
 */
static void copyrec(recnum, image)
int recnum;
int image;
{
  unsigned char copyrec_buf[SECTSIZE];
  int i = 1;

  if (cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
		progname);
	exit(EXIT_MGL_MISORDER);
  }
  while (i > 0) {
	if ((i = read(image, copyrec_buf, SECTSIZE)) < 0) {
		perror("temporary file");
		exit(EXIT_READ);
	}
	if (i)
		putrec(recnum, (char *)copyrec_buf, i);
  }
}



/*
 * Initialize a load record
 */
static void initrec(recnum, segment, flags, vendor_size)
int recnum;
int segment;
int flags;
int vendor_size;
{
  if (++cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
		progname);
	exit(EXIT_MGL_MISORDER);
  }

  if (cur_rec_num > 0)
	cur_rec = (struct load_record *)((unsigned char *)cur_rec +
					((cur_rec->rlength << 2) & 0x3c) +
					((cur_rec->rlength >> 2) & 0x3c));
  cur_rec->rlength      = (sizeof(struct load_record)) >> 2;
  cur_rec->rlength     |= ((vendor_size + 3) & 0x3c) << 2;
  cur_rec->rtag1        = recnum + VENDOR_OFF;
  cur_rec->rflags       = flags;
  assign(cur_rec->address.low, htot(low_word((unsigned long) segment << 4)));
  assign(cur_rec->address.high, htot(high_word((unsigned long) segment << 4)));
}



/*
 * Dump the load record information to stderr
 */
static void dump_header(lh)
struct load_header *lh;
{
  static char *s_tags[] = { /* PROGNUM */    "compiled program"};
  static char *s_flags[]= { "absolute address", "after previous segment",
			    "at end of memory", "before previos segment"};
  struct load_record *lr;
  int    i, num = 0;

  fprintf(stderr,"\n"
	  "Load record information:\n"
	  "  Magic number:     0x%08lX\n"
	  "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	  "  Flags:            0x%08lX\n"
	  "  Location address: %04X:%04X\n"
	  "  Execute address:  %04X:%04X\n"
	  "  Vendor data:      %s\n"
	  "\n",
	  get_long(lh->magic),
	  (lh->hlength << 2) & 0x3c,
	  (lh->hlength >> 2) & 0x3c,
	  (unsigned long)lh->hflags1 +
		((unsigned long)lh->hflags2 << 8) +
		((unsigned long)lh->hflags3 << 16),
	  ttoh(getval(lh->locn.segment)), ttoh(getval(lh->locn.offset)),
	  ttoh(getval(lh->execute.segment)), ttoh(getval(lh->execute.offset)),
	  lh->dummy);

  i  = ((lh->hlength >> 2) & 0x3c) + ((lh->hlength << 2) & 0x3c);
  lr = (struct load_record *)&(((__u8 *)lh)[i]);

  for (;;) {
  	fprintf(stderr,
	    "Record #%d:\n"
	    "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	    "  Vendor tag:       0x%02X (%s)\n"
	    "  Reserved flags:   0x%02X\n"
	    "  Flags:            0x%02X (%s%s)\n"
	    "  Load address:     0x%08lX%s\n"
	    "  Image length:     0x%08lX bytes\n"
	    "  Memory length:    0x%08lX bytes\n"
	    "\n",
	    ++num,
	    (lr->rlength << 2) & 0x3c,
	    (lr->rlength >> 2) & 0x3c,
	    (int)lr->rtag1,
	    lr->rtag1 < 16 || lr->rtag1-16 >= NUM_RECORDS ? "unknown" : s_tags[lr->rtag1-16],
	    (int)lr->rtag2,
	    (int)lr->rflags, s_flags[lr->rflags & 0x03],
	    lr->rflags & FLAG_EOF ? ", last record" : "",
	    get_long(lr->address),
	    get_long(lr->address) >= 0x100000L ? " (high memory)" : "",
	    get_long(lr->ilength),
	    get_long(lr->mlength));

	if (lr->rflags & FLAG_EOF)
		break;

	i  = ((lr->rlength >> 2) & 0x3c) + ((lr->rlength << 2) & 0x3c);
	lr = (struct load_record *)&(((__u8 *)lr)[i]);
  }
}



/*
 * Print the usage
 */
static void usage()
{
  fprintf(stderr,
	"Usage: %s [-c] [-g] [-0|-1|-3] [-x] [[-i] <source file>] [-o] <outfile>\n"
	"       %s [-h]\n"
	"       %s [-v]\n",
	  progname, progname, progname);
  exit(EXIT_USAGE);
}



/*
 * Main program
 */
void main(argc, argv)
int argc;
char **argv;
{
  char *cp;
  int vendor_size;
  int tmpfile;
  int i;

  /* Determine my own name for error output */
  if ((cp = strrchr(argv[0], '/')) == NULL)
	progname = argv[0];
  else
	progname = ++cp;

  /* Initialize option argments */
  strcpy(mglname, "");
  strcpy(outname, "");

  /* Parse options */
  opterr = 0;
  while ((i = getopt(argc, argv, "013cghi:o:vx")) != EOF)
	switch (i) {
		case 'i': strncpy(mglname, optarg, MAXPATHLEN-1);
		          mglname[MAXPATHLEN-1] = '\0';
		          break;
		case 'o': strncpy(outname, optarg, MAXPATHLEN-1);
		          outname[MAXPATHLEN-1] = '\0';
		          break;
		case 'v': fprintf(stderr, VERSION"\n");
		          exit(EXIT_SUCCESS);
		case '0': is186 = FALSE;
		          is386 = FALSE;
		          break;
		case '1': is186 = TRUE;
		          is386 = FALSE;
		          break;
		case '3': is186 = TRUE;
		          is386 = TRUE;
		          break;
		case 'g': debug++;
		          break;
		case 'c': noheader++;
		          break;
		case 'x': verbose++;
		          break;
		case 'h': /* fall thru */
		default:  usage();
	}

  /* Parse additional arguments */
  if (optind < argc) {
	strncpy(mglname, argv[optind++], MAXPATHLEN-1);
	mglname[MAXPATHLEN-1] = '\0';
  }
  if (optind < argc) {
	strncpy(outname, argv[optind++], MAXPATHLEN-1);
	outname[MAXPATHLEN-1] = '\0';
  }
  if (optind != argc || !*outname)
	usage();

  /* Open the output file */
  if ((outfile = creat(outname, 0644)) < 0) {
	perror(outname);
	exit(EXIT_MGL_IMGCREATE);
  }
  if (verbose > 0) {
	if (!*mglname)
		fprintf(stderr, "Source file name  = <stdin>\n");
	else
		fprintf(stderr, "Source file name  = %s\n", mglname);
	fprintf(stderr, "Output file name  = %s\n", outname);
  }

  /* Initialize the boot header */
  vendor_size = ((sizeof(VENDOR_ID) + 1) / sizeof(__u32)) * sizeof(__u32);
  memset(&header, 0, sizeof(header));
  assign(header.magic.low,       htot(low_word(HEADER_MAGIC)));
  assign(header.magic.high,      htot(high_word(HEADER_MAGIC)));
  assign(header.locn.segment,    htot(HEADERSEG));
  assign(header.locn.offset,     htot(0));
  assign(header.execute.segment, htot(PROGSEG));
  assign(header.execute.offset,  htot(0));
  assign(header.bootsig,         htot(BOOT_SIGNATURE));
  header.hlength  = (__u8)(((int)header.dummy - (int)&header) /
                     sizeof(__u32)) & 0x0f;
  header.hlength |= (__u8)((vendor_size/sizeof(__u32)) << 4) & 0xf0;
  header.hflags1  = HEADER_RETFLAG;
  memcpy(header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  if (!noheader && write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit(EXIT_WRITE);
  }

  /* Initialize pointer to first load record */
  cur_rec = (struct load_record *)&(header.dummy[vendor_size]);

  /* Compile the source file */
  initrec(PROGNUM, PROGSEG, FLAG_EOF, 0);
  tmpfile = gencode(mglname);
  copyrec(PROGNUM, tmpfile);
  assign(cur_rec->mlength.low, htot(low_word(PROGMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(PROGMSIZE)));

  /* After writing out all this stuff, finally update the boot header */
  if (!noheader) {
	if (lseek(outfile, 0, 0) != 0) {
		perror(outname);
		exit(EXIT_SEEK);
	}
	if (write(outfile, &header, sizeof(header)) < 0) {
		perror(outname);
		exit(EXIT_WRITE);
	}
  }

  /* If user asked for detailed output, parse the header and output all of */
  /* the load record information */
  if (verbose > 1)
	dump_header(&header);

  exit(EXIT_SUCCESS);
}
