!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  utility.S
!* Purpose: Various functions for the image loader routines, which are
!*          better written in assembler
!* Entries: _convmem, _extmem, _lmove, _exec_image
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>


!
!**************************************************************************
!
! Layout of descriptors in GDT:
!
gdt_length	equ	0		! length of segment
gdt_base	equ	2		! base of segment
gdt_flags	equ	5		! descriptor flags
gdt_res		equ	6		! reserved

gdt_size	equ	8		! size of descriptor

gdt_stdflag	equ	$93		! standard flags for GDT entries


!
!**************************************************************************
!
! BSS segment
!
	.bss

	.lcomm	gdt_table,6 * gdt_size	! descriptor table for move routine


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	_convmem		! define entry points
	public	_extmem
	public	_lmove
	public	_exec_image

	extrn	_fatal
	extrn	reg_packet
	extrn	unreg_packet


!
!**************************************************************************
!
! Get amount of conventional memory in bytes.
! Input:  none
! Output: memory amount
!
_convmem:

	int	$12			! just call the BIOS
	jmp	extm1


!
!**************************************************************************
!
! Get amount of extended memory in bytes.
! Input:  none
! Output: memory amount
!
_extmem:

	mov	ah,#$88
	int	$15			! just call the BIOS
extm1:
#ifdef OPT386
	shl	eax,#16			! delete the upper 6 bits and then
	shr	eax,#6			! multiply by 1024
#else
	mov	dx,ax
	shift	(shr,dx,6)		! multiply value with 1024 to get
	shift	(shl,ax,10)		! number of bytes
#endif
	ret


!
!**************************************************************************
!
! Move a memory block into extended memory.
! Input:  1. arg  -  linear address to destination (long word)
!         2. arg  -  near pointer to source memory block
!         3. arg  -  length of memory block
! Output: none
!
_lmove:

	penter
	getcarg	(bx,3)
#ifdef IS386
	getcarg	(edx,0)			! check for copy into lower memory
	cmp	edx,#$00100000
#else
	getcarg	(ax,0)
	getcarg	(dx,1)			! check for copy into lower memory
	cmp	dx,#$0010
#endif
	jae	lmove1

! Handle moves into lower memory

	cld
	push	es
	push	si
	push	di
#ifdef IS386
	mov	di,dx
	shr	edx,#4			! convert linear address into seg:ofs
#else
	mov	di,ax
	shift	(shr,ax,4)		! convert linear address into seg:ofs
	shift	(shl,dx,12)
	or	dx,ax
#endif
	and	di,#$000F		! load destination pointer
	mov	es,dx
	getcarg	(si,2)			! load source pointer
	mov	cx,bx
	inc	cx			! convert byte count into word count
	shr	cx,#1
	rep
	movsw				! do the copy
	pop	di
	pop	si
	pop	es
	jmp	lmove9

! Handle moves into high memory

lmove1:
#ifdef IS386
	mov	dword ptr [gdt_table + 3 * gdt_size + gdt_base + 0],edx
#else
	mov	word ptr [gdt_table + 3 * gdt_size + gdt_base + 0],ax
	mov	byte ptr [gdt_table + 3 * gdt_size + gdt_base + 2],dl
#endif
	mov	byte ptr [gdt_table + 3 * gdt_size + gdt_flags],#gdt_stdflag
	mov	word ptr [gdt_table + 3 * gdt_size + gdt_length],bx
	mov	word ptr [gdt_table + 3 * gdt_size + gdt_res],#0
#ifdef IS386
	xor	eax,eax
	mov	ax,ds
	shl	eax,#4
	getlarg	(edx,2,word)
	add	eax,edx
	mov	dword ptr [gdt_table + 2 * gdt_size + gdt_base + 0],eax
#else
	mov	ax,ds
	shift	(rol,ax,4)
	mov	cx,ax
	and	ax,#$FFF0		! convert segment and offset into
	and	cx,#$000F		! a linear address
	getcarg	(dx,2)
	add	ax,dx
	adc	cx,#0
	mov	word ptr [gdt_table + 2 * gdt_size + gdt_base + 0],ax
	mov	byte ptr [gdt_table + 2 * gdt_size + gdt_base + 2],cl
#endif
	mov	byte ptr [gdt_table + 2 * gdt_size + gdt_flags],#gdt_stdflag
	mov	word ptr [gdt_table + 2 * gdt_size + gdt_length],bx
	mov	word ptr [gdt_table + 2 * gdt_size + gdt_res],#0

	push	es
	push	si
	mov	cx,bx
	inc	cx			! convert byte count into word count
	shr	cx,#1
	mov	ax,ds
	mov	es,ax
	mov	si,#gdt_table
	mov	ah,#$87
	int	$15			! call BIOS to do the copy
	pop	si
	pop	es
lmove9:	pleave
	ret


!
!**************************************************************************
!
! Execute the boot image
! Input: 1. arg  -  header mode
!        2. arg  -  far pointer to execution point
!        3. arg  -  far pointer to header
!        4. arg  -  near pointer to parameter block
! Output: Routine does not return, except for mode 3. In this case, a non-
!         zero value in AX means to reload a new image.
!
_exec_image:

	penter
	getcarg	(ax,0)
	cmp	ax,#3			! check for mode 3
	jne	exec2

! With mode 3 the loaded image will eventually use bootrom interrupt
! vectors and may also return back to the bootrom, so we should not
! restore the interrupt vector table, and not deinitialize the packet
! driver and all other modules. This also involves not to change the
! stack. Of course the loaded image should be careful when using the
! bootroms stack.

	xor	ax,ax
	call	unreg_packet		! unregister all packet driver handles
	push	si			! save index registers for return
	push	di			! to C code
	push	ds
	push	es
#ifdef IS186
	push	#$1234			! push magic ID
#else
	mov	ax,#$1234
	push	ax
#endif
	getcarg	(es,2)			! get the execution pointer
	getcarg	(bx,1)			! into ES:BX
	getcarg	(ds,4)			! get the header pointer
	getcarg	(si,3)			! into DS:SI
	getcarg	(di,5)			! get the parameter pointer
#ifdef IS186
	push	#NEWDATA
#else
	mov	cx,#NEWDATA		! into CX:DI
	push	cx
#endif
	push	di			! push the parameter pointer
	push	ds
	push	si			! push the header pointer
	push	cs
#ifdef IS186
	push	#exec3
#else
	mov	ax,#exec3		! push the return pointer
	push	ax
#endif
	push	es			! jump to new image
	push	bx
	retf

exec3:	add	sp,#4 * 2		! restore stack pointer
	pop	bx
	cmp	bx,#$1234		! check for magic ID
	jne	exech
	pop	es
	pop	ds			! restore all previously saved
	pop	di			! registers
	pop	si
	push	ax			! save return value
	call	reg_packet		! reregister all packet driver handles
	pop	ax
	jc	exech
	pleave
	ret

exech:	mov	ax,#NEWDATA		! fatal() needs correct data segment
	mov	ds,ax			! for prnchr routine!
	jmp	near _fatal

! First clean up the system by restoring the interrupt vector table and
! disabling all resident programs and the packet driver.

exec2:
#ifdef USEMODE
	push	ax
#endif
	push	bp
	mov	ah,#1
	int	SERVICES		! handled in kernel.S module
	pop	bp

#ifdef USEMODE
	pop	ax
	cmp	ax,#1			! check for mode 1
	jne	exec1

! Handle mode number 1. This means copying the header block to
! the execution point, and setting the stack just below there.
! Note that the image doesnt get anything on its stack. It
! would be too complicated to check for overwriting the bootp
! block and/or stack with the header block.

	getcarg	(es,2)			! get the pointer to the boot
	getcarg	(di,1)			! area into ES:DI
	mov	bx,di
	getcarg	(ds,4)			! put the pointer to the header
	getcarg	(si,3)			! block into DS:SI
	mov	cx,#512 / 2
	rep
	movsw				! now place the boot block into
	cli				! its final place
	mov	ax,es
	mov	sp,bx
	mov	ss,ax			! set new stack
	sti
	xor	dx,dx			! DL should contain the boot drive ;-))
	jmp	exec9			! jump to the boot image

#endif

! Handle mode 2. This means just to assign a new stack, and then
! push the two parameters onto it before calling the new image.

exec1:	getcarg	(es,2)			! get the execution pointer
	getcarg	(bx,1)			! into ES:BX
	getcarg	(ds,4)			! get the header pointer
	getcarg	(si,3)			! into DS:SI
	getcarg	(di,5)			! get the parameter pointer
	mov	cx,#NEWDATA		! into CX:DI
	cli
	mov	ax,#OLDINTS
	mov	sp,#VECTSIZE		! use the old interrupt table as
	mov	ss,ax			! the new stack
	sti
	push	cx
	push	di			! push the parameter pointer
	push	ds
	push	si			! push the header pointer

! Jump to the boot image. The pointer is in ES:BX.

exec9:	push	cs
#ifdef IS186
	push	#exech
#else
	mov	ax,#exech		! push the return pointer
	push	ax
#endif
	push	es			! jump to new image
	push	bx
	retf

!
!**************************************************************************
!
	end

