/*
 * bltInit.c --
 *
 *	This module initials the BLT toolkit, registering its commands
 *	with the Tcl/Tk interpreter.
 *
 * Copyright 1991-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.  
 */

#include <bltInt.h>

/*
 * The inclusion of contributed commands/widgets can be suppressed by
 * defining the respective preprocessor symbol.
 */

#ifndef NO_HTEXT
extern Tcl_AppInitProc Blt_HtextInit;
#endif
#ifndef NO_GRAPH
extern Tcl_AppInitProc Blt_GraphInit;
#endif
#ifndef NO_TABLE
extern Tcl_AppInitProc Blt_TableInit;
#endif
#ifndef NO_BUSY
extern Tcl_AppInitProc Blt_BusyInit;
#endif
#ifndef NO_WINOP
extern Tcl_AppInitProc Blt_WinOpInit;
#endif
#ifndef NO_BITMAP
extern Tcl_AppInitProc Blt_BitmapInit;
#endif
#ifndef NO_BGEXEC
extern Tcl_AppInitProc Blt_BgExecInit;
#endif
#ifndef NO_DRAGDROP
extern Tcl_AppInitProc Blt_DragDropInit;
#endif
#ifndef NO_DEBUG
extern Tcl_AppInitProc Blt_DebugInit;
#endif
#ifndef NO_WATCH
extern Tcl_AppInitProc Blt_WatchInit;
#endif
#ifndef NO_VECTOR
extern Tcl_AppInitProc Blt_VectorInit;
#endif
#ifndef NO_SPLINE
extern Tcl_AppInitProc Blt_SplineInit;
#endif
#ifndef NO_BELL
extern Tcl_AppInitProc Blt_BeepInit;
#endif
#ifndef NO_CUTBUFFER
extern Tcl_AppInitProc Blt_CutbufferInit;
#endif
#ifndef NO_TILEFRAME
extern Tcl_AppInitProc Blt_FrameInit;
#endif
#ifndef NO_TILEBUTTON
extern Tcl_AppInitProc Blt_ButtonInit;
#endif
#ifndef NO_TILESCROLLBAR
extern Tcl_AppInitProc Blt_ScrollbarInit;
#endif

#if (BLT_MAJOR_VERSION == 3)
#ifndef NO_MOUNTAIN
extern Tcl_AppInitProc Blt_MountainInit;
#endif
#ifndef NO_TED
extern Tcl_AppInitProc Blt_TedInit;
#endif
#endif

static Tcl_AppInitProc *initProcArr[] =
{
#ifndef NO_HTEXT
    Blt_HtextInit,
#endif
#ifndef NO_GRAPH
    Blt_GraphInit,
#endif
#ifndef NO_TABLE
    Blt_TableInit,
#endif
#ifndef NO_BUSY
    Blt_BusyInit,
#endif
#ifndef NO_WINOP
    Blt_WinOpInit,
#endif
#ifndef NO_BITMAP
    Blt_BitmapInit,
#endif
#ifndef NO_BGEXEC
    Blt_BgExecInit,
#endif
#ifndef NO_DRAGDROP
    Blt_DragDropInit,
#endif
#ifndef NO_DEBUG
    Blt_DebugInit,
#endif
#ifndef NO_WATCH
    Blt_WatchInit,
#endif
#ifndef NO_VECTOR
    Blt_VectorInit,
#endif
#ifndef NO_SPLINE
    Blt_SplineInit,
#endif
#ifndef NO_BELL
    Blt_BeepInit,
#endif
#ifndef NO_CUTBUFFER
    Blt_CutbufferInit,
#endif
#ifndef NO_TILEFRAME
    Blt_FrameInit,
#endif
#ifndef NO_TILEBUTTON
    Blt_ButtonInit,
#endif
#ifndef NO_TILESCROLLBAR
    Blt_ScrollbarInit,
#endif

#if (BLT_MAJOR_VERSION == 3)
#ifndef NO_MOUNTAIN
    Blt_MountainInit,
#ifndef NO_TED
    Blt_TedInit,
#endif
#endif
#endif
    (Tcl_AppInitProc *) NULL
};

/* ARGSUSED */
static int
ExprMinProc(clientData, interp, argsPtr, resultPtr)
    ClientData clientData;	/* Unused */
    Tcl_Interp *interp;
    Tcl_Value *argsPtr;
    Tcl_Value *resultPtr;
{
    Tcl_Value *op1Ptr, *op2Ptr;
    
    op1Ptr = argsPtr, op2Ptr = argsPtr + 1;
    if ((op1Ptr->type == TCL_INT) && (op2Ptr->type == TCL_INT)) {
	resultPtr->intValue = MIN(op1Ptr->intValue, op2Ptr->intValue);
	resultPtr->type = TCL_INT;
    } else {
	double a, b;

	a = (op1Ptr->type == TCL_INT) 
	    ? (double)op1Ptr->intValue : op1Ptr->doubleValue;
	b = (op2Ptr->type == TCL_INT) 
	    ? (double)op2Ptr->intValue : op2Ptr->doubleValue;
	resultPtr->doubleValue = MIN(a, b);
	resultPtr->type = TCL_DOUBLE;
    }
    return TCL_OK;
}

/*ARGSUSED*/
static int
ExprMaxProc(clientData, interp, argsPtr, resultPtr)
    ClientData clientData;	/* Unused */
    Tcl_Interp *interp;
    Tcl_Value *argsPtr;
    Tcl_Value *resultPtr;
{
    Tcl_Value *op1Ptr, *op2Ptr;
    
    op1Ptr = argsPtr, op2Ptr = argsPtr + 1;
    if ((op1Ptr->type == TCL_INT) && (op2Ptr->type == TCL_INT)) {
	resultPtr->intValue = MAX(op1Ptr->intValue, op2Ptr->intValue);
	resultPtr->type = TCL_INT;
    } else {
	double a, b;

	a = (op1Ptr->type == TCL_INT) 
	    ? (double)op1Ptr->intValue : op1Ptr->doubleValue;
	b = (op2Ptr->type == TCL_INT) 
	    ? (double)op2Ptr->intValue : op2Ptr->doubleValue;
	resultPtr->doubleValue = MAX(a, b);
	resultPtr->type = TCL_DOUBLE;
    }
    return TCL_OK;
}


#ifndef BLT_LIBRARY
#define BLT_LIBRARY "/usr/local/lib/blt"
#endif

#if HAVE_NAMESPACES
static char envVarName[] = "::env";
#else
static char envVarName[] = "env";
#endif

#if (TCL_MAJOR_VERSION >= 8) 

extern Tcl_Namespace *Tcl_CreateNamespace _ANSI_ARGS_((Tcl_Interp *interp, 
	char *name, ClientData clientData, Tcl_NamespaceDeleteProc *nsDelProc));
extern void Tcl_DeleteNamespace _ANSI_ARGS_((Tcl_Namespace *spacePtr));

/*LINTLIBRARY*/
int
Blt_Init(interp)
    Tcl_Interp *interp;		/* Interpreter to add extra commands */
{
    Tk_Window tkwin;
    char *libDir;
    char version[20];
    register Tcl_AppInitProc **procPtrPtr;
    Tcl_ValueType argTypes[2];
    Tcl_Namespace *spacePtr;

    libDir = Tcl_GetVar2(interp, envVarName, "BLT_LIBRARY", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	libDir = BLT_LIBRARY;
    }
    if (Tcl_SetVar(interp, "blt_library", libDir, TCL_GLOBAL_ONLY) == NULL) {
	return TCL_ERROR;
    }
    sprintf(version, "%d.%d", BLT_MAJOR_VERSION, BLT_MINOR_VERSION);
    if (Tcl_SetVar(interp, "blt_version", version, TCL_GLOBAL_ONLY) == NULL) {
	return TCL_ERROR;
    }
    spacePtr = Tcl_CreateNamespace(interp, "blt::tile", (ClientData)0, 
	   (Tcl_NamespaceDeleteProc *)NULL);
    if (spacePtr == NULL) {
	return TCL_ERROR;
    }
    for (procPtrPtr = initProcArr; *procPtrPtr != NULL; procPtrPtr++) {
	if ((**procPtrPtr) (interp) != TCL_OK) {
	    Tcl_DeleteNamespace(spacePtr);
	    return TCL_ERROR;
	}
    }
    if (Tcl_PkgRequire(interp, "Tk", TK_VERSION, 1) == NULL) {
        return TCL_ERROR;
    }
    if (Tcl_PkgProvide(interp, "BLT", version) != TCL_OK) {
        return TCL_ERROR;
    }
    argTypes[0] = argTypes[1] = TCL_EITHER;
    Tcl_CreateMathFunc(interp, "min", 2, argTypes, ExprMinProc, (ClientData)0);
    Tcl_CreateMathFunc(interp, "max", 2, argTypes, ExprMaxProc, (ClientData)0);

    tkwin = Tk_MainWindow(interp);
    Blt_InitBitmapGC(tkwin);
    return TCL_OK;
}

#else

/*LINTLIBRARY*/
int
Blt_Init(interp)
    Tcl_Interp *interp;		/* Interpreter to add extra commands */
{
    Tk_Window tkwin;
    char *libDir;
    char version[20];
    register Tcl_AppInitProc **procPtrPtr;
    Tcl_ValueType argTypes[2];

#ifdef ITCL_NAMESPACES
    Itcl_Namespace dummy, spaceId;/* Token for "blt" namespace created, used
				   * to delete the namespace on errors. */
#endif
    libDir = Tcl_GetVar2(interp, envVarName, "BLT_LIBRARY", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	libDir = BLT_LIBRARY;
    }
    if (Tcl_SetVar(interp, "blt_library", libDir, TCL_GLOBAL_ONLY) == NULL) {
	return TCL_ERROR;
    }
    sprintf(version, "%d.%d", BLT_MAJOR_VERSION, BLT_MINOR_VERSION);
    if (Tcl_SetVar(interp, "blt_version", version, TCL_GLOBAL_ONLY) == NULL) {
	return TCL_ERROR;
    }
#ifdef ITCL_NAMESPACES
    if (Itcl_CreateNamesp(interp, "blt", (ClientData)0, 
	(Itcl_DeleteProc *)0, &spaceId) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Itcl_CreateNamesp(interp, "blt::tile", (ClientData)0, 
	(Itcl_DeleteProc *)0, &dummy) != TCL_OK) {
	return TCL_ERROR;
    }
#endif
    if (Tcl_PkgRequire(interp, "Tk", TK_VERSION, 1) == NULL) {
        return TCL_ERROR;
    }
    if (Tcl_PkgProvide(interp, "BLT", version) != TCL_OK) {
        return TCL_ERROR;
    }
    for (procPtrPtr = initProcArr; *procPtrPtr != NULL; procPtrPtr++) {
	if ((**procPtrPtr) (interp) != TCL_OK) {
#ifdef ITCL_NAMESPACES
	    Itcl_DeleteNamesp(spaceId);
#endif
	    return TCL_ERROR;
	}
    }
    argTypes[0] = argTypes[1] = TCL_EITHER;
    Tcl_CreateMathFunc(interp, "min", 2, argTypes, ExprMinProc, (ClientData)0);
    Tcl_CreateMathFunc(interp, "max", 2, argTypes, ExprMaxProc, (ClientData)0);

    tkwin = Tk_MainWindow(interp);
    Blt_InitBitmapGC(tkwin);
    return TCL_OK;
}

#endif /* TCL_MAJOR_VERION >= 8 */

/*LINTLIBRARY*/
int
Blt_SafeInit(interp)
    Tcl_Interp *interp;		/* Interpreter to add extra commands */
{
    return Blt_Init(interp);
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_InitCmd --
 *
 *      Given the name of a command, return a pointer to the
 *      clientData field of the command.
 *
 * Results:
 *      A standard TCL result. If the command is found, TCL_OK
 *	is returned and clientDataPtr points to the clientData
 *	field of the command (if the clientDataPtr in not NULL).
 *
 * Side effects:
 *      If the command is found, clientDataPtr is set to the address
 *	of the clientData of the command.  If not found, an error
 *	message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */
int
Blt_InitCmd(interp, nameSpace, specPtr)
    Tcl_Interp *interp;
    char *nameSpace;
    Blt_CmdSpec *specPtr;
{
    Tk_Window tkwin;
    Tcl_CmdInfo cmdInfo;
    char *cmdPath;
    Tcl_DString dStr;
    
    Tcl_DStringInit(&dStr);
#if HAVE_NAMESPACES
    if (nameSpace != NULL) {
	Tcl_DStringAppend(&dStr, nameSpace, -1);
    } 
    Tcl_DStringAppend(&dStr, "::", -1);
#endif /* HAVE_NAMESPACES */
    Tcl_DStringAppend(&dStr, specPtr->name, -1);

    cmdPath = Tcl_DStringValue(&dStr);
    if (Tcl_GetCommandInfo(interp, cmdPath, &cmdInfo)) {
	return TCL_OK;		/* Assume command was already initialized */
    }
    tkwin = Tk_MainWindow(interp);
    if (tkwin == NULL) {
	Tcl_AppendResult(interp, "\"", cmdPath, "\" requires Tk", (char *)NULL);
	return TCL_ERROR;
    }
    if (specPtr->clientData == (ClientData)0) {
	specPtr->clientData = (ClientData)tkwin;
    }
    Tcl_CreateCommand(interp, cmdPath, specPtr->cmdProc, specPtr->clientData, 
	specPtr->cmdDeleteProc);

#if (HAVE_NAMESPACES) && (TCL_MAJOR_VERSION >= 8)
    /* 
     * There must be a better way to export commands 
     */
    if (Tcl_VarEval(interp, "namespace ", nameSpace, " { namespace export ", 
		specPtr->name, " }", (char *)NULL) != TCL_OK) {
	return TCL_ERROR;
    }
#endif /* TCL_MAJOR_VERSION >= 8 */
    Tcl_DStringFree(&dStr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_InitCmds --
 *
 *      Given the name of a command, return a pointer to the
 *      clientData field of the command.
 *
 * Results:
 *      A standard TCL result. If the command is found, TCL_OK
 *	is returned and clientDataPtr points to the clientData
 *	field of the command (if the clientDataPtr in not NULL).
 *
 * Side effects:
 *      If the command is found, clientDataPtr is set to the address
 *	of the clientData of the command.  If not found, an error
 *	message is left in interp->result.
 *
 *----------------------------------------------------------------------
 */
int
Blt_InitCmds(interp, nameSpace, specPtr, numCmds)
    Tcl_Interp *interp;
    char *nameSpace;
    Blt_CmdSpec *specPtr;
    int numCmds;
{
    register int i;

    for (i = 0; i < numCmds; i++) {
	if (Blt_InitCmd(interp, nameSpace, specPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	specPtr++;
    }
    return TCL_OK;
}
