/*
 * bltGrMarker.c --
 *
 *	This module implements markers for the BLT graph widget.
 *
 * Copyright 1993-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.
 */

#include "bltGraph.h"
#include "bltGrElem.h"
#include <ctype.h>

static int CoordinatesParse _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int offset));
static char *CoordinatesPrint _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption coordsOption =
{
    CoordinatesParse, CoordinatesPrint, (ClientData)0
};

extern Tk_CustomOption bltXAxisOption;
extern Tk_CustomOption bltYAxisOption;
extern Tk_CustomOption bltLengthOption;
extern Tk_CustomOption bltDashesOption;
extern Tk_CustomOption bltPadOption;
extern Tk_CustomOption bltColorPairOption;
extern Tk_CustomOption bltMappedOption;


/*
 * Marker types:
 *
 * 	A marker can be either a text, bitmap, image, line, polygon,
 *	or window.
 */

typedef struct Marker Marker;

/*
 * -------------------------------------------------------------------
 *
 * MarkerType --
 *
 *	Enumerate types of the various varieties of markers.
 *
 * -------------------------------------------------------------------
 */
typedef enum {
    MARKER_TYPE_UNKNOWN = -1,
    MARKER_TYPE_BITMAP,
    MARKER_TYPE_IMAGE,
    MARKER_TYPE_LINE,
    MARKER_TYPE_POLYGON,
    MARKER_TYPE_TEXT,
    MARKER_TYPE_WINDOW
} MarkerType;

static char *markerClasses[] =
{
    "BitmapMarker",
    "ImageMarker",
    "LineMarker",
    "PolygonMarker",
    "TextMarker",
    "WindowMarker"
};

/*
 * This structure corresponds with the specific types of markers.
 * Don't change this structure without changing the individual
 * marker structures of each type below.
 */

typedef void (MarkerDrawProc) _ANSI_ARGS_((Marker * markerPtr));
typedef void (MarkerFreeProc) _ANSI_ARGS_((Graph *graphPtr, Marker * markerPtr));
typedef int (MarkerConfigProc) _ANSI_ARGS_((Marker * markerPtr));
typedef void (MarkerCoordsProc) _ANSI_ARGS_((Marker * markerPtr));
typedef void (MarkerPrintProc) _ANSI_ARGS_((Marker * markerPtr));

/*
 * -------------------------------------------------------------------
 *
 * Marker --
 *
 *	Structure defining the generic marker.  In C++ parlance
 *	this would be the base type from which all markers are
 *	derived.
 *
 * -------------------------------------------------------------------
 */
struct Marker {
    char *name;			/* Identifier for marker in list */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points in above array */
    Tk_ConfigSpec *configSpecs;	/* Marker configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;

    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. This can be a performance
				 * penalty because the graph must be redraw
				 * entirely each time the marker is
				 * redrawn. */
    int hidden;			/* If non-zero, don't display the marker. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset from graph position */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;
};

#define DEF_MARKER_ANCHOR	"center"
#define DEF_MARKER_BG_COLOR	WHITE
#define DEF_MARKER_BG_MONO	WHITE
#define DEF_MARKER_BITMAP	(char *)NULL
#define DEF_MARKER_COORDS	(char *)NULL
#define DEF_MARKER_DASHES	(char *)NULL
#define DEF_MARKER_ELEMENT	(char *)NULL
#define DEF_MARKER_FG_COLOR	BLACK
#define DEF_MARKER_FG_MONO	BLACK
#define DEF_MARKER_FILL_COLOR	RED
#define DEF_MARKER_FILL_MONO	WHITE
#define DEF_MARKER_GAP_COLOR	PINK
#define DEF_MARKER_GAP_MONO	BLACK
#define DEF_MARKER_FONT		STD_FONT
#define DEF_MARKER_HEIGHT	"0"
#define DEF_MARKER_JUSTIFY	"left"
#define DEF_MARKER_LINE_WIDTH	"1"
#define DEF_MARKER_HIDE		"no"
#define DEF_MARKER_MAP_X	"x"
#define DEF_MARKER_MAP_Y	"y"
#define DEF_MARKER_NAME		(char *)NULL
#define DEF_MARKER_OUTLINE_COLOR BLACK
#define DEF_MARKER_OUTLINE_MONO	BLACK
#define DEF_MARKER_PAD		"4"
#define DEF_MARKER_ROTATE	"0.0"
#define DEF_MARKER_SCALE	"1.0"
#define DEF_MARKER_STIPPLE	(char *)NULL
#define DEF_MARKER_TEXT		(char *)NULL
#define DEF_MARKER_UNDER	"no"
#define DEF_MARKER_WIDTH	"0"
#define DEF_MARKER_WINDOW	(char *)NULL
#define DEF_MARKER_X_OFFSET	"0"
#define DEF_MARKER_Y_OFFSET	"0"

/*
 * -------------------------------------------------------------------
 *
 * TextMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* The graph this marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* If non-zero, don't display the marker. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* pixel offset from anchor */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Text specific fields and attributes
     */
    char *text;			/* Text to display in graph (malloc'ed) or
				 * NULL. */
#ifdef notdef
    char *textVarName;		/* Name of variable (malloc'ed) or NULL. If
				 * non-NULL, graph displays the contents of
				 * this variable. */
#endif
    TextAttributes attr;	/* Attributes (font, fg, bg, anchor, etc) */
    double rotate;		/* Requested rotation of the text */
    int x, y;			/* Window x, y position of marker */
} TextMarker;


static Tk_ConfigSpec textConfigSpecs[] =
{
    {TK_CONFIG_ANCHOR, "-anchor", "anchor", "MarkerAnchor",
	DEF_MARKER_ANCHOR, Tk_Offset(TextMarker, attr.anchor), 0},
    {TK_CONFIG_COLOR, "-background", "background", "MarkerBackground",
	DEF_MARKER_BG_COLOR, Tk_Offset(TextMarker, attr.bgColorPtr),
	TK_CONFIG_NULL_OK | TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-background", "background", "MarkerBackground",
	DEF_MARKER_BG_MONO, Tk_Offset(TextMarker, attr.bgColorPtr),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-bg", "background", "Background",
	(char *)NULL, 0, 0},
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(TextMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(TextMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", "Foreground",
	(char *)NULL, 0, 0},
    {TK_CONFIG_FONT, "-font", "font", "Font",
	DEF_MARKER_FONT, Tk_Offset(TextMarker, attr.font), 0},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FG_COLOR, Tk_Offset(TextMarker, attr.fgColorPtr),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FG_MONO, Tk_Offset(TextMarker, attr.fgColorPtr),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_JUSTIFY, "-justify", "justify", "Justify",
	DEF_MARKER_JUSTIFY, Tk_Offset(TextMarker, attr.justify),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(TextMarker, hidden), TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(TextMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(TextMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(TextMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(TextMarker, name), 0},
    {TK_CONFIG_CUSTOM, "-padx", "padX", "PadX",
	DEF_MARKER_PAD, Tk_Offset(TextMarker, attr.padX),
	TK_CONFIG_DONT_SET_DEFAULT, &bltPadOption},
    {TK_CONFIG_CUSTOM, "-pady", "padY", "PadY",
	DEF_MARKER_PAD, Tk_Offset(TextMarker, attr.padY),
	TK_CONFIG_DONT_SET_DEFAULT, &bltPadOption},
    {TK_CONFIG_DOUBLE, "-rotate", "rotate", "Rotate",
	DEF_MARKER_ROTATE, Tk_Offset(TextMarker, rotate),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-text", "text", "Text",
	DEF_MARKER_TEXT, Tk_Offset(TextMarker, text), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(TextMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(TextMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(TextMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


/*
 * -------------------------------------------------------------------
 *
 * WindowMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* Indicates if the marker is currently
				 * hidden or not. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset from anchor */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Window specific attributes
     */
    char *pathName;		/* Name of child window to be displayed */
    Tk_Window tkwin;		/* Window to display */
    int reqWidth, reqHeight;	/* Requested window extents */
    int width, height;		/* Actual window extents */
    Tk_Anchor anchor;		/* Anchor */
    int x, y;

} WindowMarker;

static Tk_ConfigSpec windowConfigSpecs[] =
{
    {TK_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_MARKER_ANCHOR, Tk_Offset(WindowMarker, anchor), 0},
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(WindowMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(WindowMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-height", "height", "Height",
	DEF_MARKER_HEIGHT, Tk_Offset(WindowMarker, reqHeight),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(WindowMarker, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(WindowMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(WindowMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(WindowMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(WindowMarker, name), 0},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(WindowMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-width", "width", "Width",
	DEF_MARKER_WIDTH, Tk_Offset(WindowMarker, reqWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_STRING, "-window", "window", "Window",
	DEF_MARKER_WINDOW, Tk_Offset(WindowMarker, pathName), 0},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(WindowMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(WindowMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * BitmapMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* Indicates if the marker is currently
				 * hidden or not. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset from origin of bitmap */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Bitmap specific attributes
     */
    Pixmap bitmap;		/* Original bitmap. May be further scaled or
				 * rotated */
    double rotate;		/* Requested rotation of the bitmap */
    double theta;		/* Normalized rotation (0..360 degrees) */
    XColor *normalFg;		/* foreground color */
    XColor *normalBg;		/* background color */

    GC gc;			/* Private graphic context */
    GC fillGC;			/* Shared graphic context */
    int x, y;			/* Origin of the bitmap */
    Pixmap transBitmap;		/* Transformed bitmap (rotated or scaled) */
    Pixmap draw;		/* Bitmap to be drawn */
    int width, height;		/* Final dimensions of the bitmap */
    XPoint polygonArr[4];	/* Polygon forming the background of the
				 * bitmap. It's used to draw the background
				 * of rotated bitmaps.  We also use it to print
				 * a background in PostScript. */
} BitmapMarker;

static Tk_ConfigSpec bitmapConfigSpecs[] =
{
    {TK_CONFIG_COLOR, "-background", "background", "Background",
	DEF_MARKER_BG_COLOR, Tk_Offset(BitmapMarker, normalBg),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-background", "background", "Background",
	DEF_MARKER_BG_MONO, Tk_Offset(BitmapMarker, normalBg),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_BITMAP, "-bitmap", "bitmap", "Bitmap",
	DEF_MARKER_BITMAP, Tk_Offset(BitmapMarker, bitmap), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(BitmapMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(BitmapMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FG_COLOR, Tk_Offset(BitmapMarker, normalFg),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_MARKER_FG_MONO, Tk_Offset(BitmapMarker, normalFg),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(BitmapMarker, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(BitmapMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(BitmapMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(BitmapMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(BitmapMarker, name), 0},
    {TK_CONFIG_DOUBLE, "-rotate", "rotate", "Rotate",
	DEF_MARKER_ROTATE, Tk_Offset(BitmapMarker, rotate),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(BitmapMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(BitmapMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(BitmapMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};


/*
 * -------------------------------------------------------------------
 *
 * ImageMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* Indicates if the marker is currently
				 * hidden or not. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset from anchor */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Image specific attributes
     */
    char *imageName;		/* Name of image to be displayed */
    Tk_Image imageToken;	/* Image to be displayed */
    Tk_Anchor anchor;		/* anchor */
    int x, y;			/* Window x,y position of the image */

} ImageMarker;

static Tk_ConfigSpec imageConfigSpecs[] =
{
    {TK_CONFIG_ANCHOR, "-anchor", "anchor", "Anchor",
	DEF_MARKER_ANCHOR, Tk_Offset(ImageMarker, anchor),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(ImageMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(ImageMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(ImageMarker, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(ImageMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_STRING, "-image", "image", "Image",
	(char *)NULL, Tk_Offset(ImageMarker, imageName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(ImageMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(ImageMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(ImageMarker, name), 0},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(ImageMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(ImageMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(ImageMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * LineMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type is MARKER_TYPE_LINE */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* Indicates if the marker is currently
				 * hidden or not. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Line specific attributes
     */
    ColorPair color;		/* Pair of foreground/background colors */
    int lineWidth;		/* line width */
    Dashes dashes;		/* Dash list values (max 11) */
    GC gc;			/* Private graphic context */
    XSegment *segArr;
    int numSegments;
} LineMarker;

static Tk_ConfigSpec lineConfigSpecs[] =
{
    {TK_CONFIG_CUSTOM, "-color", "color", "Color",
	DEF_MARKER_FG_COLOR, Tk_Offset(LineMarker, color),
	TK_CONFIG_COLOR_ONLY, &bltColorPairOption},
    {TK_CONFIG_CUSTOM, "-color", "color", "Color",
	DEF_MARKER_FG_MONO, Tk_Offset(LineMarker, color),
	TK_CONFIG_MONO_ONLY, &bltColorPairOption},
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(LineMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_CUSTOM, "-dashes", "dashes", "Dashes",
	DEF_MARKER_DASHES, Tk_Offset(LineMarker, dashes),
	TK_CONFIG_NULL_OK, &bltDashesOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(LineMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-linewidth", "lineWidth", "LineWidth",
	DEF_MARKER_LINE_WIDTH, Tk_Offset(LineMarker, lineWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(LineMarker, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(LineMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(LineMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(LineMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(LineMarker, name), 0},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(LineMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(LineMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(LineMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/*
 * -------------------------------------------------------------------
 *
 * PolygonMarker --
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *name;			/* Identifier for marker */
    Graph *graphPtr;		/* Graph marker belongs to */
    MarkerType type;		/* Type of marker */
    int flags;
    Tcl_HashEntry *hashPtr;
    Blt_ListItem *listPtr;
    Point2D *coordArr;		/* Coordinate array to position marker */
    int numCoords;		/* Number of points */
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *elemName;		/* Element associated with marker */
    Axis2D axes;
    int drawUnder;		/* If non-zero, draw the marker underneath
				 * any elements. There can be a performance
				 * because the graph must be redraw entirely
				 * each time this marker is redrawn. */
    int hidden;			/* Indicates if the marker is currently
				 * hidden or not. */
    int clipped;		/* Indicates if the marker is totally clipped
				 * by the plotting area. */
    int xOffset, yOffset;	/* Pixel offset */

    MarkerDrawProc *drawProc;
    MarkerFreeProc *freeProc;
    MarkerConfigProc *configProc;
    MarkerCoordsProc *coordsProc;
    MarkerPrintProc *printProc;

    /*
     * Polygon specific attributes and fields
     */
    ColorPair outline;
    ColorPair fill;

    Pixmap stipple;		/* stipple pattern */
    int lineWidth;		/* line width */
    Dashes dashes;		/* dash list value */

    GC outlineGC;		/* Private graphic context */
    GC fillGC;			/* Private graphic context */

    XPoint *polygonArr;		/* Points needed to draw polygon */
    int numPoints;		/* Number of points in above array */

} PolygonMarker;

static Tk_ConfigSpec polygonConfigSpecs[] =
{
    {TK_CONFIG_CUSTOM, "-coords", "coords", "Coords",
	DEF_MARKER_COORDS, Tk_Offset(PolygonMarker, coordArr),
	TK_CONFIG_NULL_OK, &coordsOption},
    {TK_CONFIG_CUSTOM, "-dashes", "dashes", "Dashes",
	DEF_MARKER_DASHES, Tk_Offset(PolygonMarker, dashes),
	TK_CONFIG_NULL_OK, &bltDashesOption},
    {TK_CONFIG_STRING, "-element", "element", "Element",
	DEF_MARKER_ELEMENT, Tk_Offset(PolygonMarker, elemName), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-fill", "fill", "Fill",
	DEF_MARKER_FILL_COLOR, Tk_Offset(PolygonMarker, fill),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK, &bltColorPairOption},
    {TK_CONFIG_CUSTOM, "-fill", "fill", "Fill",
	DEF_MARKER_FILL_MONO, Tk_Offset(PolygonMarker, fill),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_NULL_OK, &bltColorPairOption},
    {TK_CONFIG_CUSTOM, "-linewidth", "lineWidth", "LineWidth",
	DEF_MARKER_LINE_WIDTH, Tk_Offset(PolygonMarker, lineWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_MARKER_HIDE, Tk_Offset(PolygonMarker, hidden),
	TK_CONFIG_DONT_SET_DEFAULT},

    {TK_CONFIG_CUSTOM, "-mapped", "mapped", "Mapped",
	(char *)NULL, Tk_Offset(PolygonMarker, hidden), 0, &bltMappedOption},

    {TK_CONFIG_CUSTOM, "-mapx", "mapX", "MapX",
	DEF_MARKER_MAP_X, Tk_Offset(PolygonMarker, axes.x), 0, &bltXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "mapY", "MapY",
	DEF_MARKER_MAP_Y, Tk_Offset(PolygonMarker, axes.y), 0, &bltYAxisOption},
    {TK_CONFIG_STRING, "-name", (char *)NULL, (char *)NULL,
	DEF_MARKER_NAME, Tk_Offset(PolygonMarker, name), 0},
    {TK_CONFIG_CUSTOM, "-outline", "outline", "Outline",
	DEF_MARKER_OUTLINE_COLOR, Tk_Offset(PolygonMarker, outline),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK, &bltColorPairOption},
    {TK_CONFIG_CUSTOM, "-outline", "outline", "Outline",
	DEF_MARKER_OUTLINE_MONO, Tk_Offset(PolygonMarker, outline),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_NULL_OK, &bltColorPairOption},
    {TK_CONFIG_BITMAP, "-stipple", "stipple", "Stipple",
	DEF_MARKER_STIPPLE, Tk_Offset(PolygonMarker, stipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-under", "under", "Under",
	DEF_MARKER_UNDER, Tk_Offset(PolygonMarker, drawUnder),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xoffset", "xOffset", "XOffset",
	DEF_MARKER_X_OFFSET, Tk_Offset(PolygonMarker, xOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yoffset", "yOffset", "YOffset",
	DEF_MARKER_Y_OFFSET, Tk_Offset(PolygonMarker, yOffset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};



/*  
 * Only for polygons right now.  Will be removed with polygon clip routine.
 */
INLINE static void
BoundPoint(pointPtr)
    XPoint *pointPtr;
{
    /* Should really figure out a good offset value and test for that because
     * we could still generate bogus numbers */

    if (pointPtr->x >= SHRT_MAX) {
	pointPtr->x = SHRT_MAX - 1000;
    } else if (pointPtr->x <= SHRT_MIN) {
	pointPtr->x = SHRT_MIN + 1000;
    }
    if (pointPtr->y >= SHRT_MAX) {
	pointPtr->y = SHRT_MAX - 1000;
    } else if (pointPtr->y <= SHRT_MIN) {
	pointPtr->y = SHRT_MIN + 1000;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * BoxesDontOverlap --
 *
 *	Tests if the bounding box of a marker overlaps the plotting
 *	area in any way.  If so, the marker will be drawn.  Just do a
 *	min/max test on the extents of both boxes.
 *
 * Results:
 *	Returns 0 is the marker is visible in the plotting area, and
 *	1 otherwise (marker is clipped).
 *
 * ----------------------------------------------------------------------
 */
static int
BoxesDontOverlap(graphPtr, extentsPtr)
    Graph *graphPtr;
    Extents2D *extentsPtr;
{
    return (((double)graphPtr->xMax < extentsPtr->xMin) ||
	((double)graphPtr->yMax < extentsPtr->yMin) ||
	(extentsPtr->xMax < (double)graphPtr->xMin) ||
	(extentsPtr->yMax < (double)graphPtr->yMin));
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintCoordinate --
 *
 * 	Convert the double precision value into its string
 * 	representation.  The only reason this routine is used in
 * 	instead of sprintf, is to handle the "elastic" bounds.  That
 * 	is, convert the values DBL_MAX and -(DBL_MAX) into "+Inf" and
 * 	"-Inf" respectively.
 *
 * Results:
 *	The return value is a standard Tcl result.  The string of the
 * 	expression is passed back via string.
 *
 * ---------------------------------------------------------------------- */
static char *
PrintCoordinate(interp, x)
    Tcl_Interp *interp;
    double x;			/* Numeric value */
{
    if (x == bltPosInfinity) {
	return "+Inf";
    } else if (x == bltNegInfinity) {
	return "-Inf";
    } else {
	static char string[TCL_DOUBLE_SPACE + 1];

	Tcl_PrintDouble(interp, x, string);
	return string;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ParseCoordinates --
 *
 *	If no coordinates are specified (coordList is NULL), this
 *	routine returns the coordinates of a given marker. Otherwise,
 *	the Tcl coordinate list is converted to their floating point
 *	values. It will then replace the current marker coordinates.
 *
 *	Since different marker types require different number of
 *	coordinates this must be checked here.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side effects:
 *	If the marker coordinates are reset, the graph is eventually redrawn
 *	with at the new marker coordinates.
 *
 * ----------------------------------------------------------------------
 */

static int
ParseCoordinates(interp, markerPtr, numExprs, exprArr)
    Tcl_Interp *interp;
    Marker *markerPtr;
    int numExprs;
    char **exprArr;
{
    int numCoords;
    int minArgs, maxArgs;
    double x, y;
    Point2D *newArr;
    register int i;
    register Point2D *pointPtr;

    /* Split the list of coordinates and check the values */

    if (numExprs == 0) {
	return TCL_OK;
    }
    if (numExprs & 1) {
	interp->result = "odd number of marker coordinates specified";
	return TCL_ERROR;
    }
    switch (markerPtr->type) {
    case MARKER_TYPE_LINE:
	minArgs = 4, maxArgs = 0;
	break;
    case MARKER_TYPE_POLYGON:
	minArgs = 6, maxArgs = 0;
	break;
    case MARKER_TYPE_WINDOW:
    case MARKER_TYPE_IMAGE:
    case MARKER_TYPE_TEXT:
	minArgs = 2, maxArgs = 2;
	break;
    case MARKER_TYPE_BITMAP:
	minArgs = 2, maxArgs = 4;
	break;
    default:
	interp->result = "unknown marker type";
	return TCL_ERROR;
    }

    if (numExprs < minArgs) {
	interp->result = "too few marker coordinates specified";
	return TCL_ERROR;
    }
    if ((maxArgs > 0) && (numExprs > maxArgs)) {
	interp->result = "too many marker coordinates specified";
	return TCL_ERROR;
    }
    numCoords = numExprs / 2;
    newArr = (Point2D *) malloc(numCoords * sizeof(Point2D));
    if (newArr == NULL) {
	interp->result = "can't allocate new coordinate array";
	return TCL_ERROR;
    }
    /*
     * A new coordinate array is allocated each time so that we
     * can check the coordinates without overwriting the current
     * marker coordinates.
     */
    pointPtr = newArr;
    for (i = 0; i < numExprs; i += 2) {
	if ((Blt_GetCoordinate(interp, exprArr[i], &x) != TCL_OK) ||
	    (Blt_GetCoordinate(interp, exprArr[i + 1], &y) != TCL_OK)) {
	    free((char *)newArr);
	    return TCL_ERROR;
	}
	pointPtr->x = x, pointPtr->y = y;
	pointPtr++;
    }
    if (markerPtr->coordArr != NULL) {
	free((char *)markerPtr->coordArr);
    }
    markerPtr->coordArr = newArr;
    markerPtr->numCoords = numCoords;
    markerPtr->flags |= COORDS_NEEDED;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * CoordinatesParse --
 *
 *	Given a Tcl list of numeric expression representing the element
 *	values, convert into an array of double precision values. In
 *	addition, the minimum and maximum values are saved.  Since
 *	elastic values are allow (values which translate to the
 *	min/max of the graph), we must try to get the non-elastic
 *	minimum and maximum.
 *
 * Results:
 *	The return value is a standard Tcl result.  The vector is passed
 *	back via the vecPtr.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CoordinatesParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Tcl list of numeric expressions */
    char *widgRec;		/* Marker record */
    int offset;			/* not used */
{
    Marker *markerPtr = (Marker *) widgRec;
    int numExprs;
    char **exprArr;
    int result;

    if ((value == NULL) || (*value == '\0')) {
      noCoordinates:
	if (markerPtr->coordArr != NULL) {
	    free((char *)markerPtr->coordArr);
	}
	markerPtr->numCoords = 0;
	return TCL_OK;
    }
    if (Tcl_SplitList(interp, value, &numExprs, &exprArr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (numExprs == 0) {
	goto noCoordinates;
    }
    result = ParseCoordinates(interp, markerPtr, numExprs, exprArr);
    free((char *)exprArr);
    return (result);
}

/*
 * ----------------------------------------------------------------------
 *
 * CoordinatesPrint --
 *
 *	Convert the vector of floating point values into a Tcl list.
 *
 * Results:
 *	The string representation of the vector is returned.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
CoordinatesPrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Marker record */
    int offset;			/* not used */
    Tcl_FreeProc **freeProcPtr;	/* Memory deallocation scheme to use */
{
    Marker *markerPtr = (Marker *) widgRec;
    Tcl_DString dStr;
    char *result;
    register int i;
    register Point2D *pointPtr;

    if (markerPtr->numCoords < 1) {
	return "";
    }
    Tcl_DStringInit(&dStr);
    pointPtr = markerPtr->coordArr;
    for (i = 0; i < markerPtr->numCoords; i++, pointPtr++) {
	Tcl_DStringAppendElement(&dStr,
	    PrintCoordinate(markerPtr->graphPtr->interp, pointPtr->x));
	Tcl_DStringAppendElement(&dStr,
	    PrintCoordinate(markerPtr->graphPtr->interp, pointPtr->y));
    }
    result = Tcl_DStringValue(&dStr);

    /*
     * If memory wasn't allocated for the dynamic string, do it here (it's
     * currently on the stack), so that Tcl can free it normally.
     */
    if (result == dStr.staticSpace) {
	result = strdup(result);
    }
    *freeProcPtr = (Tcl_FreeProc *)free;
    return (result);
}


/* Map graph coordinates to normalized coordinates [0..1] */
#define NORMALIZE(A,x) 	(((x) - (A)->limitsPtr->min) / (A)->limitsPtr->range)

/*
 * ----------------------------------------------------------------------
 *
 * HMap --
 *
 *	Map the given graph coordinate value to its axis, returning a window
 *	position.
 *
 * Results:
 *	Returns a double precision number representing the window coordinate
 *	position on the given axis.
 *
 * ----------------------------------------------------------------------
 */
static double
HMap(graphPtr, axisPtr, x)
    Graph *graphPtr;
    VirtualAxis *axisPtr;
    double x;
{
    register double norm;

    if (x == bltPosInfinity) {
	norm = 1.0;
    } else if (x == bltNegInfinity) {
	norm = 0.0;
    } else {
	if (axisPtr->logScale) {
	    if (x > 0.0) {
		x = log10(x);
	    } else if (x < 0.0) {
		x = 0.0;
	    }
	}
	norm = NORMALIZE(axisPtr, x);
    }
    if (axisPtr->logScale) {
	if (x > 0.0) {
	    x = log10(x);
	} else if (x < 0.0) {
	    x = 0.0;
	}
    }
    norm = NORMALIZE(axisPtr, x);
    if (axisPtr->descending) {
	norm = 1.0 - norm;
    }
    /* Horizontal transformation */
    return ((norm * (graphPtr->hRange)) + graphPtr->hOffset);
}

/*
 * ----------------------------------------------------------------------
 *
 * VMap --
 *
 *	Map the given graph coordinate value to its axis, returning a window
 *	position.
 *
 * Results:
 *	Returns a double precision number representing the window coordinate
 *	position on the given axis.
 *
 * ----------------------------------------------------------------------
 */
static double
VMap(graphPtr, axisPtr, y)
    Graph *graphPtr;
    VirtualAxis *axisPtr;
    double y;
{
    register double norm;

    if (y == bltPosInfinity) {
	norm = 1.0;
    } else if (y == bltNegInfinity) {
	norm = 0.0;
    } else {
	if (axisPtr->logScale) {
	    if (y > 0.0) {
		y = log10(y);
	    } else if (y < 0.0) {
		y = 0.0;
	    }
	}
	norm = NORMALIZE(axisPtr, y);
    }
    if (axisPtr->logScale) {
	if (y > 0.0) {
	    y = log10(y);
	} else if (y < 0.0) {
	    y = 0.0;
	}
    }
    norm = NORMALIZE(axisPtr, y);
    if (axisPtr->descending) {
	norm = 1.0 - norm;
    }
    /* Vertical transformation */
    return (((1.0 - norm) * (graphPtr->vRange)) + graphPtr->vOffset);
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformMarkerPt --
 *
 *	Maps the given graph x,y coordinate values to a window position.
 *
 * Results:
 *	Returns a XPoint structure containing the window coordinates of
 *	the given graph x,y coordinate.
 *
 * ----------------------------------------------------------------------
 */

static Point2D
TransformMarkerPt(graphPtr, x, y, axesPtr)
    Graph *graphPtr;
    double x, y;		/* Graph x and y coordinates */
    Axis2D *axesPtr;		/* Specifies which axes to use */
{
    Point2D point;

    if (graphPtr->inverted) {
	point.x = HMap(graphPtr, axesPtr->y, y);
	point.y = VMap(graphPtr, axesPtr->x, x);
    } else {
	point.x = HMap(graphPtr, axesPtr->x, x);
	point.y = VMap(graphPtr, axesPtr->y, y);
    }
    return point;
}



static void
DestroyMarker(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;

    /* Free the resources allocated for the particular type of marker */
    (*markerPtr->freeProc) (graphPtr, markerPtr);
    if (markerPtr->coordArr != NULL) {
	free((char *)markerPtr->coordArr);
    }
    Tk_FreeOptions(markerPtr->configSpecs, (char *)markerPtr,
	graphPtr->display, 0);
    if (markerPtr->hashPtr != NULL) {
	Tcl_DeleteHashEntry(markerPtr->hashPtr);
    }
    if (markerPtr->listPtr != NULL) {
	Blt_ListDeleteItem(markerPtr->listPtr);
    }
    if (markerPtr->name != NULL) {
	free(markerPtr->name);
    }
    free((char *)markerPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureBitmap --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a bitmap marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as bitmap pixmap, colors, rotation,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
ConfigureBitmap(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *) markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    if (bmPtr->bitmap == None) {
	return TCL_OK;
    }
    bmPtr->theta = FMOD(bmPtr->rotate, 360.0);
    if (bmPtr->theta < 0.0) {
	bmPtr->theta += 360.0;
    }
    gcValues.foreground = bmPtr->normalFg->pixel;
    gcMask = GCForeground;
    if (bmPtr->normalBg != NULL) {
	gcValues.background = bmPtr->normalBg->pixel;
	gcMask |= GCBackground;
    } else {
	gcValues.clip_mask = bmPtr->bitmap;
	gcMask |= GCClipMask;
    }

    /*
     * Note that while this is a "shared" GC. We're going to change
     * the clip origin right before the bitmap is drawn anyways.  I'm
     * assuming that any drawing code using this GC (with the
     * GCClipMask set) is going to set the clip origin before it draws
     * anyway.
     */
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (bmPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->gc);
    }
    bmPtr->gc = newGC;

    /* Create background GC color */

    if (bmPtr->normalBg != NULL) {
	gcValues.foreground = bmPtr->normalBg->pixel;
	newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
	if (bmPtr->fillGC != NULL) {
	    Tk_FreeGC(graphPtr->display, bmPtr->fillGC);
	}
	bmPtr->fillGC = newGC;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TranformBitmap --
 *
 * 	This procedure gets called each time the layout of the graph
 *	changes.  The x, y window coordinates of the bitmap marker are
 *	saved in the marker structure.
 *
 *	Additionly, if no background color was specified, the
 *	GCTileStipXOrigin and GCTileStipYOrigin attributes are set in
 *	the private GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window coordinates are saved and if no background color was
 * 	set, the GC stipple origins are changed to calculated window
 *	coordinates.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformBitmap(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *) markerPtr;
    int width, height, bmWidth, bmHeight, scaledWidth, scaledHeight;
    double x, y;
    Point2D corner1, corner2;
    Extents2D extents;
    double sx, sy;
    register int i;

    if (bmPtr->transBitmap != None) {
	Tk_FreePixmap(graphPtr->display, bmPtr->transBitmap);
    }
    bmPtr->transBitmap = None;
    if (bmPtr->bitmap == None) {
	return;
    }
    Tk_SizeOfBitmap(graphPtr->display, bmPtr->bitmap, &width, &height);
    bmWidth = width;
    bmHeight = height;
    corner1 = TransformMarkerPt(graphPtr, bmPtr->coordArr[0].x,
	bmPtr->coordArr[0].y, &bmPtr->axes);
    if (bmPtr->numCoords > 1) {
	corner2 = TransformMarkerPt(graphPtr, bmPtr->coordArr[1].x,
	    bmPtr->coordArr[1].y, &bmPtr->axes);
	scaledWidth = ROUND(corner1.x) - ROUND(corner2.x);
	scaledHeight = ROUND(corner1.y) - ROUND(corner2.y);
	scaledWidth = ABS(scaledWidth) + 1;
	scaledHeight = ABS(scaledHeight) + 1;
	x = MIN(corner1.x, corner2.x);
	y = MIN(corner1.y, corner2.y);
    } else {
	scaledWidth = width;
	scaledHeight = height;
	x = corner1.x;
	y = corner1.y;
    }

    x = ROUND(x) + bmPtr->xOffset;
    y = ROUND(y) + bmPtr->yOffset;

    /*
     * Determine the bounding box of the bitmap and test to see if it
     * is at least partially contained within the plotting area.
     */
    extents.xMin = x;
    extents.yMax = y + scaledHeight;
    extents.yMin = y;
    extents.xMax = x + scaledWidth;
    bmPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
    if (bmPtr->clipped) {
	return;			/* Bitmap is offscreen. Don't generate
				 * rotated or scaled bitmaps. */
    }
    /* Start of pointing to the original bitmap */
    bmPtr->draw = bmPtr->bitmap;
    if (bmPtr->theta != 0.0) {
	bmPtr->draw = Blt_RotateBitmap(graphPtr->display,
	    Tk_WindowId(graphPtr->tkwin), bmPtr->draw, width, height,
	    bmPtr->theta, &width, &height);
	bmPtr->transBitmap = bmPtr->draw;
    }
    if ((scaledWidth != width) || (scaledHeight != height)) {
	int regionX, regionY, regionWidth, regionHeight;
	int xMin, xMax, yMin, yMax;

	/*
	 * Determine if the bitmap is clipped by the graph.
	 */
	xMin = MAX((int)x, graphPtr->xMin);
	yMin = MAX((int)y, graphPtr->yMin);
	xMax = MIN((int)x + scaledWidth - 1, graphPtr->xMax);
	yMax = MIN((int)y + scaledHeight - 1, graphPtr->yMax);

	regionX = xMin - (int)x, regionY = xMin - (int)y;
	regionWidth = xMax - xMin + 1;
	regionHeight = yMax - yMin + 1;

	bmPtr->draw = Blt_ScaleBitmapRegion(graphPtr->display,
	    Tk_WindowId(graphPtr->tkwin), bmPtr->draw, width, height, scaledWidth,
	    scaledHeight, regionX, regionY, regionWidth, regionHeight);
	if (bmPtr->transBitmap != None) {
	    Tk_FreePixmap(graphPtr->display, bmPtr->transBitmap);
	}
	bmPtr->transBitmap = bmPtr->draw;

	/* Reset bitmap location and coordinates to that of the region */
	width = regionWidth;
	height = regionHeight;
	x = xMin;
	y = yMin;
    }
    bmPtr->x = x;
    bmPtr->y = y;
    bmPtr->width = width;
    bmPtr->height = height;

    /*
     * Compute the background area of the bitmap.  This is needed to
     * draw a background for arbitrarily rotated bitmaps.  We also use
     * it to print a background in PostScript. For right-angle
     * rotations, this is the bounding box same as above.
     */
    Blt_GetBoundingBox(bmWidth, bmHeight, bmPtr->theta, &width, &height,
	bmPtr->polygonArr);
    sx = (double)scaledWidth / (double)width;
    sy = (double)scaledHeight / (double)height;
    for (i = 0; i < 4; i++) {
	x = (double)bmPtr->polygonArr[i].x * sx;
	y = (double)bmPtr->polygonArr[i].y * sy;
	bmPtr->polygonArr[i].x = ROUND(x) + bmPtr->x + (scaledWidth / 2);
	bmPtr->polygonArr[i].y = ROUND(y) + bmPtr->y + (scaledHeight / 2);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawBitmap --
 *
 *	This procedure is invoked to draw a bitmap marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	GC stipple origins are changed to current window coordinates.
 *	Commands are output to X to draw the marker in its current mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawBitmap(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *) markerPtr;
    double theta;

    if (bmPtr->bitmap == None) {
	return;
    }
    theta = FMOD(bmPtr->theta, (double)90.0);
    if ((bmPtr->normalBg == NULL) || (theta != 0.0)) {
	if (bmPtr->normalBg != NULL) {
	    XFillPolygon(graphPtr->display, graphPtr->pixwin, bmPtr->fillGC,
		bmPtr->polygonArr, 4, Convex, CoordModeOrigin);
	}
	XSetClipMask(graphPtr->display, bmPtr->gc, bmPtr->draw);
	XSetClipOrigin(graphPtr->display, bmPtr->gc, bmPtr->x, bmPtr->y);
    }
    XCopyPlane(graphPtr->display, bmPtr->draw, graphPtr->pixwin, bmPtr->gc,
	0, 0, bmPtr->width, bmPtr->height, bmPtr->x, bmPtr->y, 1);
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintBitmap --
 *
 *	This procedure is invoked to print a bitmap marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
PrintBitmap(markerPtr)
    Marker *markerPtr;		/* Marker to be printed */
{
    Graph *graphPtr = markerPtr->graphPtr;
    BitmapMarker *bmPtr = (BitmapMarker *) markerPtr;

    if (bmPtr->bitmap == None) {
	return;
    }
    if (bmPtr->normalBg != NULL) {
	Blt_BackgroundToPostScript(graphPtr, bmPtr->normalBg);
	Blt_PolygonToPostScript(graphPtr, bmPtr->polygonArr, 4);
    }
    Blt_ForegroundToPostScript(graphPtr, bmPtr->normalFg);

    Blt_PostScriptAppend(graphPtr,
	 "  gsave\n    %d %d translate\n    %d %d scale\n",
	 bmPtr->x, bmPtr->y + bmPtr->height, bmPtr->width, -bmPtr->height);
    Blt_PostScriptAppend(graphPtr, "    %d %d true [%d 0 0 %d 0 %d] {",
	bmPtr->width, bmPtr->height, bmPtr->width, -bmPtr->height, bmPtr->height);
    Blt_BitmapToPostScript(graphPtr, bmPtr->draw, bmPtr->width, bmPtr->height);
    Blt_PostScriptAppend(graphPtr, "    } imagemask\n  grestore\n");
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyBitmap --
 *
 *	Destroys the structure containing the attributes of the bitmap
 * 	marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Bitmap attributes (GCs, colors, bitmap, etc) get destroyed.
 *	Memory is released, X resources are freed, and the graph is
 *	redrawn.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyBitmap(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    BitmapMarker *bmPtr = (BitmapMarker *) markerPtr;

    if (bmPtr->gc != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->gc);
    }
    if (bmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, bmPtr->fillGC);
    }
    if (bmPtr->transBitmap != None) {
	Tk_FreePixmap(graphPtr->display, bmPtr->transBitmap);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateBitmap --
 *
 *	Allocate memory and initialize methods for the new bitmap marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the bitmap marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateBitmap()
{
    BitmapMarker *bmPtr;

    bmPtr = (BitmapMarker *) calloc(1, sizeof(BitmapMarker));
    if (bmPtr != NULL) {
	bmPtr->configSpecs = bitmapConfigSpecs;
	bmPtr->configProc = ConfigureBitmap;
	bmPtr->freeProc = DestroyBitmap;
	bmPtr->drawProc = DrawBitmap;
	bmPtr->coordsProc = TransformBitmap;
	bmPtr->printProc = PrintBitmap;
	bmPtr->type = MARKER_TYPE_BITMAP;
    }
    return ((Marker *) bmPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ImageChangedProc
 *
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
/* ARGSUSED */
static void
ImageChangedProc(clientData, x, y, width, height, imageWidth, imageHeight)
    ClientData clientData;
    int x, y, width, height;	/* Not used */
    int imageWidth, imageHeight;/* Not used */
{
    ImageMarker *imPtr = (ImageMarker *) clientData;

    imPtr->graphPtr->flags |= UPDATE_PIXMAP;
    Blt_RedrawGraph(imPtr->graphPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureImage --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a image marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as image pixmap, colors, rotation,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
 /* ARGSUSED */
static int
ConfigureImage(markerPtr)
    Marker *markerPtr;
{
    ImageMarker *imPtr = (ImageMarker *) markerPtr;

    if (Blt_ConfigModified(imPtr->configSpecs, "-image", (char *)NULL)) {
	Graph *graphPtr = markerPtr->graphPtr;

	if (imPtr->imageToken != NULL) {
	    Tk_FreeImage(imPtr->imageToken);
	    imPtr->imageToken = NULL;
	}
	if (imPtr->imageName[0] != '\0') {
	    imPtr->imageToken = Tk_GetImage(graphPtr->interp, graphPtr->tkwin,
		imPtr->imageName, ImageChangedProc, (ClientData)imPtr);
	    if (imPtr->imageToken == NULL) {
		Tcl_AppendResult(graphPtr->interp, "can't find an image \"",
		    imPtr->imageName, "\"", (char *)NULL);
		free((char *)imPtr->imageName);
		imPtr->imageName = NULL;
		return TCL_ERROR;
	    }
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TranformImage --
 *
 * 	This procedure gets called each time the layout of the graph
 *	changes.  The x, y window coordinates of the image marker are
 *	saved in the marker structure.
 *
 *	Additionly, if no background color was specified, the
 *	GCTileStipXOrigin and GCTileStipYOrigin attributes are set in
 *	the private GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window coordinates are saved and if no background color was
 * 	set, the GC stipple origins are changed to calculated window
 *	coordinates.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformImage(markerPtr)
    Marker *markerPtr;
{
    ImageMarker *imPtr = (ImageMarker *) markerPtr;
    int width, height;
    Point2D p;
    int x, y;
    Extents2D extents;

    if (imPtr->imageToken == NULL) {
	return;
    }
    Tk_SizeOfImage(imPtr->imageToken, &width, &height);
    p = TransformMarkerPt(imPtr->graphPtr, imPtr->coordArr[0].x,
	imPtr->coordArr[0].y, &imPtr->axes);
    x = ROUND(p.x) + imPtr->xOffset;
    y = ROUND(p.y) + imPtr->yOffset;
    p = Blt_TranslateBoxCoords(x, y, width, height, imPtr->anchor);

    imPtr->x = (int)p.x;
    imPtr->y = (int)p.y;

    /*
     * Determine the bounding box of the image and test to see if it
     * is at least partially contained within the plotting area.
     */
    extents.xMin = (double)imPtr->x;
    extents.yMax = (double)imPtr->y + height;
    extents.xMax = (double)imPtr->x + width;
    extents.yMin = (double)imPtr->y;
    imPtr->clipped = BoxesDontOverlap(imPtr->graphPtr, &extents);

}

/*
 * ----------------------------------------------------------------------
 *
 * DrawImage --
 *
 *	This procedure is invoked to draw a image marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	GC stipple origins are changed to current window coordinates.
 *	Commands are output to X to draw the marker in its current mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawImage(markerPtr)
    Marker *markerPtr;
{
    ImageMarker *imPtr = (ImageMarker *) markerPtr;
    int width, height;

    if ((imPtr->imageToken == NULL) || (Blt_ImageDeleted(imPtr->imageToken))) {
	return;
    }
    Tk_SizeOfImage(imPtr->imageToken, &width, &height);
    Tk_RedrawImage(imPtr->imageToken, 0, 0, width, height,
	imPtr->graphPtr->pixwin, imPtr->x, imPtr->y);
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintImage --
 *
 *	This procedure is invoked to print a image marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
PrintImage(markerPtr)
    Marker *markerPtr;		/* Marker to be printed */
{
    ImageMarker *imPtr = (ImageMarker *) markerPtr;
    Tk_PhotoHandle photoToken;

    if ((imPtr->imageToken == NULL) || (Blt_ImageDeleted(imPtr->imageToken))) {
	return;
    }
    photoToken = Tk_FindPhoto(imPtr->imageName);
    if (photoToken == NULL) {
	return;			/* image doesn't exist or isn't a photo image */
    }
    Blt_PhotoToPostScript(imPtr->graphPtr, photoToken, imPtr->x, imPtr->y);
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyImage --
 *
 *	Destroys the structure containing the attributes of the image
 * 	marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Image attributes (GCs, colors, image, etc) get destroyed.
 *	Memory is released, X resources are freed, and the graph is
 *	redrawn.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static void
DestroyImage(graphPtr, markerPtr)
    Graph *graphPtr;		/* Unused */
    Marker *markerPtr;
{
    ImageMarker *imPtr = (ImageMarker *) markerPtr;

    if (imPtr->imageToken != NULL) {
	Tk_FreeImage(imPtr->imageToken);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateImage --
 *
 *	Allocate memory and initialize methods for the new image marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the image marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateImage()
{
    ImageMarker *imPtr;

    imPtr = (ImageMarker *) calloc(1, sizeof(ImageMarker));
    if (imPtr != NULL) {
	imPtr->configSpecs = imageConfigSpecs;
	imPtr->configProc = ConfigureImage;
	imPtr->freeProc = DestroyImage;
	imPtr->drawProc = DrawImage;
	imPtr->coordsProc = TransformImage;
	imPtr->printProc = PrintImage;
	imPtr->type = MARKER_TYPE_IMAGE;
    }
    return ((Marker *) imPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureText --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a text marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureText(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    TextMarker *tmPtr = (TextMarker *) markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    tmPtr->attr.theta = FMOD(tmPtr->rotate, 360.0);
    if (tmPtr->attr.theta < 0.0) {
	tmPtr->attr.theta += 360.0;
    }
    gcValues.foreground = tmPtr->attr.fgColorPtr->pixel;
    gcValues.font = Tk_FontId(tmPtr->attr.font);
    gcMask = (GCFont | GCForeground);
    if (tmPtr->attr.bgColorPtr != NULL) {
	gcValues.background = tmPtr->attr.bgColorPtr->pixel;
	gcMask |= GCBackground;
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (tmPtr->attr.textGC != NULL) {
	Tk_FreeGC(graphPtr->display, tmPtr->attr.textGC);
    }
    tmPtr->attr.textGC = newGC;
    if (tmPtr->attr.fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, tmPtr->attr.fillGC);
	tmPtr->attr.fillGC = NULL;
    }
    if (tmPtr->attr.bgColorPtr != NULL) {
	gcValues.foreground = tmPtr->attr.bgColorPtr->pixel;
	newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
	tmPtr->attr.fillGC = newGC;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TranformText --
 *
 *	Calculate the layout position for a text marker.  Positional
 *	information is saved in the marker.  If the text is rotated,
 *	a bitmap containing the text is created.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If no background color has been specified, the GC stipple
 *	origins are changed to current window coordinates. For both
 *	rotated and non-rotated text, if any old bitmap is leftover,
 *	it is freed.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformText(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    TextMarker *tmPtr = (TextMarker *) markerPtr;
    Point2D p;			/* Upper left corner of text region */
    Extents2D extents;
    Dimension textDim;

    if (tmPtr->text == NULL) {
	return;
    }
    p = TransformMarkerPt(graphPtr, tmPtr->coordArr[0].x, tmPtr->coordArr[0].y,
	&tmPtr->axes);
    tmPtr->x = ROUND(p.x) + tmPtr->xOffset;
    tmPtr->y = ROUND(p.y) + tmPtr->yOffset;

    /*
     * Determine the bounding box of the text and test to see if it
     * is at least partially contained within the plotting area.
     */
    textDim = Blt_GetTextExtents(tmPtr->attr.font, tmPtr->text, 0);
    Blt_GetBoundingBox(textDim.width, textDim.height, tmPtr->attr.theta,
	&(textDim.width), &(textDim.height), (XPoint *)NULL);
    extents.xMin = (double)tmPtr->x;
    extents.yMax = (double)tmPtr->y + textDim.height;
    extents.xMax = (double)tmPtr->x + textDim.width;
    extents.yMin = (double)tmPtr->y;
    tmPtr->clipped = BoxesDontOverlap(graphPtr, &extents);

}

/*
 * ----------------------------------------------------------------------
 *
 * DrawText --
 *
 *	Draw the text marker on the graph given. If the text is not
 *	rotated, simply use the X text drawing routines. However, if
 *	the text has been rotated, stencil the bitmap representing
 *	the text. Since stencilling is very expensive, we try to
 *	draw right angle rotations with XCopyArea.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to draw the marker in its current mode.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawText(markerPtr)
    Marker *markerPtr;
{
    TextMarker *tmPtr = (TextMarker *) markerPtr;

    if (tmPtr->text != NULL) {
	Graph *graphPtr = markerPtr->graphPtr;

	Blt_DrawText(graphPtr->tkwin, graphPtr->pixwin, tmPtr->text,
	    &(tmPtr->attr), tmPtr->x, tmPtr->y);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintText --
 *
 *	Outputs PostScript commands to draw a text marker at a given
 *	x,y coordinate, rotation, anchor, and font.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript font and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
static void
PrintText(markerPtr)
    Marker *markerPtr;
{
    TextMarker *tmPtr = (TextMarker *) markerPtr;

    if (tmPtr->text != NULL) {
	Blt_PrintText(markerPtr->graphPtr, tmPtr->text, &(tmPtr->attr),
	    tmPtr->x, tmPtr->y);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyText --
 *
 *	Destroys the structure containing the attributes of the text
 * 	marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Text attributes (GCs, colors, stipple, font, etc) get destroyed.
 *	Memory is released, X resources are freed, and the graph is
 *	redrawn.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyText(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    TextMarker *tmPtr = (TextMarker *) markerPtr;

    if (tmPtr->attr.textGC != NULL) {
	Tk_FreeGC(graphPtr->display, tmPtr->attr.textGC);
    }
    if (tmPtr->attr.fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, tmPtr->attr.fillGC);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateText --
 *
 *	Allocate memory and initialize methods for the new text marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the text marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateText()
{
    TextMarker *tmPtr;

    tmPtr = (TextMarker *) calloc(1, sizeof(TextMarker));
    if (tmPtr != NULL) {
	tmPtr->configSpecs = textConfigSpecs;
	tmPtr->configProc = ConfigureText;
	tmPtr->freeProc = DestroyText;
	tmPtr->drawProc = DrawText;
	tmPtr->coordsProc = TransformText;
	tmPtr->printProc = PrintText;
	tmPtr->type = MARKER_TYPE_TEXT;
	tmPtr->attr.justify = TK_JUSTIFY_CENTER;
	tmPtr->attr.padLeft = tmPtr->attr.padRight = 4;
	tmPtr->attr.padTop = tmPtr->attr.padBottom = 4;
    }
    return ((Marker *) tmPtr);
}


static void ChildEventProc _ANSI_ARGS_((ClientData clientData,
	XEvent *eventPtr));
static void ChildGeometryProc _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin));

static void ChildCustodyProc _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin));

static Tk_GeomMgr winMarkerMgrInfo =
{
    "graph",			/* Name of geometry manager used by winfo */
    ChildGeometryProc,		/* Procedure to for new geometry requests */
    ChildCustodyProc,		/* Procedure when window is taken away */
};

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureWindow --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a window marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as window pathname, placement,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureWindow(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    WindowMarker *wmPtr = (WindowMarker *) markerPtr;
    Tk_Window tkwin;

    if (wmPtr->pathName == NULL) {
	return TCL_OK;
    }
    tkwin = Tk_NameToWindow(graphPtr->interp, wmPtr->pathName, graphPtr->tkwin);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    if (Tk_Parent(tkwin) != graphPtr->tkwin) {
	Tcl_AppendResult(graphPtr->interp, "\"", wmPtr->pathName,
	    "\" is not a child of \"", Tk_PathName(graphPtr->tkwin), "\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    if (tkwin != wmPtr->tkwin) {
	if (wmPtr->tkwin != NULL) {
	    Tk_DeleteEventHandler(wmPtr->tkwin, StructureNotifyMask,
		ChildEventProc, (ClientData)wmPtr);
	    Tk_ManageGeometry(wmPtr->tkwin, (Tk_GeomMgr *) 0, (ClientData)0);
	    Tk_UnmapWindow(wmPtr->tkwin);
	}
	Tk_CreateEventHandler(tkwin, StructureNotifyMask, ChildEventProc,
	    (ClientData)wmPtr);
	Tk_ManageGeometry(tkwin, &winMarkerMgrInfo, (ClientData)wmPtr);
    }
    wmPtr->tkwin = tkwin;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformWindow --
 *
 *	Calculate the layout position for a window marker.  Positional
 *	information is saved in the marker.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformWindow(markerPtr)
    Marker *markerPtr;
{
    WindowMarker *wmPtr = (WindowMarker *) markerPtr;
    Graph *graphPtr = markerPtr->graphPtr;
    Point2D p;
    Extents2D extents;
    int x, y, width, height;

    if (wmPtr->tkwin == (Tk_Window)NULL) {
	return;
    }
    p = TransformMarkerPt(graphPtr, wmPtr->coordArr[0].x, wmPtr->coordArr[0].y,
	&wmPtr->axes);

    width = (wmPtr->reqWidth > 0) ? wmPtr->reqWidth : Tk_ReqWidth(wmPtr->tkwin);
    height = (wmPtr->reqHeight > 0) ? wmPtr->reqHeight : Tk_ReqHeight(wmPtr->tkwin);
    x = ROUND(p.x) + wmPtr->xOffset;
    y = ROUND(p.y) + wmPtr->yOffset;
    p = Blt_TranslateBoxCoords(x, y, width, height, wmPtr->anchor);

    wmPtr->x = (int)p.x;
    wmPtr->y = (int)p.y;
    wmPtr->width = width;
    wmPtr->height = height;
    /*
     * Determine the bounding box of the window and test to see if it
     * is at least partially contained within the plotting area.
     */
    extents.xMin = (double)wmPtr->x;
    extents.yMax = (double)wmPtr->y + wmPtr->height;
    extents.xMax = (double)wmPtr->x + wmPtr->width;
    extents.yMin = (double)wmPtr->y;
    wmPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
}

/*ARGSUSED*/
static void
DrawWindow(markerPtr)
    Marker *markerPtr;
{
    WindowMarker *wmPtr = (WindowMarker *) markerPtr;

    if (wmPtr->tkwin == NULL) {
	return;
    }
    if ((wmPtr->height != Tk_Height(wmPtr->tkwin)) ||
	(wmPtr->width != Tk_Width(wmPtr->tkwin)) ||
	(wmPtr->x != Tk_X(wmPtr->tkwin)) ||
	(wmPtr->y != Tk_Y(wmPtr->tkwin))) {
	Tk_MoveResizeWindow(wmPtr->tkwin, wmPtr->x, wmPtr->y,
	    wmPtr->width, wmPtr->height);
    }
    if (!Tk_IsMapped(wmPtr->tkwin)) {
	Tk_MapWindow(wmPtr->tkwin);
    }
}

static void
PrintWindow(markerPtr)
    Marker *markerPtr;
{
    WindowMarker *wmPtr = (WindowMarker *) markerPtr;

    if (wmPtr->tkwin == NULL) {
	return;
    }
    if (Tk_IsMapped(wmPtr->tkwin)) {
	Blt_WindowToPostScript(wmPtr->graphPtr, wmPtr->tkwin, wmPtr->x,
	    wmPtr->y, wmPtr->width, wmPtr->height);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyWindow --
 *
 *	Destroys the structure containing the attributes of the window
 *      marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Window is destroyed and removed from the screen.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
DestroyWindow(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    WindowMarker *wmPtr = (WindowMarker *) markerPtr;

    if (wmPtr->tkwin != NULL) {
	Tk_DeleteEventHandler(wmPtr->tkwin, StructureNotifyMask,
	    ChildEventProc, (ClientData)wmPtr);
	Tk_ManageGeometry(wmPtr->tkwin, (Tk_GeomMgr *) 0, (ClientData)0);
	Tk_DestroyWindow(wmPtr->tkwin);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateWindow --
 *
 *	Allocate memory and initialize methods for the new window marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the window marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateWindow()
{
    WindowMarker *wmPtr;

    wmPtr = (WindowMarker *) calloc(1, sizeof(WindowMarker));
    if (wmPtr != NULL) {
	wmPtr->configSpecs = windowConfigSpecs;
	wmPtr->configProc = ConfigureWindow;
	wmPtr->freeProc = DestroyWindow;
	wmPtr->drawProc = DrawWindow;
	wmPtr->coordsProc = TransformWindow;
	wmPtr->printProc = PrintWindow;
	wmPtr->type = MARKER_TYPE_WINDOW;
    }
    return ((Marker *) wmPtr);
}

/*
 * --------------------------------------------------------------
 *
 * ChildEventProc --
 *
 *	This procedure is invoked whenever StructureNotify events
 *	occur for a window that's managed as part of a graph window
 *	marker. This procedure's only purpose is to clean up when
 *	windows are deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The window is disassociated from the window item when it is
 *	deleted.
 *
 * --------------------------------------------------------------
 */
static void
ChildEventProc(clientData, eventPtr)
    ClientData clientData;	/* Pointer to record describing window item. */
    XEvent *eventPtr;		/* Describes what just happened. */
{
    WindowMarker *wmPtr = (WindowMarker *) clientData;

    if (eventPtr->type == DestroyNotify) {
	wmPtr->tkwin = NULL;
    }
}

/*
 * --------------------------------------------------------------
 *
 * ChildGeometryProc --
 *
 *	This procedure is invoked whenever a window that's associated
 *	with a window item changes its requested dimensions.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The size and location on the window of the window may change,
 *	depending on the options specified for the window item.
 *
 * --------------------------------------------------------------
 */
/* ARGSUSED */
static void
ChildGeometryProc(clientData, tkwin)
    ClientData clientData;	/* Pointer to record for window item. */
    Tk_Window tkwin;		/* Window that changed its desired size. */
{
    WindowMarker *wmPtr = (WindowMarker *) clientData;

    if (wmPtr->reqWidth == 0) {
	wmPtr->width = Tk_ReqWidth(tkwin);
    }
    if (wmPtr->reqHeight == 0) {
	wmPtr->height = Tk_ReqHeight(tkwin);
    }
}

/*
 * --------------------------------------------------------------
 *
 * ChildCustodyProc --
 *
 *	This procedure is invoked when a slave window has been
 *	stolen by another geometry manager.  The information and
 *	memory associated with the slave window is released.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arranges for the widget formerly associated with the slave
 *	window to have its layout re-computed and arranged at the
 *	next idle point.
 *
 * --------------------------------------------------------------
 */
 /* ARGSUSED */
static void
ChildCustodyProc(clientData, tkwin)
    ClientData clientData;	/* Window marker to be destroyed. */
    Tk_Window tkwin;		/* Not used. */
{
    Marker *markerPtr = (Marker *) clientData;
    Graph *graphPtr;

    graphPtr = markerPtr->graphPtr;
    DestroyMarker(markerPtr);
    /*
     * Not really needed. We should get an Expose event when the
     * child window is unmapped
     */
    Blt_RedrawGraph(graphPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureLine --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a line marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as line width, colors, dashes,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureLine(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    LineMarker *lmPtr = (LineMarker *) markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    gcMask = (GCForeground | GCLineWidth | GCLineStyle | GCCapStyle |
	GCJoinStyle);
    gcValues.foreground = lmPtr->color.fgColor->pixel;
    if (lmPtr->color.bgColor != COLOR_NONE) {
	gcMask |= GCBackground;
	gcValues.background = lmPtr->color.bgColor->pixel;
    }
    gcValues.cap_style = CapRound;
    gcValues.join_style = JoinRound;
    gcValues.dash_offset = 0;
    gcValues.line_width = LineWidth(lmPtr->lineWidth);
    gcValues.line_style =
	(lmPtr->dashes.numValues > 0) ? LineOnOffDash : LineSolid;
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (lmPtr->gc != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lmPtr->gc);
    }
    if (lmPtr->dashes.numValues > 0) {
	XSetDashes(graphPtr->display, newGC, 0, lmPtr->dashes.valueArr,
	    lmPtr->dashes.numValues);
    }
    lmPtr->gc = newGC;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformLine --
 *
 *	Calculate the layout position for a line marker.  Positional
 *	information is saved in the marker.  The line positions are
 *	stored in an array of points (malloc'ed).
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformLine(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    LineMarker *lmPtr = (LineMarker *) markerPtr;
    Point2D p1, p2;
    register Point2D *p;
    XSegment *segArr;
    int count;
    register int i;
    Extents2D extents;

    lmPtr->numSegments = 0;
    if (lmPtr->segArr != NULL) {
	free((char *)lmPtr->segArr);
    }
    if (lmPtr->numCoords < 2) {
	return;			/* Too few points */
    }
    Blt_SetClipRegion(graphPtr, &extents);
    segArr = (XSegment *)calloc(lmPtr->numCoords, sizeof(XSegment));

    count = 0;
    p = lmPtr->coordArr;
    p1 = TransformMarkerPt(graphPtr, p->x, p->y, &lmPtr->axes);
    p1.x += (double)lmPtr->xOffset;
    p1.y += (double)lmPtr->yOffset;

    p++;
    for (i = 1; i < lmPtr->numCoords; i++, p++) {
	p2 = TransformMarkerPt(graphPtr, p->x, p->y, &lmPtr->axes);
	p2.x += (double)lmPtr->xOffset;
	p2.y += (double)lmPtr->yOffset;
	if (Blt_ClipSegment(&extents, &p1, &p2, segArr + count)) {
	    count++;
	}
	p1 = p2;
    }
    lmPtr->numSegments = count;
    lmPtr->segArr = segArr;
    lmPtr->clipped = (lmPtr->numSegments == 0);
}

static void
DrawLine(markerPtr)
    Marker *markerPtr;
{
    LineMarker *lmPtr = (LineMarker *) markerPtr;

    if (lmPtr->numSegments > 0) {
	Graph *graphPtr = markerPtr->graphPtr;

	XDrawSegments(graphPtr->display, graphPtr->pixwin, lmPtr->gc,
	    lmPtr->segArr, lmPtr->numSegments);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintLine --
 *
 *	Prints postscript commands to display the connect line.
 *	Dashed lines need to be handled specially, especially if a
 *	a background color is designated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript output commands are saved in the interpreter
 *	(infoPtr->interp) result field.
 *
 * ----------------------------------------------------------------------
 */
static void
PrintLine(markerPtr)
    Marker *markerPtr;
{
    LineMarker *lmPtr = (LineMarker *) markerPtr;

    if (lmPtr->numSegments > 0) {
	Graph *graphPtr = markerPtr->graphPtr;

	Blt_LineWidthToPostScript(graphPtr, lmPtr->lineWidth);
	Blt_ForegroundToPostScript(graphPtr, lmPtr->color.fgColor);
	Blt_LineDashesToPostScript(graphPtr, &(lmPtr->dashes));
	if ((lmPtr->dashes.numValues > 0) &&
	    (lmPtr->color.bgColor != COLOR_NONE)) {
	    Blt_PostScriptAppend(graphPtr, "/DashesProc {\n  gsave\n    ");
	    Blt_BackgroundToPostScript(graphPtr, lmPtr->color.bgColor);
	    Blt_PostScriptAppend(graphPtr, "    ");
	    Blt_LineDashesToPostScript(graphPtr, (Dashes *)NULL);
	    Blt_PostScriptAppend(graphPtr, "stroke\n  grestore\n} def\n");
	} else {
	    Blt_PostScriptAppend(graphPtr, "/DashesProc {} def\n");
	}
	Blt_SegmentsToPostScript(graphPtr, lmPtr->segArr, lmPtr->numSegments);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyLine --
 *
 *	Destroys the structure and attributes of a line marker.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Line attributes (GCs, colors, stipple, etc) get released.
 *	Memory is deallocated, X resources are freed.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyLine(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    LineMarker *lmPtr = (LineMarker *) markerPtr;

    Blt_FreeColorPair(&lmPtr->color);
    if (lmPtr->gc != NULL) {
	Blt_FreePrivateGC(graphPtr->display, lmPtr->gc);
    }
    if (lmPtr->segArr != NULL) {
	free((char *)lmPtr->segArr);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateLine --
 *
 *	Allocate memory and initialize methods for a new line marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the line marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreateLine()
{
    LineMarker *lmPtr;

    lmPtr = (LineMarker *) calloc(1, sizeof(LineMarker));
    if (lmPtr != NULL) {
	lmPtr->configSpecs = lineConfigSpecs;
	lmPtr->configProc = ConfigureLine;
	lmPtr->freeProc = DestroyLine;
	lmPtr->drawProc = DrawLine;
	lmPtr->coordsProc = TransformLine;
	lmPtr->printProc = PrintLine;
	lmPtr->type = MARKER_TYPE_LINE;
    }
    return ((Marker *) lmPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigurePolygon --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a polygon marker.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as polygon color, dashes, fillstyle,
 *	etc. get set for markerPtr;  old resources get freed, if there
 *	were any.  The marker is eventually redisplayed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigurePolygon(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *) markerPtr;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    gcMask = (GCLineWidth | GCLineStyle | GCCapStyle | GCJoinStyle);
    if (pmPtr->outline.fgColor != COLOR_NONE) {
	gcMask |= GCForeground;
	gcValues.foreground = pmPtr->outline.fgColor->pixel;
    }
    if (pmPtr->outline.bgColor != COLOR_NONE) {
	gcMask |= GCBackground;
	gcValues.background = pmPtr->outline.bgColor->pixel;
    }
    gcValues.cap_style = CapRound;
    gcValues.join_style = JoinRound;
    gcValues.line_style = LineSolid;
    gcValues.dash_offset = 0;
    gcValues.line_width = LineWidth(pmPtr->lineWidth);
    if (pmPtr->dashes.numValues > 0) {
	gcValues.line_style = (pmPtr->outline.bgColor == COLOR_NONE)
	    ? LineOnOffDash : LineDoubleDash;
    }
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (pmPtr->dashes.numValues > 0) {
	XSetDashes(graphPtr->display, newGC, 0, pmPtr->dashes.valueArr,
	    pmPtr->dashes.numValues);
    }
    if (pmPtr->outlineGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, pmPtr->outlineGC);
    }
    pmPtr->outlineGC = newGC;

    gcMask = 0;
    if (pmPtr->fill.fgColor != COLOR_NONE) {
	gcMask |= GCForeground;
	gcValues.foreground = pmPtr->fill.fgColor->pixel;
    }
    if (pmPtr->fill.bgColor != COLOR_NONE) {
	gcMask |= GCBackground;
	gcValues.background = pmPtr->fill.bgColor->pixel;
    }
    if (pmPtr->stipple != None) {
	gcValues.stipple = pmPtr->stipple;
	gcValues.fill_style = (pmPtr->fill.bgColor != COLOR_NONE)
	    ? FillOpaqueStippled : FillStippled;
	gcMask |= (GCStipple | GCFillStyle);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (pmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, pmPtr->fillGC);
    }
    pmPtr->fillGC = newGC;

    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformPolygon --
 *
 *	Calculate the layout position for a polygon marker.  Positional
 *	information is saved in the polygon in an array of points
 *	(malloc'ed).
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
TransformPolygon(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *) markerPtr;
    register int i;
    XPoint *polygonArr;
    register XPoint *pointPtr;
    Point2D point;
    Extents2D extents;
    int numPoints;

    if (pmPtr->polygonArr != NULL) {
	free((char *)pmPtr->polygonArr);
	pmPtr->polygonArr = NULL;
    }
    pmPtr->numPoints = 0;
    if (pmPtr->numCoords < 3) {
	return;			/* Too few points */
    }
    numPoints = pmPtr->numCoords + 1;
    polygonArr = (XPoint *)malloc(numPoints * sizeof(XPoint));
    if (polygonArr == NULL) {
	return;			/* Can't allocate point array */
    }
    pointPtr = polygonArr;

    /*
     * Determine the bounding box of the polygon and test to see if it
     * is at least partially contained within the plotting area.
     */
    extents.yMin = extents.xMin = bltPosInfinity;
    extents.yMax = extents.xMax = bltNegInfinity;
    for (i = 0; i < pmPtr->numCoords; i++) {
	point = TransformMarkerPt(graphPtr, pmPtr->coordArr[i].x,
	    pmPtr->coordArr[i].y, &pmPtr->axes);
	point.x += (double)pmPtr->xOffset;
	point.y += (double)pmPtr->yOffset;
	pointPtr->x = ROUND(point.x);
	if (point.x > extents.xMax) {
	    extents.xMax = point.x;
	} else if (point.x < extents.xMin) {
	    extents.xMin = point.x;
	}
	pointPtr->y = ROUND(point.y);
	if (point.y > extents.yMax) {
	    extents.yMax = point.y;
	} else if (point.y < extents.yMin) {
	    extents.yMin = point.y;
	}
	BoundPoint(pointPtr);	/* Bound the points of the polygon to
				 * fit the size of a signed short int.
				 * We'll let X do the clipping here,
				 * because we really don't want to
				 * write a polygon clipping routine.
				 * Just yet. */
	pointPtr++;
    }
    *pointPtr = polygonArr[0];	/* Ensure that the polygon is closed */

    pmPtr->clipped = BoxesDontOverlap(graphPtr, &extents);
    if (pmPtr->clipped) {
	pmPtr->numPoints = 0;
	free((char *)polygonArr);
    } else {
	pmPtr->numPoints = numPoints;
	pmPtr->polygonArr = polygonArr;
    }
}


static void
DrawPolygon(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *) markerPtr;

    if (pmPtr->numPoints < 3) {
	return;
    }
    if (pmPtr->fill.fgColor != COLOR_NONE) {
	XFillPolygon(graphPtr->display, graphPtr->pixwin, pmPtr->fillGC,
	    pmPtr->polygonArr, pmPtr->numPoints, Complex,
	    CoordModeOrigin);
    }
    if ((pmPtr->lineWidth > 0) &&
	(pmPtr->outline.fgColor != COLOR_NONE)) {
	XDrawLines(graphPtr->display, graphPtr->pixwin, pmPtr->outlineGC,
	    pmPtr->polygonArr, pmPtr->numPoints, CoordModeOrigin);
    }
}


static void
PrintPolygon(markerPtr)
    Marker *markerPtr;
{
    Graph *graphPtr = markerPtr->graphPtr;
    PolygonMarker *pmPtr = (PolygonMarker *) markerPtr;

    if (pmPtr->numPoints < 3) {
	return;
    }
    if (pmPtr->fill.fgColor != COLOR_NONE) {
	/*
	 * If the background fill color was specified, draw the polygon
	 * in a solid fashion with that color.
	 */
	if (pmPtr->fill.bgColor != COLOR_NONE) {
	    Blt_BackgroundToPostScript(graphPtr, pmPtr->fill.bgColor);
	    Blt_PolygonToPostScript(graphPtr, pmPtr->polygonArr, pmPtr->numPoints);
	}
	if (pmPtr->stipple != None) {
	    int width, height;
	    /*
	     * Draw the stipple in the foreground color.
	     */
	    Blt_ForegroundToPostScript(graphPtr, pmPtr->fill.fgColor);
	    Blt_LinesToPostScript(graphPtr, pmPtr->polygonArr, pmPtr->numPoints);
	    Blt_PostScriptAppend(graphPtr, "closepath\n");
	    Tk_SizeOfBitmap(graphPtr->display, pmPtr->stipple, &width, &height);
	    Blt_StippleToPostScript(graphPtr, pmPtr->stipple, width, height, True);
	}
    }
    /*
     * Draw the outline in the foreground color.
     */
    if ((pmPtr->lineWidth > 0) ||
	(pmPtr->outline.fgColor != COLOR_NONE)) {
	/*
	 * Set up the line attributes.  There may be no dashes.  This
	 * will draw a possibly stippled line in the foreground color.
	 * But, it will call the following "DashesProc" routine first.
	 */
	Blt_LineWidthToPostScript(graphPtr, pmPtr->lineWidth);
	Blt_ForegroundToPostScript(graphPtr, pmPtr->outline.fgColor);
	Blt_LineDashesToPostScript(graphPtr, &(pmPtr->dashes));
	if ((pmPtr->outline.bgColor != COLOR_NONE) &&
	    (pmPtr->dashes.numValues > 0)) {
	    /*
	     * This routine gets called if there is a background color
	     * specified *and* there are some dashes. It will draw a
	     * solid line in the background color.
	     */
	    Blt_PostScriptAppend(graphPtr, "/DashesProc {\n  gsave\n    ");
	    Blt_BackgroundToPostScript(graphPtr, pmPtr->outline.bgColor);
	    Blt_PostScriptAppend(graphPtr, "    ");
	    Blt_LineDashesToPostScript(graphPtr, (Dashes *)NULL);
	    Blt_PostScriptAppend(graphPtr, "stroke\n  grestore\n} def\n");
	} else {
	    Blt_PostScriptAppend(graphPtr, "/DashesProc {} def\n");
	}
	Blt_PrintLine(graphPtr, pmPtr->polygonArr, pmPtr->numPoints);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyPolygon --
 *
 *	Release memory and resources allocated for the polygon element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the polygon element is freed up.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyPolygon(graphPtr, markerPtr)
    Graph *graphPtr;
    Marker *markerPtr;
{
    PolygonMarker *pmPtr = (PolygonMarker *) markerPtr;

    if (pmPtr->fillGC != NULL) {
	Tk_FreeGC(graphPtr->display, pmPtr->fillGC);
    }
    if (pmPtr->outlineGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, pmPtr->outlineGC);
    }
    if (pmPtr->polygonArr != NULL) {
	free((char *)pmPtr->polygonArr);
    }
    Blt_FreeColorPair(&pmPtr->outline);
    Blt_FreeColorPair(&pmPtr->fill);
}

/*
 * ----------------------------------------------------------------------
 *
 * CreatePolygon --
 *
 *	Allocate memory and initialize methods for the new polygon marker.
 *
 * Results:
 *	The pointer to the newly allocated marker structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the polygon marker structure.
 *
 * ----------------------------------------------------------------------
 */
static Marker *
CreatePolygon()
{
    PolygonMarker *pmPtr;

    pmPtr = (PolygonMarker *) calloc(1, sizeof(PolygonMarker));
    if (pmPtr != NULL) {
	pmPtr->configSpecs = polygonConfigSpecs;
	pmPtr->configProc = ConfigurePolygon;
	pmPtr->freeProc = DestroyPolygon;
	pmPtr->drawProc = DrawPolygon;
	pmPtr->coordsProc = TransformPolygon;
	pmPtr->printProc = PrintPolygon;
	pmPtr->type = MARKER_TYPE_POLYGON;
    }
    return ((Marker *) pmPtr);
}


static Marker *
NameToMarker(graphPtr, name)
    Graph *graphPtr;
    char *name;
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(graphPtr->markerTable), name);
    if (hPtr == NULL) {
	Tcl_AppendResult(graphPtr->interp, "can't find marker \"", name,
	    "\" in \"", Tk_PathName(graphPtr->tkwin), (char *)NULL);
	return NULL;
    }
    return (Marker *) Tcl_GetHashValue(hPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * NameToMarkerType --
 *
 *	Convert the marker type string value into a numeric value.
 *
 * Results:
 *	The value representing the marker type is returned.
 *
 * ----------------------------------------------------------------------
 */
static MarkerType
NameToMarkerType(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    MarkerType type;
    unsigned int length;
    char c;

    c = name[0];
    length = strlen(name);
    if ((c == 't') && (strncmp(name, "text", length) == 0)) {
	type = MARKER_TYPE_TEXT;
    } else if ((c == 'l') && (strncmp(name, "line", length) == 0)) {
	type = MARKER_TYPE_LINE;
    } else if ((c == 'b') && (strncmp(name, "bitmap", length) == 0)) {
	type = MARKER_TYPE_BITMAP;
    } else if ((c == 'i') && (strncmp(name, "image", length) == 0)) {
	type = MARKER_TYPE_IMAGE;
    } else if ((c == 'p') && (strncmp(name, "polygon", length) == 0)) {
	type = MARKER_TYPE_POLYGON;
    } else if ((c == 'w') && (strncmp(name, "window", length) == 0)) {
	type = MARKER_TYPE_WINDOW;
    } else {
	Tcl_AppendResult(interp, "unknown marker type \"", name,
	    "\": should be \"text\", \"line\", \"polygon\", \"bitmap\", \"image\", or \
\"window\"", (char *)NULL);
	return MARKER_TYPE_UNKNOWN;
    }
    return type;
}

/*
 * ----------------------------------------------------------------------
 *
 * NameOfMarkerType --
 *
 *	Convert the marker type value into a string.
 *
 * Results:
 *	The string representing the marker type is returned.
 *
 * ----------------------------------------------------------------------
 */
static char *
NameOfMarkerType(type)
    MarkerType type;
{
    switch (type) {
    case MARKER_TYPE_BITMAP:
	return "bitmap";
    case MARKER_TYPE_IMAGE:
	return "image";
    case MARKER_TYPE_LINE:
	return "line";
    case MARKER_TYPE_POLYGON:
	return "polygon";
    case MARKER_TYPE_TEXT:
	return "text";
    case MARKER_TYPE_WINDOW:
	return "window";
    default:
	return "unknown marker type";
    }
}

static int
RenameMarker(graphPtr, markerPtr, oldName, newName)
    Graph *graphPtr;
    Marker *markerPtr;
    char *oldName;
    char *newName;
{
    int isNew;
    Tcl_HashEntry *hPtr;

    /*
     * Rename the marker only if no marker already exists by that name
     */
    hPtr = Tcl_CreateHashEntry(&(graphPtr->markerTable), newName, &isNew);
    if (!isNew) {
	Tcl_AppendResult(graphPtr->interp, "can't rename marker: \"", newName,
	    "\" already exists", (char *)NULL);
	return TCL_ERROR;
    }
    markerPtr->name = strdup(newName);
    markerPtr->hashPtr = hPtr;
    Tcl_SetHashValue(hPtr, (char *)markerPtr);
    /* Delete the old hash entry */
    hPtr = Tcl_FindHashEntry(&(graphPtr->markerTable), oldName);
    Tcl_DeleteHashEntry(hPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * NamesOp --
 *
 *	Returns a list of marker identifiers in interp->result;
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
static int
NamesOp(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char **argv;
{
    Marker *markerPtr;
    Blt_ListItem *itemPtr;
    register int i;

    for (itemPtr = Blt_ListFirstItem(&(graphPtr->markerList)); itemPtr != NULL;
	itemPtr = Blt_ListNextItem(itemPtr)) {
	markerPtr = (Marker *) Blt_ListGetValue(itemPtr);
	if (argc == 3) {
	    Tcl_AppendElement(graphPtr->interp, markerPtr->name);
	    continue;
	}
	for (i = 3; i < argc; i++) {
	    if (Tcl_StringMatch(markerPtr->name, argv[i])) {
		Tcl_AppendElement(graphPtr->interp, markerPtr->name);
		break;
	    }
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * CgetOp --
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CgetOp(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    int argc;			/* not used */
    char *argv[];		/* Contains the markerId and the
				 * option to be queried */
{
    Marker *markerPtr;

    markerPtr = NameToMarker(graphPtr, argv[3]);
    if (markerPtr == NULL) {
	return TCL_ERROR;
    }
    if (Tk_ConfigureValue(graphPtr->interp, graphPtr->tkwin,
	    markerPtr->configSpecs, (char *)markerPtr, argv[4], 0) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureOp(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    int argc;			/* Number of options */
    char *argv[];		/* List of marker options */
{
    Marker *markerPtr;
    int flags = TK_CONFIG_ARGV_ONLY;
    char *oldName;
    int numNames, numOpts;
    char **options;
    register int i;

    /* Figure out where the option value pairs begin */
    argc -= 3;
    argv += 3;
    for (i = 0; i < argc; i++) {
	if (argv[i][0] == '-') {
	    break;
	}
	if (NameToMarker(graphPtr, argv[i]) == NULL) {
	    return TCL_ERROR;	/* Can't find named marker */
	}
    }
    numNames = i;		/* Number of element names specified */
    numOpts = argc - i;		/* Number of options specified */
    options = argv + numNames;	/* Start of options in argv  */

    for (i = 0; i < numNames; i++) {
	markerPtr = NameToMarker(graphPtr, argv[i]);
	if (numOpts == 0) {
	    return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		    markerPtr->configSpecs, (char *)markerPtr, (char *)NULL,
		    flags));
	} else if (numOpts == 1) {
	    return (Tk_ConfigureInfo(graphPtr->interp, graphPtr->tkwin,
		    markerPtr->configSpecs, (char *)markerPtr, options[0], flags));
	}
	oldName = strdup(markerPtr->name);
	if (Tk_ConfigureWidget(graphPtr->interp, graphPtr->tkwin,
		markerPtr->configSpecs, numOpts, options, (char *)markerPtr,
		flags) != TCL_OK) {
	    goto error;
	}
	if (strcmp(oldName, markerPtr->name) != 0) {
	    if (RenameMarker(graphPtr, markerPtr, oldName,
		    markerPtr->name) != TCL_OK) {
		free(markerPtr->name);
		markerPtr->name = oldName;
		return TCL_ERROR;
	    }
	}
	free(oldName);
	if ((*markerPtr->configProc) (markerPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (markerPtr->hidden) {
	    markerPtr->flags |= COORDS_NEEDED;
	    if (markerPtr->drawUnder) {
		graphPtr->flags |= UPDATE_PIXMAP;
	    }
	    Blt_RedrawGraph(graphPtr);
	}
    }
    return TCL_OK;
  error:
    free(oldName);
    return TCL_ERROR;
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateOp --
 *
 *	This procedure creates and initializes a new marker.
 *
 * Results:
 *	The return value is a pointer to a structure describing
 *	the new element.  If an error occurred, then the return
 *	value is NULL and an error message is left in interp->result.
 *
 * Side effects:
 *	Memory is allocated, etc.
 *
 * ----------------------------------------------------------------------
 */
static int
CreateOp(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    int argc;
    char *argv[];
{
    Marker *markerPtr;
    Tcl_HashEntry *hPtr;
    int isNew;
    MarkerType type;
    register int i;
    char *name;
    char string[200];
    unsigned int length;

    markerPtr = NULL;

    /* Create the new marker based upon the given type */

    type = NameToMarkerType(graphPtr->interp, argv[3]);
    switch (type) {
    case MARKER_TYPE_LINE:
	markerPtr = CreateLine();
	break;
    case MARKER_TYPE_WINDOW:
	markerPtr = CreateWindow();
	break;
    case MARKER_TYPE_BITMAP:
	markerPtr = CreateBitmap();
	break;
    case MARKER_TYPE_IMAGE:
	markerPtr = CreateImage();
	break;
    case MARKER_TYPE_POLYGON:
	markerPtr = CreatePolygon();
	break;
    case MARKER_TYPE_TEXT:
	markerPtr = CreateText();
	break;
    default:
	return TCL_ERROR;
    }
    assert(markerPtr);
    markerPtr->graphPtr = graphPtr;
    markerPtr->type = type;
    markerPtr->hidden = markerPtr->drawUnder = FALSE;
    markerPtr->flags |= COORDS_NEEDED;

    /* Scan for "-name" option. We need it for the component name */
    name = NULL;
    for (i = 4; i < argc; i += 2) {
	length = strlen(argv[i]);
	if ((length > 1) && (strncmp(argv[i], "-name", length) == 0)) {
	    name = argv[i + 1];
	    break;
	}
    }
    if (name == NULL) {

	/* No name was given for the marker, so make up one. */

	sprintf(string, "marker%d", graphPtr->nextMarkerId++);
	name = string;
    }
    markerPtr->name = strdup(name);
    if (Blt_ConfigureWidgetComponent(graphPtr->interp, graphPtr->tkwin,
	    markerPtr->name, markerClasses[type], markerPtr->configSpecs,
	    argc - 4, argv + 4, (char *)markerPtr, 0) != TCL_OK) {
	DestroyMarker(markerPtr);
	return TCL_ERROR;
    }
    if ((*markerPtr->configProc) (markerPtr) != TCL_OK) {
	DestroyMarker(markerPtr);
	return TCL_ERROR;
    }
    hPtr = Tcl_CreateHashEntry(&(graphPtr->markerTable), markerPtr->name,
	&isNew);
    if (!isNew) {
	Marker *oldMarkerPtr;
	/*
	 * Marker by the same name already exists.  Delete the old
	 * marker and it's list entry.  But save the hash entry.
	 */
	oldMarkerPtr = (Marker *) Tcl_GetHashValue(hPtr);
	oldMarkerPtr->hashPtr = NULL;
	DestroyMarker(oldMarkerPtr);
    }
    Tcl_SetHashValue(hPtr, (ClientData)markerPtr);
    markerPtr->hashPtr = hPtr;

    markerPtr->listPtr = Blt_ListAppend(&(graphPtr->markerList), (char *)markerPtr,
	(ClientData)markerPtr);
    if (markerPtr->drawUnder) {
	graphPtr->flags |= UPDATE_PIXMAP;
    }
    Blt_RedrawGraph(graphPtr);
    Tcl_SetResult(graphPtr->interp, markerPtr->name, TCL_STATIC);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * DeleteOp --
 *
 *	Deletes the marker given by markerId.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new display list.
 *
 * ----------------------------------------------------------------------
 */
static int
DeleteOp(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char *argv[];
{
    Marker *markerPtr;
    register int i;

    for (i = 3; i < argc; i++) {
	markerPtr = NameToMarker(graphPtr, argv[i]);
	if (markerPtr == NULL) {
	    return TCL_ERROR;
	}
	DestroyMarker(markerPtr);
    }
    Blt_RedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * RelinkOp --
 *
 *	Reorders the marker (given by the first markerId) before/after
 *	the another marker (given by the second markerId) in the
 *	marker display list.  If no second markerId is given, the
 *	marker is placed at the beginning/end of the list.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new display list.
 *
 * ----------------------------------------------------------------------
 */
static int
RelinkOp(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;
    char *argv[];
{
    Blt_ListItem *itemPtr, *placePtr;
    Marker *markerPtr;

    /* Find the new marker to be inserted into the display list */
    markerPtr = NameToMarker(graphPtr, argv[3]);
    if (markerPtr == NULL) {
	return TCL_ERROR;
    }
    /* Now use the marker to find the entry in the display list */
    itemPtr = markerPtr->listPtr;

    placePtr = NULL;
    if (argc == 5) {
	markerPtr = NameToMarker(graphPtr, argv[4]);
	if (markerPtr == NULL) {
	    return TCL_ERROR;
	}
	placePtr = markerPtr->listPtr;
    }
    /* Unlink the list item and relink it at the new location */
    Blt_ListUnlinkItem(itemPtr);

    if (argv[2][0] == 'a') {
	Blt_ListLinkAfter(&(graphPtr->markerList), itemPtr, placePtr);
    } else {
	Blt_ListLinkBefore(&(graphPtr->markerList), itemPtr, placePtr);
    }
    Blt_RedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ExistsOp --
 *
 *	Returns if marker by a given ID currently exists.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ExistsOp(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;			/* not used */
    char **argv;
{
    Tcl_HashEntry *hPtr;

    hPtr = Tcl_FindHashEntry(&(graphPtr->markerTable), argv[3]);
    graphPtr->interp->result = (hPtr != NULL) ? "1" : "0";
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TypeOp --
 *
 *	Returns a symbolic name for the type of the marker whose ID is
 *	given.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *	interp->result contains the symbolic type of the marker.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
TypeOp(graphPtr, argc, argv)
    Graph *graphPtr;
    int argc;			/* not used */
    char **argv;
{
    Marker *markerPtr;

    markerPtr = NameToMarker(graphPtr, argv[3]);
    if (markerPtr == NULL) {
	return TCL_ERROR;
    }
    Tcl_SetResult(graphPtr->interp, NameOfMarkerType(markerPtr->type),
	TCL_STATIC);
    return TCL_OK;
}

/* Public routines */

/*
 * --------------------------------------------------------------
 *
 * Blt_MarkerOp --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 * --------------------------------------------------------------
 */

static Blt_OpSpec markerOps[] =
{
    {"after", 1, (Blt_Operation)RelinkOp, 4, 5, "markerId ?afterMarkerId?",},
    {"before", 1, (Blt_Operation)RelinkOp, 4, 5, "markerId ?beforeMarkerId?",},
    {"cget", 2, (Blt_Operation)CgetOp, 5, 5, "markerId option",},
    {"configure", 2, (Blt_Operation)ConfigureOp, 4, 0,
	"markerId ?markerId?... ?option value?...",},
    {"create", 2, (Blt_Operation)CreateOp, 3, 0,
	"markerType ?option value?...",},
    {"delete", 1, (Blt_Operation)DeleteOp, 3, 0, "?markerId?...",},
    {"exists", 1, (Blt_Operation)ExistsOp, 4, 4, "markerId",},
    {"names", 1, (Blt_Operation)NamesOp, 3, 0, "?pattern?...",},
    {"type", 1, (Blt_Operation)TypeOp, 4, 4, "markerId",},
};
static int numMarkerOps = sizeof(markerOps) / sizeof(Blt_OpSpec);

/*ARGSUSED*/
int
Blt_MarkerOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;
    Tcl_Interp *interp;		/* not used */
    int argc;
    char **argv;
{
    Blt_Operation proc;
    int result;

    proc = Blt_LookupOperation(graphPtr->interp, numMarkerOps, markerOps,
	BLT_OPER_ARG2, argc, argv);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (graphPtr, argc, argv);
    return (result);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_PrintMarkers --
 *
 * -----------------------------------------------------------------
 */
void
Blt_PrintMarkers(graphPtr, under)
    Graph *graphPtr;
    int under;
{
    Blt_ListItem *itemPtr;
    register Marker *markerPtr;

    for (itemPtr = Blt_ListFirstItem(&(graphPtr->markerList)); itemPtr != NULL;
	itemPtr = Blt_ListNextItem(itemPtr)) {
	markerPtr = (Marker *) Blt_ListGetValue(itemPtr);
	if ((markerPtr->printProc == NULL) || (markerPtr->numCoords == 0)) {
	    continue;
	}
	if (markerPtr->drawUnder != under) {
	    continue;
	}
	if (markerPtr->hidden) {
	    continue;
	}
	if (markerPtr->elemName != NULL) {
	    Tcl_HashEntry *hPtr;

	    hPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), markerPtr->elemName);
	    if (hPtr != NULL) {
		Element *elemPtr;

		elemPtr = (Element *)Tcl_GetHashValue(hPtr);
		if (elemPtr->hidden) {
		    continue;
		}
	    }
	}
	Blt_PostScriptAppend(graphPtr, "\n%% Marker \"%s\" is a %s.\n\n",
		markerPtr->name, NameOfMarkerType(markerPtr->type));
	(*markerPtr->printProc) (markerPtr);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_DrawMarkers --
 *
 *	Calls the individual drawing routines (based on marker type)
 *	for each marker in the display list.
 *
 *	A marker will not be drawn if
 *
 *	1) there is an element linked with the marker (whose name is
 *	   elemName) and that element is not currently being displayed.
 *
 *	2) there are no coordinates available for the marker.
 *
 *	3) the smarkere at which we're drawing is different from how
 *	   the marker wants to be displayed (either above/below the
 *	   elements).
 *
 * Results:
 *	None
 *
 * Side Effects:
 *	Markers are drawn into the drawable (pixmap) which will eventually
 *	be displayed in the graph window.
 *
 * -----------------------------------------------------------------
 */
void
Blt_DrawMarkers(graphPtr, under)
    Graph *graphPtr;
    int under;
{
    Blt_ListItem *iPtr;
    Marker *markerPtr;

    for (iPtr = Blt_ListFirstItem(&(graphPtr->markerList)); iPtr != NULL;
	iPtr = Blt_ListNextItem(iPtr)) {
	markerPtr = (Marker *) Blt_ListGetValue(iPtr);

	if ((markerPtr->numCoords == 0) || (markerPtr->drawUnder != under) ||
	    (markerPtr->hidden) || (markerPtr->clipped)) {
	    continue;
	}
	if (markerPtr->elemName != NULL) {
	    Tcl_HashEntry *hPtr;

	    hPtr = Tcl_FindHashEntry(&(graphPtr->elemTable), markerPtr->elemName);
	    if (hPtr != NULL) {
		Element *elemPtr;

		elemPtr = (Element *)Tcl_GetHashValue(hPtr);
		if (elemPtr->hidden) {
		    continue;
		}
	    }
	}
	(*markerPtr->drawProc) (markerPtr);
    }
}


void
Blt_TransformMarkers(graphPtr)
    Graph *graphPtr;
{
    Blt_ListItem *itemPtr;
    Marker *markerPtr;

    for (itemPtr = Blt_ListFirstItem(&(graphPtr->markerList));
	itemPtr != NULL; itemPtr = Blt_ListNextItem(itemPtr)) {
	markerPtr = (Marker *) Blt_ListGetValue(itemPtr);
	if ((markerPtr->numCoords == 0) || (markerPtr->hidden)) {
	    continue;
	}
	if ((graphPtr->flags & COORDS_ALL_PARTS) ||
	    (markerPtr->flags & COORDS_NEEDED)) {
	    (*markerPtr->coordsProc) (markerPtr);
	    markerPtr->flags &= ~COORDS_NEEDED;
	}
    }
}


void
Blt_DestroyMarkers(graphPtr)
    Graph *graphPtr;
{
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch cursor;
    Marker *markerPtr;

    for (hPtr = Tcl_FirstHashEntry(&(graphPtr->markerTable), &cursor);
	hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	markerPtr = (Marker *) Tcl_GetHashValue(hPtr);
	/*
	 * Mark the hash pointer to NULL to prevent them from being
	 * deleted.  It's much more efficient (and safe!) to destroy
	 * the entire table.
	 */
	markerPtr->hashPtr = NULL;
	DestroyMarker(markerPtr);
    }
    Tcl_DeleteHashTable(&(graphPtr->markerTable));
    Blt_ListReset(&(graphPtr->markerList));
}
