
#ifndef lint
static char SccsId[] = "%W%  %G%";
#endif

/* Module:	readreal.c (Read Real)
 * Purpose:	Scale real image data to fit in a short (int*2) buffer
 * Subroutine:	scale_data_r4()			returns: void
 * Subroutine:	scale_data_r8()			returns: void
 * Xlib calls:	none
 * Copyright:	1988 Smithsonian Astrophysical Observatory
 *		You may do anything you like with this file except remove
 *		this copyright.  The Smithsonian Astrophysical Observatory
 *		makes no representations about the suitability of this
 *		software for any purpose.  It is provided "as is" without
 *		express or implied warranty.
 * Modified:	{0} Michael VanHilst	initial version	     31 October 1988
 *		{n} <who> -- <does what> -- <when>
 * Modified:	27 Mar 94 R. W. Wilson put changes suggested by Mike in
 *		scale_data_r[48] so that NaNs in the input file will be
 *		accepted.  Also changed the Linux fixes so that they will
 *		work for any Bytereversed machine.
 * Modified:    12 May 1995 G. Privett applied changes suggested by 
 *              R.W.Wilson to cope with word length on DEC Alphas.
 */

#include "hfiles/image.h"	/* image struct */
#include "hfiles/scale.h"	/* define SCALEWIDTH, etc. */
#

#include <sys/types.h>		/* needed for ntohl (net to host long) */
#ifndef VMS
#include <netinet/in.h>		/* needed for ntohl (net to host long) */
#endif

#if 0
#ifdef LINUX
#include <sys/types.h>          /* for ntohl */
#include <netinet/in.h>
#endif
#endif 0


/*
 * Subroutine:	scale_data_r4
 * Purpose:	scale 32 bit real data into the (short) img array
 */
void scale_data_r4 ( image, imgbuf, databuf, vals, verbose )
     struct imageRec *image;
     short *imgbuf;
     float *databuf;
     int vals;
     int verbose;		/* whether to print explanatory messages */
{
  register float *fbuf, *fbufend;
  float fmin, fmax;


  if (image->byte_swap){

#ifndef ALPHA
    unsigned int *swapbuf;
#else
    unsigned long *swapbuf;
#endif
    int i;

#ifndef ALPHA
    swapbuf = (unsigned int *) databuf;
#else
    swapbuf = (unsigned long *) databuf;
#endif
    for(i=0;i<vals;i++,swapbuf++)
      *swapbuf = ntohl(*swapbuf);
  }

  /* set buf start and end pointers for a pass through fbuf */
  fbuf = databuf;
  fbufend = fbuf + vals;
  /* find the min and the max */
  
  while(isnan(*fbuf)) ++fbuf;

  fmin = fmax = *fbuf;
  /* skip the first val since we just used it */
  while( ++fbuf < fbufend ) {
    if(!isnan(*fbuf)) {

      if( *fbuf < fmin )
        fmin = *fbuf;
      else if( *fbuf > fmax )
        fmax = *fbuf;
    }
  }
  /* make announcement if requested */
  if( verbose ) {
    (void)printf("Data min and max as read: %g, %g\n", fmin, fmax);
    if( image->fimin < image->fimax ) {
      (void)printf("Using given limits: %g, %g\n", image->fimin, image->fimax);
    }
  }
  /* apply preset limits if given */
  if( image->fimin < image->fimax ) {
    fmin = image->fimin;
    fmax = image->fimax;
  }  
  {
    float scale, bias;
    float ftemp;
    register short *sbuf;

    /* set bias to offset values to zero center the range */
    bias = -((fmin + fmax) / (float)2.0);
    /* get the scale factor */
    if( (fmax - fmin) > 0.0 ) {
      scale = (float)SCALEWIDTH / (fmax - fmin);
    } else {
      scale = 1.0;
    }
    /* reset buf for another pass through fbuf */
    /* while loop changed to current form because Sun compiler was */ 
    /* inc'ing buf by 8 bytes */
    fbuf = databuf;
    sbuf = imgbuf;
    /* use min and max to mark out limits */
    fmin = (float)SCALEMIN;
    fmax = (float)SCALEMAX;
    /* scale the picture */
    do {
      if(isnan(*fbuf)) {
	*sbuf++ = -SCALEOFF;
      } else {
	ftemp = (*fbuf + bias) * scale;
	if( ftemp < 0.0 ) {
	  ftemp -= 0.5;
	  if( ftemp < fmin )
	    *sbuf++ = fmin;
	  else
	    *sbuf++ = (short)ftemp;
	} else {
	  ftemp += 0.5;
	  if( ftemp > fmax )
	    *sbuf++ = fmax;
	    else
	      *sbuf++ = (short)ftemp;
	}
      }
    } while( ++fbuf < fbufend );
    image->fiscaled = 1;
    image->fibias = (double)(-bias);
    image->fiscale = 1.0 / (double)scale;
  }
}

/*
 * Subroutine:	scale_data_r8
 * Purpose:	Scale 64 bit real data into the (short) img array
 */
void scale_data_r8 ( image, imgbuf, databuf, vals, verbose )
     struct imageRec *image;
     short *imgbuf;
     double *databuf;
     int vals;
     int verbose;		/* whether to print explanatory messages */
{
  register double *dbuf, *dbufend;
  double dmin, dmax;


  if (image->byte_swap){
#ifndef ALPHA
    unsigned int *swapbuf;
#else
    unsigned long *swapbuf;
#endif
    int i;
    unsigned long temp;

#ifndef ALPHA
    swapbuf = (unsigned int *) databuf;
#else
    swapbuf = (unsigned long *) databuf;
#endif
    for(i=0;i<vals;i++,swapbuf+=2){
      temp = ntohl(*swapbuf);
      *swapbuf = ntohl(*(swapbuf+1));
      *(swapbuf+1)=temp;
    }
  }


  /* set buf start and end pointers for a pass through fbuf */
  dbuf = databuf;
  dbufend = dbuf + vals;
  /* find the min and the max */
  while(isnan(*dbuf)) ++dbuf;

  dmin = dmax = *dbuf;
  /* skip the first val since we just used it */
  while( ++dbuf < dbufend ) {
   if( !isnan(*dbuf)) {
      if( *dbuf < dmin )
        dmin = *dbuf;
      else if( *dbuf > dmax )
        dmax = *dbuf;
    }
  }
  /* make announcement if requested */
  if( verbose ) {
    (void)printf("Data min and max as read: %g, %g\n", dmin, dmax);
    if( image->fimin < image->fimax ) {
      (void)printf("Using given limits: %g, %g\n", image->fimin, image->fimax);
    }
  }
  /* apply preset limits if given */
  if( image->fimin < image->fimax ) {
    dmin = image->fimin;
    dmax = image->fimax;
  }  
  {
    double scale, bias;
    double dtemp;
    register short *sbuf;

    /* set bias to offset values to zero center the range */
    bias = -((dmin + dmax) / (double)2.0);
    /* get the scale factor */
    if( (dmax - dmin) > 0.0 ) {
      scale = (double)SCALEWIDTH / (dmax - dmin);
    } else {
      scale = 1.0;
    }
    /* reset buf for another pass through fbuf */
    dbuf = databuf;
    sbuf = imgbuf;
    /* use min and max to mark out limits */
    dmin = (double)SCALEMIN;
    dmax = (double)SCALEMAX;
    /* scale the picture */
    do {
      if(isnan(*dbuf)) {
	*dbuf++ = -SCALEOFF;
      } else {
	dtemp = (*dbuf + bias) * scale;
	if( dtemp < 0.0 ) {
	  dtemp -= 0.5;
	  if( dtemp < dmin )
	    *sbuf++ = dmin;
	  else
	    *sbuf++ = (short)dtemp;
	} else {
	  dtemp += 0.5;
	  if( dtemp > dmax )
	    *sbuf++ = dmax;
	else
	  *sbuf++ = (short)dtemp;
	}
      }
    } while( ++dbuf < dbufend );
    image->fiscaled = 1;
    image->fibias = -bias;
    image->fiscale = 1.0 / scale;
  }
}
