/*
 * mknbi.c  -  MaKe NetBoot Image for DOS
 *
 * Version 1.1 <Sat Nov  9 22:16:53 MET 1996>
 *
 * Copyright (C) 1996 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../common.h"
#include "mknbi.h"

#ifndef _MKNBI_H_DOS_
#error Included wrong header file
#endif


char *progname;
char outname[MAXPATHLEN];	/* Name of output file */
char rdname[MAXPATHLEN];	/* Name of ramdisk image file */
int outfile;			/* File handle for output file */
int rdimage;			/* File handle for ramdisk image */
int rdsize;			/* Size of ramdisk in kB */
int verbose = 0;		/* Verbosity level; currently just 0, 1 or 2 */

int cur_rec_num = -1;		/* Number of current load record */
struct load_header header;	/* Load header */
struct load_record *cur_rec;	/* Pointer to current load record */




/*
 * Read one sector from ram disk image file
 */
static void readsect(unsigned char *buf)
{
  int i;

  if ((i = read(rdimage, buf, SECTSIZE)) < 0) {
	perror(rdname);
	exit(EXIT_FAILURE);
  }
  if (i != SECTSIZE) {
	fprintf(stderr, "%s: Unexpected end of ram disk image file\n",
								progname);
	exit(EXIT_FAILURE);
  }
}



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(int recnum, char *src, int size)
{
  unsigned long l;
  int isize;
  char *buf;

  if (cur_rec_num != recnum) {
    fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
	    progname);
    exit(EXIT_FAILURE);
  }
  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  if ((buf = malloc(isize)) == NULL) {
	perror(progname);
	exit(EXIT_FAILURE);
  }
  memset(buf, isize, 0);
  memcpy(buf, src, size);
  if (write(outfile, buf, isize) != isize) {
	perror(outname);
	exit(EXIT_FAILURE);
  }
  free(buf);
  l = get_long(cur_rec->ilength) + isize;
  cur_rec->ilength.low  = htot(low_word(l));
  cur_rec->ilength.high = htot(high_word(l));
  l = get_long(cur_rec->mlength) + isize;
  cur_rec->mlength.low  = htot(low_word(l));
  cur_rec->mlength.high = htot(high_word(l));
}



/*
 * Initialize a load record
 */
static void initrec(int recnum, int segment, int flags, int vendor_size)
{
  if (++cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
		progname);
	exit(EXIT_FAILURE);
  }

  if (cur_rec_num > 0)
	(unsigned char *)cur_rec += ((cur_rec->rlength << 2) & 0x3c) +
					((cur_rec->rlength >> 2) & 0x3c);
  cur_rec->rlength      = (sizeof(struct load_record) -
                           sizeof(union vendor_data)) >> 2;
  cur_rec->rlength     |= ((vendor_size + 3) & 0x3c) << 2;
  cur_rec->rtag1        = recnum + VENDOR_OFF;
  cur_rec->rflags       = flags;
  cur_rec->address.low  = htot(low_word((unsigned long) segment << 4));
  cur_rec->address.high = htot(high_word((unsigned long) segment << 4));
}



/*
 * Write a simple partition table into the boot image file
 */
static void putpart(struct boot_record *boot, int use16)
{
  struct partition *part;
  unsigned char part_buf[SECTSIZE];
  unsigned int sector, cylinder;
  int i;

  /* Setup the partition table */
  memset(part_buf, 0, SECTSIZE);
  *((__u16 *)&(part_buf[BOOT_SIG_OFF])) = BOOT_SIGNATURE;
  part = (struct partition *)&(part_buf[BOOT_PART_OFF]);
  part->status = BOOT_ACTIVE;
  part->type = use16 ? BOOT_TYPE_16 : BOOT_TYPE_12;
  part->number_sectors.low  = boot->sect_num;
  part->number_sectors.high = 0;
  part->first_abs_sector     = boot->hidden_num;

  /* Determine starting sector of partition */
  sector = get_long(boot->hidden_num) % ttoh(boot->sect_per_track);
  cylinder = get_long(boot->hidden_num) / ttoh(boot->sect_per_track);
  part->first_sector = sector + 1;
  part->first_track = cylinder / ttoh(boot->head_num);
  part->first_head = cylinder % ttoh(boot->head_num);

  /* Determine last sector of partition */
  sector = (get_long(boot->hidden_num) + ttoh(boot->sect_num) - 1) %
							ttoh(boot->sect_per_track);
  cylinder = (get_long(boot->hidden_num) + ttoh(boot->sect_num) - 1) /
							ttoh(boot->sect_per_track);
  part->last_sector = sector + 1;
  part->last_track = cylinder / ttoh(boot->head_num);
  part->last_head = cylinder % ttoh(boot->head_num);

  /* Write partition table */
  putrec(RAMDISKNUM, part_buf, SECTSIZE);

  /* Insert all hidden sectors */
  memset(part_buf, 0, SECTSIZE);
  for (i = 0; i < (get_long(boot->hidden_num) - 1); i++)
	putrec(RAMDISKNUM, part_buf, SECTSIZE);
}



/*
 * Read the FAT from the ram disk image and convert it into 16 bit format.
 */
static unsigned char *readfat(int oldnum, int newnum, int copynum, int use16)
{
  unsigned char *oldbuf, *newbuf, *cp;
  unsigned long entry;
  int entries, i;

  /*
   * Read the old FATs, and check that the first two FATs are identical. Also
   * ensure that all copies in the buffer are cleared. If we don't have to
   * create a 16 bit FAT table, we can then just return.
   */
  if ((i = (oldnum + 1) * copynum) < newnum)
	i = newnum;
  if ((oldbuf = malloc(i * SECTSIZE)) == NULL) {
	fprintf(stderr, "%s: Not enough memory for old FAT\n", progname);
	exit(EXIT_FAILURE);
  }
  memset(oldbuf, 0, i * SECTSIZE);

  cp = oldbuf;
  for (i = 0; i < (oldnum * copynum); i++) {
	readsect(cp);
	cp += SECTSIZE;
  }
  cp = oldbuf + oldnum * SECTSIZE;
  if (copynum > 1) {
	if (memcmp(oldbuf, cp, oldnum * SECTSIZE)) {
		fprintf(stderr, "%s: First two FATs in ram disk image differ\n",
								progname);
		exit(EXIT_FAILURE);
	}
	memset(cp, 0, oldnum * (copynum - 1) * SECTSIZE);
  }
  if (!use16)
	return(oldbuf);

  /*
   * Copy the old FAT into the new buffer while adjusting each entry
   * to 16 bit entries.
   */
  if ((newbuf = malloc(newnum * SECTSIZE)) == NULL) {
	fprintf(stderr, "%s: Not enough memory for new FAT\n", progname);
	exit(EXIT_FAILURE);
  }
  memset(newbuf, 0, newnum * SECTSIZE);
  entries = (oldnum * SECTSIZE * 8) / 12;
  if ((((entries * 2) / SECTSIZE) + 1) > newnum) {
	fprintf(stderr, "%s: Internal error; wrong new FAT size\n", progname);
	exit(EXIT_FAILURE);
  }
  cp = newbuf;
  for (i = 0; i < entries; i++) {
	entry  = ttoh(*((__u16 *)(oldbuf + (i * 12) / 8)));
	entry |= ttoh(*((__u16 *)(oldbuf + (i * 12) / 8 + 2))) << 16;
	entry >>= (i * 12) % 8;
	entry &= 0x0fff;
	if (entry >= 0x0ff0)
		entry |= 0xf000;
	*((__u16 *)cp)++ = htot((__u16)entry);
  }

  free(oldbuf);
  return(newbuf);
}



/*
 * Determine number of last used cluster
 */
static unsigned int getlast(unsigned char *fatbuf, int sectnum, int use16)
{
  __u8 *cp;
  int last;

  cp = (__u8 *)fatbuf + (sectnum * SECTSIZE) - 1;
  while (cp > (__u8 *)fatbuf && !*cp)
	cp--;
  if ((last = cp - (__u8 *)fatbuf) < 2)
	return(0);
  if (use16)
	last = ((last * 8) + 15) / 16;
  else
	last = ((last * 8) + 11) / 12;
  if (last > 2)
	last -= 2;
  else
	last = 0;
  return(last);
}



/*
 * Process ramdisk image file
 */
static void do_ramdisk(void)
{
  unsigned long sectnum;		/* number of sectors on ram disk */
  unsigned long clusternum;		/* number of cluster in partition */
  unsigned char *fatbuf;		/* buffer for FAT */
  struct disk_geometry geometry;	/* ramdisk geometry */
  struct boot_record old_boot;		/* old boot record */
  struct boot_record *new_boot;		/* new boot record */
  static unsigned char boot_buf[SECTSIZE];
  static unsigned char copyrec_buf[SECTSIZE];
  unsigned long l;
  int use_fat16 = 0;			/* use 16 bit fat */
  int i;

  /*
   * Read the boot record of the ram disk image
   */
  readsect(boot_buf);
  if (*((__u16 *)&(((__u8 *)boot_buf)[BOOT_SIG_OFF])) != htot(BOOT_SIGNATURE)) {
	fprintf(stderr, "%s: Ram disk image has no boot signature\n", progname);
	exit(EXIT_FAILURE);
  }
  old_boot = *((struct boot_record *)boot_buf);

  /*
   * Check that the boot record of the ram disk image is correct
   */
  if (strncmp(old_boot.oem_name, BOOT_OEM_NAME, sizeof(old_boot.oem_name))) {
	fprintf(stderr, "%s: Ram disk image is no MS-DOS disk\n", progname);
	exit(EXIT_FAILURE);
  }
  if (strncmp(old_boot.fat_name, BOOT_FAT12_NAME, sizeof(old_boot.fat_name))) {
	fprintf(stderr, "%s: Ram disk image has invalid filesystem\n", progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.bytes_per_sect != htot(SECTSIZE)) {
	fprintf(stderr, "%s: Ram disk image has wrong sector size\n", progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.hidden_num.low != 0 || old_boot.hidden_num.high != 0) {
	fprintf(stderr, "%s: Ram disk image has hidden sectors\n", progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.sect_num_32.low != 0 || old_boot.sect_num_32.high != 0) {
	fprintf(stderr, "%s: Ram disk image has invalid number of sectors\n",
								progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.fat_num != 2) {
	fprintf(stderr, "%s: Ram disk image has invalid number of FATs\n",
								progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.reserved_sect != htot(1)) {
	fprintf(stderr, "%s: Ram disk image has too many reserved sectors\n",
								progname);
	exit(EXIT_FAILURE);
  }
  if (old_boot.head_num != htot(2)) {
	fprintf(stderr, "%s: Ram disk image has to be double sided\n",
								progname);
	exit(EXIT_FAILURE);
  }
  sectnum = ttoh(old_boot.sect_num);
  if (sectnum != 720  && sectnum != 1440 && sectnum != 2400 && sectnum != 2880) {
	fprintf(stderr, "%s: Invalid ram disk image size (%d sectors)\n",
								progname, sectnum);
	exit(EXIT_FAILURE);
  }

  /*
   * Now create the new boot record. We have to change a couple of values
   * not only to simulate a hard disk boot, but also to take care for
   * the new 16-bit FAT.
   */
  new_boot = (struct boot_record *)boot_buf;
  new_boot->media_id = BOOT_MEDIA_ID;
  new_boot->boot_id = BOOT_DEVICE;
  new_boot->hidden_num.low = old_boot.sect_per_track;
  /* Determine new size of ramdisk image */
  sectnum = ttoh(old_boot.sect_num);
  if (rdsize > 0) {
	if (rdsize < (sectnum / 2)) {
		fprintf(stderr, "%s: Specified ram disk size is smaller "
		                "than image\n", progname);
		exit(EXIT_FAILURE);
	}
	sectnum = rdsize * 2;
  }
  /* Determine number of clusters in data area */
  sectnum -= ttoh(old_boot.sect_per_fat) * old_boot.fat_num;
  sectnum -= ttoh(old_boot.reserved_sect);
  sectnum -= (ttoh(old_boot.dir_num) * 32) / SECTSIZE;
  clusternum = sectnum / old_boot.sect_per_cluster;
  if ((sectnum % old_boot.sect_per_cluster) > 0) {
	clusternum++;
	sectnum = clusternum * old_boot.sect_per_cluster;
  }
  /* Compute new size of FAT */
  if (clusternum > 0x0ff6) {
	use_fat16 = 1;
	new_boot->sect_per_fat = htot((((clusternum + 2) * 2) / SECTSIZE) + 1);
	strncpy(new_boot->fat_name, BOOT_FAT16_NAME, sizeof(new_boot->fat_name));
  } else {
	use_fat16 = 0;
	new_boot->sect_per_fat = htot((((clusternum + 2) * 12) / (8 * SECTSIZE)) + 1);
	strncpy(new_boot->fat_name, BOOT_FAT12_NAME, sizeof(new_boot->fat_name));
  }
  /* Compute new total size of partition */
  sectnum += (ttoh(new_boot->dir_num) * 32) / SECTSIZE;
  sectnum += ttoh(new_boot->reserved_sect);
  sectnum += ttoh(new_boot->sect_per_fat) * new_boot->fat_num;
  new_boot->sect_num = htot(sectnum);

  /*
   * Generate the load record which contains the disk geometry, and
   * then write the partition table, all hidden sectors and the new
   * boot record.
   */
  initrec(RAMDISKNUM, 0, FLAG_B1 | FLAG_EOF, sizeof(struct disk_geometry));
  geometry.cylinders = htot(((ttoh(new_boot->sect_num) +
				get_long(new_boot->hidden_num)) /
				(ttoh(new_boot->sect_per_track) * 2)) + 1);
  geometry.sect_per_track = new_boot->sect_per_track;
  geometry.num_sectors = htot(ttoh(geometry.cylinders) *
				ttoh(geometry.sect_per_track) * 2);
  cur_rec->vendor_data.geometry = geometry;
  putpart(new_boot, use_fat16);
  putrec(RAMDISKNUM, boot_buf, SECTSIZE);

  /*
   * Read the FAT from the ram disk image, convert it into 16 bit format,
   * and write it back into the output image. Also find the last used
   * sector in the ram disk image.
   */
  fatbuf = readfat(ttoh(old_boot.sect_per_fat), ttoh(new_boot->sect_per_fat),
						old_boot.fat_num, use_fat16);
  *((__u8 *)fatbuf) = BOOT_MEDIA_ID;
  for (i = 0; i < new_boot->fat_num; i++)
	putrec(RAMDISKNUM, fatbuf, ttoh(new_boot->sect_per_fat) * SECTSIZE);

  sectnum = getlast(fatbuf, ttoh(new_boot->sect_per_fat), use_fat16);
  if (sectnum > clusternum)
	sectnum = clusternum;
  sectnum *= new_boot->sect_per_cluster;
  sectnum += (ttoh(new_boot->dir_num) * 32) / SECTSIZE;

  /*
   * Write the remainder of the ramdisk image.
   */
  while (sectnum > 0) {
	readsect(copyrec_buf);
	putrec(RAMDISKNUM, copyrec_buf, SECTSIZE);
	sectnum--;
  }
  if (get_long(cur_rec->ilength) > (RAMDISKSIZE * 1024L)) {
	fprintf(stderr, "%s: Ram disk image too large\n", progname);
	exit(EXIT_FAILURE);
  }
  l = ttoh(geometry.num_sectors) * SECTSIZE;
  cur_rec->mlength.low  = htot(low_word(l));
  cur_rec->mlength.high = htot(high_word(l));
  cur_rec->address.low  = htot(low_word(l + SECTSIZE));
  cur_rec->address.high = htot(high_word(l + SECTSIZE));
}


/*
 * Dump the load record information to stderr
 */
static void dump_header(struct load_header *lh)
{
  static char *s_tags[] = { [LOADERNUM] "primary boot loader",
			    [RAMDISKNUM]"ramdisk image"};
  static char *s_flags[]= { "absolute address", "after previous segment",
			    "at end of memory", "before previos segment"};
  struct load_record *lr;
  int    i, num = 0;

  fprintf(stderr,"\n"
	  "Load record information:\n"
	  "  Magic number:     0x%08lX\n"
	  "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	  "  Flags:            0x%08lX\n"
	  "  Location address: %04X:%04X\n"
	  "  Execute address:  %04X:%04X\n"
	  "  Vendor data:      %s\n"
	  "\n",
	  get_long(lh->magic),
	  (lh->hlength << 2) & 0x3c,
	  (lh->hlength >> 2) & 0x3c,
	  (unsigned long)lh->hflags1 +
		((unsigned long)lh->hflags2 << 8) +
		((unsigned long)lh->hflags3 << 16),
	  ttoh(lh->locn.segment), ttoh(lh->locn.offset),
	  ttoh(lh->execute.segment), ttoh(lh->execute.offset),
	  lh->dummy);

  i  = ((lh->hlength >> 2) & 0x3c) + ((lh->hlength << 2) & 0x3c);
  lr = (struct load_record *)&(((__u8 *)lh)[i]);

  for (;;) {
    fprintf(stderr,
	    "Record #%d:\n"
	    "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	    "  Vendor tag:       0x%02X (%s)\n"
	    "  Reserved flags:   0x%02X\n"
	    "  Flags:            0x%02X (%s%s)\n"
	    "  Load address:     0x%08lX%s\n"
	    "  Image length:     0x%08lX bytes\n"
	    "  Memory length:    0x%08lX bytes\n",
	    ++num,
	    (lr->rlength << 2) & 0x3c,
	    (lr->rlength >> 2) & 0x3c,
	    (int)lr->rtag1,
	    lr->rtag1 < 16 || lr->rtag1-16 >= NUM_RECORDS ? "unknown" : s_tags[lr->rtag1-16],
	    (int)lr->rtag2,
	    (int)lr->rflags, s_flags[lr->rflags & 0x03],
	    lr->rflags & FLAG_EOF ? ", last record" : "",
	    get_long(lr->address),
	    get_long(lr->address) >= 0x100000L &&
	    (lr->rflags & 0x03) == 0? " (high memory)" : "",
	    get_long(lr->ilength),
	    get_long(lr->mlength));
    if (lr->rtag1-16 == RAMDISKNUM) {
      int cyls = ttoh(lr->vendor_data.geometry.cylinders);
      int sects = ttoh(lr->vendor_data.geometry.num_sectors);
      int spt = ttoh(lr->vendor_data.geometry.sect_per_track);

      fprintf(stderr,
              "  Vendor data:      %d cylinders; %d heads; %d sectors\n"
              "\n",
              cyls,
              (sects + (cyls * spt) - 1) / (cyls * spt),
              spt);
    } else {
      fprintf(stderr,
              "  Vendor data:      %s\n"
              "\n",
              lr->rlength & 0xf0 ? "unknown" : "none");
    }

    if (lr->rflags & FLAG_EOF)
      break;

    i  = ((lr->rlength >> 2) & 0x3c) + ((lr->rlength << 2) & 0x3c);
    lr = (struct load_record *)&(((__u8 *)lr)[i]);
  }
}


/*
 * Print the usage
 */
static void usage(void)
{
  fprintf(stderr,
	"Usage: %s [-s <ramdisk size>] [-r] <ramdisk image> [[-o] <outfile>] [-x]\n"
	"       %s [-h]\n"
	"       %s [-v]\n",
	  progname, progname, progname);
  exit(EXIT_FAILURE);
}


/*
 * Main program
 */
void main(int argc, char **argv)
{
  char *cp;
  int vendor_size, i;

  /* Determine my own name for error output */
  if ((cp = strrchr(argv[0], '/')) == NULL)
	progname = argv[0];
  else
	progname = ++cp;

  /* Initialize option argments */
  strcpy(outname, "");
  strcpy(rdname, "");
  rdsize = 0;

  /* Parse options */
  opterr = 0;
  while ((i = getopt(argc, argv, "ho:r:s:vx")) != EOF)
	switch (i) {
		case 'o': strncpy(outname, optarg, MAXPATHLEN-1);
		          outname[MAXPATHLEN-1] = '\0';
		          break;
		case 'r': strncpy(rdname, optarg, MAXPATHLEN-1);
		          rdname[MAXPATHLEN-1] = '\0';
		          break;
		case 's': if ((i = atoi(optarg)) > RAMDISKSIZE) {
				fprintf(stderr, "%s: Ramdisk size must "
				                "be < %d kB\n",
						progname, RAMDISKSIZE);
				exit(EXIT_FAILURE);
		          }
		          rdsize = i;
		          break;
		case 'v': fprintf(stderr, VERSION"\n");
		          exit(EXIT_FAILURE);
		case 'x': verbose++;
		          break;
		case 'h': /* fall thru */
		default:  usage();
	}

  /* Parse additional arguments */
  if (optind < argc) {
	strncpy(rdname, argv[optind++], MAXPATHLEN-1);
	rdname[MAXPATHLEN-1] = '\0';
  }
  if (optind < argc) {
	strncpy(outname, argv[optind++], MAXPATHLEN-1);
	outname[MAXPATHLEN-1] = '\0';
  }
  if (optind != argc || !*rdname)
	usage();

  /* Open the input and output files */
  if (strlen(outname) == 0 || !strcmp(outname, "-")) {
	outfile = STDOUT_FILENO;
	strcpy(outname, "stdout");
  } else if ((outfile = creat(outname, 0664)) < 0) {
	perror(outname);
	exit(EXIT_FAILURE);
  }
  if ((rdimage = open(rdname, O_RDONLY)) < 0) {
	perror(rdname);
	exit(EXIT_FAILURE);
  }
  if (verbose > 0) {
	fprintf(stderr, "Ramdisk filename = %s\n", rdname);
	fprintf(stderr, "Output file name = %s\n", outname);
  }

  /* Initialize the boot header */
  vendor_size = (sizeof(VENDOR_ID) / sizeof(__u32) + 1) * sizeof(__u32);
  memset(&header, sizeof(header), 0);
  header.magic.low       = htot(low_word(HEADER_MAGIC));
  header.magic.high      = htot(high_word(HEADER_MAGIC));
  header.hlength         = (__u8)(((int)&header.dummy - (int)&header)
                           / sizeof(__u32)) & 0x0f;
  header.hlength        |= (__u8)((vendor_size/sizeof(__u32)) << 4) & 0xf0;
  header.locn.segment    = htot(HEADERSEG);
  header.locn.offset     = 0;
  header.execute.segment = htot(LOADERSEG);
  header.execute.offset  = 0;
  header.bootsig         = htot(BOOT_SIGNATURE);
  memcpy(&header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit(EXIT_FAILURE);
  }

  /* Initialize pointer to first load record */
  cur_rec = (struct load_record *)&(header.dummy[vendor_size]);

  /* Process the boot loader record */
  if (first_data_size > LOADERLSIZE) {
	fprintf(stderr, "%s: Boot loader too large\n", progname);
	exit(EXIT_FAILURE);
  }
  initrec(LOADERNUM, LOADERSEG, 0, 0);
  putrec(LOADERNUM, (char *) &first_data, first_data_size);
  cur_rec->mlength.low  = htot(low_word(LOADERMSIZE));
  cur_rec->mlength.high = htot(high_word(LOADERMSIZE));

  /* Process the ramdisk image */
  do_ramdisk();

  /* After writing out all these stuff, finally update the boot header */
  if (lseek(outfile, 0, 0) != 0) {
	perror(outname);
	exit(EXIT_FAILURE);
  }
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit(EXIT_FAILURE);
  }

  /* If user asked for detailed output, parse the header and output all of */
  /* the load record information */
  if (verbose > 1)
	dump_header(&header);

  exit(EXIT_SUCCESS);
}
