/* cfengine for GNU
 
        Copyright (C) 1995
        Free Software Foundation, Inc.
 
   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/
 

/*******************************************************************/
/*                                                                 */
/*  Parse Zone for cfengine                                        */
/*                                                                 */
/*  This is wide screen entertainment. Resize                      */
/*  your window before viewing!                                    */
/*                                                                 */
/*  The routines here are called by the lexer and the yacc parser  */
/*                                                                 */
/*******************************************************************/

 /*
    The parser uses the classification of strings into id's items
    paths, varpaths etc. Each complete action is gradually assembled
    by setting flags based on what action, class etc is being
    parsed. e.g. HAVEPATH, FROM_LINK etc.. these are typed as
    "flag" in the variables file. When all the necessary critera
    are met, or the beginning of a new definition is found
    the action gets completed and installed in the actions lists.

  */

#define INET

#include <stdio.h>
#include "cf.defs.h"
#include "cf.extern.h"

/*******************************************************************/

SetAction (action)

enum actions action;

{
InstallPending(ACTION);   /* Flush any existing actions */

if (DEBUG || D1)
   {
   printf ("\nBEGIN NEW ACTION %s\n",ACTIONTEXT[action]);
   }

ACTION = action;
strcpy(ACTIONBUFF,ACTIONTEXT[action]);

if (ACTION == files || ACTION == makepath || ACTION == tidy || ACTION == disable)
   {
   InitFileAction();
   }

strcpy(CLASSBUFF,"any");    /* default class */
CLASS=soft;
}

/*******************************************************************/

HandleId(id)

char *id;

{
if (DEBUG || D1)
   {
   printf("HandleId(%s)\n",id);
   }

if (ACTION == control)
   {
   switch(ScanVariable(id))
      {
      case fac:    
      case sit:     CLASS = faculty;
                    return;

      case sysad:   CLASS = sysadm;
                    return;

      case dom:     CLASS = domain;
                    return;

      case timez:   CLASS = timezon;
                    return;

      case netmsk:  CLASS = netmask;
                    return;

      case nfstp:   CLASS = nfstype;
                    return;

      case ssize:   CLASS = sensiblesize;
                    return;

      case scount:  CLASS = sensiblecount; 
                    return;

      case esize:   CLASS = editfilesize;
                    return;

      case actseq:  CLASS = actionsequence;
                    return;

      case acess:   CLASS = accesss;
                    return;

      case mpat:    CLASS = mountpath;
                    return;

      case hpat:    CLASS = homepat;
                    return;

      case adclass: CLASS = addclasses;
                    return;

      default:      RecordMacroId(id);
                    CLASS = soft;
                    return;
      }
   }


if (ACTION == groups)
   { int count = 1;
     char *cid = id;

while (*cid != '\0')
   {
   if (*cid++ == '.')
      {
      count++;
      }
   }

   if (strcmp(id,"any") == 0)
      {
      yyerror("Reserved class <any>");
      }

   if (count > 1)                              /* compound class */
      {
      yyerror("Group with compound identifier");
      FatalError("Dots [.] not allowed in group identifiers");
      }

   if (IsHardClass(id))
      {
      yyerror("Reserved class name (choose a different name)");
      }

   strcpy(GROUPBUFF,id);
   }

strcpy(CLASSBUFF,"any");    /* default class */
CLASS=soft;
}

/*******************************************************************/

HandleClass (id)

char *id;

{ int members;

InstallPending(ACTION);

if (DEBUG || D1)
   {
   printf("HandleClass(%s)\n",id);
   }

if ((members = CompoundId(id)) > 1)             /* Parse compound id */
   {
   if (DEBUG || D1)
      {
      printf("Compound identifier-class = (%s) with %d members\n",id,members);
      }
   }
else
   {
   if (DEBUG || D1)
      {
      printf("simple identifier-class = (%s)\n",CLASSTEXT[CLASS]);
      }
   }
}

/*******************************************************************/

HandleItem (item)

char *item;

{
if (DEBUG || D1)
   {
   printf("HandleItem(%s)\n",item);
   }

if (item[0] == '+')                               /* Lookup in NIS */
   {
   item++;

   if (item[0] == '@')                               /* Irrelevant */
      {
      item++;
      }

   if (DEBUG || D1)
      {
      printf ("Netgroup item, lookup NIS group (%s)\n",item);
      }

   strcpy(CURRENTITEM,item);

   if (ACTION == groups)
      {
      HandleGroupItem(item,netgroup);
      }
   else
      {
      yyerror("Netgroup reference outside group: action or illegal octal number");
      FatalError("Netgroups may only define internal groups or illegal octal file action.");
      }
   }
else if (item[0] == '-') 
   {
   item++;

   if (item[0] == '@')                               /* Irrelevant */
      {
      item++;

      if (ACTION == groups)
         {
         HandleGroupItem(item,groupdeletion);
         }
      else
         {
         yyerror("Netgroup reference outside group: action or illegal octal number");
         FatalError("Netgroups may only define internal groups or illegal octal file action.");
         }
      }
   else
      {
      if (ACTION == groups)
         {
         HandleGroupItem(item,deletion);
         }
      else
         {
         yyerror("Illegal deletion sign or octal number");
         FatalError("The deletion operator may only be in the groups: action");
         }
      }


   }
else
   {
   if (DEBUG || D1) 
      {
      printf("simple item = (%s)\n",item);
      }
  
   /* CLASS set by CompoundId via HandleId */

   switch(ACTION)
      {
      case control:     InstallLocalInfo(CLASS,item);
                        break;
      case groups:      HandleGroupItem(item,simple);
                        break;
      case processes:   yyerror("Absolute path required for a process");
                        break;
      case resolve:     AppendNameServer(item);
                        break;
      case files:       HandleFileItem(item);
                        break;
      case tidy:        strcpy(CURRENTITEM,item);
                        break;
      case homeservers: InstallHomeserverItem(item);
                        break;
      case binservers:  InstallBinserverItem(item);
                        break;
      case mailserver:  yyerror("Give whole rpc path to mailserver");
                        break;
      case required:    yyerror("Required filesystem must be an absolute path");
                        FatalError("Fatal error");
                        break;
      case mountables:  yyerror("Mountables should be specified by an absolute pathname");
                        break;
      case links:       yyerror("Links requires path or varitem");
                        break;
      case import:      AppendImport(item);
                        break;
      case shellcommands:
                        yyerror("Script / command must have an absolute pathname");
                        FatalError("Fatal error");
                        break;
      case makepath:    yyerror("makepath needs an abolute pathname");
                        FatalError("Fatal Error");
      case disable:     yyerror("disable needs an absolute path name");
                        FatalError("Fatal Error");
      case broadcast:   InstallBroadcastItem(item);
                        break;
      case defaultroute:InstallDefaultRouteItem(item);
                        break;
      case misc_mounts: if (MOUNT_FROM && MOUNT_ONTO && (strcmp(item,"rw") == 0 || strcmp(item,"ro") == 0))
                           {
                           AppendMiscMount(MOUNTFROM,MOUNTONTO,item);
                           CURRENTITEM[0] = NULL;
                           MOUNT_FROM = false;
                           MOUNT_ONTO = false;
                           }
                        else
                           {
                           yyerror("misc_mounts: host:/frompath /mounton_path [ro|rw]\n");
                           }
                        break;

      case unmount:     yyerror("Umount must be in the format machine:directory");
                        break;

      case editfiles:   /* action recorded in CURRENTITEM, installed from qstring in lexer */
                        strcpy(CURRENTITEM,item);
                        break;

      case ignore:      AppendIgnore(item);
                        break;

      default:          FatalError("Internal software failure (***Unknown action***)\n");
      }
   }
}

/***************************************************************************/

HandlePath (path)

char *path;

{
InstallPending(ACTION);           /* Flush any existing actions */
InitFileAction();                 /* Clear data for files/dirs  */

if (DEBUG || D1)
   {
   printf("path = (%s)\n",path);
   }

strcpy(CURRENTPATH,path);               /* Yes this must be here */

ACTIONPENDING = true;                 /* we're parsing an action */

if (ACTION_IS_LINK)                 /* to link (after the arrow) */
   {
   InstallLinkItem(LINKFROM,CURRENTPATH);
   ACTION_IS_LINK = false;
   LINKFROM[0] = NULL;
   }
else if (ACTION_IS_LINKCHILDREN)
   {
   InstallLinkChildrenItem(LINKFROM,CURRENTPATH);
   ACTION_IS_LINKCHILDREN = false;
   LINKFROM[0] = NULL;
   }
else
   {
   switch (ACTION)
      {
      case control:  if (CLASS == mountpath)
                          {
                          SetMountPath(path);
                          }

                       if (CLASS == homepat)
                          {
                          yyerror("Path relative to mountpath required");
                          FatalError("Absolute path was specified\n");
                          }

                      if (CLASS == soft)
                         {
                         AddMacroValue(CURRENTITEM,path);
                         }

                       break;
      case import:     AppendImport(path);
                       break;
      case links:      /* from link */
                       break;
      case required:   InstallRequiredPath(path);
                       break;
      case processes:
                       AppendProcess(path);
                       break;
      case shellcommands:
                       AppendScript(path);
                       break;
      case mountables: AppendMountable(path);
                       break;
      case mailserver: InstallMailserverPath(path);
                       break;
      case tidy:       strcpy(CURRENTITEM,path);
                       break;
      case disable:    strcpy(CURRENTPATH,path);
                       ACTIONPENDING = true;
                       break;
      case makepath:   strcpy(CURRENTPATH,path);
                       break;
      case ignore:     AppendIgnore(path);
                       break;

      case misc_mounts:if (! MOUNT_FROM)
                          {
                          MOUNT_FROM = true;
                          strcpy(MOUNTFROM,CURRENTPATH);
                          }
                       else
                          {
                          if (MOUNT_ONTO)
			    {
                            yyerror ("Path not expected");
                            FatalError("misc_mount: syntax error");
                            }
                          MOUNT_ONTO = true;
                          strcpy(MOUNTONTO,CURRENTPATH);
                          }
                       break;

      case unmount:    AppendUmount(path);
                       break;

      case files:      
                       break;

      case editfiles:  /* file recorded in CURRENTPATH */
                       break;

      default:         printf ("HandlePath() DEFAULT\n");
                       printf ("Fatal action is %s\n",ACTIONTEXT[ACTION]);
                       FatalError("Internal software error in HandlePath()");
      }
   }
}

/*******************************************************************/

HandleVarpath(varpath)         

  /* Expand <binserver> and <fac> etc. Note that the right hand  */
  /* side of links etc. gets expanded at runtime. <binserver> is */
  /* only legal on the right hand side.                          */

char *varpath;

{
InstallPending(ACTION);   /* Flush any existing actions */
InitFileAction();

if (DEBUG || D1)
   {
   printf("HandleVarpath(%s)\n",varpath);
   }

if (IsWildCard(varpath) && ! (ACTION == files || ACTION == tidy))
   {
   yyerror("Wildcards can only be used in pathnames in tidy: and files:");
   }

strcpy(CURRENTPATH,varpath);

ACTIONPENDING = true;
 
if (ACTION_IS_LINK)
   {
   InstallLinkItem(LINKFROM,varpath);
   ACTION_IS_LINK = false;
   LINKFROM[0] = NULL;
   }
else if (ACTION_IS_LINKCHILDREN)
   {
   InstallLinkChildrenItem(LINKFROM,varpath);
   ACTION_IS_LINKCHILDREN = false;
   LINKFROM[0] = NULL;
   }
else 
   {
   switch (ACTION)
      {
      case tidy:       strcpy(CURRENTITEM,varpath);
                       break;
      case required:   InstallRequiredPath(varpath);
                       break;
      case makepath:   strcpy(CURRENTPATH,varpath);
                       break;
      case control:    if (CLASS == mountpath)
                          {
                          SetMountPath(varpath);
                          }

                        if (CLASS == homepat)
                           {
                           yyerror("Home-pattern should be relative to mount-path, not absolute");
                           }

                        if (CLASS == soft)
                           {
                           yyerror("Nested macros not permitted");
                           }
                        break;
      case files:  
                        break;
      }
   }
}

/*******************************************************************/

HandleWildcard(wildcard)

char *wildcard;

{
if (DEBUG || D1)
   {
   printf("wildcard = (%s)\n",wildcard);
   }

ACTIONPENDING = true;

if (ACTION == ignore)
   {
   AppendIgnore(wildcard);
   }
else if (ACTION == control && CLASS == homepat)
   {
   if (*wildcard == '/')
      {
      yyerror("Home pattern was specified as absolute path (should be relative to mountpath)");
      }

   if (DEBUG || D1)
      {
      printf(">>Installing wildcard %s as a home pattern\n",wildcard);
      }

   HandleHomePattern(wildcard);
   }
else if (ACTION == control && CLASS == soft)
   {
   AddMacroValue(CURRENTITEM,wildcard);
   }
else if (ACTION == control)
   {
   RecordMacroId(wildcard);
   }
else if (ACTION == files)
   {
   HandleOptionalFileAttribute(wildcard);
   }
else if (ACTION == tidy)
   {
   HandleOptionalTidyAttribute(wildcard);
   }
else if (ACTION == makepath)
   {
   HandleOptionalDirAttribute(wildcard);
   }
else if (ACTION == disable)
   {
   HandleOptionalDisableAttribute(wildcard);
   }
else
   {
   yyerror("Wildcards can only be used in pathnames in tidy: and files:");
   }
}

/*******************************************************************/

HandleEdit(file,edit,string)      /* child routines in edittools.c */

char *file, *edit, *string;

{
if (DEBUG || D1)
   {
   printf("Handling Edit of %s, action [%s] with data <%s>\n",file,edit,string);
   }

if (EditFileExists(file))
   {
   AddEditAction(file,edit,string);
   }
else
   {
   InstallEditFile(file,edit,string);
   }
}

/*******************************************************************/
/* Level 2                                                         */
/*******************************************************************/

CompoundId(id)                       /* check for dots in the name */

char *id;

{ int count = 1;
  char *cid = id;

while (*id != '\0')
   {
   if (*id++ == '.')
      {
      count++;
      }
   }

strcpy(CLASSBUFF,cid);

switch (count)
   {
   case 1: 
           if (IsHardClass(cid))
              {
              CLASS = ClassStringToCode(cid);
              }
           else
              {                              /* <soft> class is a group */
              CLASS = soft;
              }

           break;

   default: CLASS = soft;
            break;
   }

return(count);
}

/*******************************************************************/

InstallLocalInfo (class,value)

enum classes class;
char *value;

{ int number = -1;

if ( ! IsDefinedClass(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",value);
   return;
   }

if (DEBUG || D1)
   {
   printf(">>(Action is control, storing class [%s=%s])\n",CLASSTEXT[class],value);
   }

switch (class)
   {
   case site:
   case faculty:  if (VFACULTY[0] != '\0')
                     {
                     yyerror("Multiple declaration of variable faculty / site");
                     FatalError("Redefinition of basic system variable");
                     }

                  strcpy(VFACULTY,value);
                  break;

   case domain:  if (VDOMAIN[0] != '\0')
                     {
                     yyerror("Multiple declaration of variable domain");
                     FatalError("Redefinition of basic system variable");
                     }

                  strcpy(VDOMAIN,value);
                  break;

   case sysadm:  /* Can be redefined */

                  strcpy(VSYSADM,value);
                  break;

   case netmask:  if (VNETMASK[0] != '\0')
                     {
                     yyerror("Multiple declaration of variable netmask");
                     FatalError("Redefinition of basic system variable");
                     }
                  strcpy(VNETMASK,value);
                  break;


   case mountpath: SetMountPath(value);
                   break;

   case homepat:  
                  if (DEBUG || D1)
                     {
                     printf(">>Installing %s as home pattern\n",value);
                     }
                  AppendItem(&VHOMEPATLIST,value,NULL);
                  break;


   case timezon:  if (VTIMEZONE[0] != '\0')
                     {
                     yyerror ("Multiple declaration of variable timezone");
                     FatalError("Redefinition of basic system variable");
                     }

                  strcpy(VTIMEZONE,value);
                  break;

   case sensiblesize: 
                  sscanf(value,"%d",&number);
                  if (number > 0)
                     {
                     SENSIBLEFSSIZE = number;
                     }
                  else
                     {
                     yyerror("Silly value for sensiblesize");
                     }
                  break;

   case sensiblecount:
                  sscanf(value,"%d",&number);
                  if (number > 0)
                     {
                     SENSIBLEFILECOUNT = number;
                     }
                  else
                     {
                     yyerror("Silly value for sensiblecount");
                     }

                  break;

   case editfilesize:
                  sscanf(value,"%d",&number);

                  if (number > 10)
                     {
                     EDITFILESIZE = number;
                     }
                  else
                     {
                     yyerror("Silly value for editfilesize");
                     }

                  break;

   case actionsequence:
                  AppendToActionSequence(value);
                  break;

   case accesss:
                  AppendToAccessList(value);
                  break;

   case nfstype:

                  strcpy(VNFSTYPE,value); 
                  break;

   case addclasses:
                  AddCompoundClass(value);
                  break;

   default:       AddMacroValue(CURRENTITEM,value);
                  break;
                  
   }
}

/*******************************************************************/

HandleOptionalFileAttribute(item)

char *item;

{ char value[maxvarsize];

value[0]=NULL;

sscanf(item,"%*[^=]=%s",value);

if (value[0] == NULL)
   {
   yyerror("Attribute with no value");
   }

if (DEBUG || D1)
   {
   printf("HandleOptionalFileAttribute(%s)\n",value);
   }

switch(GetFileAttribute(item))
   {
   case frecurse:  HandleRecurse(value);
                   break;
   case fmode:     ParseModeString(value,&PLUSMASK,&MINUSMASK);
                   break;
   case fowner:    strcpy(VUIDNAME,value);
                   break;
   case fgroup:    strcpy(VGIDNAME,value);
                   break;
   case faction:   FILEACTION = GetFileAction(value);
                   break;
   case flinks:    HandleTravLinks(value);
                   break;

   default:        yyerror("Illegal file attribute");
   }
}

/******************************************************************/

HandleOptionalTidyAttribute(item)

char *item;

{ char value[maxvarsize];

value[0]=NULL;

sscanf(item,"%*[^=]=%s",value);

if (value[0] == NULL)
   {
   yyerror("Attribute with no value");
   }

if (DEBUG || D1)
   {
   printf("HandleOptionalTidyAttribute(%s)\n",value);
   }

switch(GetFileAttribute(item))
   {
   case frecurse:  HandleRecurse(value);
                   break;

   case fpattern:  strcpy(CURRENTITEM,value);
                   break;

   case fage:      HandleAge(value);
                   break;

   case flinks:    HandleTravLinks(value);
                   break;

   default:        yyerror("Illegal tidy attribute");
   }
}

/******************************************************************/

HandleOptionalDirAttribute(item)

char *item;

{ char value[maxvarsize];

value[0]=NULL;

sscanf(item,"%*[^=]=%s",value);

if (value[0] == NULL)
   {
   yyerror("Attribute with no value");
   }

if (DEBUG || D1)
   {
   printf("HandleOptionalDirAttribute(%s)\n",value);
   }

switch(GetFileAttribute(item))
   {
   case fmode:     ParseModeString(value,&PLUSMASK,&MINUSMASK);
                   break;
   case fowner:    strcpy(VUIDNAME,value);
                   break;
   case fgroup:    strcpy(VGIDNAME,value);
                   break;
   default:        yyerror("Illegal directory attribute");
   }
}


/*******************************************************************/

HandleOptionalDisableAttribute(item)

char *item;

{ char value[maxvarsize];

value[0]=NULL;

sscanf(item,"%*[^=]=%s",value);

if (value[0] == NULL)
   {
   yyerror("Attribute with no value");
   }

if (DEBUG || D1)
   {
   printf("HandleOptionalDisableAttribute(%s)\n",value);
   }

switch(GetFileAttribute(item))
   {
   case ftype:     HandleDisableFileType(value);
                   break;

   default:        yyerror("Illegal disable attribute");
   }
}
/*******************************************************************/

HandleFileItem(item)

char *item;

{ int i;
  char err[100];

if (strcmp(item,"home") == 0)
   {
   ACTIONPENDING=true;
   strcpy(CURRENTPATH,"home");
   return;
   }

sprintf(err,"Unknown file attribute %s",item);
yyerror(err);
}

/*******************************************************************/

InitFileAction()                                   /* Set defaults */

{
if (DEBUG || D1)
   {
   printf ("InitFileAction()\n");
   }

PLUSMASK = 0;
MINUSMASK = 0;
VRECURSE = 0;
VAGE = 99999;
VUIDNAME[0] = '*';
VUIDNAME[1] = NULL;
VGIDNAME[0] = '*';
VGIDNAME[1] = NULL;
HAVEPATH=1;
FILEACTION=warnall;
CURRENTPATH[0] = NULL;
ACTIONPENDING = false;
PTRAVLINKS = (short) '?';                   /* Take from command line */
}

/*******************************************************************/

InstallBroadcastItem(item)

char *item;

{
if (DEBUG || D1)
   {
   printf (">>Install broadcast mode (%s)\n",item);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

if (VBROADCAST[0] != '\0')
   {
   yyerror("Multiple declaration of variable broadcast");
   FatalError("Redefinition of basic system variable");
   }

if (strcmp("ones",item) == 0)
   {
   strcpy(VBROADCAST,"one");
   return;
   }

if (strcmp("zeroes",item) == 0)
   {
   strcpy(VBROADCAST,"zero");
   return;
   }

if (strcmp("zeros",item) == 0)
   {
   strcpy(VBROADCAST,"zero");
   return;
   }

yyerror ("Unknown broadcast mode (should be ones, zeros or zeroes)");
FatalError("Unknown broadcast mode");
}

/*******************************************************************/

InstallDefaultRouteItem(item)

char *item;

{ struct sockaddr_in *sin;
  struct hostent *hp;
  struct in_addr inaddr;

if (DEBUG || D1)
   {
   printf (">>Install defaultroute mode (%s)\n",item);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

if (VDEFAULTROUTE[0] != '\0')
   {
   yyerror("Multiple declaration of variable defaultroute");
   FatalError("Redefinition of basic system variable");
   }

if (inet_addr(item) == -1)
   {
   if ((hp = gethostbyname(item)) == NULL)
      {
      perror("InstallDefaultRouteItem: gethostbyname: ");
      yyerror ("Bad specification of default packet route: hostname or decimal IP address");
      FatalError("Stopped.");
      }
   else
      {
      bcopy(hp->h_addr,&inaddr, hp->h_length);
      strcpy(VDEFAULTROUTE,inet_ntoa(inaddr));
      }
   }
else
   {
   strcpy(VDEFAULTROUTE,item);
   }
}

/*******************************************************************/

HandleGroupItem(item,type)

char *item;
enum itemtypes type;

{ char *machine, *user, *domain;

if (DEBUG || D1)
   {
   printf(">>Handling item (%s) in group (%s)\n",item,GROUPBUFF);
   }

switch (type)
   {
   case simple:    if (strcmp(item,VSYSNAME.nodename) == 0)
                      {
                      AddClassToHeap(GROUPBUFF);
                      }

                   break;

   case netgroup:  setnetgrent(item);

                   while (getnetgrent(&machine,&user,&domain))
                      {
                      if (strcmp(machine,VSYSNAME.nodename) == 0)
                         {
                         if (DEBUG || D1) printf("Matched %s in netgroup %s\n",machine,item);
                         AddClassToHeap(GROUPBUFF);
                         break;
                         }
                      }
                   
                   endnetgrent();
                   break;


   case groupdeletion: 

                   setnetgrent(item);

                   while (getnetgrent(&machine,&user,&domain))
                      {
                      if (strcmp(machine,VSYSNAME.nodename) == 0)
                         {
                         if (DEBUG || D1) printf("Matched delete item %s in netgroup %s\n",machine,item);
                         DeleteItemStarting(&VHEAP,GROUPBUFF);
                         break;
                         }
                      }
                   
                   endnetgrent();
                   break;

   case deletion:  if (strcmp(item,VSYSNAME.nodename) == 0)
                      {
                      DeleteItemStarting(&VHEAP,GROUPBUFF);
                      }
                   break;

   default:        yyerror("Software error");
                   FatalError("Unknown item type");
   }
}

/*******************************************************************/

HandleHomePattern(pattern)

char *pattern;

{
VBUFF[0]=NULL;
ExpandVarstring(pattern,VBUFF,"");
AppendItem(&VHOMEPATLIST,VBUFF,NULL);
}

/*******************************************************************/

AppendNameServer(item)

char *item;

{ 
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the nameserver list\n",item);
   }


if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

AppendItem(&VRESOLVE,item,NULL);
}

/*******************************************************************/

AppendImport(item)

char *item;

{ 
if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

if (strcmp(item,VCURRENTFILE) == 0)
   {
   yyerror("A file cannot import itself");
   FatalError("Infinite self-reference in class inheritance");
   }

if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the import list\n",item);
   }


AppendItem(&VIMPORT,item,NULL);
}


/*******************************************************************/

InstallHomeserverItem(item)

char *item;

{
if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in  list for homeservers (%s) in group (%s)\n",item,CLASSBUFF,GROUPBUFF);
   }

AppendItem(&VHOMESERVERS,item,NULL);
}

/*******************************************************************/

InstallBinserverItem(item)           /* Install if matches classes */

char *item;

{
if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

AppendItem(&VBINSERVERS,item,NULL);
}

/*******************************************************************/

InstallMailserverPath(path)

char *path;

{
if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }

if (VMAILSERVER[0] != NULL)
   {
   FatalError("Redefinition of mailserver");
   }

strcpy(VMAILSERVER,path);

if (DEBUG || D1)
   {
   printf(">>Installing mailserver (%s) for group (%s)",path,GROUPBUFF);
   }
}

/*******************************************************************/

AppendScript(item)

char *item;

{
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the script list\n",item);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",item);
   return;
   }

VBUFF[0]=NULL;
ExpandVarstring(item,VBUFF,"");
AppendItem(&VSCRIPT,VBUFF,CLASSBUFF);
}

/*******************************************************************/

InstallLinkItem (from,to)

char *from, *to;

{ struct Link *ptr;
  char *spfrom,*spto,*spe;

if (DEBUG || D1) 
   {
   printf(">>Storing Link: %s -> %s\n",from,to);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing link no match\n");
   return;
   }

VBUFF[0]=NULL;                         /* Expand any variables */
ExpandVarstring(from,VBUFF,"");


if ((ptr = (struct Link *)malloc(sizeof(struct Link))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallListItem() #1");
   }

if ((spfrom = malloc(strlen(VBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallListItem() #2");
   }

if ((spto = malloc(strlen(to)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallListItem() #3");
   }

if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallListItem() #4");
   }

if (VLINKTOP == NULL)                 /* First element in the list */
   {
   VLINK = ptr;
   }
else
   {
   VLINKTOP->next = ptr;
   }

strcpy (spfrom,VBUFF);
strcpy (spto,to);
strcpy (spe,CLASSBUFF);

ptr->classes = spe;
ptr->from = spfrom;
ptr->to = spto;
ptr->force = FORCELINK;
ptr->next = NULL;
VLINKTOP = ptr;
}

/*******************************************************************/

InstallLinkChildrenItem (from,to)

char *from, *to;

{ struct Link *ptr;
  char *spfrom,*spto, *spe;

if (DEBUG || D1) 
   {
   printf(">>Storing Linkchildren item: %s -> %s\n",from,to);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing linkchildren no match\n");
   return;
   }

VBUFF[0]=NULL;                                /* Expand any variables */
ExpandVarstring(from,VBUFF,"");

if ((ptr = (struct Link *)malloc(sizeof(struct Link))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallListChildrenItem() #1");
   }

if ((spfrom = malloc(strlen(VBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallLinkchildrenItem() #2");
   }

if ((spto = malloc(strlen(to)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallLinkChildrenItem() #3");
   }


if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallLinkChildrenItem() #3");
   }

if (VCHLINKTOP == NULL)                 /* First element in the list */
   {
   VCHLINK = ptr;
   }
else
   {
   VCHLINKTOP->next = ptr;
   }

strcpy (spfrom,VBUFF);
strcpy (spto,to);
strcpy (spe,CLASSBUFF);

ptr->classes = spe;
ptr->from = spfrom;
ptr->to = spto;
ptr->force = FORCELINK;
ptr->next = NULL;
VCHLINKTOP = ptr;
}

/*******************************************************************/

InstallTidyItem (path,wild,rec,age,travlinks)

char *wild, *path;
short age;
int rec, travlinks;

{ struct Tidy *ptr;
  char *sp, *spe;
  char *pp;

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing tidy item, no match\n");
   return;
   }

if (DEBUG || D1)
   {  
   printf("InstallTidyItem(%s,%s,%d,%d,%c)\n",path,wild,rec,age,travlinks);
   }

VBUFF[0]=NULL;                                /* Expand any variables */
ExpandVarstring(path,VBUFF,"");

if ((ptr = (struct Tidy *)malloc(sizeof(struct Tidy))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #1");
   }

if ((sp = malloc(strlen(wild)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #2");
   }

if ((pp = malloc(strlen(VBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #3");
   }

if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallTidyItem() #2");
   }

if (VTIDYTOP == NULL)                 /* First element in the list */
   {
   VTIDY = ptr;
   }
else
   {
   VTIDYTOP->next = ptr;
   }

strcpy (sp,wild);
strcpy (pp,VBUFF);
strcpy (spe,CLASSBUFF);

ptr->classes= spe;
ptr->wild = sp;
ptr->path = pp;
ptr->recurse = rec;
ptr->age = age;
ptr->travlinks = travlinks;
ptr->next = NULL;
VTIDYTOP = ptr;
InitFileAction();
}

/*******************************************************************/

InstallRequiredPath(path)

char *path;

{ 
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the required list\n",path);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }

AppendItem(&VREQUIRED,path,CLASSBUFF);
}

/*******************************************************************/

AppendProcess(path)

char *path;

{ 
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the processes list\n",path);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }

/* AppendItem(&VPROCESSES,path,CLASSBUFF); */
}

/*******************************************************************/

AppendDisable(path,type)

char *path, *type;

{ 
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the disable list\n",path);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }

if (strlen(type) == 0)
   {
   sprintf(VBUFF,"%s=%s","all",path);
   }
else
   {
   sprintf(VBUFF,"%s=%s",type,path);
   }

AppendItem(&VDISABLELIST,VBUFF,CLASSBUFF);
}

/*******************************************************************/

InstallMakePath(path,plus,minus,uidnames,gidnames)

char *path;
mode_t plus,minus;
char *uidnames;
char *gidnames;

{ struct File *ptr;
  char *sp, *spe;
  int i;

if (DEBUG || D1)
   {
   printf("InstallMakePath (%s) (+%o)(-%o)(%s)(%s)\n",path,plus,minus,uidnames,gidnames);
   }


if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing directory item, no match\n");
   return;
   }

VBUFF[0]=NULL;                                /* Expand any variables */
ExpandVarstring(path,VBUFF,"");

if ((ptr = (struct File *)malloc(sizeof(struct File))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallMakepath() #1");
   }

if ((sp = malloc(strlen(VBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallMakepath() #2");
   }

if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallMakepath() #3");
   }

if (VMAKEPATHTOP == NULL)                 /* First element in the list */
   {
   VMAKEPATH = ptr;
   }
else
   {
   VMAKEPATHTOP->next = ptr;
   }

strcpy (sp,VBUFF);
strcpy (spe,CLASSBUFF);

ptr->classes = spe;
ptr->path = sp;
ptr->plus = plus;
ptr->minus = minus;
ptr->recurse = 0;
ptr->action = fixdirs;
ptr->uid = MakeUidList(uidnames);
ptr->gid = MakeGidList(gidnames);
ptr->next = NULL;
VMAKEPATHTOP = ptr;
InitFileAction();
}

/*******************************************************************/

AppendMountable(path)

char *path;

{ 
if (DEBUG || D1)
   {
   printf(">>Adding mountable %s to list\n",path);
   }

AppendItem(&VMOUNTABLES,path,NULL);
}

/*******************************************************************/

AppendUmount(path)

char *path;

{ 
if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }

if (DEBUG || D1)
   {
   printf(">>Adding unmount %s to list\n",path);
   }

AppendItem(&VUNMOUNT,path,CLASSBUFF);
}

/*******************************************************************/

AppendMiscMount(from,onto,perm)

char *from, *onto, *perm;

{ struct MiscMount *ptr;
  char *sp1,*sp2,*sp3, *spe;

if (DEBUG || D1)
   {
   printf(">>Adding misc mountable %s %s (%s) to list\n",from,onto,perm);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",from);
   return;
   }

if ((ptr = (struct MiscMount *)malloc(sizeof(struct MiscMount))) == NULL)
   {
   FatalError("Memory Allocation failed for AppendMiscMount #1");
   }

if ((sp1 = malloc(strlen(from)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for AppendMiscMount() #2");
   }

if ((sp2 = malloc(strlen(onto)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for AppendMiscMount() #3");
   }

if ((sp3 = malloc(strlen(perm)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for AppendMiscMount() #4");
   }

if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for AppendMiscMount() #5");
   }

strcpy(sp1,from);
strcpy(sp2,onto);
strcpy(sp3,perm);
strcpy(spe,CLASSBUFF);

if (VMISCMOUNTTOP == NULL)                 /* First element in the list */
   {
   VMISCMOUNT = ptr;
   }
else
   {
   VMISCMOUNTTOP->next = ptr;
   }

ptr->classes = CLASSBUFF;
ptr->from = sp1;
ptr->onto = sp2;
ptr->options  = sp3;
ptr->next = NULL;
VMISCMOUNTTOP = ptr;
}


/*******************************************************************/

AppendIgnore(path)

char *path;

{
if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the ignore list\n",path);
   }

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing %s, no match\n",path);
   return;
   }


AppendItem(&VIGNORE,path,NULL);
}

/*******************************************************************/

AppendToActionSequence (action)

char *action;

{ int i,j = 0;
  char *sp,cbuff[bufsize],actiontxt[bufsize];
  struct Item *ip;

if (DEBUG || D1)
   {
   printf(">>Installing item (%s) in the action sequence list\n",action);
   }

AppendItem(&VACTIONSEQ,action,NULL);

cbuff[0]='\0';
actiontxt[0]='\0';
sp = action;

while (*sp != NULL)
   {
   ++j;
   sscanf(sp,"%[^.]",cbuff);

   while ((*sp != NULL) && (*sp !='.'))
      {
      sp++;
      }
 
   if (*sp == '.')
      {
      sp++;
      }
 
   if (IsHardClass(cbuff))
      {
      yyerror("Error in action sequence: %s\n",action);
      yyerror("You cannot add a reserved class!");
      return;
      }
 
   if (j == 1)
      {
      strcpy(actiontxt,cbuff);
      continue;
      }
   else
      {
      AppendItem(&VALLADDCLASSES,cbuff,NULL);
      }
   }
}

/*******************************************************************/

AppendToAccessList (user)

char *user;

{ char id[maxvarsize];
  struct passwd *pw;

if (DEBUG || D1 || D2)
   {
   printf("Adding to access list for %s\n",user);
   }

if (isalpha(user[0]))
   {
   if ((pw = getpwnam(user)) == NULL)
      {
      yyerror("No such user in password database");
      return;
      }

   sprintf(id,"%d",pw->pw_uid);
   AppendItem(&VACCESSLIST,id,NULL);
   }
else
   {
   AppendItem(&VACCESSLIST,user,NULL);
   }
}


/*******************************************************************/

InstallPending(action)

enum actions action;

{
if (DEBUG || D1)
   {
   if (ACTIONPENDING)
      {
      printf("Installing outstanding %s\n",ACTIONTEXT[action]);
      }
   else
      {
      printf("(No actions pending in %s)\n",ACTIONTEXT[action]);
      }
   }

if (! ACTIONPENDING)
   {
   return;
   }

switch (action)
   {
   case files:
                  InstallFileListItem(CURRENTPATH,PLUSMASK,MINUSMASK,FILEACTION,VUIDNAME,VGIDNAME,VRECURSE,(char)PTRAVLINKS);
                  break;

   case tidy:     if (VAGE >= 99999)
                     {
                     yyerror("Must specify an age for tidy actions");
                     return;
                     }
                  InstallTidyItem(CURRENTPATH,CURRENTITEM,VRECURSE,VAGE,(char)PTRAVLINKS);
                  break;

   case makepath: InstallMakePath(CURRENTPATH,PLUSMASK,MINUSMASK,VUIDNAME,VGIDNAME);
                  break;

   case disable:  AppendDisable(CURRENTPATH,CURRENTITEM);
                  break;
   }

ACTIONPENDING = false;
CURRENTITEM[0] = '\0';
CURRENTITEM[0] = '\0';
VRECURSE = 0;
VAGE=99999;
}

/*******************************************************************/
/* Level 3                                                         */
/*******************************************************************/

HandleTravLinks(value)

char *value;

{
if (ACTION == tidy && strncmp(CURRENTPATH,"home",4) == 0)
   {
   yyerror("Can't use links= option with special variable home");
   yyerror("Use command line options instead.\n");
   }

if (PTRAVLINKS != '?')
   {
   Warning("redefinition of links= option");
   }

if ((strcmp(value,"stop") == 0) || (strcmp(value,"false") == 0))
   {
   PTRAVLINKS = (short) 'F';
   return;
   }

if ((strcmp(value,"traverse") == 0) || (strcmp(value,"follow") == 0) || (strcmp(value,"true") == 0))
   {
   PTRAVLINKS = (short) 'T';
   return;
   }

if ((strcmp(value,"tidy"))==0)
   {
   PTRAVLINKS = (short) 'K';
   return;
   }

yyerror("Illegal links= specifier");
}

/*******************************************************************/

GetFileAction(action)

char *action;

{ int i;

for (i = 0; FILEACTIONTEXT[i] != NULL; i++)
   {
   if (strcmp(action,FILEACTIONTEXT[i]) == 0)
      {
      return i;
      }
   }

yyerror("Unknown action type");
return (int) warnall;
}

/*********************************************************************/

SetMountPath (value)

char *value;

{ struct Item *ip;
  char *sp;

if (VMOUNTPATH[0] != NULL)
   {
   yyerror("Redefinition of system variable mountpath");
   FatalError("");
   }

for (sp = value; /* No exit */ ; sp++)       /* check for varitems */
   {
   sscanf(sp,"%[^<]",CURRENTITEM);

   strcat(VMOUNTPATH,CURRENTITEM);
   sp += strlen(CURRENTITEM);

   CURRENTITEM[0] = NULL;

   if (*sp == NULL)
      {
      if (strcmp("/",VMOUNTPATH+strlen(VMOUNTPATH)-1) != 0)
         {
         strcat(VMOUNTPATH,"/");
         }

      break;
      }
   else
      {
      sscanf(++sp,"%[^>]",CURRENTITEM);               /* reduce item */
      if (strcmp(CURRENTITEM,"faculty") == 0 || strcmp(CURRENTITEM,"site") == 0)
         {
         if (VFACULTY[0] == NULL)
            {
            yyerror("<faculty> / <site> undefined variable");
            }
         strcat(VMOUNTPATH,VFACULTY);
         sp += strlen(CURRENTITEM);
         CURRENTITEM[0] = NULL;
         }
      else if (strcmp(CURRENTITEM,"host") == 0)
         {
         strcat(VMOUNTPATH,VSYSNAME.nodename);
         sp += strlen(CURRENTITEM);
         CURRENTITEM[0] = NULL;
         }
      else
         {
         printf("Bad varpath in mountpath <%s>\n",CURRENTITEM);
         FatalError("No such variable allowed");
         }
      }
   }
}

/*********************************************************************/

struct UidList *MakeUidList(uidnames)

char *uidnames;

{ struct UidList *uidlist;
  struct Item *ip, *tmplist;
  char uidbuff[bufsize];
  char *sp;
  int offset;
  struct passwd *pw;
  char *machine, *user, *domain;
  int uid;
  int tmp;

uidlist = NULL;

for (sp = uidnames; *sp != NULL; sp+=strlen(uidbuff))
   {
   if (*sp == ',')
      {
      sp++;
      }

   if (sscanf(sp,"%[^,]",uidbuff))
      {
      if (uidbuff[0] == '+')        /* NIS group - have to do this in a roundabout     */
         {                          /* way because calling getpwnam spoils getnetgrent */
         offset = 1;
         if (uidbuff[1] == '@')
            {
            offset++;
            }

         setnetgrent(uidbuff+offset);
         tmplist = NULL;

         while (getnetgrent(&machine,&user,&domain))
            {
            if (user != NULL)
               {
               AppendItem(&tmplist,user,NULL);
               }
            }
                   
         endnetgrent();

         for (ip = tmplist; ip != NULL; ip=ip->next)
            {
            if ((pw = getpwnam(ip->name)) == NULL)
               {
               if (VERBOSE || DEBUG || D1)
                  {
                  sprintf(CURRENTITEM,"Unknown user [%s]\n",ip->name);
                  Warning(CURRENTITEM);
                  sprintf(CURRENTITEM,"User in netgroup [%s] not in passwd file\n",uidbuff+offset);
                  Warning(CURRENTITEM);
                  }
               continue;
               }
            else
               {
               uid = pw->pw_uid;
               }
            AddSimpleUidItem(&uidlist,uid); 
            }

         DeleteItemList(tmplist);
         continue;
         }

      if (isdigit(uidbuff[0]))
         {
         sscanf(uidbuff,"%d",&tmp);
         uid = (uid_t)tmp;
         }
      else
         {
         if (strcmp(uidbuff,"*") == 0)
            {
            uid = -1;                     /* signals wildcard */
            }
         else if ((pw = getpwnam(uidbuff)) == NULL)
            {
            printf("Unknown user %s\n",uidbuff);
            Warning("User is not known in this passwd domain");
            continue;
            }
         else
            {
            uid = pw->pw_uid;
            }
         }
      AddSimpleUidItem(&uidlist,uid);
      }
   }

return (uidlist);
}

/*********************************************************************/

struct GidList *MakeGidList(gidnames)

char *gidnames;

{ struct GidList *gidlist;
  char gidbuff[bufsize];
  char *sp;
  struct group *gr;
  int gid;
  int tmp;

gidlist = NULL;

for (sp = gidnames; *sp != NULL; sp+=strlen(gidbuff))
   {
   if (*sp == ',')
      {
      sp++;
      }

   if (sscanf(sp,"%[^,]",gidbuff))
      {
      if (isdigit(gidbuff[0]))
         {
         sscanf(gidbuff,"%d",&tmp);
         gid = (gid_t)tmp;
         }
      else
         {
         if (strcmp(gidbuff,"*") == 0)
            {
            gid = -1;                     /* signals wildcard */
            }
         else if ((gr = getgrnam(gidbuff)) == NULL)
            {
            sprintf(CURRENTITEM,"Unknown group %s\n",gidbuff);
            Warning(CURRENTITEM);
            continue;
            }
         else
            {
            gid = gr->gr_gid;
            }
         }
      AddSimpleGidItem(&gidlist,gid);
      }
   }


return(gidlist);
}


/*******************************************************************/

InstallFileListItem(path,plus,minus,action,uidnames,gidnames,recurse,travlinks)

char *path;
mode_t plus,minus;
enum fileactions action;
char *uidnames;
char *gidnames;
int recurse;
char travlinks;

{ struct File *ptr;
  char *sp, *spe;
  char buf[bufsize];
  int i;

if (DEBUG || D1)
   {
   printf("InstallFileaction (%s) (+%o)(-%o) (%s) (%d) (%c)\n",path,plus,minus,FILEACTIONTEXT[action],action,travlinks);
   }

for (i=0; i<bufsize; i++)
   {
   buf[i] = NULL;
   }

ExpandVarstring(path,buf,"");

if ( ! IsInstallable(CLASSBUFF))
   {
   if (DEBUG || D1) printf("Not installing file item, no match\n");
   return;
   }

if ((ptr = (struct File *)malloc(sizeof(struct File))) == NULL)
   {
   FatalError("Memory Allocation failed for InstallFileListItem() #1");
   }

if ((sp = malloc(strlen(buf)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallFileListItem() #2");
   }

if ((spe = malloc(strlen(CLASSBUFF)+1)) == NULL)
   {
   FatalError("Memory Allocation failed for InstallFileListItem() #3");
   }


if (VFILETOP == NULL)                 /* First element in the list */
   {
   VFILE = ptr;
   }
else
   {
   VFILETOP->next = ptr;
   }

strcpy (sp,buf);
strcpy (spe,CLASSBUFF);

ptr->classes = spe;
ptr->path = sp;
ptr->action = action;
ptr->plus = plus;
ptr->minus = minus;
ptr->recurse = recurse;
ptr->uid = MakeUidList(uidnames);
ptr->gid = MakeGidList(gidnames);
ptr->travlinks = travlinks;
ptr->next = NULL;
VFILETOP = ptr;
InitFileAction();
}

/*******************************************************************/

GetFileAttribute(s)

char *s;

{ int i;
  char comm[maxvarsize];

for (i = 0; s[i] != NULL; i++)
   {
   s[i] = ToLower(s[i]);
   }

comm[0]=NULL;

sscanf(s,"%[^=]",comm);

if (DEBUG || D1)
   {
   printf("GetFileAttribute(%s)\n",comm);
   }

for (i = 0; FILEATTRIBUTES[i] != NULL; i++)
   {
   if (strncmp(FILEATTRIBUTES[i],comm,strlen(comm)) == 0)
      {
      if (DEBUG || D1)
         {
         printf("GetFileAttribute - got: %s\n",FILEATTRIBUTES[i]);
         }
      return i;
      }
   }

return fbad;
}

/*******************************************************************/

HandleRecurse(value)

char *value;

{ int n = -1;

if (strcmp(value,"inf") == 0)
   {
   VRECURSE = INFINITERECURSE;
   }
else
   {
   if (strncmp(CURRENTPATH,"home",4) == 0)
      {
      yyerror("Recursion is always infinite for home");
      return;
      }

   sscanf(value,"%d",&n);

   if (n == -1)
      {
      yyerror("Illegal recursion specifier");
      }
   else
      {
      VRECURSE = n;
      }
   }
}

/*******************************************************************/

HandleDisableFileType(value)

{
if (strlen(CURRENTITEM) != 0)
   {
   Warning("Redefinition of filetype in disable");
   }

if (strcmp(value,"link") == 0 || strcmp(value,"links") == 0)
   {
   strcpy(CURRENTITEM,"link");
   }
else if (strcmp(value,"plain") == 0 || strcmp(value,"file") == 0)
   {
   strcpy(CURRENTITEM,"file");
   }
else
   {
   yyerror("Disable filetype unknown");
   }
}

/*******************************************************************/

HandleAge(days)

char *days;

{ 
sscanf(days,"%d",&VAGE);

if (DEBUG || D1)
   { 
   printf ("HandleAge(%d)\n",VAGE);
   }
}

/*******************************************************************/
/* Level 4                                                         */
/*******************************************************************/

AddSimpleUidItem(uidlist,uid)

struct UidList **uidlist;
int uid;

{ struct UidList *ulp, *u;

if ((ulp = (struct UidList *)malloc(sizeof(struct UidList))) == NULL)
   {
   FatalError("cfengine: malloc() failed #1 in AddSimpleUidItem()");
   }

ulp->uid = uid;
ulp->next = NULL;

if (*uidlist == NULL)
   {
   *uidlist = ulp;
   }
else
   {
   for (u = *uidlist; u->next != NULL; u = u->next)
      {
      }
   u->next = ulp;
   }
}

/*******************************************************************/

AddSimpleGidItem(gidlist,gid)

struct GidList **gidlist;
int gid;

{ struct GidList *glp,*g;

if ((glp = (struct GidList *)malloc(sizeof(struct GidList))) == NULL)
   {
   FatalError("cfengine: malloc() failed #1 in AddSimpleGidItem()");
   }
 
glp->gid = gid;
glp->next = NULL;

if (*gidlist == NULL)
   {
   *gidlist = glp;
   }
else
   {
   for (g = *gidlist; g->next != NULL; g = g->next)
      {
      }
   g->next = glp;
   }

}



/* EOF */
