/* This file is Copyright 1993 by Clifford A. Adams */
/* sgmisc.c
 *
 * group scan mode misc. functions, plus the sg_ext (editing) functions
 */

/* trim this list */
#include "EXTERN.h"
#include "common.h"
#ifdef SCAN
#include "cache.h"
#include "intrp.h"
#include "ng.h"		/* for NG_PREV, etc... */
#include "ngdata.h"	/* abs1st[] */
#include "trn.h"
#include "rcln.h"
#include "rcstuff.h"
#include "scan.h"
#include "scmd.h"
#include "sdisp.h"
#include "smisc.h"
#include "sgroup.h"
#include "sgdata.h"
#include "sgfile.h"
#include "term.h"
#include "util.h"
#include "INTERN.h"
#include "sgmisc.h"

bool
sg_eligible(ent)
long ent;
{
    long grp;
    long num;

    switch (sg_ents[ent].type) {
	case 2:		/* file type */
	case 3:		/* restriction list */
	case 4:		/* virtual newsgroup */
	case 5:		/* text */
	    return(TRUE);
    }
    /* check newsgroup */
    sg_toread(sg_ents[ent].groupnum,FALSE);
    grp = sg_groups[sg_ents[ent].groupnum].rcnum;
    if ((grp<0) || (grp>nextrcline)) {
	/* ick.  let's not crash though... */
	return(FALSE);
    }
    num = toread[grp];
    if (ngmax[grp]<abs1st[grp])		/* empty group */
	num = TR_NONE;			/* Icky bug workaround */
    if (sg_mode_allgroups)
	return(TRUE);
    if (sg_mode_unsub && (num != TR_UNSUB))
	return(FALSE);
    switch (num) {
	case TR_JUNK:
	case TR_BOGUS:
	    return(FALSE);
	case TR_UNSUB:
	    if (sg_mode_unsub)
		return(TRUE);
	    return(FALSE);
	case TR_NONE:
	    if (sg_mode_zero) {
		if (ngmax[grp]<abs1st[grp])		/* empty group */
		    return(FALSE);
		return(TRUE);
	    }
	    return(FALSE);
	default:
	    /* later consider number-of-article limitations? */
	    return(TRUE);
    }
}

char *
sg_get_desc(ent,line,trunc)
long ent;
int line;
bool_int trunc;		/* should this be truncated? */
{
    static char desc_buf[256];
    static char lbuf[256];
    char *s;

    desc_buf[0] = '\0';
    switch (line) {
	case 1:
	    /* name has same function for any type */
	    sprintf(lbuf,"%s",sg_ents[ent].name);
	    strcat(desc_buf,lbuf);
	    break;
	case 2:
	    s = sg_groups[sg_ents[ent].groupnum].desc;
	    if (!s || !*s)
		strcpy(desc_buf,"<No group description>");
	    else
		strcpy(desc_buf,s);
	    break;
    }
    if (trunc)
	desc_buf[s_desc_cols] = '\0';
    return(desc_buf);
}

/* fixed 10-character line */
char *
sg_get_statchars(ent,line)
long ent;
int line;
{
    static char statbuf[256];
    int num;
    int grp;
    int i;
    char *p;

    if (line==2) {
	strcpy(statbuf,"          ");
	return(statbuf);
    }
    switch (sg_ents[ent].type) {
	case 2:		/* file */
	    /* strings copied so they can be altered */
	    strcpy(statbuf,"      file");
	    return(statbuf);
	case 3:		/* restriction list */
	    strcpy(statbuf,"      list");
	    return(statbuf);
	case 4:		/* virtual newsgroup */
	    strcpy(statbuf,"      virt");
	    return(statbuf);
	case 5:		/* text */
	    strcpy(statbuf,"      text");
	    return(statbuf);
    }
    grp = sg_groups[sg_ents[ent].groupnum].rcnum;
    if ((grp<0) || (grp>nextrcline)) {
	/* ick.  let's not crash though... */
	strcpy(statbuf,"*BUG*");	/* get attention? */
	return(statbuf);
    }
    for (i=0;i<5;i++)
	statbuf[i] = '.';
    p = statbuf;
    sg_toread(sg_ents[ent].groupnum,FALSE);
    switch (toread[grp]) {
	case TR_JUNK:
	    *p = 'J';
	    break;
	case TR_BOGUS:
	    *p = 'B';
	    break;
	case TR_UNSUB:
	    *p = '!';
	    break;
	case TR_NONE:
	    if (ngmax[grp]<abs1st[grp])		/* empty group */
		*p = '0';
	    else
		*p = '-';
	    break;
	default:
	    *p = '+';
	    break;
    }
    if (sg_groups[sg_ents[ent].groupnum].flags & 1)	/* moderated */
	statbuf[1] = 'M';
    num = toread[grp];
    if (ngmax[grp]<abs1st[grp])			/* empty group */
	sprintf(statbuf+5," NONE");
    else
    if (num>TR_UNSUB) {
	if (num<100000) {	/* will fit in 5 columns */
	     sprintf(statbuf+5,"%5ld",(long)num);
	} else {
	     sprintf(statbuf+5,"%s"," ++++");
	}
    }
    if (num<=TR_UNSUB) {
	/* unsub, bogus, and junk don't get counts */
	sprintf(statbuf+5,"%s"," ****");
    }
    statbuf[s_status_cols] = '\0';
    return(statbuf);
}

void
sg_refresh_top()
{
    char lbuf[256];
    char *s;

    standout();
    if (sg_num_contexts==1)
	printf("Top Level |");
    else
	printf("Level %d |",sg_num_contexts);
    if (sg_mode_allgroups)
	printf(" All");
    if (sg_mode_unsub)
	printf(" Unsubscribed");
    if (!(sg_mode_allgroups || sg_mode_unsub)) {
	if (sg_mode_zero)
	    printf(" Unread+Read");
	else
	    printf(" Unread");
    }
    printf(" Newsgroups |");
    /* print up to 40 characters of title (change to less later?) */
    s = sg_contexts[sg_num_contexts-1].title;
    if (strlen(s)<41)
	printf(" %s",s);
    else {
	/* actually use 43 characters of line */
	strncpy(lbuf,s,40);
	lbuf[40] = '\0';
	printf(" %s...",lbuf);
    }
    un_standout();
    erase_eol();
    printf("\n") FLUSH;
}

void
sg_refresh_bot()
{
    standout();
    s_mail_and_place();
    un_standout();
    erase_eol();
    fflush(stdout);
}

int
sg_ent_lines(ent)
long ent;
{
    if (sg_use_desc) {
	switch (sg_ents[ent].type) {
	    case 1:		/* newsgroup */
		if (sg_group_desc(sg_ents[ent].groupnum))
		    return(2);
		/* FALL THROUGH */
	    case 2:		/* filename */
	    case 3:		/* list */
	    case 4:		/* virtual newsgroup */
		return(1);
	}
    }
    return(1);
}

/* returns TRUE if screen/window is big enough, FALSE otherwise */
/* for now, the requirements are 15 columns and 5 lines */
bool
sg_check_screen()
{
    if (LINES<5 || COLS<5)
	return(FALSE);
    return(TRUE);
}

/* sets up shape of screen */
/* assumes s_initscreen has been called */
void
sg_set_screen()
{
    /* One size fits all for now. */
    s_top_lines = 1;
    s_bot_lines = 1;
    s_status_cols = 10;
    s_cursor_cols = 2;
    /* (scr_width-1) keeps last character blank. */
    s_desc_cols = (scr_width-1)-s_status_cols-s_cursor_cols;
}

/* do something useful related to the group scan lookahead */
void
sg_lookahead()
{
#ifdef PENDING
    long flags;

#ifdef NICEBG
    if (wait_key_pause(10))	/* spend one second waiting for a key */
	return;			/* give user commands first priority */
#endif
    while (sg_look_num<sg_num_groups) {
	if (input_pending())
	    return;
	flags = sg_groups[sg_look_num].flags;
	if (!(flags & 2))
	    sg_toread(sg_look_num,FALSE);
	if (sg_use_desc && !(flags & 4))
	    (void)sg_group_desc(sg_look_num);
	sg_look_num++;
    }
#endif
}

/* subscribe to grp */
void
sg_subscribe(grp)
long grp;		/* index into sg_groups */
{
    long ng;		/* index into rc arrays */
    char *cp;

    sg_toread(grp,TRUE);
    ng = sg_groups[grp].rcnum;
    if ((ng<0) || (ng>nextrcline)) {
	/* ick.  deal with it. */
	return;
    }
    if (toread[ng] > TR_UNSUB)
	return;		/* already subscribed */

    cp = rcline[ng] + rcnums[ng];
    rcchar[ng] = (*cp && cp[1] == '0' ? '0' : ':');
    sg_toread(grp,TRUE);
}

/* unsubscribe to grp */
void
sg_unsubscribe(grp)
long grp;		/* index into sg_groups */
{
    long ng;		/* index into rc arrays */

    sg_toread(grp,TRUE);
    ng = sg_groups[grp].rcnum;
    if ((ng<0) || (ng>nextrcline)) {
	/* ick.  deal with it. */
	return;
    }
    if (ng < nextrcline && toread[ng] >= TR_NONE) {
			/* unsubscribable? */
	rcchar[ng] = NEGCHAR;
			/* unsubscribe to (from?) it */
	toread[ng] = TR_UNSUB;
			/* and make line invisible */
    }
    sg_toread(grp,TRUE);
}

void sg_ext_setup _((void));
void sg_ext_add  _((void));
void sg_ext_add_write _((char *));
void sg_ext_add_grp _((void));
void sg_ext_add_list _((void));
void sg_ext_add_vgr _((void));
void sg_ext_add_file _((void));

void
sg_ext_menu()
{
    bool q_done;	/* if TRUE, we are finished with current question */
    int i;
    char ch;

    printf("Group scan mode editing menu.\n") FLUSH;
    q_done = FALSE;
    while (!q_done) {
	printf("0) Exit.\n") FLUSH;
	printf("1) Add something to this file.\n");
	printf("   (newsgroups, sub-lists, files, and virtual groups.)\n");
	printf("2) Edit the current group scan file.\n");
	printf("3) Create a personalized top file for group scan mode.\n");
	printf("   (You only need to do this once.)\n");
	ch = menu_get_char();
	q_done = TRUE;
	switch (ch) {
	    case '0':
		return;
	    case '1':
		sg_ext_add();
		q_done = FALSE;		/* give the user another chance */
		return;
	    case '2':
		for (i=sg_num_contexts-1;i>=0;i--)
		    if (sg_contexts[i].filename)
			break;
		if (i<0) {
		    s_beep();
		    break;
		}
		sg_edit_file();
		/* consider handling return value later */
		(void)sg_reuse_file();
		s_refill = TRUE;
		s_ref_all = TRUE;
		break;
	    case '3':
		sg_ext_setup();
		return;
	    case 'h':
		printf("No help available (yet).\n") FLUSH;
		q_done = FALSE;
		break;
	    default:
		q_done = FALSE;
		break;
	}
    }
}

/* set up an initial group scan file. */
void
sg_ext_setup()
{
    static char lbuf[LBUFLEN];
    FILE *fp;
    bool has_default;

    /* try to open an old "top" file */
    strcpy(lbuf,getval("SCANGROUPS","%p/scangroups"));
    strcat(lbuf,"/top");
    /* the following line works because filexp uses a private buffer */
    strcpy(lbuf,filexp(lbuf));
    fp = fopen(lbuf,"r");
    if (fp) {	/* there was an old top file */
	fclose(fp);
	printf("\nYou have an old top file named:\n%s\n",lbuf) FLUSH;
	printf("If you wish to create a new top-level file, you will have\n");
	printf("to remove the old file.\n");
	menu_pause_key();
	return;
    }

    printf("Creating a new group scan top file.\n") FLUSH;
    fp = fopen(filexp("%X/default_topic"),"r");
    if (fp) {
	fclose(fp);
	printf("Found the system-wide default topic file.\n") FLUSH;
    } else {
	printf("System-wide default topic file not found.\n") FLUSH;
    }
    /* For now, pretend that all systems have the default_topic file.
     * Later, consider not adding the default_topic file if it was not found.
     */
    has_default = TRUE;
    
    makedir(lbuf,MD_FILE);
    fp = fopen(lbuf,"w");
    if (!fp) {
	printf("Could not open file for writing:\n%s\n",lbuf) FLUSH;
	menu_pause_key();
	return;
    }
    fprintf(fp,"### Strn group scan mode top file.\n");
    fprintf(fp,"\n\"All newsgroups in .newsrc\" *\n");
    if (has_default) {
	fprintf(fp,"\n\"System default file\" %%X/default_topic\n");
	fprintf(fp,"\n\"Groups by hierarchies\" %%X/hier_groups\n");
    }
    fprintf(fp,"\n\"Hotlist\" @%%p/hotlist.strn\n");
    fclose(fp);
    printf("Done.  Quit strn and restart to use the new top file.\n") FLUSH;
    menu_pause_key();
}

void
sg_ext_add()
{
    bool q_done;	/* if TRUE, we are finished with current question */
    char ch;


    q_done = FALSE;
    while (!q_done) {
	printf("\nAdd something to this file:\n");
	printf("0) Exit.\n") FLUSH;
	printf("1) Add one or more newsgroups to this file.\n");
	printf("2) Add a sub-list of newsgroups to this file.\n");
	printf("3) Add a virtual group to this file.\n");
	printf("4) Add a link to another file.\n");
	ch = menu_get_char();
	q_done = TRUE;
	switch (ch) {
	    case '0':
		return;
	    case '1':
		sg_ext_add_grp();
		break;
	    case '2':
		sg_ext_add_list();
		break;
	    case '3':
		sg_ext_add_vgr();
		break;
	    case '4':
		sg_ext_add_file();
		break;
	    case 'h':
		printf("No help available (yet).\n") FLUSH;
		q_done = FALSE;
		break;
	    default:
		q_done = FALSE;
		break;
	}
    }
    menu_pause_key();
}

void
sg_ext_add_write(str)
char *str;
{
    char *fname = Nullch;
    FILE *fp;
    int i;

    for (i=sg_num_contexts-1;i>=0;i--)
	if ((fname=sg_contexts[i].filename))	/* EQUALS */
	    break;
    if (i<0) {	/* no file found */
	printf("\nCould not find the current group scan file.\n") FLUSH;
	return;
    }
    fp = fopen(filexp(fname),"a");
    if (!fp) {
	printf("\nCould not open the following file (for writing):\n%s\n",
	       filexp(fname));
	return;
    }
    fprintf(fp,"%s\n",str);
    fclose(fp);
}

/* Make sure the destination file exists (can be opened for reading). */
/* returns TRUE if file opened successfully. */
bool
sg_ext_touch(str)
char *str;
{
    static char lbuf[LBUFLEN];
    int i;
    char *fname = Nullch;
    FILE *fp;
    char *s;
    char *kind;

    if (*str == '@') {	/* virtual group file */
	kind = "virtual group";
	str++;		/* skip the @ character */
    } else
	kind = "group scan file";

    switch (*str) {
	case ':':
	    for (i=sg_num_contexts-1;i>=0;i--)
		if ((fname=sg_contexts[i].filename))	/* EQUALS */
		    break;
	    if (i<0) {	/* no file found */
		printf("\nCould not find current group scan file.\n") FLUSH;
		return(FALSE);
	    }
	    strcpy(lbuf,fname);
	    s = rindex(lbuf,'/');
	    if (!s) {
		printf("\nCurrent file name has no '/'.\n") FLUSH;
		return(FALSE);
	    }
	    s++;			/* point to after slash */
	    strcpy(s,str+1);	/* copy final filename after slash */
	    fname = lbuf;
	    break;
	case '~':
	case '%':
	case '/':
	    fname = str;
	    break;
	default:
	    printf("Bad file name: %s\n",str);
	    return(FALSE);
    }
    fp = fopen(filexp(fname),"r");
    if (!fp) {
	makedir(filexp(fname),MD_FILE);
	fp = fopen(filexp(fname),"w");
	if (!fp) {
	    printf("\nCould not open (for writing) the file:\n%s\n",
		   filexp(fname));
	    return(FALSE);
	}
	fprintf(fp,"### Strn %s\n",kind);
	fclose(fp);
    } else {
	/* the file existed before */
	fclose(fp);
    }
    return(TRUE);
}

char *
sg_ext_get_desc()
{
    static char descbuf[LBUFLEN];
    char *s,*p;

    printf("\nEnter the description that will appear in group scan mode:\n");
    s = menu_get_text();
    if (!s || !*s)
	return(Nullch);
    p = descbuf;
    for (;*s;s++) {
	switch (*s) {
	    case '\"':
		*p++ = '\'';
		*p++ = '\'';
		break;
	    case '\t':
		*p++ = ' ';
		break;
	    default:
		*p++ = *s;
		break;
	}
    }
    *p = '\0';
    return(descbuf);
}

char *
sg_ext_get_groups()
{
    static char groupbuf[LBUFLEN];
    char *s;

    printf("\nEnter a list of newsgroups separated by commas or spaces:\n");
    printf("(The '*' character can be used as a shortcut, as in alt.*)\n");
    s = menu_get_text();
    if (!s || !*s)
	return(Nullch);
    strcpy(groupbuf,s);
    return(groupbuf);
}

char *
sg_ext_get_filename(is_vgr)
bool_int is_vgr;
{
    static char namebuf[LBUFLEN];
    char *s,*p;
    bool q_done;	/* if TRUE, we are finished with current question */
    char ch;

    s = namebuf;
    if (is_vgr)
	*s++ = '@';
    q_done = FALSE;
    while (!q_done) {
	printf("0) Exit.\n") FLUSH;
	printf("1) Enter a full path.\n");
	printf("2) Enter a file name relative to the current directory.\n");
	printf("h) Help.\n");
	ch = menu_get_char();
	q_done = TRUE;
	switch (ch) {
	    case '0':
		return(Nullch);
	    case '1':
		printf("Enter the full path name:\n") FLUSH;
		p = menu_get_text();
		if (!p || !*p) {
		    q_done = FALSE;
		    break;
		}
		strcpy(s,p);
		return(namebuf);
	    case '2':
		printf("Enter the file name or path:\n")FLUSH;
		p = menu_get_text();
		if (!p || !*p) {
		    q_done = FALSE;
		    break;
		}
		*s++ = ':';	/* add the ':' character */
		strcpy(s,p);
		return(namebuf);
	    case 'h':
		printf("Here are some samples of full file names:\n");
		printf("/u3/caadams/News/scangroups/foo_bar\n");
		printf("%%p/bar_ick\n");
		printf("~mary/News/virt/foo\n\n");
		printf("Names relative to the current directory are more\n");
		printf("portable.  They look like:\n");
		printf("foo_bar\n");
		printf("fantasy/top\n\n") FLUSH;
		q_done = FALSE;
		break;
	    default:
		q_done = FALSE;
		break;
	}
    }
    return(Nullch);
}

void
sg_ext_add_grp()
{
    char *groups;

    groups = sg_ext_get_groups();
    if (!groups)
	return;

    sg_ext_add_write(groups);
    sg_reuse_file();
}

void
sg_ext_add_list()
{
    static char lbuf[LBUFLEN];
    char *groups;
    char *desc;

    desc = sg_ext_get_desc();
    if (!desc)
	return;

    groups = sg_ext_get_groups();
    if (!groups)
	return;

    sprintf(lbuf,"\"%s\" %s",desc,groups);
    sg_ext_add_write(lbuf);
    sg_reuse_file();
}

void
sg_ext_add_vgr()
{
    static char lbuf[LBUFLEN];
    char *fname;
    char *desc;

    desc = sg_ext_get_desc();
    if (!desc)
	return;
    printf("\nEnter the filename of the virtual group:\n") FLUSH;
    fname = sg_ext_get_filename(TRUE);
    if (!fname)
	return;
    if (!sg_ext_touch(fname))
	return;
    sprintf(lbuf,"\"%s\" %s",desc,fname);
    sg_ext_add_write(lbuf);
    sg_reuse_file();
}

void
sg_ext_add_file()
{
    static char lbuf[LBUFLEN];
    char *fname;
    char *desc;

    desc = sg_ext_get_desc();
    if (!desc)
	return;
    printf("\nEnter the filename you wish to link to:\n") FLUSH;
    fname = sg_ext_get_filename(FALSE);
    if (!fname)
	return;
    if (!sg_ext_touch(fname))
	return;
    sprintf(lbuf,"\"%s\" %s",desc,fname);
    sg_ext_add_write(lbuf);
    sg_reuse_file();
}
#endif /* SCAN */
